<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Builder; // برای Type Hinting در Scopeها

class Contact extends Model
{
    // 🛡️ Mass Assignment Protection
    protected $fillable = [
        'name', 'phone_number', 'fax_number', 'email', 'address',
        'phone_type', 'is_public', 'created_by', 'group_id', 'tags',
    ];

    // 🔄 Type Casting
    protected $casts = [
        'is_public' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'tags' => 'array',
    ];

    // 💡 Appended Attributes (Accessors)
    protected $appends = [
        'avatar_color',
        'initials',
        'call_link',
        'phone_type_label',
        'jalali_created_at',
        'display_name',
    ];

    // --- 🔗 Relations ---

    public function group(): BelongsTo
    {
        return $this->belongsTo(ContactGroup::class, 'group_id');
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function callReminders(): HasMany
    {
        return $this->hasMany(CallReminder::class);
    }

    // --- 🪄 Accessors & Mutators ---

    // 📞 هوشمند: نمایش نام — اگر نام نبود، از شماره استفاده کن
    public function getDisplayNameAttribute(): string
    {
        return $this->name ?: ($this->phone_number ?: 'بدون نام');
    }

    // 🎨 تولید رنگ آواتار بر اساس نام
    public function getAvatarColorAttribute(): string
    {
        $colors = [
            '#6366F1', '#8B5CF6', '#EC4899', '#F97316',
            '#10B981', '#06B6D4', '#0EA5E9', '#3B82F6'
        ];
        // از display_name استفاده شود که همیشه مقدار دارد
        $hash = abs(crc32($this->display_name)) % count($colors);
        return $colors[$hash];
    }

    // 🅰️ تولید حروف اول نام
    public function getInitialsAttribute(): string
    {
        $name = $this->display_name;
        $words = array_filter(array_map('trim', explode(' ', $name)));

        // اگر حداقل دو کلمه وجود داشت
        if (count($words) >= 2) {
            // از اولین حرف دو کلمه اول استفاده کن
            return strtoupper(mb_substr($words[0], 0, 1) . mb_substr($words[1], 0, 1));
        }
        // در غیر این صورت، دو حرف اول را برگردان (مناسب برای نام‌های تک‌کلمه‌ای)
        return strtoupper(mb_substr($name, 0, 2));
    }

    // 🔗 لینک تماس (tel:)
    public function getCallLinkAttribute(): string
    {
        // فقط اعداد و علامت + را حفظ می‌کند
        $phone = preg_replace('/[^0-9\+]/', '', $this->phone_number);
        return 'tel:' . ltrim($phone, '+');
    }

    // 🏷️ برچسب نوع تلفن فارسی
    public function getPhoneTypeLabelAttribute(): string
    {
        $map = [
            'mobile'   => 'تلفن همراه',
            'work'     => 'کاری',
            'home'     => 'منزل',
            'internal' => 'داخلی',
            'fax'      => 'فکس',
        ];
        // اگر در نقشه نبود، عنوان‌بندی (Title Case) استاندارد انگلیسی را برگردان.
        return $map[$this->phone_type] ?? Str::title($this->phone_type ?? '');
    }

    // 📅 تاریخ شمسی ایجاد
    public function getJalaliCreatedAtAttribute(): string
    {
        if (class_exists(\Morilog\Jalali\Jalalian::class)) {
            // از تابع Static برای تاریخ شمسی استفاده می‌شود (نیاز به نصب پکیج)
            return \Morilog\Jalali\Jalalian::fromCarbon($this->created_at)->format('Y/m/d');
        }
        // Fallback به میلادی
        return $this->created_at->format('Y/m/d');
    }


    // --- 🔭 Query Scopes ---

    /**
     * Scope: مخاطبینی که برای کاربر مورد نظر قابل مشاهده هستند.
     * (مخاطبین ایجاد شده توسط کاربر یا مخاطبین عمومی)
     */
    public function scopeVisibleToUser(Builder $query, int $userId): Builder
    {
        return $query->where(function ($q) use ($userId) {
            $q->where('created_by', $userId)
                ->orWhere('is_public', true);
        });
    }

    /**
     * Scope: فیلتر کردن بر اساس گروه.
     */
    public function scopeInGroup(Builder $query, int $groupId): Builder
    {
        return $query->where('group_id', $groupId);
    }

    /**
     * Scope: جستجوی هوشمند در فیلدهای کلیدی.
     */
    public function scopeSearch(Builder $query, ?string $term): Builder
    {
        // 💡 از متغیرها یا فیلدهای مدل در کوئری اسکوپ استفاده نکنید (مانند $this->tags).
        // شما باید مستقیماً فیلد دیتابیس را بررسی کنید.

        if (empty($term)) {
            return $query;
        }

        // 1. حذف % و _ از رشته جستجو برای جلوگیری از حملات SQL Wildcard
        $safeTerm = str_replace(['%', '_'], ['\\%', '\\_'], $term);
        $likeTerm = "%{$safeTerm}%";

        return $query->where(function ($q) use ($likeTerm) {
            $q->where('name', 'like', $likeTerm)
                ->orWhere('phone_number', 'like', $likeTerm)
                ->orWhere('email', 'like', $likeTerm)
                ->orWhere('fax_number', 'like', $likeTerm);

            // اگر tags به صورت JSON/String ذخیره می‌شود، می‌توان آن را جستجو کرد.
            // اگر از MySQL >= 5.7 یا MariaDB >= 10.2 استفاده می‌کنید، استفاده از JSON_CONTAINS (برای جستجوی دقیق در آرایه‌ها) بهتر است.
            $q->orWhere('tags', 'like', $likeTerm);
        });
    }
}
