<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo; // ✅ جدید
use Illuminate\Database\Eloquent\Builder; // ✅ برای Type Hinting

class ContactGroup extends Model
{
    // 🛡️ Mass Assignment Protection
    protected $fillable = ['name', 'created_by', 'color'];

    // 🔄 Type Casting
    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // 💡 Appended Attributes (Accessors)
    protected $appends = [
        'safety_color',
        'contrast_color',
        'inline_color_style',
    ];

    // --- 🔗 Relations ---

    /**
     * @return HasMany مخاطبین عضو این گروه
     */
    public function contacts(): HasMany
    {
        return $this->hasMany(Contact::class, 'group_id');
    }

    /**
     * @return BelongsTo مالک (کاربر ایجاد کننده) گروه
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }


    // --- 🎨 Accessors برای رنگ و UI ---

    // ✅ رنگ ایمن — همیشه یک کد هگزای معتبر برمی‌گرداند.
    public function getSafetyColorAttribute(): string
    {
        // بررسی می‌کند آیا رنگ موجود و یک کد هگزا ۶ رقمی معتبر است یا خیر
        return $this->color && preg_match('/^#[0-9A-Fa-f]{6}$/', $this->color)
            ? $this->color
            : '#4361EE'; // رنگ دیفالت ثابت و معتبر
    }

    // ✅ رنگ متضاد برای متن روی پس‌زمینه (سیاه یا سفید)
    public function getContrastColorAttribute(): string
    {
        $hex = ltrim($this->safety_color, '#');
        // بررسی اطمینان از طول هگزا (اگرچه Safety Color آن را تضمین می‌کند)
        if (strlen($hex) !== 6) return '#000';

        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        // محاسبه YIQ برای تعیین روشنایی (Luminance)
        $yiq = (($r * 299) + ($g * 587) + ($b * 114)) / 1000;

        return $yiq >= 128 ? '#000' : '#fff'; // آستانه استاندارد ۱۲۸
    }

    // ✅ برای Livewire/UI: رنگ به شکل inline style
    public function getInlineColorStyleAttribute(): string
    {
        // استفاده از دو Accessor قبلی برای تولید استایل کامل
        return "background-color: {$this->safety_color}; color: {$this->contrast_color};";
    }

    // --- 🔭 Query Scopes ---

    /**
     * Scope: فیلتر کردن گروه‌ها بر اساس منطق دسترسی کاربر.
     * یک گروه قابل مشاهده است اگر: ۱. توسط کاربر ایجاد شده باشد یا
     * ۲. حداقل یک مخاطب قابل مشاهده (عمومی یا ایجاد شده توسط کاربر) داشته باشد.
     *
     * @param Builder $query
     * @param int $userId شناسه کاربر جاری
     * @return Builder
     */
    public function scopeVisibleToUser(Builder $query, int $userId): Builder
    {
        // 1. تعریف Scope مخاطبین قابل مشاهده از مدل Contact
        // این کار از تکرار منطق دسترسی جلوگیری می‌کند.
        $visibleContactsScope = function (Builder $q) use ($userId) {
            // فرض بر وجود Scope 'visibleToUser' در مدل Contact است
            $q->visibleToUser($userId);
        };

        // 2. اعمال فیلتر گروه
        return $query->where(function ($q) use ($userId, $visibleContactsScope) {
            // گروه‌های ایجاد شده توسط کاربر فعلی (مالکیت)
            $q->where('created_by', $userId)
                // یا گروه‌هایی که حداقل یک مخاطب قابل مشاهده دارند
                ->orWhereHas('contacts', $visibleContactsScope);
        });
    }
}
