<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne; // <<--- این باید وجود داشته باشد
use Illuminate\Support\Str;

class Letter extends Model
{
    protected $fillable = [
        'title',
        'code',
      //  'unique_code',
        'classification_id',
        'type',
        'user_id',
        'sender_id',
        'project_id',
        'content',
        'from_where',
        'from_who',
        'background_id',
        'status',
        'signed_at',
        'priority',
        'confidential',
        'due_date',
        'signature_position',

        // ✨ فیلدهای جدید برای نامه‌های وارده و ارجاع اولیه
        'external_sender',
        'external_number',
        'external_date',
        'recipient_user_id', // گیرنده داخلی/مسئول اولیه


    ];

    protected $casts = [
        'confidential' => 'boolean',
        'signed_at' => 'datetime',
    //    'due_date' => 'date',
        'due_date' => 'datetime',
        'external_date' => 'date', // ✨ Cast برای تاریخ خارجی
    ];

    const TYPE_INTERNAL = 'internal';
    const TYPE_EXTERNAL = 'external';
    const TYPE_INCOMING = 'incoming'; // ✨ تعریف ثابت TYPE_INCOMING

    const STATUS_DRAFT = 'draft';
    const STATUS_PENDING = 'pending';
    const STATUS_IN_PROGRESS = 'in_progress';

    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';
    const STATUS_ARCHIVED = 'archived';

    const PRIORITY_LOW = 'low';
    const PRIORITY_NORMAL = 'normal';
    const PRIORITY_HIGH = 'high';
    const PRIORITY_URGENT = 'urgent';

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($letter) {
            if (empty($letter->code)) {
                $letter->code = $letter->generateCode();
            }
            if (empty($letter->unique_code)) {
                $letter->unique_code = Str::uuid();
            }
        });

        static::updating(function ($letter) {
            if ($letter->isDirty('status')) {
                LetterStatusLog::create([
                    'letter_id'     => $letter->id,
                    'user_id'       => auth()->id(), // یا از جایی که مناسب است
                    'status_before' => $letter->getOriginal('status'),
                    'status_after'  => $letter->status,
                    'message'       => null, // یا بر اساس context
                ]);
            }
        });
    }


//    protected static function boot()
//    {
//        parent::boot();
//
//        static::creating(function ($letter) {
//            if (empty($letter->code)) {
//                $letter->code = $letter->generateCode();
//            }
//            if (empty($letter->unique_code)) {
//                $letter->unique_code = Str::uuid();
//            }
//        });
//    }

    // اسکوپ‌ها
    public function scopeInternal($query)
    {
        return $query->where('type', self::TYPE_INTERNAL);
    }

    public function scopeExternal($query)
    {
        return $query->where('type', self::TYPE_EXTERNAL);
    }

    public function scopeDraft($query)
    {
        return $query->where('status', self::STATUS_DRAFT);
    }

    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    public function scopeApproved($query)
    {
        return $query->where('status', self::STATUS_APPROVED);
    }

    public function scopeUrgent($query)
    {
        return $query->where('priority', self::PRIORITY_URGENT);
    }

    public function scopeConfidential($query)
    {
        return $query->where('confidential', true);
    }

    public function scopeRecent($query, $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    // روابط
    public function sender(): BelongsTo
    {
        return $this->belongsTo(User::class, 'sender_id')->withDefault([
            'name' => 'سیستم'
        ]);
    }
    // ✨ تعریف گیرنده اولیه/مسئول پیگیری (اختیاری)
    public function recipientUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'recipient_user_id')->withDefault();
    }


    public function receivers(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'letter_user', 'letter_id', 'user_id')
            ->withPivot(['is_read', 'read_at'])
            ->withTimestamps();
    }

    public function classification(): BelongsTo
    {
        return $this->belongsTo(Classification::class)->withDefault();
    }

    public function attachments(): HasMany
    {
        return $this->hasMany(Attachment::class)->orderBy('created_at');
    }

    public function referrals(): HasMany
    {
        return $this->hasMany(Referral::class)->orderBy('created_at');
    }

    public function creator()
    {
        // فرض بر این است که creator یک User است و کلید خارجی 'creator_id' است.
        // اگر مدل User شما در جای دیگری قرار دارد، مسیر آن را اصلاح کنید.
        return $this->belongsTo(User::class, 'creator_id');
    }


    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class)->withDefault();
    }

    public function views(): HasMany
    {
        return $this->hasMany(LetterView::class);
    }

    public function approvals(): HasMany
    {
        return $this->hasMany(LetterAproval::class)->orderBy('created_at');
    }

    public function events(): HasMany
    {
        return $this->hasMany(LetterEvent::class)->orderBy('created_at');
    }

    public function background(): BelongsTo
    {
        return $this->belongsTo(LetterForm::class, 'background_id')->withDefault();
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id')->withDefault([
            'name' => 'سیستم'
        ]);
    }

    public function copies(): HasMany
    {
        return $this->hasMany(LetterCopy::class);
    }

    // متدهای کمکی
    public function sign(): bool
    {
        if ($this->signed_at) {
            return false;
        }

        return $this->update([
            'unique_code' => Str::uuid(),
            'signed_at' => now(),
            'status' => self::STATUS_APPROVED
        ]);
    }

    public function unsign(): bool
    {
        if (!$this->signed_at) {
            return false;
        }

        return $this->update([
            'unique_code' => null,
            'signed_at' => null,
            'status' => self::STATUS_PENDING
        ]);
    }

    public function isSigned(): bool
    {
        return !is_null($this->signed_at);
    }

    public function isDraft(): bool
    {
        return $this->status === self::STATUS_DRAFT;
    }

    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    public function isApproved(): bool
    {
        return $this->status === self::STATUS_APPROVED;
    }

    public function statusLabel(): string
    {
        $labels = [
            self::STATUS_DRAFT => 'پیش‌نویس',
            self::STATUS_PENDING => 'در انتظار بررسی',
            self::STATUS_IN_PROGRESS => 'در دست اقدام', // فرض می‌کنیم این ثابت وجود دارد
            self::STATUS_APPROVED => 'تایید شده',
            self::STATUS_REJECTED => 'رد شده',
            self::STATUS_ARCHIVED => 'بایگانی شده'
        ];

        return $labels[$this->status] ?? 'نامشخص';
    }

    public function priorityLabel(): string
    {
        $labels = [
            self::PRIORITY_LOW => 'کم',
            self::PRIORITY_NORMAL => 'عادی',
            self::PRIORITY_HIGH => 'بالا',
            self::PRIORITY_URGENT => 'فوری'
        ];

        return $labels[$this->priority] ?? 'نامشخص';
    }

    public function priorityColor(): string
    {
        $colors = [
            self::PRIORITY_LOW => 'blue',
            self::PRIORITY_NORMAL => 'green',
            self::PRIORITY_HIGH => 'orange',
            self::PRIORITY_URGENT => 'red'
        ];

        return $colors[$this->priority] ?? 'gray';
    }

    private function generateCode(): string
    {
        $year = now()->year;
        $count = static::where('project_id', $this->project_id)->count() + 1;
        return "LTR/{$this->project->code}/{$year}/{$count}";
    }

    public function markAsRead(User $user): void
    {
        $this->receivers()->updateExistingPivot($user->id, [
            'is_read' => true,
            'read_at' => now()
        ]);
    }

    public function hasReceiver(User $user): bool
    {
        return $this->receivers()->where('user_id', $user->id)->exists();
    }

    public function getExcerpt($length = 100): string
    {
        $content = strip_tags($this->content);
        return Str::limit($content, $length);
    }
    public static function insertReferrals(array $referrals): void
    {
        $now = now();
        $data = array_map(function ($referral) use ($now) {
            return [
                'letter_id' => $referral->letter_id,
                'user_id' => $referral->user_id,
                'type' => $referral->type,
                'status' => $referral->status,
                'created_by' => $referral->created_by,
                'parent_id' => $referral->parent_id ?? null,
                'created_at' => $referral->created_at ?? $now,
                'updated_at' => $referral->updated_at ?? $now,
            ];
        }, $referrals);

        \DB::table('referrals')->insert($data);
    }

    public function getDueDateShamsiAttribute(): ?string
    {
        if ($this->due_date) {
            // باید مطمئن شوید کلاس Jalalian را Import کرده‌اید
            // use Morilog\Jalali\Jalalian;
            return \Morilog\Jalali\Jalalian::fromCarbon($this->due_date)->format('Y/m/d');
        }
        return null;
    }






//    public function unreadReferrals(): HasMany
//    {
//        return $this->hasMany(Referral::class)->where('status', 'pending');
//    }
    public function unreadReferrals(): HasMany
    {
        return $this->hasMany(Referral::class)
            ->where('status', Referral::STATUS_PENDING);
    }

    public function scopeWithUnreadReferrals($query)
    {
        return $query->whereHas('referrals', function ($q) {
            $q->where('status', Referral::STATUS_PENDING);
        });
    }


    /**
     * رابطه با آخرین رویداد
     */
    public function latestEvent(): \Illuminate\Database\Eloquent\Relations\HasOne
    {
        return $this->hasOne(LetterEvent::class)->latestOfMany();
    }

    /**
     * اسکوپ برای نامه‌های کاربر جاری
     */
    public function scopeForCurrentUser($query)
    {
        return $query->where('sender_id', auth()->id());
    }

    /**
     * اسکوپ برای نامه‌های دارای ارجاع خوانده نشده
     */
//    public function scopeWithUnreadReferrals($query)
//    {
//        return $query->whereHas('referrals', function ($q) {
//            $q->where('status', 'pending');
//        });
//    }

    /**
     * بررسی آیا نامه ارجاع خوانده نشده دارد
     */
    public function hasUnreadReferrals(): bool
    {
        return $this->unreadReferrals()->exists();
    }

    /**
     * تعداد ارجاعات خوانده نشده
     */
    public function getUnreadReferralsCountAttribute(): int
    {
        return $this->unreadReferrals()->count();
    }


    public function statusColor(): string
    {
        return match ($this->status) {
            self::STATUS_PENDING => 'bg-yellow-100 text-yellow-800',
            self::STATUS_IN_PROGRESS => 'bg-blue-100 text-blue-800',
            self::STATUS_APPROVED => 'bg-green-100 text-green-800',
            self::STATUS_ARCHIVED => 'bg-gray-100 text-gray-800',
            self::STATUS_REJECTED => 'bg-red-100 text-red-800',
            default => 'bg-gray-200 text-gray-700',
        };
    }


    public function archive(): bool
    {
        // بایگانی فقط برای وضعیت‌های نهایی (تایید، رد یا پیش‌نویس‌های قدیمی) مجاز است.
        if ($this->status === self::STATUS_APPROVED || $this->status === self::STATUS_REJECTED || $this->status === self::STATUS_DRAFT) {
            return $this->update(['status' => self::STATUS_ARCHIVED]);
        }
        // اگر در وضعیت Pending یا In Progress باشد، از بایگانی جلوگیری می‌کنیم.
        return false;
    }

    public function unarchive(): bool
    {
        // بازیابی به وضعیت قبلی
        // در اینجا فرض می‌کنیم به وضعیت 'approved' بازگردد یا اگر پیش‌نویس بوده به 'draft'
        $newStatus = ($this->getOriginal('status') === self::STATUS_DRAFT) ? self::STATUS_DRAFT : self::STATUS_APPROVED;

        return $this->update(['status' => $newStatus]);
    }

// اسکوپ جدید برای فیلتر کردن نامه‌های بایگانی نشده
    public function scopeNotArchived($query)
    {
        return $query->where('status', '!=', self::STATUS_ARCHIVED);
    }


    /**
     * درصد خوانده شدن توسط گیرندگان اصلی
     */
    public function getReadPercentageAttribute(): int
    {
        $totalReceivers = $this->receivers()->count();

        if ($totalReceivers === 0) {
            return 0;
        }

        $readCount = $this->receivers()->wherePivot('read_at', '!=', null)->count();

        return round(($readCount / $totalReceivers) * 100);
    }

    /**
     * تعداد گیرندگان خوانده شده
     */
    public function getReadCountAttribute(): string
    {
        $totalReceivers = $this->receivers()->count();
        $readCount = $this->receivers()->wherePivot('read_at', '!=', null)->count();

        return "{$readCount}/{$totalReceivers}";
    }

    /**
     * رابطه با آخرین ارجاع فعال (Pending) ثبت شده برای این نامه
     */
    public function latestActiveReferral(): HasOne
    {
        // از Referral::class مطمئن شوید که موجود است
        return $this->hasOne(Referral::class)
            ->where('status', Referral::STATUS_PENDING)
            ->latest('created_at');
    }

    /**
     * آخرین ارجاع ثبت شده (شامل تکمیل شده‌ها)
     */
    public function latestReferral(): HasOne
    {
        return $this->hasOne(Referral::class)->latestOfMany();
    }


}
