<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class MasterDocument extends Model
{
    protected $fillable = [
        'project_id', 'category_id', 'document_number', 'title', 'discipline_id', 'document_type',
         'last_revision', 'status', 'planned_issue_date', 'actual_issue_date',
        'description', 'person_hour', 'wf', 'progress_percent', 'approval_status', 'remark', 'weight',

        // 'revision',
    ];

    // وضعیت‌های مجاز
    const STATUSES = [
        'draft'       => 'در حال تهیه',
        'submitted'   => 'ارسال شده',
        'approved'    => 'تایید شده',
        'rejected'    => 'رد شده',
        'pending'     => 'در انتظار',
    ];

// Accessor برای نمایش فارسی
    public function getDisplayStatusAttribute()
    {
        return self::STATUSES[$this->status] ?? $this->status;
    }

    // ارتباط با پروژه
    public function project()
    {
        return $this->belongsTo(Project::class);
    }

    // ارتباط با دسته‌بندی
    public function category()
    {
        return $this->belongsTo(DocumentCategory::class, 'category_id');
    }

    // ارتباط با دیسپلین
    public function discipline()
    {
        return $this->belongsTo(Discipline::class, 'discipline_id');
    }

    // ارتباط با نسخه‌ها (Document Revision)
    public function revisions()
    {
        return $this->hasMany(DocumentRevision::class, 'master_document_id');
    }

    // ارتباط با فایل‌های مستندات
    public function documentFiles()
    {
        return $this->morphMany(DocumentFile::class, 'attachable');
    }

    // ارتباط با تاریخچه مستندات
    public function histories()
    {
        return $this->morphMany(DocumentHistory::class, 'documentable');
    }

    public function transmittalDocuments()
    {
        return $this->hasMany(\App\Models\TransmittalDocument::class, 'master_document_id');
    }



    public function transmittals()
    {
        return $this->hasManyThrough(
            \App\Models\Transmittal::class,
            \App\Models\TransmittalDocument::class,
            'master_document_id', // Foreign key on TransmittalDocument model
            'id', // Foreign key on Transmittal model
            'id', // Local key on MasterDocument model
            'master_document_id' // Local key on TransmittalDocument model
        );
    }
    public function getProgressPercentAttribute()
    {
        // گرفتن تمام POIهای منحصربه‌فرد از رویژن‌ها (به ترتیب تاریخ)
        $poiTypes = $this->revisions()
            ->whereIn('poi_type', array_keys(DocumentRevision::POI_TYPES))
            ->orderBy('date', 'asc')
            ->pluck('poi_type')
            ->unique()
            ->values();

        if ($poiTypes->isEmpty()) {
            return 0;
        }

        // جمع درصدها از تنظیمات پروژه
        $total = 0;
        foreach ($poiTypes as $poi) {
            $weight = $this->project->poiWeights()
                ->where('poi_type', $poi)
                ->value('weight_percent');
            $total += (float) ($weight ?? 0);
        }

        // سقف 100%
        return min(100, round($total, 2));
    }

}
