<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Morilog\Jalali\Jalalian;
use Illuminate\Database\Eloquent\Casts\Attribute; // ✅ این خط اضافه شد


class PaymentRequest extends Model
{
    protected $fillable = [
        'user_id',
        'purchase_request_id',
        'project_id',
        'recipient_name',
        'payee_id', // 💡 تکمیل شده
        'amount',
        'amount_gross', // 💡 تکمیل شده
        'tax_amount',   // 💡 تکمیل شده
        'purpose',
        'status',
        'due_date',
    ];

    protected $casts = [
        // 💡 اضافه کردن Casting برای پایداری
        'due_date' => 'date',
        'amount' => 'decimal:2',
        'amount_gross' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function project()
    {
        return $this->belongsTo(Project::class);
    }

    public function purchaseRequest()
    {
        return $this->belongsTo(PurchaseRequest::class);
    }

    public function approvals()
    {
        return $this->hasMany(PaymentRequestApproval::class);
    }
    public function payee(): BelongsTo // 💡 رابطه جدید
    {
        return $this->belongsTo(Payee::class);
    }

    public function attachments(): HasMany
    {
        return $this->hasMany(PaymentRequestAttachment::class);
    }
    protected function createdAtShamsi(): Attribute
    {
        return Attribute::get(fn () => $this->created_at ? Jalalian::fromCarbon($this->created_at)->format('Y/m/d') : null);
    }

    protected function dueDateShamsi(): Attribute
    {
        return Attribute::get(fn () => $this->due_date ? Jalalian::fromCarbon($this->due_date)->format('Y/m/d') : '—');
    }



    // متد مشابه PurchaseRequest برای تعیین تأییدکننده بعدی
    public function getNextApprover(): ?PaymentRequestApproval
    {
        return $this->approvals()
            ->whereNull('is_approved')
            ->orderBy('priority', 'asc')
            ->first();
    }

    // متد به‌روزرسانی وضعیت کلی
    public function updateStatus(): void
    {
        $approvals = $this->approvals;
        if ($approvals->where('is_approved', false)->isNotEmpty()) {
            $this->update(['status' => 'rejected']);
            return;
        }
        if ($approvals->whereNull('is_approved')->isEmpty()) {
            $this->update(['status' => 'approved']);
        } else {
            $this->update(['status' => 'pending']);
        }
    }
}
