<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes; // ۱. این خط را اضافه کنید

class PettyCash extends Model
{
    protected $table = 'petty_cash';
    use SoftDeletes; // ۲. این تریت را داخل کلاس اضافه کنید

    protected $fillable = [
        'user_id',
        'amount',
        'reason',
        'status',
        'details_json',
    ];

    protected $casts = [
        'details_json' => 'array',
        'amount' => 'decimal:2',
    ];

    // وضعیت‌ها ثابت هستند
    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';
    const STATUS_PAID = 'paid';

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function approvals(): HasMany
    {
        return $this->hasMany(PettyCashApproval::class, 'petty_cash_id');
    }

    // متد کمکی برای یافتن قانون مرتبط (بهینه‌سازی شده)
    protected function getApprovalRule(): ?PettyCashApprovalRule
    {
        return PettyCashApprovalRule::where('min_amount', '<=', $this->amount)
            ->where(function ($q) {
                $q->whereNull('max_amount')->orWhere('max_amount', '>=', $this->amount);
            })
            ->orderBy('min_amount', 'desc')
            ->first();
    }

    /**
     * تابع: دریافت تعداد تأییدیه‌های مورد نیاز (فراخوانی با پرانتز: $req->getRequiredApprovalsCount())
     */
    public function getRequiredApprovalsCount(): int
    {
        $rule = $this->getApprovalRule();
        return $rule ? $rule->required_approvals : 1;
    }

    /**
     * تابع: دریافت لیست کاربران تأیید کننده (فراخوانی با پرانتز: $req->getRequiredApprovers())
     */
    public function getRequiredApprovers(): \Illuminate\Support\Collection
    {
        $rule = $this->getApprovalRule();
        return $rule ? User::whereIn('id', $rule->approver_ids)->get() : collect();
    }

    /**
     * Accessor: تعداد تأییدیه‌های انجام شده (فراخوانی بدون پرانتز: $req->approved_count)
     */
    public function getApprovedCountAttribute(): int
    {
        return $this->approvals()->where('status', PettyCashApproval::STATUS_APPROVED)->count();
    }

    public function isFullyApproved(): bool
    {
        return $this->approved_count >= $this->getRequiredApprovalsCount();
    }

    public function updateStatusBasedOnApprovals()
    {
        if ($this->isFullyApproved()) {
            $this->update(['status' => self::STATUS_APPROVED]);
            return;
        }

        if ($this->approvals()->where('status', PettyCashApproval::STATUS_REJECTED)->exists()) {
            $this->update(['status' => self::STATUS_REJECTED]);
        }
    }

    public function isApprover($userId): bool
    {
        $approvers = $this->getRequiredApprovers()->pluck('id')->toArray();
        return in_array($userId, $approvers);
    }
    public function projects(): BelongsToMany
    {
        return $this->belongsToMany(Project::class, 'petty_cash_project');
    }
    public function statusLabel(): string
    {
        return match ($this->status) {
            self::STATUS_PENDING => 'در انتظار',
            self::STATUS_APPROVED => 'تأیید شده',
            self::STATUS_REJECTED => 'رد شده',
            self::STATUS_PAID => 'پرداخت شده',
            default => 'نامشخص',
        };
    }

    public function statusColor(): string
    {
        return match ($this->status) {
            self::STATUS_PENDING => 'warning',
            self::STATUS_APPROVED => 'primary',
            self::STATUS_REJECTED => 'danger',
            self::STATUS_PAID => 'success',
            default => 'secondary',
        };
    }


    public function getApprovalProgressAttribute()
    {
        $required = $this->getRequiredApprovalsCount();
        $current = $this->approved_count;
        return ($required > 0) ? round(($current / $required) * 100) : 0;
    }
}
