<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Post extends Model
{
    protected $fillable = [
        'author_id',
        'blog_category_id',
        'title',
        'content',
        'image',
        'media_file',
        'is_published',
        'slug',
    ];

    protected $casts = [
        'is_published' => 'boolean',
    ];

    /**
     * رابطه با نویسنده
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'author_id');
    }

    /**
     * رابطه با دسته‌بندی
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(BlogCategory::class, 'blog_category_id');
    }



    public function comments(): HasMany
    {
        return $this->hasMany(\App\Models\PostComment::class);
    }

    public function likes(): HasMany
    {
        return $this->hasMany(\App\Models\PostLike::class);
    }



    /**
     * اسکوپ برای پست‌های منتشر شده
     */
    public function scopePublished(Builder $query): Builder
    {
        return $query->where('is_published', true);
    }

    /**
     * اسکوپ برای پیش‌نویس‌ها
     */
    public function scopeDraft(Builder $query): Builder
    {
        return $query->where('is_published', false);
    }

    /**
     * اسکوپ جستجو روی عنوان
     */
    public function scopeSearch(Builder $query, ?string $term): Builder
    {
        if (!filled($term)) {
            return $query;
        }

        $term = trim($term);

        return $query->where('title', 'like', "%{$term}%");
    }

    /**
     * آدرس کامل تصویر شاخص (اگر ذخیره در storage/public)
     */
    public function getImageUrlAttribute(): ?string
    {
        if (!$this->image) {
            return null;
        }

        // اگر فقط اسم/مسیر داخل ستون image ذخیره شده:
        return asset('storage/' . ltrim($this->image, '/'));
    }

    /**
     * آدرس کامل فایل رسانه‌ای
     */
    public function getMediaUrlAttribute(): ?string
    {
        if (!$this->media_file) {
            return null;
        }

        return asset('storage/' . ltrim($this->media_file, '/'));
    }

    /**
     * نوع فایل رسانه (بر اساس پسوند، برای آیکن متفاوت)
     */
    public function getMediaExtensionAttribute(): ?string
    {
        if (!$this->media_file) {
            return null;
        }

        return strtolower(pathinfo($this->media_file, PATHINFO_EXTENSION));
    }

    /**
     * اگر خواستی روت‌ها با slug کار کنن
     */
    public function getRouteKeyName(): string
    {
        return 'id';
    }
}
