<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\SoftDeletes; // ۱. اضافه کردن این خط
use Illuminate\Database\Eloquent\Casts\Attribute; // برای استفاده از Accessor جدید

use Illuminate\Database\Eloquent\Relations\HasMany;


class Project extends Model
{
    use SoftDeletes; // ۲. استفاده از تریت در کلاس
    protected $fillable = ['name','project_type','code'];

    public function users()
    {
        return $this->belongsToMany(User::class);
     }

    public function sections()
    {
        return $this->hasMany(Section::class);
    }

    public function details()
    {
        return $this->hasone(ProjectDetail::class);
    }

    public function schedule()
    {
        return $this->hasone(ProjectSchedule::class);
    }
    public function projectfinancialdetail()
    {
        return $this->hasOne(projectFinancialDetail::class);

    }

    public function insurances()
    {
        return $this->hasMany(ProjectInsurance::class);

    }
    public function projectTypes()
    {
        return $this->belongsToMany(ProjectType::class, 'project_project_type');
    }
    public function tasks(): HasMany
    {
        return $this->hasMany(Task::class);
    }

    public function purchaseRequests()
    {
        return $this->hasMany(PurchaseRequest::class);
    }

    // ارتباط با مدارک پروژه (MDR)
    public function masterDocuments()
    {
        return $this->hasMany(MasterDocument::class);
    }

    // ارتباط با ترنسمیتال‌های پروژه
    public function transmittals()
    {
        return $this->hasMany(Transmittal::class);
    }


    public function members()
    {
        return $this->belongsToMany(User::class, 'user_roles', 'project_id', 'user_id')
            ->withPivot('role_id', 'status')
            ->withTimestamps();
    }

    public function parties() // برای شرکت‌ها (کارفرما، پیمانکار، مشاور)
    {
        return $this->belongsToMany(\App\Models\Company::class, 'project_parties', 'project_id', 'company_id')
            ->withPivot('party_role_type');
    }

    public function managementUsers() // برای مدیران کلیدی
    {
        return $this->belongsToMany(\App\Models\User::class, 'project_management_users', 'project_id', 'user_id')
            ->withPivot('management_role');
    }




    public function approvalRules(): HasMany
    {
        return $this->hasMany(ApprovalRule::class);
    }


    public function roles()
    {
        return $this->hasMany(ProjectRole::class);
        }

    public function missionRequests()
    {
        // فقط رکوردهای type=mission
        return $this->hasMany(LeaveMissionRequest::class)->where('type','mission');
    }



    public function contracts()
    {
        return $this->hasMany(ProjectContract::class);
    }

    public function mainContract()
    {
        return $this->hasOne(ProjectContract::class)->where('is_main', true);
    }

    public function poiWeights()
    {
        return $this->hasMany(ProjectPoiWeight::class);
    }
    public function scopeActive($query)
    {
        return $query->where('is_active', true); // یا where('status', 'active')
    }

    // 💡 Accessor: تعریف فیلد 'budget' به صورت مجازی
    // بودجه پروژه را برابر با مبلغ کل قرارداد اصلی (Contract Value + Addenda) در نظر می‌گیرد.
    protected function budget(): Attribute
    {
        return Attribute::make(
            get: function ($value, $attributes) {
                // تلاش برای استفاده از getTotalAmountAttribute از مدل ProjectContract
                $mainContract = $this->mainContract;

                if ($mainContract) {
                    // 💡 استفاده از Accessor در مدل ProjectContract
                    return $mainContract->getTotalAmountAttribute();
                }

                // اگر هیچ قرارداد اصلی پیدا نشد، به دنبال ProjectFinancialDetail بگرد
                $financialDetail = $this->details; // فرض می‌کنیم details به ProjectFinancialDetail برمی‌گردد
                if ($financialDetail) {
                    return $financialDetail->contract_value ?? 0;
                }

                return 0; // اگر هیچ داده‌ای نبود، صفر برمی‌گرداند
            }
        );
    }

    public function getTotalProgressPercentAttribute()
    {
        $docs = $this->masterDocuments;

        if ($docs->isEmpty()) {
            return 0;
        }

        $totalWeight = $docs->sum('weight');
        if ($totalWeight == 0) {
            return 0;
        }

        $weightedSum = $docs->sum(fn($doc) => ($doc->progress_percent / 100) * $doc->weight);

        return round(($weightedSum / $totalWeight) * 100, 2);
    }












    /**
     * رابطه با نامه‌های پروژه
     */
    public function letters(): HasMany
    {
        return $this->hasMany(Letter::class);
    }

    /**
     * رابطه با نامه‌های پیش‌نویس
     */
    public function draftLetters(): HasMany
    {
        return $this->hasMany(Letter::class)->where('status', Letter::STATUS_DRAFT);
    }

    /**
     * رابطه با نامه‌های تایید شده
     */
    public function approvedLetters(): HasMany
    {
        return $this->hasMany(Letter::class)->where('status', Letter::STATUS_APPROVED);
    }

    /**
     * رابطه با نامه‌های فوری
     */
    public function urgentLetters(): HasMany
    {
        return $this->hasMany(Letter::class)->where('priority', Letter::PRIORITY_URGENT);
    }




//    public function pettyCashes() {
//        return $this->hasMany(PettyCash::class);
//    }
    public function pettyCashes(): BelongsToMany
    {
        return $this->belongsToMany(PettyCash::class, 'petty_cash_project');
    }

    protected function totalPettyCashAmount(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->pettyCashes()->sum('amount') ?? 0
        );
    }
    // فایل Project.php
//    public function scopeActive($query)
//    {
//        // اگر نام ستون شما status است و مقدارش active
//        return $query->where('status', 'active');
//    }

//    protected function budget(): Attribute
//    {
//        return Attribute::make(
//            get: function () {
//                // ۱. اولویت با قرارداد اصلی
//                if ($this->mainContract) {
//                    return $this->mainContract->total_amount;
//                }
//
//                // ۲. اولویت دوم با جزئیات مالی
//                return $this->projectfinancialdetail->contract_value ?? 0;
//            }
//        );
//    }

// اضافه کردن این متد برای محاسبات مالی سریع‌تر
    public function getRemainingBudgetAttribute()
    {
        return $this->budget - $this->purchaseRequests()->where('status', 'paid')->sum('amount');
    }


    }


