<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Morilog\Jalali\Jalalian;

class ProjectFile extends Model
{
    protected $fillable = ['name', 'path', 'section_id', 'project_id', 'user_id', 'size', 'type','description','tags','download_count',  'folder_id', 'last_editor_id'];
    protected $casts = [
        'tags' => 'array',
    ];

    public function getUpdatedAtJalaliAttribute()
    {
        return Jalalian::fromDateTime($this->updated_at)->format('Y/m/d H:i');
    }

    public function getCreatedAtJalaliAttribute()
    {
        return Jalalian::fromDateTime($this->created_at)->format('Y/m/d H:i');
    }

    public function getTagsAttribute($value)
    {
        return $value ? json_decode($value, true) : [];
    }

    public function getFormattedSizeAttribute(): string
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = $this->size;

        for ($i = 0; $bytes > 1024; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, 2) . ' ' . $units[$i];
    }

    public function getIconAttribute(): string
    {
        return match(true) {
            str_contains($this->type, 'image/') => 'fa-file-image text-primary',
            str_contains($this->type, 'pdf') => 'fa-file-pdf text-danger',
            str_contains($this->type, 'word') => 'fa-file-word text-info',
            str_contains($this->type, 'zip') => 'fa-file-archive text-warning',
            default => 'fa-file text-secondary'
        };
    }

    // رابطه با پروژه
    public function project()
    {
        return $this->belongsTo(Project::class);
    }

    // رابطه با بخش
    public function section()
    {
        return $this->belongsTo(Section::class);
    }

    // رابطه با کاربر آپلودکننده (Original Uploader)
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    // رابطه با کاربر ویرایشگر (Last Editor)
    public function lastEditor()
    {
        return $this->belongsTo(User::class, 'last_editor_id');
    }

    // FIX: اضافه کردن رابطه با جدول فعالیت‌ها (Audit Log)
    public function activities()
    {
        return $this->hasMany(FileActivity::class, 'project_file_id');
    }

}
