<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;


class PurchaseRequest extends Model
{
    protected $fillable = [
        'user_id', 'project_id', 'description', 'status',
        'purchase_type',
        'cost_center_id',
        'required_by',
    ];
    protected $casts = [
        // 💡 اضافه کردن casts برای فیلدهای زمانی
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'required_by' => 'datetime', // اگر فیلد required_by را دارید

        // اگر این مدل یک Accessor totalPrice دارد، آن را حفظ کنید.
    ];
    /**
     * Accessor: محاسبه مبلغ کل درخواست به صورت خودکار.
     * این Accessor از رابطه items (که HasMany به PurchaseRequestItem است) استفاده می‌کند.
     * در View از $request->total_price استفاده کنید.
     */

    // App\Models\PurchaseRequest.php - Accessor ایمن‌تر
    protected function totalPrice(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->items->sum(fn($item) =>
                // 💡 ایمن‌سازی: اگر quantity یا price برابر null بود، مقدار آن را صفر در نظر بگیر
                ($item->quantity ?? 0) * ($item->price ?? 0)
            ),
        );
    }


    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    /**
     * رابطه HasMany به مدل واسط/آیتم‌ها (PurchaseRequestItem)
     */
    public function items(): HasMany
    {
        return $this->hasMany(PurchaseRequestItem::class, 'purchase_request_id');
    }

    public function approvals(): HasMany
    {
        return $this->hasMany(PurchaseRequestApproval::class);
    }

    public function costCenter(): BelongsTo
    {
        // نیاز به مدل CostCenter دارید
        return $this->belongsTo(CostCenter::class, 'cost_center_id');
    }
}
