<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Casts\Attribute; // برای Computed properties در مدل
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PurchaseRequestItem extends Model
{
    // 💡 نام جدول میانی شما در اینجا PurchaseRequestItem است و با PurchaseRequest لینک می‌شود.
    protected $table = 'purchase_request_items';

    protected $fillable = [
        'purchase_request_id',
        'item_id', // nullable برای خدمات آزاد
        'quantity',
        'price',
        'description', // برای شرح خدمات آزاد (Free Text)
    ];

    // 💡 تضمین می‌کند که مقادیر عددی به صورت float/int ذخیره و بازیابی شوند
    protected $casts = [
        'quantity' => 'integer',
        'price' => 'float',
    ];


    public function purchaseRequest(): BelongsTo
    {
        return $this->belongsTo(PurchaseRequest::class, 'purchase_request_id');
    }

    /**
     * ارتباط با مدل Item (ممکن است برای خدمات آزاد null باشد)
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class, 'item_id');
    }

    /**
     * 💡 Accessor: محاسبه جمع جزء (Subtotal) به صورت خودکار
     * در ویو می‌توانید مستقیماً از $item->subtotal استفاده کنید.
     */
    protected function subtotal(): Attribute
    {
        return Attribute::make(
            get: fn (mixed $value, array $attributes) =>
            round($attributes['quantity'] * $attributes['price'], 2),
        );
    }

    /**
     * 💡 Accessor: نمایش نام آیتم یا شرح خدمات آزاد
     * در ویو می‌توانید مستقیماً از $item->display_name استفاده کنید.
     */
    protected function displayName(): Attribute
    {
        return Attribute::make(
            get: fn (mixed $value, array $attributes) =>
                $this->item->name ?? $attributes['description'] ?? 'خدمت ناشناس',
        );
    }

    /**
     * 💡 متد کمکی: بررسی می‌کند که آیا این ردیف یک خدمت آزاد است یا خیر
     */
    public function isService(): bool
    {
        return is_null($this->item_id);
    }
}
