<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;
use Carbon\Carbon;

/**
 * @property int $id
 * @property int $letter_id
 * @property int $user_id
 * @property string $type
 * @property string $status
 * @property int $created_by
 * @property int|null $parent_id
 * @property string|null $correction_message
 * @property Carbon $created_at
 * @property Carbon $updated_at
 *
 * @method static Builder pending()
 * @method static Builder completed()
 * @method static Builder forAction()
 * @method static Builder forApproval()
 * @method static Builder createdBy($userId)
 */
class Referral extends Model
{
    // از ویژگی HasFactory در صورت استفاده در تست‌ها/Seedها استفاده کنید
    // use HasFactory;

    protected $fillable = [
        'letter_id',
        'user_id',
        'type',
        'created_by',
        'status',
        'parent_id',
        'correction_message',
    ];

    protected $casts = [
        'letter_id' => 'integer',
        'user_id' => 'integer',
        'created_by' => 'integer',
        'parent_id' => 'integer',
        'created_at' => 'datetime', // افزودن Type Casting
        'updated_at' => 'datetime',
    ];

    // ----------------------------------------------------
    // ✨ ثابت‌های پیشرفته (Constants)
    // ----------------------------------------------------

    // --- Type (نوع ارجاع) ---
    const TYPE_ACTION = 'action';           // اقدام و اجرا
    const TYPE_INFORMATION = 'information'; // صرفاً جهت اطلاع
    const TYPE_APPROVAL = 'approval';       // جهت تأیید/امضا
    const TYPE_CORRECTION = 'correction';   // برگشت جهت تصحیح
    const TYPE_CONSULTATION = 'consultation'; // ✨ جدید: برای مشورت/نظر فنی

    // --- Status (وضعیت ارجاع) ---
    const STATUS_PENDING = 'pending';       // در انتظار اقدام
    const STATUS_COMPLETED = 'completed';   // انجام شده (بسته شده)
    const STATUS_CANCELED = 'canceled';     // لغو شده (معمولاً توسط فرستنده)
    const STATUS_VIEWED = 'viewed';         // دیده شده، اما اقدام نشده
    const STATUS_REJECTED = 'rejected';     // ✨ جدید: رد شده (تایید نشده)
    const TYPE_OTHER = 'other'; // ✨ جدید: سایر اعمال/عملیات


    // ----------------------------------------------------
    // روابط (Relationships)
    // ----------------------------------------------------

    /**
     * رابطه با مدل Letter (نامه اصلی)
     */
    public function letter(): BelongsTo
    {
        return $this->belongsTo(Letter::class);
    }

    /**
     * رابطه با کاربر دریافت کننده این ارجاع (Recipient)
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * رابطه با کاربر ارجاع دهنده (Creator/Sender)
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * رابطه درختی: ارجاع والد (Parent)
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(self::class, 'parent_id');
    }

    /**
     * رابطه درختی: ارجاعات فرزند (Children)
     */
    public function children(): HasMany
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    /**
     * کامنت‌های مرتبط با این ارجاع
     * (فرض بر وجود مدل ReferralComment)
     */
    public function comments(): HasMany
    {
        return $this->hasMany(ReferralComment::class)
            ->whereNull('parent_id')
            ->with('user', 'replies')
            ->orderBy('created_at', 'asc'); // مرتب‌سازی صعودی برای خط زمانی
    }


    // ----------------------------------------------------
    // ✨ Scopeهای کوئری (Query Scopes)
    // ----------------------------------------------------

    /**
     * فقط ارجاعات در وضعیت "در انتظار"
     */
    public function scopePending(Builder $query): void
    {
        $query->where('status', self::STATUS_PENDING);
    }

    /**
     * ارجاعات مربوط به اقدام (Action) یا اجرا
     */
    public function scopeForAction(Builder $query): void
    {
        $query->where('type', self::TYPE_ACTION);
    }

    /**
     * ارجاعات مربوط به تایید (Approval) یا امضا
     */
    public function scopeForApproval(Builder $query): void
    {
        $query->where('type', self::TYPE_APPROVAL);
    }

    /**
     * ارجاعاتی که توسط یک کاربر خاص ایجاد شده‌اند
     */
    public function scopeCreatedBy(Builder $query, int $userId): void
    {
        $query->where('created_by', $userId);
    }


    // ----------------------------------------------------
    // متدهای کمکی (Helper Methods)
    // ----------------------------------------------------

    /**
     * برچسب خوانای نوع ارجاع
     */
    public function typeLabel(): string
    {
        $labels = [
            self::TYPE_ACTION => 'اقدام و اجرا',
            self::TYPE_INFORMATION => 'صرفاً جهت اطلاع',
            self::TYPE_APPROVAL => 'جهت تأیید',
            self::TYPE_CORRECTION => 'برگشت جهت تصحیح',
            self::TYPE_CONSULTATION => 'مشاوره/نظریه',
            self::TYPE_OTHER => 'سایر اعمال', // ✨ اضافه شدن لیبل
        ];

        return $labels[$this->type] ?? 'نامشخص';
    }

    /**
     * برچسب خوانای وضعیت ارجاع
     */
    public function statusLabel(): string
    {
        $labels = [
            self::STATUS_PENDING => 'در انتظار اقدام',
            self::STATUS_COMPLETED => 'انجام شده',
            self::STATUS_CANCELED => 'لغو شده',
            self::STATUS_VIEWED => 'دیده شده',
            self::STATUS_REJECTED => 'رد شده (عدم تأیید)'
        ];

        return $labels[$this->status] ?? 'نامشخص';
    }

    /**
     * متد کمکی برای چک کردن وضعیت بحرانی
     */
    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    /**
     * متد کمکی برای چک کردن نوع امضا
     */
    public function isApproval(): bool
    {
        return $this->type === self::TYPE_APPROVAL;
    }
}
