<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Morilog\Jalali\Jalalian;

class SalesInvoice extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'project_id',
        'invoice_number',
        'client_name',
        'issue_date',
        'due_date',
        'currency',
        'exchange_rate',
        'net_amount',
        'tax_amount',
        'total_amount',
        'total_foreign',
        'total_amount_irr',
        'description',
        'status',
    ];

    protected $casts = [
        'issue_date' => 'datetime',
        'due_date' => 'datetime',
        'exchange_rate' => 'float',
        'net_amount' => 'float',
        'tax_amount' => 'float',
        'total_amount' => 'float',
        'total_amount_irr' => 'float',
        'total_foreign' => 'float',
    ];

    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(SalesInvoiceItem::class, 'sales_invoice_id');
    }

    // Accessor: محاسبه مبلغ ریالی نهایی
    protected function totalIrr(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->total_amount * $this->exchange_rate,
        );
    }

    // 💡 FIX: Accessor برای مبلغ نهایی ارزی (که در دیتابیس total_foreign است)
    protected function totalAmount(): Attribute
    {
        // در ویو از $invoice->total_amount استفاده می‌شود، که باید مقدار total_foreign را برگرداند.
        return Attribute::make(
            get: fn () => $this->total_foreign,
        );
    }

    public function attachments(): HasMany
    {
        return $this->hasMany(SalesInvoiceAttachment::class, 'sales_invoice_id');
    }



    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    // Accessors برای تاریخ شمسی (که در ویو استفاده می‌شود)
    protected function createdAtShamsi(): Attribute
    {
        return Attribute::get(fn () => $this->created_at ? Jalalian::fromCarbon($this->created_at)->format('Y/m/d H:i') : '—');
    }

    protected function dueDateShamsi(): Attribute
    {
        return Attribute::get(fn () => $this->due_date ? Jalalian::fromCarbon($this->due_date)->format('Y/m/d') : '—');
    }

}
