<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Morilog\Jalali\Jalalian;

class Task extends Model
{
    protected $fillable = [
        'project_id', 'section_id', 'assigned_to', 'created_by',
        'title', 'description', 'status', 'priority', 'type',
        'start_date', 'due_date', 'parent_id', 'dependency_id',
        'progress_percent', 'attachment'
    ];

    // ✅ تغییر cast به 'date' اگر ستون‌ها از نوع DATE هستند
    // اگر ستون‌ها DATETIME هستند، 'datetime' بگذارید
    protected $casts = [
        'start_date' => 'date',  // ← توجه: 'date' نه 'datetime'
        'due_date'   => 'date',
        'progress_percent' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // --- Relations ---
    public function project()      { return $this->belongsTo(Project::class); }
    public function section()      { return $this->belongsTo(Section::class); }
    public function assignedUser() { return $this->belongsTo(User::class, 'assigned_to'); }
    public function creator()      { return $this->belongsTo(User::class, 'created_by'); }
    public function parent()       { return $this->belongsTo(Task::class, 'parent_id'); }
    public function children()     { return $this->hasMany(Task::class, 'parent_id'); }
    public function dependency()   { return $this->belongsTo(Task::class, 'dependency_id'); }
    public function dependents()   { return $this->hasMany(Task::class, 'dependency_id'); }

    // --- Accessors (برای نمایش شمسی) ---
    public function startDateJalali(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->start_date
                ? Jalalian::fromCarbon($this->start_date)->format('Y/m/d')
                : null
        );
    }

    public function dueDateJalali(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->due_date
                ? Jalalian::fromCarbon($this->due_date)->format('Y/m/d')
                : null
        );
    }

    // --- Helper Attributes ---
    public function getStatusLabelAttribute()
    {
        return match ($this->status) {
            'new' => 'جدید',
            'in_progress' => 'در حال انجام',
            'done' => 'انجام‌شده',
            'on_hold' => 'متوقف‌شده',
            default => $this->status,
        };
    }

    public function getProgressBadgeAttribute()
    {
        $p = (int) $this->progress_percent;
        $color = match (true) {
            $p === 100 => 'success',
            $p > 50     => 'primary',
            $p > 0      => 'warning',
            default     => 'secondary',
        };
        return "<span class=\"badge bg-{$color} fs-7\">{$p}%</span>";
    }
}
