<?php


namespace App\Models;


use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Collection;


class User extends Authenticatable
{
    use HasFactory, Notifiable;

    protected $fillable = [
        'name',
        'ncode',
        'email',
        'password',
        'mobile',
        'role',
        'degree',
        'image',
        'sign',
        'gender',
        'status',
        'is_active',
        'last_login_at',
        'last_activity_request_at',
        'company_id',
        'department_id',
        'internal_number',

    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'last_login_at' => 'datetime',
        'is_active' => 'boolean',
            'status' => 'string',
         'role' =>'string',
        'internal_number' => 'string',
        'last_activation_request_at' => 'datetime', // اضافه شده

    ];

    const STATUS_PENDING = 'pending'; // وضعیت تایید نشده
    const STATUS_ACTIVE = 'active';  // وضعیت فعال


    // نقش‌های کاربر
    const ROLE_ADMIN = 'admin';
    const ROLE_MANAGER = 'manager';
    const ROLE_USER = 'user';
    const ROLE_GUEST = 'guest';
    const ROLE_EDITOR = 'editor';


    protected static function boot()
    {
        parent::boot();

        // 💡 ایونت 'created' پس از ذخیره‌سازی کاربر در دیتابیس اجرا می‌شود.
        static::created(function ($user) {

            // ۱. بررسی تعداد کل کاربران
            // اگر تعداد کل کاربران (شامل همین کاربر تازه ایجاد شده) برابر ۱ باشد، یعنی اولین کاربر است.
            if (User::count() === 1) {

                // ۲. پیدا کردن نقش ادمین
                // فرض می‌کنیم نقش ادمین در جدول roles، نام 'admin' دارد.
                $adminRole = Role::where('name', self::ROLE_ADMIN)->first();

                if ($adminRole) {
                    // ۳. تخصیص نقش ادمین

                    // از آنجایی که شما از فیلد role_id استفاده می‌کنید، آن را به روزرسانی می‌کنیم.
                    // 🛑 به روزرسانی مستقیم فیلد role_id
                    $user->update(['role_id' => $adminRole->id]);

                    // 💡 اگر هنوز از فیلد 'role' استرینگ قدیمی استفاده می‌کنید:
                    // $user->update(['role' => self::ROLE_ADMIN]);
                }
            }
        });
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function department()
    {
        return $this->belongsTo(Department::class);
    }



    /**
     * رابطه با پروژه‌ها
     */
    public function projects(): BelongsToMany
    {
        return $this->belongsToMany(Project::class, 'user_roles', 'user_id', 'project_id')
            ->withPivot(['role_id', 'status'])
            ->withTimestamps();
    }

    /**
     * رابطه با بخش‌ها
     */
    public function sections(): HasMany
    {
        return $this->hasMany(Section::class);
    }

    /**
     * پروژه‌های مدیریت شده
     */
    public function managedProjects(): HasMany
    {
        return $this->hasMany(ProjectDetail::class, 'manager_id');
    }

    /**
     * ترنسمیتال‌های ارسالی
     */
    public function sentTransmittals(): HasMany
    {
        return $this->hasMany(Transmittal::class, 'sender_id');
    }

    /**
     * ترنسمیتال‌های دریافتی
     */
    public function receivedTransmittals(): HasMany
    {
        return $this->hasMany(Transmittal::class, 'receiver_id');
    }

    /**
     * فایل‌های آپلود شده
     */
    public function documentFiles(): HasMany
    {
        return $this->hasMany(DocumentFile::class, 'uploaded_by');
    }

    /**
     * تاریخچه تغییرات
     */
    public function documentHistories(): HasMany
    {
        return $this->hasMany(DocumentHistory::class, 'user_id');
    }

    /**
     * تسک‌های واگذار شده
     */
    public function assignedTasks(): HasMany
    {
        return $this->hasMany(Task::class, 'assigned_to');
    }

    /**
     * تسک‌های ایجاد شده
     */
    public function createdTasks(): HasMany
    {
        return $this->hasMany(Task::class, 'created_by');
    }

    /**
     * پیام‌های اطلاع‌رسانی دریافتی
     */
    public function receivedAnnouncements()
    {
        return $this->belongsToMany(Announcement::class, 'announcement_recipients')
            ->withPivot('read_at')
            ->withTimestamps();
    }

    /**
     * پیام‌های اطلاع‌رسانی ارسالی
     */
    public function sentAnnouncements()
    {
        return $this->hasMany(Announcement::class, 'sender_id');
    }

    /**
     * پست‌های ایجاد شده توسط کاربر
     */
    public function posts()
    {
        return $this->hasMany(Post::class, 'author_id');
    }

    /**
     * نامه‌های دریافتی
     */
    public function receivedLetters(): BelongsToMany
    {
        return $this->belongsToMany(Letter::class, 'letter_user', 'user_id', 'letter_id')
            ->withPivot(['read_at', 'received_at', 'archived_at'])
            ->withTimestamps();
    }

    /**
     * موقعیت‌های شغلی
     * موقعیت‌های شغلی
     */
    public function positions(): HasMany
    {
        return $this->hasMany(UserPosition::class);
    }

    /**
     * نامه‌های ارسالی
     */
    public function sentLetters(): HasMany
    {
        return $this->hasMany(Letter::class, 'sender_id');
    }

    /**
     * نامه‌های ایجاد شده
     */
    public function createdLetters(): HasMany
    {
        return $this->hasMany(Letter::class, 'user_id');
    }

    /**
     * بررسی آیا کاربر مدیر است
     */
//    public function isAdmin(): bool
//    {
//        // 🛑 به جای خواندن $this->role، باید از طریق رابطه roleObject() نام نقش را چک کنیم.
//        // فرض می‌کنیم نقش ادمین در جدول roles، نام 'admin' دارد.
//        return $this->roleObject?->name === self::ROLE_ADMIN;
//
////        return $this->role === self::ROLE_ADMIN;
//    }

    public function isAdmin(): bool
    {
        // چک کردن رابطه (اگر از جدول roles استفاده می‌کنید)
        if ($this->roleObject && $this->roleObject->name === self::ROLE_ADMIN) {
            return true;
        }

        // چک کردن فیلد استرینگ (به عنوان لایه پشتیبان)
        return $this->role === self::ROLE_ADMIN;
    }

// App\Models\User.php
    public function roleObject() // تغییر نام برای جلوگیری از تداخل با فیلد role فعلی
    {
        return $this->belongsTo(Role::class, 'role_id');
    }

    /**
     * بررسی آیا کاربر مدیر پروژه است
     */
    public function isProjectManager(): bool
    {
        return $this->managedProjects()->exists();
    }

    /**
     * دریافت تمام پروژه‌های قابل دسترسی
     */
//    public function accessibleProjects(): Collection
//    {
//        if ($this->isAdmin()) {
//            return Project::all();
//        }
//
//        return $this->projects()->get()->merge($this->managedProjects()->get());
//    }
    public function canAccessProject($projectId): bool
    {
        // قانون اول: ادمین به همه جا دسترسی دارد
        if ($this->isAdmin()) {
            return true;
        }

        // قانون دوم: بررسی عضویت در پروژه خاص
        return $this->projects()->where('project_id', $projectId)->exists() ||
            $this->managedProjects()->where('project_id', $projectId)->exists();
    }

    public static function getStatuses(): array
    {
        return [
            self::STATUS_ACTIVE => 'فعال',
            self::STATUS_PENDING => 'در انتظار تایید',
        ];
    }

    /**
     * بررسی دسترسی کاربر
     */
    public function hasPermission(string $permission): bool
    {
        $permissions = config('permissions.roles.' . $this->role, []);
        return in_array($permission, $permissions);
    }

    /**
     * فعال کردن حساب کاربری
     */
    public function activate(): void
    {
        $this->update(['is_active' => true]);
    }


    /**
     * غیرفعال کردن حساب کاربری
     */
    public function deactivate(): void
    {
        $this->update(['is_active' => false]);
    }

    /**
     * به‌روزرسانی زمان آخرین ورود
     */
    public function updateLastLogin(): void
    {
        $this->update(['last_login_at' => now()]);
    }

    // در مدل User
    public function projectRoles()
    {
        return $this->belongsToMany(ProjectRole::class, 'user_roles', 'user_id', 'role_id')
            ->withPivot('project_id', 'status')
            ->withTimestamps();
    }

    public function purchaseRequests(): HasMany
    {
        return $this->hasMany(PurchaseRequest::class, 'user_id');
    }

    public function contacts(): HasMany
    {
        return $this->hasMany(Contact::class, 'created_by');
}
}
