<?php

namespace App\Notifications;

use App\Models\Letter;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\BroadcastMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;

class LetterReferralSent extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public int $letterId,
        public int $senderId,
        public string $purpose
    ) {}

    public function via($notifiable): array
    {
        //database برای ذخیره در جدول notifications و broadcast برای ارسال لحظه‌ای Reverb
        return ['database', 'broadcast'];
    }

    public function toDatabase($notifiable): array
    {
        // بارگذاری مدل‌ها با قفل محافظتی برای جلوگیری از خطای Null
        $letter = Letter::find($this->letterId);
        $sender = User::find($this->senderId);

        if (!$letter || !$sender) {
            Log::error("Notification Failed: Letter or Sender not found.", ['id' => $this->letterId]);
            return [];
        }

        return [
            'letter_id' => $letter->id,
            'letter_title' => $letter->title,
            'sender_id' => $sender->id,
            'sender_name' => $sender->name,
            'purpose' => $this->purpose,
            'message' => "«{$sender->name}» نامه‌ای جهت «{$this->purpose}» برای شما ارسال کرد.",
            'icon' => 'ti ti-mail-forward',
            'color' => 'primary',
            // 💡 نکته: مطمئن شوید این روت در php artisan route:list وجود دارد
            'url' => route('letter.show', $letter->id),
        ];
    }

    public function toBroadcast($notifiable): BroadcastMessage
    {
        $letter = Letter::find($this->letterId);
        $sender = User::find($this->senderId);

        // اگر اطلاعات نبود، یک پیام خالی بفرست تا جاب FAIL نشود
        if (!$letter || !$sender) {
            return new BroadcastMessage(['error' => 'Data not found']);
        }

        return new BroadcastMessage([
            'type' => 'referral',
            'title' => 'ارجاع جدید',
            'message' => "«{$sender->name}» نامه‌ای جهت «{$this->purpose}» برای شما ارسال کرد.",
            'letter_id' => $letter->id,
            'sender_name' => $sender->name,
            'url' => route('letter.show', $letter->id),
        ]);
    }

    // ✨ نکته طلایی برای رفع FAIL: اگر جاب اول پیدا نکرد، ۵ ثانیه صبر کند و دوباره امتحان کند
    public $tries = 3;
    public $backoff = 5;
}
