<?php

namespace App\Notifications;

use App\Models\PettyCash;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;

class PettyCashStatusUpdated extends Notification
{
    use Queueable;

    /**
     * ایجاد نوتیفیکیشن با دریافت درخواست هزینه جزء
     */
    public function __construct(public PettyCash $pettyCash)
    {
        //
    }

    /**
     * کانال‌های ارسال اعلان: دیتابیس + ایمیل (اختیاری)
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail']; // اگر فقط دیتابیس می‌خواهی، 'mail' را حذف کن
    }

    /**
     * فرمت اعلان برای ذخیره در دیتابیس
     */
    public function toDatabase(object $notifiable): array
    {
        $statusLabels = [
            PettyCash::STATUS_APPROVED => 'تأیید شد',
            PettyCash::STATUS_REJECTED => 'رد شد',
            PettyCash::STATUS_PAID => 'پرداخت شد',
        ];

        $statusText = $statusLabels[$this->pettyCash->status] ?? 'به‌روزرسانی شد';

        return [
            'petty_cash_id' => $this->pettyCash->id,
            'amount' => $this->pettyCash->amount,
            'status' => $this->pettyCash->status,
            'message' => "درخواست هزینه جزء شما (مبلغ: " . number_format($this->pettyCash->amount) . " تومان) {$statusText}.",
            'url' => route('admin.petty-cash.show', $this->pettyCash->id),
            'icon' => $this->pettyCash->status === PettyCash::STATUS_APPROVED ? 'check-circle' : 'times-circle',
            'timestamp' => now()->toISOString(),
        ];
    }

    /**
     * فرمت ایمیل (اختیاری — اگر از ایمیل استفاده نمی‌کنی، می‌توانی حذفش کنی)
     */
    public function toMail(object $notifiable): MailMessage
    {
        $statusLabels = [
            PettyCash::STATUS_APPROVED => 'تأیید شده',
            PettyCash::STATUS_REJECTED => 'رد شده',
            PettyCash::STATUS_PAID => 'پرداخت شده',
        ];

        $statusText = $statusLabels[$this->pettyCash->status] ?? 'به‌روزرسانی شده';

        return (new MailMessage)
            ->subject("وضعیت درخواست هزینه جزء #{$this->pettyCash->id} {$statusText}")
            ->greeting('سلام ' . $notifiable->name)
            ->line("درخواست هزینه جزء شما با مبلغ **" . number_format($this->pettyCash->amount) . " تومان** {$statusText} شد.")
            ->line('دلیل: ' . $this->pettyCash->reason)
            ->action('مشاهده جزئیات', route('admin.petty-cash.show', $this->pettyCash->id))
            ->line('با تشکر از شما!');
    }

    /**
     * نمایش در API یا لاگ (اختیاری)
     */
    public function toArray(object $notifiable): array
    {
        return $this->toDatabase($notifiable);
    }
}
