<?php

namespace App\Notifications;

use App\Models\Letter;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\BroadcastMessage;
use Illuminate\Notifications\Notification;

class ReferralActionTaken extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public int $letterId,
        public int $senderId, // User ID who performed the action (Auth::id())
        public string $purpose // The action type: 'completed' or 'correction'
    ) {}

    public function via($notifiable): array
    {
        return ['database', 'broadcast'];
    }

    // ✨ تابع کمکی برای ترجمه هدف (Purpose)
    protected function getTranslatedAction(string $purpose): string
    {
        return match ($purpose) {
            'completed' => 'تکمیل شده است.',
            'correction' => 'جهت تصحیح بازگشت داده شده است.',
            default => 'وضعیت آن به‌روزرسانی شده است.',
        };
    }

    public function toDatabase($notifiable)
    {
        $letter = Letter::find($this->letterId);
        $sender = User::find($this->senderId);
        $translatedAction = $this->getTranslatedAction($this->purpose);

        if (!$letter || !$sender) {
            \Log::warning('Letter or Sender model not found during toDatabase serialization.', ['letter_id' => $this->letterId, 'sender_id' => $this->senderId]);
            return [];
        }

        return [
            'letter_id' => $letter->id,
            'letter_title' => $letter->title,
            'sender_id' => $sender->id,
            'sender_name' => $sender->name,
            'purpose' => $this->purpose,
            // ✨ محتوای پیام دیتابیس (خوانا برای جدول نوتیفیکیشن)
            'message' => "نامه «{$letter->title}» توسط «{$sender->name}» به وضعیت «{$translatedAction}»",
            'icon' => match($this->purpose) {
                'completed' => 'ti ti-check',
                'correction' => 'ti ti-refresh',
                default => 'ti-mail-forward',
            },
            'color' => match($this->purpose) {
                'completed' => 'success',
                'correction' => 'warning',
                default => 'primary',
            },
            'url' => route('letter.show', $letter->id), // فرض می‌کنیم route('letter.show') مسیر عمومی را می‌دهد
        ];
    }

    public function toBroadcast($notifiable)
    {
        $letter = Letter::find($this->letterId);
        $sender = User::find($this->senderId);
        $translatedAction = $this->getTranslatedAction($this->purpose);

        if (!$letter || !$sender) {
            \Log::warning('Letter or Sender model not found during toBroadcast.', ['letter_id' => $this->letterId, 'sender_id' => $this->senderId]);

            return new BroadcastMessage([
                'type' => 'error',
                'title' => 'خطا در ارجاع',
                'message' => 'اطلاعات نامه‌ی اقدام شده یافت نشد.',
            ]);
        }

        return new BroadcastMessage([
            'type' => 'referral_action', // ✨ تغییر نوع برای تمایز از ارجاع جدید
            'title' => 'وضعیت ارجاع به‌روزرسانی شد',
            'message' => "نامه «{$letter->title}» توسط «{$sender->name}» به وضعیت «{$translatedAction}»",
            'icon' => match($this->purpose) {
                'completed' => 'ti-check',
                'correction' => 'ti-refresh',
                default => 'ti-mail-forward',
            },
            'color' => match($this->purpose) {
                'completed' => 'success',
                'correction' => 'warning',
                default => 'primary',
            },
            'time' => now()->diffForHumans(),
            'url' => route('letter.show', $letter->id),
        ]);
    }
}
