<?php

namespace App\Policies;

use App\Models\Project;
use App\Models\User;
use Illuminate\Auth\Access\Response;
use Illuminate\Database\Eloquent\Builder;

class ProjectPolicy
{
    /**
     * قبل از اجرای هر متدی در Policy، اگر کاربر ادمین باشد، دسترسی کامل بده.
     */
    public function before(User $user, string $ability): bool|null
    {
        // با فرض اینکه متد isAdmin() در مدل User تعریف شده است.
        if ($user->isAdmin()) {
            return true;
        }

        return null;
    }

    /**
     * تعیین اینکه آیا کاربر می‌تواند لیستی از پروژه‌ها را ببیند.
     * (عملاً لیست پروژه‌هایی که به آن‌ها دسترسی دارد.)
     *
     * @param User $user
     * @return bool
     */
    public function viewAny(User $user): bool
    {
        // در این حالت، هر کاربر وارد شده‌ای می‌تواند لیست پروژه‌های خود را ببیند.
        // ادمین قبلاً توسط متد before() تأیید شده است.
        return true;
    }

    /**
     * تعیین اینکه آیا کاربر می‌تواند یک پروژه خاص را ببیند.
     * دسترسی فقط برای ادمین‌ها یا اعضای پروژه.
     *
     * @param User $user
     * @param Project $project
     * @return Response|bool
     */
    public function view(User $user, Project $project): Response|bool
    {
        // ادمین قبلاً توسط متد before() تأیید شده است.

        // بررسی اینکه آیا کاربر عضو فعال پروژه است.
        $isMember = $project->members()
            ->where('user_id', $user->id)
            // 💡 اگر نیاز به چک کردن وضعیت عضویت (status) هم هست، اینجا اضافه کنید:
            // ->wherePivot('status', 'active')
            ->exists();

        return $isMember
            ? true
            : Response::deny('شما عضو این پروژه نیستید و اجازه مشاهده ندارید.');
    }

    /**
     * تعیین اینکه آیا کاربر می‌تواند یک پروژه جدید ایجاد کند.
     *
     * @param User $user
     * @return bool
     */
    public function create(User $user): bool
    {
        // ادمین قبلاً توسط متد before() تأیید شده است.
        // فقط کاربرانی با نقش‌های مدیریتی یا خاص اجازه ایجاد پروژه را دارند.
        // با فرض اینکه نقش 'manager' می‌تواند ایجاد کند:
        return $user->role === User::ROLE_MANAGER;
        // یا بر اساس سیستم مجوزهای شما:
        // return $user->hasPermission('create_project');
    }

    /**
     * تعیین اینکه آیا کاربر می‌تواند یک پروژه خاص را ویرایش کند.
     * دسترسی فقط برای ادمین‌ها یا مدیران پروژه.
     *
     * @param User $user
     * @param Project $project
     * @return Response|bool
     */
    public function update(User $user, Project $project): Response|bool
    {
        // ادمین قبلاً توسط متد before() تأیید شده است.

        // 1. بررسی آیا کاربر مدیر پروژه است (فرض بر استفاده از ProjectDetail)
        $isProjectManager = $project->details->manager_id === $user->id;

        // 2. بررسی آیا کاربر یک نقش مدیریتی (مثلاً نقش "مدیر پروژه" در project_roles) دارد.
        $hasManagementRole = $project->members()
            ->where('user_id', $user->id)
            ->whereHas('roleObject', function (Builder $query) {
                $query->where('name', 'project_manager'); // نام نقش مدیریتی پروژه
            })
            ->exists();

        // اجازه ویرایش به مدیر رسمی پروژه یا کاربر دارای نقش مدیریتی در پروژه داده شود.
        if ($isProjectManager || $hasManagementRole) {
            return true;
        }

        return Response::deny('شما اجازه ویرایش این پروژه را ندارید.');
    }

    /**
     * تعیین اینکه آیا کاربر می‌تواند یک پروژه را حذف کند.
     *
     * @param User $user
     * @param Project $project
     * @return Response|bool
     */
    public function delete(User $user, Project $project): Response|bool
    {
        // ادمین قبلاً توسط متد before() تأیید شده است.
        // اجازه حذف فقط به ادمین‌های اصلی یا نقش‌های بسیار ارشد.
        return Response::deny('حذف پروژه ممنوع است.');
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Project $project): bool
    {
        // معمولاً فقط برای ادمین‌ها یا کاربران دارای مجوز خاص.
        return false;
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Project $project): bool
    {
        // معمولاً فقط برای ادمین‌ها (که توسط before چک شد)
        return false;
    }
}
