/**
 * @license Copyright (c) 2003-2025, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-licensing-options
 */
import { logWarning, EmitterMixin, CKEditorError, compareArrays, toArray, toMap, isIterable, ObservableMixin, count, EventInfo, Collection, keyCodes, isText, env, remove as remove$1, insertAt, diff, fastDiff, isNode, isComment, indexOf, global, isValidAttributeName, first, getAncestors, DomEmitterMixin, getCode, isArrowKeyCode, scrollViewportToShowTarget, uid, spliceArray, priorities, isInsideSurrogatePair, isInsideCombinedSymbol, isInsideEmojiSequence } from '@ckeditor/ckeditor5-utils/dist/index.js';
import { isObject, get, merge, set, isPlainObject, extend, debounce, isEqualWith, cloneDeep, isEqual, clone } from 'es-toolkit/compat';

// Each document stores information about its placeholder elements and check functions.
const documentPlaceholders = new WeakMap();
let hasDisplayedPlaceholderDeprecationWarning = false;
/**
 * A helper that enables a placeholder on the provided view element (also updates its visibility).
 * The placeholder is a CSS pseudo–element (with a text content) attached to the element.
 *
 * To change the placeholder text, change value of the `placeholder` property in the provided `element`.
 *
 * To disable the placeholder, use {@link module:engine/view/placeholder~disableViewPlaceholder `disableViewPlaceholder()`} helper.
 *
 * @param options Configuration options of the placeholder.
 * @param options.view Editing view instance.
 * @param options.element Element that will gain a placeholder. See `options.isDirectHost` to learn more.
 * @param options.isDirectHost If set `false`, the placeholder will not be enabled directly
 * in the passed `element` but in one of its children (selected automatically, i.e. a first empty child element).
 * Useful when attaching placeholders to elements that can host other elements (not just text), for instance,
 * editable root elements.
 * @param options.text Placeholder text. It's **deprecated** and will be removed soon. Use
 * {@link module:engine/view/placeholder~PlaceholderableViewElement#placeholder `options.element.placeholder`} instead.
 * @param options.keepOnFocus If set `true`, the placeholder stay visible when the host element is focused.
 */ function enableViewPlaceholder({ view, element, text, isDirectHost = true, keepOnFocus = false }) {
    const doc = view.document;
    // Use a single post fixer per—document to update all placeholders.
    if (!documentPlaceholders.has(doc)) {
        documentPlaceholders.set(doc, new Map());
        // If a post-fixer callback makes a change, it should return `true` so other post–fixers
        // can re–evaluate the document again.
        doc.registerPostFixer((writer)=>updateDocumentPlaceholders(documentPlaceholders.get(doc), writer));
        // Update placeholders on isComposing state change since rendering is disabled while in composition mode.
        doc.on('change:isComposing', ()=>{
            view.change((writer)=>updateDocumentPlaceholders(documentPlaceholders.get(doc), writer));
        }, {
            priority: 'high'
        });
    }
    if (element.is('editableElement')) {
        element.on('change:placeholder', (evtInfo, evt, text)=>setPlaceholder(text));
    }
    if (element.placeholder) {
        setPlaceholder(element.placeholder);
    } else if (text) {
        setPlaceholder(text);
    }
    if (text) {
        showViewPlaceholderTextDeprecationWarning();
    }
    function setPlaceholder(text) {
        const config = {
            text,
            isDirectHost,
            keepOnFocus,
            hostElement: isDirectHost ? element : null
        };
        // Store information about the element placeholder under its document.
        documentPlaceholders.get(doc).set(element, config);
        // Update the placeholders right away.
        view.change((writer)=>updateDocumentPlaceholders([
                [
                    element,
                    config
                ]
            ], writer));
    }
}
/**
 * Disables the placeholder functionality from a given element.
 *
 * See {@link module:engine/view/placeholder~enableViewPlaceholder `enableViewPlaceholder()`} to learn more.
 */ function disableViewPlaceholder(view, element) {
    const doc = element.document;
    if (!documentPlaceholders.has(doc)) {
        return;
    }
    view.change((writer)=>{
        const placeholders = documentPlaceholders.get(doc);
        const config = placeholders.get(element);
        writer.removeAttribute('data-placeholder', config.hostElement);
        hideViewPlaceholder(writer, config.hostElement);
        placeholders.delete(element);
    });
}
/**
 * Shows a placeholder in the provided element by changing related attributes and CSS classes.
 *
 * **Note**: This helper will not update the placeholder visibility nor manage the
 * it in any way in the future. What it does is a one–time state change of an element. Use
 * {@link module:engine/view/placeholder~enableViewPlaceholder `enableViewPlaceholder()`} and
 * {@link module:engine/view/placeholder~disableViewPlaceholder `disableViewPlaceholder()`} for full
 * placeholder functionality.
 *
 * **Note**: This helper will blindly show the placeholder directly in the root editable element if
 * one is passed, which could result in a visual clash if the editable element has some children
 * (for instance, an empty paragraph). Use {@link module:engine/view/placeholder~enableViewPlaceholder `enableViewPlaceholder()`}
 * in that case or make sure the correct element is passed to the helper.
 *
 * @returns `true`, if any changes were made to the `element`.
 */ function showViewPlaceholder(writer, element) {
    if (!element.hasClass('ck-placeholder')) {
        writer.addClass('ck-placeholder', element);
        return true;
    }
    return false;
}
/**
 * Hides a placeholder in the element by changing related attributes and CSS classes.
 *
 * **Note**: This helper will not update the placeholder visibility nor manage the
 * it in any way in the future. What it does is a one–time state change of an element. Use
 * {@link module:engine/view/placeholder~enableViewPlaceholder `enableViewPlaceholder()`} and
 * {@link module:engine/view/placeholder~disableViewPlaceholder `disableViewPlaceholder()`} for full
 * placeholder functionality.
 *
 * @returns `true`, if any changes were made to the `element`.
 */ function hideViewPlaceholder(writer, element) {
    if (element.hasClass('ck-placeholder')) {
        writer.removeClass('ck-placeholder', element);
        return true;
    }
    return false;
}
/**
 * Checks if a placeholder should be displayed in the element.
 *
 * **Note**: This helper will blindly check the possibility of showing a placeholder directly in the
 * root editable element if one is passed, which may not be the expected result. If an element can
 * host other elements (not just text), most likely one of its children should be checked instead
 * because it will be the final host for the placeholder. Use
 * {@link module:engine/view/placeholder~enableViewPlaceholder `enableViewPlaceholder()`} in that case or make
 * sure the correct element is passed to the helper.
 *
 * @param element Element that holds the placeholder.
 * @param keepOnFocus Focusing the element will keep the placeholder visible.
 */ function needsViewPlaceholder(element, keepOnFocus) {
    if (!element.isAttached()) {
        return false;
    }
    if (hasContent(element)) {
        return false;
    }
    const doc = element.document;
    const viewSelection = doc.selection;
    const selectionAnchor = viewSelection.anchor;
    if (doc.isComposing && selectionAnchor && selectionAnchor.parent === element) {
        return false;
    }
    // Skip the focus check and make the placeholder visible already regardless of document focus state.
    if (keepOnFocus) {
        return true;
    }
    // If the document is blurred.
    if (!doc.isFocused) {
        return true;
    }
    // If document is focused and the element is empty but the selection is not anchored inside it.
    return !!selectionAnchor && selectionAnchor.parent !== element;
}
/**
 * Anything but uiElement(s) counts as content.
 */ function hasContent(element) {
    for (const child of element.getChildren()){
        if (!child.is('uiElement')) {
            return true;
        }
    }
    return false;
}
/**
 * Updates all placeholders associated with a document in a post–fixer callback.
 *
 * @returns True if any changes were made to the view document.
 */ function updateDocumentPlaceholders(placeholders, writer) {
    const directHostElements = [];
    let wasViewModified = false;
    // First set placeholders on the direct hosts.
    for (const [element, config] of placeholders){
        if (config.isDirectHost) {
            directHostElements.push(element);
            if (updatePlaceholder(writer, element, config)) {
                wasViewModified = true;
            }
        }
    }
    // Then set placeholders on the indirect hosts but only on those that does not already have an direct host placeholder.
    for (const [element, config] of placeholders){
        if (config.isDirectHost) {
            continue;
        }
        const hostElement = getChildPlaceholderHostSubstitute(element);
        // When not a direct host, it could happen that there is no child element
        // capable of displaying a placeholder.
        if (!hostElement) {
            continue;
        }
        // Don't override placeholder if the host element already has some direct placeholder.
        if (directHostElements.includes(hostElement)) {
            continue;
        }
        // Update the host element (used for setting and removing the placeholder).
        config.hostElement = hostElement;
        if (updatePlaceholder(writer, element, config)) {
            wasViewModified = true;
        }
    }
    return wasViewModified;
}
/**
 * Updates a single placeholder in a post–fixer callback.
 *
 * @returns True if any changes were made to the view document.
 */ function updatePlaceholder(writer, element, config) {
    const { text, isDirectHost, hostElement } = config;
    let wasViewModified = false;
    // This may be necessary when updating the placeholder text to something else.
    if (hostElement.getAttribute('data-placeholder') !== text) {
        writer.setAttribute('data-placeholder', text, hostElement);
        wasViewModified = true;
    }
    // If the host element is not a direct host then placeholder is needed only when there is only one element.
    const isOnlyChild = isDirectHost || element.childCount == 1;
    if (isOnlyChild && needsViewPlaceholder(hostElement, config.keepOnFocus)) {
        if (showViewPlaceholder(writer, hostElement)) {
            wasViewModified = true;
        }
    } else if (hideViewPlaceholder(writer, hostElement)) {
        wasViewModified = true;
    }
    return wasViewModified;
}
/**
 * Gets a child element capable of displaying a placeholder if a parent element can host more
 * than just text (for instance, when it is a root editable element). The child element
 * can then be used in other placeholder helpers as a substitute of its parent.
 */ function getChildPlaceholderHostSubstitute(parent) {
    if (parent.childCount) {
        const firstChild = parent.getChild(0);
        if (firstChild.is('element') && !firstChild.is('uiElement') && !firstChild.is('attributeElement')) {
            return firstChild;
        }
    }
    return null;
}
/**
 * Displays a deprecation warning message in the console, but only once per page load.
 */ function showViewPlaceholderTextDeprecationWarning() {
    if (!hasDisplayedPlaceholderDeprecationWarning) {
        /**
		 * The "text" option in the {@link module:engine/view/placeholder~enableViewPlaceholder `enableViewPlaceholder()`}
		 * function is deprecated and will be removed soon.
		 *
		 * See the {@glink updating/guides/update-to-39#view-element-placeholder Migration to v39} guide for
		 * more information on how to apply this change.
		 *
		 * @error enableViewPlaceholder-deprecated-text-option
		 */ logWarning('enableViewPlaceholder-deprecated-text-option');
    }
    hasDisplayedPlaceholderDeprecationWarning = true;
}

/**
 * @license Copyright (c) 2003-2025, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-licensing-options
 */ /**
 * @module engine/view/typecheckable
 */ class ViewTypeCheckable {
    /* istanbul ignore next -- @preserve */ is() {
        // There are a lot of overloads above.
        // Overriding method in derived classes remove them and only `is( type: string ): boolean` is visible which we don't want.
        // One option would be to copy them all to all classes, but that's ugly.
        // It's best when TypeScript compiler doesn't see those overloads, except the one in the top base class.
        // To overload a method, but not let the compiler see it, do after class definition:
        // `MyClass.prototype.is = function( type: string ) {...}`
        throw new Error('is() method is abstract');
    }
}

/**
 * Abstract view node class.
 *
 * This is an abstract class. Its constructor should not be used directly.
 * Use the {@link module:engine/view/downcastwriter~ViewDowncastWriter} or {@link module:engine/view/upcastwriter~ViewUpcastWriter}
 * to create new instances of view nodes.
 */ class ViewNode extends /* #__PURE__ */ EmitterMixin(ViewTypeCheckable) {
    /**
	 * The document instance to which this node belongs.
	 */ document;
    /**
	 * Parent element. Null by default. Set by {@link module:engine/view/element~ViewElement#_insertChild}.
	 */ parent;
    /**
	 * Creates a tree view node.
	 *
	 * @param document The document instance to which this node belongs.
	 */ constructor(document){
        super();
        this.document = document;
        this.parent = null;
    }
    /**
	 * Index of the node in the parent element or null if the node has no parent.
	 *
	 * Accessing this property throws an error if this node's parent element does not contain it.
	 * This means that view tree got broken.
	 */ get index() {
        let pos;
        if (!this.parent) {
            return null;
        }
        // No parent or child doesn't exist in parent's children.
        if ((pos = this.parent.getChildIndex(this)) == -1) {
            /**
			 * The node's parent does not contain this node. It means that the document tree is corrupted.
			 *
			 * @error view-node-not-found-in-parent
			 */ throw new CKEditorError('view-node-not-found-in-parent', this);
        }
        return pos;
    }
    /**
	 * Node's next sibling, or `null` if it is the last child.
	 */ get nextSibling() {
        const index = this.index;
        return index !== null && this.parent.getChild(index + 1) || null;
    }
    /**
	 * Node's previous sibling, or `null` if it is the first child.
	 */ get previousSibling() {
        const index = this.index;
        return index !== null && this.parent.getChild(index - 1) || null;
    }
    /**
	 * Top-most ancestor of the node. If the node has no parent it is the root itself.
	 */ get root() {
        // eslint-disable-next-line @typescript-eslint/no-this-alias, consistent-this
        let root = this;
        while(root.parent){
            root = root.parent;
        }
        return root;
    }
    /**
	 * Returns true if the node is in a tree rooted in the document (is a descendant of one of its roots).
	 */ isAttached() {
        return this.root.is('rootElement');
    }
    /**
	 * Gets a path to the node. The path is an array containing indices of consecutive ancestors of this node,
	 * beginning from {@link module:engine/view/node~ViewNode#root root}, down to this node's index.
	 *
	 * ```ts
	 * const abc = downcastWriter.createText( 'abc' );
	 * const foo = downcastWriter.createText( 'foo' );
	 * const h1 = downcastWriter.createElement( 'h1', null, downcastWriter.createText( 'header' ) );
	 * const p = downcastWriter.createElement( 'p', null, [ abc, foo ] );
	 * const div = downcastWriter.createElement( 'div', null, [ h1, p ] );
	 * foo.getPath(); // Returns [ 1, 3 ]. `foo` is in `p` which is in `div`. `p` starts at offset 1, while `foo` at 3.
	 * h1.getPath(); // Returns [ 0 ].
	 * div.getPath(); // Returns [].
	 * ```
	 *
	 * @returns The path.
	 */ getPath() {
        const path = [];
        // eslint-disable-next-line @typescript-eslint/no-this-alias, consistent-this
        let node = this;
        while(node.parent){
            path.unshift(node.index);
            node = node.parent;
        }
        return path;
    }
    /**
	 * Returns ancestors array of this node.
	 *
	 * @param options Options object.
	 * @param options.includeSelf When set to `true` this node will be also included in parent's array.
	 * @param options.parentFirst When set to `true`, array will be sorted from node's parent to root element,
	 * otherwise root element will be the first item in the array.
	 * @returns Array with ancestors.
	 */ getAncestors(options = {}) {
        const ancestors = [];
        let parent = options.includeSelf ? this : this.parent;
        while(parent){
            ancestors[options.parentFirst ? 'push' : 'unshift'](parent);
            parent = parent.parent;
        }
        return ancestors;
    }
    /**
	 * Returns a {@link module:engine/view/element~ViewElement} or {@link module:engine/view/documentfragment~ViewDocumentFragment}
	 * which is a common ancestor of both nodes.
	 *
	 * @param node The second node.
	 * @param options Options object.
	 * @param options.includeSelf When set to `true` both nodes will be considered "ancestors" too.
	 * Which means that if e.g. node A is inside B, then their common ancestor will be B.
	 */ getCommonAncestor(node, options = {}) {
        const ancestorsA = this.getAncestors(options);
        const ancestorsB = node.getAncestors(options);
        let i = 0;
        while(ancestorsA[i] == ancestorsB[i] && ancestorsA[i]){
            i++;
        }
        return i === 0 ? null : ancestorsA[i - 1];
    }
    /**
	 * Returns whether this node is before given node. `false` is returned if nodes are in different trees (for example,
	 * in different {@link module:engine/view/documentfragment~ViewDocumentFragment}s).
	 *
	 * @param node Node to compare with.
	 */ isBefore(node) {
        // Given node is not before this node if they are same.
        if (this == node) {
            return false;
        }
        // Return `false` if it is impossible to compare nodes.
        if (this.root !== node.root) {
            return false;
        }
        const thisPath = this.getPath();
        const nodePath = node.getPath();
        const result = compareArrays(thisPath, nodePath);
        switch(result){
            case 'prefix':
                return true;
            case 'extension':
                return false;
            default:
                return thisPath[result] < nodePath[result];
        }
    }
    /**
	 * Returns whether this node is after given node. `false` is returned if nodes are in different trees (for example,
	 * in different {@link module:engine/view/documentfragment~ViewDocumentFragment}s).
	 *
	 * @param node Node to compare with.
	 */ isAfter(node) {
        // Given node is not before this node if they are same.
        if (this == node) {
            return false;
        }
        // Return `false` if it is impossible to compare nodes.
        if (this.root !== node.root) {
            return false;
        }
        // In other cases, just check if the `node` is before, and return the opposite.
        return !this.isBefore(node);
    }
    /**
	 * Removes node from parent.
	 *
	 * @internal
	 */ _remove() {
        this.parent._removeChildren(this.index);
    }
    /**
	 * @internal
	 * @param type Type of the change.
	 * @param node Changed node.
	 * @param data Additional data.
	 * @fires change
	 */ _fireChange(type, node, data) {
        this.fire(`change:${type}`, node, data);
        if (this.parent) {
            this.parent._fireChange(type, node, data);
        }
    }
    /**
	 * Converts `ViewNode` to plain object and returns it.
	 *
	 * @returns `ViewNode` converted to plain object.
	 */ toJSON() {
        const json = {
            path: this.getPath(),
            type: 'Node'
        };
        if (this !== this.root && this.root.is('rootElement')) {
            json.root = this.root.toJSON();
        }
        return json;
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewNode.prototype.is = function(type) {
    return type === 'node' || type === 'view:node';
};

/**
 * Tree view text node.
 *
 * The constructor of this class should not be used directly. To create a new text node instance
 * use the {@link module:engine/view/downcastwriter~ViewDowncastWriter#createText `ViewDowncastWriter#createText()`}
 * method when working on data downcasted from the model or the
 * {@link module:engine/view/upcastwriter~ViewUpcastWriter#createText `ViewUpcastWriter#createText()`}
 * method when working on non-semantic views.
 */ class ViewText extends ViewNode {
    /**
	 * The text content.
	 *
	 * Setting the data fires the {@link module:engine/view/node~ViewNode#event:change:text change event}.
	 */ _textData;
    /**
	 * Creates a tree view text node.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#createText
	 * @internal
	 * @param document The document instance to which this text node belongs.
	 * @param data The text's data.
	 */ constructor(document, data){
        super(document);
        this._textData = data;
    }
    /**
	 * The text content.
	 */ get data() {
        return this._textData;
    }
    /**
	 * The `_data` property is controlled by a getter and a setter.
	 *
	 * The getter is required when using the addition assignment operator on protected property:
	 *
	 * ```ts
	 * const foo = downcastWriter.createText( 'foo' );
	 * const bar = downcastWriter.createText( 'bar' );
	 *
	 * foo._data += bar.data;   // executes: `foo._data = foo._data + bar.data`
	 * console.log( foo.data ); // prints: 'foobar'
	 * ```
	 *
	 * If the protected getter didn't exist, `foo._data` will return `undefined` and result of the merge will be invalid.
	 *
	 * The setter sets data and fires the {@link module:engine/view/node~ViewNode#event:change:text change event}.
	 *
	 * @internal
	 */ get _data() {
        return this.data;
    }
    set _data(data) {
        this._fireChange('text', this);
        this._textData = data;
    }
    /**
	 * Checks if this text node is similar to other text node.
	 * Both nodes should have the same data to be considered as similar.
	 *
	 * @param otherNode Node to check if it is same as this node.
	 */ isSimilar(otherNode) {
        if (!(otherNode instanceof ViewText)) {
            return false;
        }
        return this === otherNode || this.data === otherNode.data;
    }
    /**
	 * Converts `ViewText` instance to plain object and returns it.
	 *
	 * @returns `ViewText` instance converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        json.type = 'Text';
        json.data = this.data;
        return json;
    }
    /**
	 * Clones this node.
	 *
	 * @internal
	 * @returns Text node that is a clone of this node.
	 */ _clone() {
        return new ViewText(this.document, this.data);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewText.prototype.is = function(type) {
    return type === '$text' || type === 'view:$text' || // This are legacy values kept for backward compatibility.
    type === 'text' || type === 'view:text' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
    type === 'node' || type === 'view:node';
};

/**
 * ViewTextProxy is a wrapper for substring of {@link module:engine/view/text~ViewText}. Instance of this class is created by
 * {@link module:engine/view/treewalker~ViewTreeWalker} when only a part of {@link module:engine/view/text~ViewText} needs to be returned.
 *
 * `ViewTextProxy` has an API similar to {@link module:engine/view/text~ViewText Text} and allows to do most of the common tasks performed
 * on view nodes.
 *
 * **Note:** Some `ViewTextProxy` instances may represent whole text node, not just a part of it.
 * See {@link module:engine/view/textproxy~ViewTextProxy#isPartial}.
 *
 * **Note:** `ViewTextProxy` is a readonly interface.
 *
 * **Note:** `ViewTextProxy` instances are created on the fly basing
 * on the current state of parent {@link module:engine/view/text~ViewText}.
 * Because of this it is highly unrecommended to store references to `TextProxy instances because they might get
 * invalidated due to operations on Document. Also ViewTextProxy is not a {@link module:engine/view/node~ViewNode} so it cannot be
 * inserted as a child of {@link module:engine/view/element~ViewElement}.
 *
 * `ViewTextProxy` instances are created by {@link module:engine/view/treewalker~ViewTreeWalker view tree walker}.
 * You should not need to create an instance of this class by your own.
 */ class ViewTextProxy extends ViewTypeCheckable {
    /**
	 * Reference to the {@link module:engine/view/text~ViewText} element which ViewTextProxy is a substring.
	 */ textNode;
    /**
	 * Text data represented by this text proxy.
	 */ data;
    /**
	 * Offset in the `textNode` where this `ViewTextProxy` instance starts.
	 */ offsetInText;
    /**
	 * Creates a text proxy.
	 *
	 * @internal
	 * @param textNode Text node which part is represented by this text proxy.
	 * @param offsetInText Offset in {@link module:engine/view/textproxy~ViewTextProxy#textNode text node}
	 * from which the text proxy starts.
	 * @param length Text proxy length, that is how many text node's characters, starting from `offsetInText` it represents.
	 */ constructor(textNode, offsetInText, length){
        super();
        this.textNode = textNode;
        if (offsetInText < 0 || offsetInText > textNode.data.length) {
            /**
			 * Given offsetInText value is incorrect.
			 *
			 * @error view-textproxy-wrong-offsetintext
			 */ throw new CKEditorError('view-textproxy-wrong-offsetintext', this);
        }
        if (length < 0 || offsetInText + length > textNode.data.length) {
            /**
			 * Given length value is incorrect.
			 *
			 * @error view-textproxy-wrong-length
			 */ throw new CKEditorError('view-textproxy-wrong-length', this);
        }
        this.data = textNode.data.substring(offsetInText, offsetInText + length);
        this.offsetInText = offsetInText;
    }
    /**
	 * Offset size of this node.
	 */ get offsetSize() {
        return this.data.length;
    }
    /**
	 * Flag indicating whether `ViewTextProxy` instance covers only part of the original {@link module:engine/view/text~ViewText text node}
	 * (`true`) or the whole text node (`false`).
	 *
	 * This is `false` when text proxy starts at the very beginning of {@link module:engine/view/textproxy~ViewTextProxy#textNode textNode}
	 * ({@link module:engine/view/textproxy~ViewTextProxy#offsetInText offsetInText} equals `0`) and text proxy sizes is equal to
	 * text node size.
	 */ get isPartial() {
        return this.data.length !== this.textNode.data.length;
    }
    /**
	 * Parent of this text proxy, which is same as parent of text node represented by this text proxy.
	 */ get parent() {
        return this.textNode.parent;
    }
    /**
	 * Root of this text proxy, which is same as root of text node represented by this text proxy.
	 */ get root() {
        return this.textNode.root;
    }
    /**
	 * {@link module:engine/view/document~ViewDocument View document} that owns this text proxy, or `null` if the text proxy is inside
	 * {@link module:engine/view/documentfragment~ViewDocumentFragment document fragment}.
	 */ get document() {
        return this.textNode.document;
    }
    /**
	 * Returns ancestors array of this text proxy.
	 *
	 * @param options Options object.
	 * @param options.includeSelf When set to `true`, textNode will be also included in parent's array.
	 * @param options.parentFirst When set to `true`, array will be sorted from text proxy parent to
	 * root element, otherwise root element will be the first item in the array.
	 * @returns Array with ancestors.
	 */ getAncestors(options = {}) {
        const ancestors = [];
        let parent = options.includeSelf ? this.textNode : this.parent;
        while(parent !== null){
            ancestors[options.parentFirst ? 'push' : 'unshift'](parent);
            parent = parent.parent;
        }
        return ancestors;
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewTextProxy.prototype.is = function(type) {
    return type === '$textProxy' || type === 'view:$textProxy' || // This are legacy values kept for backward compatibility.
    type === 'textProxy' || type === 'view:textProxy';
};

/**
 * Class used for handling consumption of view {@link module:engine/view/element~ViewElement elements},
 * {@link module:engine/view/text~ViewText text nodes} and
 * {@link module:engine/view/documentfragment~ViewDocumentFragment document fragments}.
 * Element's name and its parts (attributes, classes and styles) can be consumed separately. Consuming an element's name
 * does not consume its attributes, classes and styles.
 * To add items for consumption use {@link module:engine/conversion/viewconsumable~ViewConsumable#add add method}.
 * To test items use {@link module:engine/conversion/viewconsumable~ViewConsumable#test test method}.
 * To consume items use {@link module:engine/conversion/viewconsumable~ViewConsumable#consume consume method}.
 * To revert already consumed items use {@link module:engine/conversion/viewconsumable~ViewConsumable#revert revert method}.
 *
 * ```ts
 * viewConsumable.add( element, { name: true } ); // Adds element's name as ready to be consumed.
 * viewConsumable.add( textNode ); // Adds text node for consumption.
 * viewConsumable.add( docFragment ); // Adds document fragment for consumption.
 * viewConsumable.test( element, { name: true }  ); // Tests if element's name can be consumed.
 * viewConsumable.test( textNode ); // Tests if text node can be consumed.
 * viewConsumable.test( docFragment ); // Tests if document fragment can be consumed.
 * viewConsumable.consume( element, { name: true }  ); // Consume element's name.
 * viewConsumable.consume( textNode ); // Consume text node.
 * viewConsumable.consume( docFragment ); // Consume document fragment.
 * viewConsumable.revert( element, { name: true }  ); // Revert already consumed element's name.
 * viewConsumable.revert( textNode ); // Revert already consumed text node.
 * viewConsumable.revert( docFragment ); // Revert already consumed document fragment.
 * ```
 */ class ViewConsumable {
    /**
	 * Map of consumable elements. If {@link module:engine/view/element~ViewElement element} is used as a key,
	 * {@link module:engine/conversion/viewconsumable~ViewElementConsumables ViewElementConsumables} instance is stored as value.
	 * For {@link module:engine/view/text~ViewText text nodes} and
	 * {@link module:engine/view/documentfragment~ViewDocumentFragment document fragments} boolean value is stored as value.
	 */ _consumables = new Map();
    /**
	 * Adds view {@link module:engine/view/element~ViewElement element}, {@link module:engine/view/text~ViewText text node} or
	 * {@link module:engine/view/documentfragment~ViewDocumentFragment document fragment} as ready to be consumed.
	 *
	 * ```ts
	 * viewConsumable.add( p, { name: true } ); // Adds element's name to consume.
	 * viewConsumable.add( p, { attributes: 'name' } ); // Adds element's attribute.
	 * viewConsumable.add( p, { classes: 'foobar' } ); // Adds element's class.
	 * viewConsumable.add( p, { styles: 'color' } ); // Adds element's style
	 * viewConsumable.add( p, { attributes: 'name', styles: 'color' } ); // Adds attribute and style.
	 * viewConsumable.add( p, { classes: [ 'baz', 'bar' ] } ); // Multiple consumables can be provided.
	 * viewConsumable.add( textNode ); // Adds text node to consume.
	 * viewConsumable.add( docFragment ); // Adds document fragment to consume.
	 * ```
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `viewconsumable-invalid-attribute` when `class` or `style`
	 * attribute is provided - it should be handled separately by providing actual style/class.
	 *
	 * ```ts
	 * viewConsumable.add( p, { attributes: 'style' } ); // This call will throw an exception.
	 * viewConsumable.add( p, { styles: 'color' } ); // This is properly handled style.
	 * ```
	 *
	 * @param consumables Used only if first parameter is {@link module:engine/view/element~ViewElement view element} instance.
	 * @param consumables.name If set to true element's name will be included.
	 * @param consumables.attributes Attribute name or array of attribute names.
	 * @param consumables.classes Class name or array of class names.
	 * @param consumables.styles Style name or array of style names.
	 */ add(element, consumables) {
        let elementConsumables;
        // For text nodes and document fragments just mark them as consumable.
        if (element.is('$text') || element.is('documentFragment')) {
            this._consumables.set(element, true);
            return;
        }
        // For elements create new ViewElementConsumables or update already existing one.
        if (!this._consumables.has(element)) {
            elementConsumables = new ViewElementConsumables(element);
            this._consumables.set(element, elementConsumables);
        } else {
            elementConsumables = this._consumables.get(element);
        }
        elementConsumables.add(consumables ? normalizeConsumables(consumables) : element._getConsumables());
    }
    /**
	 * Tests if {@link module:engine/view/element~ViewElement view element}, {@link module:engine/view/text~ViewText text node} or
	 * {@link module:engine/view/documentfragment~ViewDocumentFragment document fragment} can be consumed.
	 * It returns `true` when all items included in method's call can be consumed. Returns `false` when
	 * first already consumed item is found and `null` when first non-consumable item is found.
	 *
	 * ```ts
	 * viewConsumable.test( p, { name: true } ); // Tests element's name.
	 * viewConsumable.test( p, { attributes: 'name' } ); // Tests attribute.
	 * viewConsumable.test( p, { classes: 'foobar' } ); // Tests class.
	 * viewConsumable.test( p, { styles: 'color' } ); // Tests style.
	 * viewConsumable.test( p, { attributes: 'name', styles: 'color' } ); // Tests attribute and style.
	 * viewConsumable.test( p, { classes: [ 'baz', 'bar' ] } ); // Multiple consumables can be tested.
	 * viewConsumable.test( textNode ); // Tests text node.
	 * viewConsumable.test( docFragment ); // Tests document fragment.
	 * ```
	 *
	 * Testing classes and styles as attribute will test if all added classes/styles can be consumed.
	 *
	 * ```ts
	 * viewConsumable.test( p, { attributes: 'class' } ); // Tests if all added classes can be consumed.
	 * viewConsumable.test( p, { attributes: 'style' } ); // Tests if all added styles can be consumed.
	 * ```
	 *
	 * @param consumables Used only if first parameter is {@link module:engine/view/element~ViewElement view element} instance.
	 * @param consumables.name If set to true element's name will be included.
	 * @param consumables.attributes Attribute name or array of attribute names.
	 * @param consumables.classes Class name or array of class names.
	 * @param consumables.styles Style name or array of style names.
	 * @returns Returns `true` when all items included in method's call can be consumed. Returns `false`
	 * when first already consumed item is found and `null` when first non-consumable item is found.
	 */ test(element, consumables) {
        const elementConsumables = this._consumables.get(element);
        if (elementConsumables === undefined) {
            return null;
        }
        // For text nodes and document fragments return stored boolean value.
        if (element.is('$text') || element.is('documentFragment')) {
            return elementConsumables;
        }
        // For elements test consumables object.
        return elementConsumables.test(normalizeConsumables(consumables));
    }
    /**
	 * Consumes {@link module:engine/view/element~ViewElement view element}, {@link module:engine/view/text~ViewText text node} or
	 * {@link module:engine/view/documentfragment~ViewDocumentFragment document fragment}.
	 * It returns `true` when all items included in method's call can be consumed, otherwise returns `false`.
	 *
	 * ```ts
	 * viewConsumable.consume( p, { name: true } ); // Consumes element's name.
	 * viewConsumable.consume( p, { attributes: 'name' } ); // Consumes element's attribute.
	 * viewConsumable.consume( p, { classes: 'foobar' } ); // Consumes element's class.
	 * viewConsumable.consume( p, { styles: 'color' } ); // Consumes element's style.
	 * viewConsumable.consume( p, { attributes: 'name', styles: 'color' } ); // Consumes attribute and style.
	 * viewConsumable.consume( p, { classes: [ 'baz', 'bar' ] } ); // Multiple consumables can be consumed.
	 * viewConsumable.consume( textNode ); // Consumes text node.
	 * viewConsumable.consume( docFragment ); // Consumes document fragment.
	 * ```
	 *
	 * Consuming classes and styles as attribute will test if all added classes/styles can be consumed.
	 *
	 * ```ts
	 * viewConsumable.consume( p, { attributes: 'class' } ); // Consume only if all added classes can be consumed.
	 * viewConsumable.consume( p, { attributes: 'style' } ); // Consume only if all added styles can be consumed.
	 * ```
	 *
	 * @param consumables Used only if first parameter is {@link module:engine/view/element~ViewElement view element} instance.
	 * @param consumables.name If set to true element's name will be included.
	 * @param consumables.attributes Attribute name or array of attribute names.
	 * @param consumables.classes Class name or array of class names.
	 * @param consumables.styles Style name or array of style names.
	 * @returns Returns `true` when all items included in method's call can be consumed,
	 * otherwise returns `false`.
	 */ consume(element, consumables) {
        if (element.is('$text') || element.is('documentFragment')) {
            if (!this.test(element, consumables)) {
                return false;
            }
            // For text nodes and document fragments set value to false.
            this._consumables.set(element, false);
            return true;
        }
        // For elements - consume consumables object.
        const elementConsumables = this._consumables.get(element);
        if (elementConsumables === undefined) {
            return false;
        }
        return elementConsumables.consume(normalizeConsumables(consumables));
    }
    /**
	 * Reverts {@link module:engine/view/element~ViewElement view element}, {@link module:engine/view/text~ViewText text node} or
	 * {@link module:engine/view/documentfragment~ViewDocumentFragment document fragment} so they can be consumed once again.
	 * Method does not revert items that were never previously added for consumption, even if they are included in
	 * method's call.
	 *
	 * ```ts
	 * viewConsumable.revert( p, { name: true } ); // Reverts element's name.
	 * viewConsumable.revert( p, { attributes: 'name' } ); // Reverts element's attribute.
	 * viewConsumable.revert( p, { classes: 'foobar' } ); // Reverts element's class.
	 * viewConsumable.revert( p, { styles: 'color' } ); // Reverts element's style.
	 * viewConsumable.revert( p, { attributes: 'name', styles: 'color' } ); // Reverts attribute and style.
	 * viewConsumable.revert( p, { classes: [ 'baz', 'bar' ] } ); // Multiple names can be reverted.
	 * viewConsumable.revert( textNode ); // Reverts text node.
	 * viewConsumable.revert( docFragment ); // Reverts document fragment.
	 * ```
	 *
	 * Reverting classes and styles as attribute will revert all classes/styles that were previously added for
	 * consumption.
	 *
	 * ```ts
	 * viewConsumable.revert( p, { attributes: 'class' } ); // Reverts all classes added for consumption.
	 * viewConsumable.revert( p, { attributes: 'style' } ); // Reverts all styles added for consumption.
	 * ```
	 *
	 * @param consumables Used only if first parameter is {@link module:engine/view/element~ViewElement view element} instance.
	 * @param consumables.name If set to true element's name will be included.
	 * @param consumables.attributes Attribute name or array of attribute names.
	 * @param consumables.classes Class name or array of class names.
	 * @param consumables.styles Style name or array of style names.
	 */ revert(element, consumables) {
        const elementConsumables = this._consumables.get(element);
        if (elementConsumables !== undefined) {
            if (element.is('$text') || element.is('documentFragment')) {
                // For text nodes and document fragments - set consumable to true.
                this._consumables.set(element, true);
            } else {
                // For elements - revert items from consumables object.
                elementConsumables.revert(normalizeConsumables(consumables));
            }
        }
    }
    /**
	 * Creates {@link module:engine/conversion/viewconsumable~ViewConsumable ViewConsumable} instance from
	 * {@link module:engine/view/node~ViewNode node} or {@link module:engine/view/documentfragment~ViewDocumentFragment document fragment}.
	 * Instance will contain all elements, child nodes, attributes, styles and classes added for consumption.
	 *
	 * @param from View node or document fragment from which `ViewConsumable` will be created.
	 * @param instance If provided, given `ViewConsumable` instance will be used
	 * to add all consumables. It will be returned instead of a new instance.
	 */ static createFrom(from, instance) {
        if (!instance) {
            instance = new ViewConsumable();
        }
        if (from.is('$text')) {
            instance.add(from);
        } else if (from.is('element') || from.is('documentFragment')) {
            instance.add(from);
            for (const child of from.getChildren()){
                ViewConsumable.createFrom(child, instance);
            }
        }
        return instance;
    }
}
/**
 * This is a private helper-class for {@link module:engine/conversion/viewconsumable~ViewConsumable}.
 * It represents and manipulates consumable parts of a single {@link module:engine/view/element~ViewElement}.
 *
 * @internal
 */ class ViewElementConsumables {
    element;
    /**
	 * Flag indicating if name of the element can be consumed.
	 */ _canConsumeName = null;
    /**
	 * A map of element's consumables.
	 * * For plain attributes the value is a boolean indicating whether the attribute is available to consume.
	 * * For token based attributes (like class list and style) the value is a map of tokens to booleans
	 * indicating whether the token is available to consume on the given attribute.
	 */ _attributes = new Map();
    /**
	 * Creates ViewElementConsumables instance.
	 *
	 * @param from View element from which `ViewElementConsumables` is being created.
	 */ constructor(from){
        this.element = from;
    }
    /**
	 * Adds consumable parts of the {@link module:engine/view/element~ViewElement view element}.
	 * Element's name itself can be marked to be consumed (when element's name is consumed its attributes, classes and
	 * styles still could be consumed):
	 *
	 * ```ts
	 * consumables.add( { name: true } );
	 * ```
	 *
	 * Attributes classes and styles:
	 *
	 * ```ts
	 * consumables.add( { attributes: [ [ 'title' ], [ 'class', 'foo' ], [ 'style', 'color'] ] } );
	 * consumables.add( { attributes: [ [ 'title' ], [ 'name' ], [ 'class', 'foo' ], [ 'class', 'bar' ] ] } );
	 * ```
	 *
	 * Note: This method accepts only {@link module:engine/view/element~ViewNormalizedConsumables}.
	 * You can use {@link module:engine/conversion/viewconsumable~normalizeConsumables} helper to convert from
	 * {@link module:engine/conversion/viewconsumable~Consumables} to `ViewNormalizedConsumables`.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `viewconsumable-invalid-attribute` when `class` or `style`
	 * attribute is provided - it should be handled separately by providing `style` and `class` in consumables object.
	 *
	 * @param consumables Object describing which parts of the element can be consumed.
	 */ add(consumables) {
        if (consumables.name) {
            this._canConsumeName = true;
        }
        for (const [name, token] of consumables.attributes){
            if (token) {
                let attributeTokens = this._attributes.get(name);
                if (!attributeTokens || typeof attributeTokens == 'boolean') {
                    attributeTokens = new Map();
                    this._attributes.set(name, attributeTokens);
                }
                attributeTokens.set(token, true);
            } else if (name == 'style' || name == 'class') {
                /**
				 * Class and style attributes should be handled separately in
				 * {@link module:engine/conversion/viewconsumable~ViewConsumable#add `ViewConsumable#add()`}.
				 *
				 * What you have done is trying to use:
				 *
				 * ```ts
				 * consumables.add( { attributes: [ 'class', 'style' ] } );
				 * ```
				 *
				 * While each class and style should be registered separately:
				 *
				 * ```ts
				 * consumables.add( { classes: 'some-class', styles: 'font-weight' } );
				 * ```
				 *
				 * @error viewconsumable-invalid-attribute
				 */ throw new CKEditorError('viewconsumable-invalid-attribute', this);
            } else {
                this._attributes.set(name, true);
            }
        }
    }
    /**
	 * Tests if parts of the {@link module:engine/view/element~ViewElement view element} can be consumed.
	 *
	 * Element's name can be tested:
	 *
	 * ```ts
	 * consumables.test( { name: true } );
	 * ```
	 *
	 * Attributes classes and styles:
	 *
	 * ```ts
	 * consumables.test( { attributes: [ [ 'title' ], [ 'class', 'foo' ], [ 'style', 'color' ] ] } );
	 * consumables.test( { attributes: [ [ 'title' ], [ 'name' ], [ 'class', 'foo' ], [ 'class', 'bar' ] ] } );
	 * ```
	 *
	 * @param consumables Object describing which parts of the element should be tested.
	 * @returns `true` when all tested items can be consumed, `null` when even one of the items
	 * was never marked for consumption and `false` when even one of the items was already consumed.
	 */ test(consumables) {
        // Check if name can be consumed.
        if (consumables.name && !this._canConsumeName) {
            return this._canConsumeName;
        }
        for (const [name, token] of consumables.attributes){
            const value = this._attributes.get(name);
            // Return null if attribute is not found.
            if (value === undefined) {
                return null;
            }
            // Already consumed.
            if (value === false) {
                return false;
            }
            // Simple attribute is not consumed so continue to next attribute.
            if (value === true) {
                continue;
            }
            if (!token) {
                // Tokenized attribute but token is not specified so check if all tokens are not consumed.
                for (const tokenValue of value.values()){
                    // Already consumed token.
                    if (!tokenValue) {
                        return false;
                    }
                }
            } else {
                const tokenValue = value.get(token);
                // Return null if token is not found.
                if (tokenValue === undefined) {
                    return null;
                }
                // Already consumed.
                if (!tokenValue) {
                    return false;
                }
            }
        }
        // Return true only if all can be consumed.
        return true;
    }
    /**
	 * Tests if parts of the {@link module:engine/view/element~ViewElement view element} can be consumed and consumes them if available.
	 * It returns `true` when all items included in method's call can be consumed, otherwise returns `false`.
	 *
	 * Element's name can be consumed:
	 *
	 * ```ts
	 * consumables.consume( { name: true } );
	 * ```
	 *
	 * Attributes classes and styles:
	 *
	 * ```ts
	 * consumables.consume( { attributes: [ [ 'title' ], [ 'class', 'foo' ], [ 'style', 'color' ] ] } );
	 * consumables.consume( { attributes: [ [ 'title' ], [ 'name' ], [ 'class', 'foo' ], [ 'class', 'bar' ] ] } );
	 * ```
	 *
	 * @param consumables Object describing which parts of the element should be consumed.
	 * @returns `true` when all tested items can be consumed and `false` when even one of the items could not be consumed.
	 */ consume(consumables) {
        if (!this.test(consumables)) {
            return false;
        }
        if (consumables.name) {
            this._canConsumeName = false;
        }
        for (const [name, token] of consumables.attributes){
            // `value` must be set, because `this.test()` returned `true`.
            const value = this._attributes.get(name);
            // Plain (not tokenized) not-consumed attribute.
            if (typeof value == 'boolean') {
                // Use Element API to collect related attributes.
                for (const [toConsume] of this.element._getConsumables(name, token).attributes){
                    this._attributes.set(toConsume, false);
                }
            } else if (!token) {
                // Tokenized attribute but token is not specified so consume all tokens.
                for (const token of value.keys()){
                    value.set(token, false);
                }
            } else {
                // Use Element API to collect related attribute tokens.
                for (const [, toConsume] of this.element._getConsumables(name, token).attributes){
                    value.set(toConsume, false);
                }
            }
        }
        return true;
    }
    /**
	 * Revert already consumed parts of {@link module:engine/view/element~ViewElement view Element}, so they can be consumed once again.
	 * Element's name can be reverted:
	 *
	 * ```ts
	 * consumables.revert( { name: true } );
	 * ```
	 *
	 * Attributes classes and styles:
	 *
	 * ```ts
	 * consumables.revert( { attributes: [ [ 'title' ], [ 'class', 'foo' ], [ 'style', 'color' ] ] } );
	 * consumables.revert( { attributes: [ [ 'title' ], [ 'name' ], [ 'class', 'foo' ], [ 'class', 'bar' ] ] } );
	 * ```
	 *
	 * @param consumables Object describing which parts of the element should be reverted.
	 */ revert(consumables) {
        if (consumables.name) {
            this._canConsumeName = true;
        }
        for (const [name, token] of consumables.attributes){
            const value = this._attributes.get(name);
            // Plain consumed attribute.
            if (value === false) {
                this._attributes.set(name, true);
                continue;
            }
            // Unknown attribute or not consumed.
            if (value === undefined || value === true) {
                continue;
            }
            if (!token) {
                // Tokenized attribute but token is not specified so revert all tokens.
                for (const token of value.keys()){
                    value.set(token, true);
                }
            } else {
                const tokenValue = value.get(token);
                if (tokenValue === false) {
                    value.set(token, true);
                }
            // Note that revert of consumed related styles is not handled.
            }
        }
    }
}
/**
 * Normalizes a {@link module:engine/conversion/viewconsumable~Consumables} or {@link module:engine/view/matcher~Match}
 * to a {@link module:engine/view/element~ViewNormalizedConsumables}.
 *
 * @internal
 */ function normalizeConsumables(consumables) {
    const attributes = [];
    if ('attributes' in consumables && consumables.attributes) {
        normalizeConsumablePart(attributes, consumables.attributes);
    }
    if ('classes' in consumables && consumables.classes) {
        normalizeConsumablePart(attributes, consumables.classes, 'class');
    }
    if ('styles' in consumables && consumables.styles) {
        normalizeConsumablePart(attributes, consumables.styles, 'style');
    }
    return {
        name: consumables.name || false,
        attributes
    };
}
/**
 * Normalizes a list of consumable attributes to a common tuple format.
 */ function normalizeConsumablePart(attributes, items, prefix) {
    if (typeof items == 'string') {
        attributes.push(prefix ? [
            prefix,
            items
        ] : [
            items
        ]);
        return;
    }
    for (const item of items){
        if (Array.isArray(item)) {
            attributes.push(item);
        } else {
            attributes.push(prefix ? [
                prefix,
                item
            ] : [
                item
            ]);
        }
    }
}

/**
 * View matcher class.
 * Instance of this class can be used to find {@link module:engine/view/element~ViewElement elements} that match given pattern.
 */ class Matcher {
    _patterns = [];
    /**
	 * Creates new instance of Matcher.
	 *
	 * @param pattern Match patterns. See {@link module:engine/view/matcher~Matcher#add add method} for more information.
	 */ constructor(...pattern){
        this.add(...pattern);
    }
    /**
	 * Adds pattern or patterns to matcher instance.
	 *
	 * ```ts
	 * // String.
	 * matcher.add( 'div' );
	 *
	 * // Regular expression.
	 * matcher.add( /^\w/ );
	 *
	 * // Single class.
	 * matcher.add( {
	 * 	classes: 'foobar'
	 * } );
	 * ```
	 *
	 * See {@link module:engine/view/matcher~MatcherPattern} for more examples.
	 *
	 * Multiple patterns can be added in one call:
	 *
	 * ```ts
	 * matcher.add( 'div', { classes: 'foobar' } );
	 * ```
	 *
	 * @param pattern Object describing pattern details. If string or regular expression
	 * is provided it will be used to match element's name. Pattern can be also provided in a form
	 * of a function - then this function will be called with each {@link module:engine/view/element~ViewElement element} as a parameter.
	 * Function's return value will be stored under `match` key of the object returned from
	 * {@link module:engine/view/matcher~Matcher#match match} or {@link module:engine/view/matcher~Matcher#matchAll matchAll} methods.
	 */ add(...pattern) {
        for (let item of pattern){
            // String or RegExp pattern is used as element's name.
            if (typeof item == 'string' || item instanceof RegExp) {
                item = {
                    name: item
                };
            }
            this._patterns.push(item);
        }
    }
    /**
	 * Matches elements for currently stored patterns. Returns match information about first found
	 * {@link module:engine/view/element~ViewElement element}, otherwise returns `null`.
	 *
	 * Example of returned object:
	 *
	 * ```ts
	 * {
	 * 	element: <instance of found element>,
	 * 	pattern: <pattern used to match found element>,
	 * 	match: {
	 * 		name: true,
	 * 		attributes: [
	 * 			[ 'title' ],
	 * 			[ 'href' ],
	 * 			[ 'class', 'foo' ],
	 * 			[ 'style', 'color' ],
	 * 			[ 'style', 'position' ]
	 * 		]
	 * 	}
	 * }
	 * ```
	 *
	 * You could use the `match` field from the above returned object as an input for the
	 * {@link module:engine/conversion/viewconsumable~ViewConsumable#test `ViewConsumable#test()`} and
	 * {@link module:engine/conversion/viewconsumable~ViewConsumable#consume `ViewConsumable#consume()`} methods.
	 *
	 * @see module:engine/view/matcher~Matcher#add
	 * @see module:engine/view/matcher~Matcher#matchAll
	 * @param element View element to match against stored patterns.
	 * @returns The match information about found element or `null`.
	 */ match(...element) {
        for (const singleElement of element){
            for (const pattern of this._patterns){
                const match = this._isElementMatching(singleElement, pattern);
                if (match) {
                    return {
                        element: singleElement,
                        pattern,
                        match
                    };
                }
            }
        }
        return null;
    }
    /**
	 * Matches elements for currently stored patterns. Returns array of match information with all found
	 * {@link module:engine/view/element~ViewElement elements}. If no element is found - returns `null`.
	 *
	 * @see module:engine/view/matcher~Matcher#add
	 * @see module:engine/view/matcher~Matcher#match
	 * @param element View element to match against stored patterns.
	 * @returns Array with match information about found elements or `null`. For more information
	 * see {@link module:engine/view/matcher~Matcher#match match method} description.
	 */ matchAll(...element) {
        const results = [];
        for (const singleElement of element){
            for (const pattern of this._patterns){
                const match = this._isElementMatching(singleElement, pattern);
                if (match) {
                    results.push({
                        element: singleElement,
                        pattern,
                        match
                    });
                }
            }
        }
        return results.length > 0 ? results : null;
    }
    /**
	 * Returns the name of the element to match if there is exactly one pattern added to the matcher instance
	 * and it matches element name defined by `string` (not `RegExp`). Otherwise, returns `null`.
	 *
	 * @returns Element name trying to match.
	 */ getElementName() {
        if (this._patterns.length !== 1) {
            return null;
        }
        const pattern = this._patterns[0];
        const name = pattern.name;
        return typeof pattern != 'function' && name && !(name instanceof RegExp) ? name : null;
    }
    /**
	 * Returns match information if {@link module:engine/view/element~ViewElement element} is matching provided pattern.
	 * If element cannot be matched to provided pattern - returns `null`.
	 *
	 * @returns Returns object with match information or null if element is not matching.
	 */ _isElementMatching(element, pattern) {
        // If pattern is provided as function - return result of that function;
        if (typeof pattern == 'function') {
            const match = pattern(element);
            // In some places we use Matcher with callback pattern that returns boolean.
            if (!match || typeof match != 'object') {
                return match;
            }
            return normalizeConsumables(match);
        }
        const match = {};
        // Check element's name.
        if (pattern.name) {
            match.name = matchName(pattern.name, element.name);
            if (!match.name) {
                return null;
            }
        }
        const attributesMatch = [];
        // Check element's attributes.
        if (pattern.attributes && !matchAttributes(pattern.attributes, element, attributesMatch)) {
            return null;
        }
        // Check element's classes.
        if (pattern.classes && !matchClasses(pattern.classes, element, attributesMatch)) {
            return null;
        }
        // Check element's styles.
        if (pattern.styles && !matchStyles(pattern.styles, element, attributesMatch)) {
            return null;
        }
        // Note the `attributesMatch` array is populated by the above calls.
        if (attributesMatch.length) {
            match.attributes = attributesMatch;
        }
        return match;
    }
}
/**
 * Returns true if the given `item` matches the pattern.
 *
 * @internal
 * @param pattern A pattern representing a key/value we want to match.
 * @param item An actual item key/value (e.g. `'src'`, `'background-color'`, `'ck-widget'`) we're testing against pattern.
 */ function isPatternMatched(pattern, item) {
    return pattern === true || pattern === item || pattern instanceof RegExp && !!String(item).match(pattern);
}
/**
 * Checks if name can be matched by provided pattern.
 *
 * @returns Returns `true` if name can be matched, `false` otherwise.
 */ function matchName(pattern, name) {
    // If pattern is provided as RegExp - test against this regexp.
    if (pattern instanceof RegExp) {
        return !!name.match(pattern);
    }
    return pattern === name;
}
/**
 * Bring all the possible pattern forms to an array of tuples where first item is a key, second is a value,
 * and third optional is a token value.
 *
 * Examples:
 *
 * Boolean pattern value:
 *
 * ```ts
 * true
 * ```
 *
 * to
 *
 * ```ts
 * [ [ true, true ] ]
 * ```
 *
 * Textual pattern value:
 *
 * ```ts
 * 'attribute-name-or-class-or-style'
 * ```
 *
 * to
 *
 * ```ts
 * [ [ 'attribute-name-or-class-or-style', true ] ]
 * ```
 *
 * Regular expression:
 *
 * ```ts
 * /^data-.*$/
 * ```
 *
 * to
 *
 * ```ts
 * [ [ /^data-.*$/, true ] ]
 * ```
 *
 * Objects (plain or with `key` and `value` specified explicitly):
 *
 * ```ts
 * {
 * 	src: /^https:.*$/
 * }
 * ```
 *
 * or
 *
 * ```ts
 * [ {
 * 	key: 'src',
 * 	value: /^https:.*$/
 * } ]
 * ```
 *
 * to:
 *
 * ```ts
 * [ [ 'src', /^https:.*$/ ] ]
 * ```
 *
 * @returns Returns an array of objects or null if provided patterns were not in an expected form.
 */ function normalizePatterns(patterns, prefix) {
    if (Array.isArray(patterns)) {
        return patterns.map((pattern)=>{
            if (typeof pattern !== 'object' || pattern instanceof RegExp) {
                return prefix ? [
                    prefix,
                    pattern,
                    true
                ] : [
                    pattern,
                    true
                ];
            }
            if (pattern.key === undefined || pattern.value === undefined) {
                // Documented at the end of matcher.js.
                logWarning('matcher-pattern-missing-key-or-value', pattern);
            }
            return prefix ? [
                prefix,
                pattern.key,
                pattern.value
            ] : [
                pattern.key,
                pattern.value
            ];
        });
    }
    if (typeof patterns !== 'object' || patterns instanceof RegExp) {
        return [
            prefix ? [
                prefix,
                patterns,
                true
            ] : [
                patterns,
                true
            ]
        ];
    }
    // Below we do what Object.entries() does, but faster
    const normalizedPatterns = [];
    for(const key in patterns){
        // Replace with Object.hasOwn() when we upgrade to es2022.
        if (Object.prototype.hasOwnProperty.call(patterns, key)) {
            normalizedPatterns.push(prefix ? [
                prefix,
                key,
                patterns[key]
            ] : [
                key,
                patterns[key]
            ]);
        }
    }
    return normalizedPatterns;
}
/**
 * Checks if attributes of provided element can be matched against provided patterns.
 *
 * @param patterns Object with information about attributes to match. Each key of the object will be
 * used as attribute name. Value of each key can be a string or regular expression to match against attribute value.
 * @param  element Element which attributes will be tested.
 * @param match An array to populate with matching tuples.
 * @returns Returns array with matched attribute names or `null` if no attributes were matched.
 */ function matchAttributes(patterns, element, match) {
    let excludeAttributes;
    // `style` and `class` attribute keys are deprecated. Only allow them in object pattern
    // for backward compatibility.
    if (typeof patterns === 'object' && !(patterns instanceof RegExp) && !Array.isArray(patterns)) {
        if (patterns.style !== undefined) {
            // Documented at the end of matcher.js.
            logWarning('matcher-pattern-deprecated-attributes-style-key', patterns);
        }
        if (patterns.class !== undefined) {
            // Documented at the end of matcher.js.
            logWarning('matcher-pattern-deprecated-attributes-class-key', patterns);
        }
    } else {
        excludeAttributes = [
            'class',
            'style'
        ];
    }
    return element._collectAttributesMatch(normalizePatterns(patterns), match, excludeAttributes);
}
/**
 * Checks if classes of provided element can be matched against provided patterns.
 *
 * @param patterns Array of strings or regular expressions to match against element's classes.
 * @param element Element which classes will be tested.
 * @param match An array to populate with matching tuples.
 * @returns Returns array with matched class names or `null` if no classes were matched.
 */ function matchClasses(patterns, element, match) {
    return element._collectAttributesMatch(normalizePatterns(patterns, 'class'), match);
}
/**
 * Checks if styles of provided element can be matched against provided patterns.
 *
 * @param patterns Object with information about styles to match. Each key of the object will be
 * used as style name. Value of each key can be a string or regular expression to match against style value.
 * @param element Element which styles will be tested.
 * @param match An array to populate with matching tuples.
 * @returns Returns array with matched style names or `null` if no styles were matched.
 */ function matchStyles(patterns, element, match) {
    return element._collectAttributesMatch(normalizePatterns(patterns, 'style'), match);
}
 /**
 * The key-value matcher pattern is missing key or value. Both must be present.
 * Refer the documentation: {@link module:engine/view/matcher~MatcherPattern}.
 *
 * @param pattern Pattern with missing properties.
 * @error matcher-pattern-missing-key-or-value
 */  /**
 * The key-value matcher pattern for `attributes` option is using deprecated `style` key.
 *
 * Use `styles` matcher pattern option instead:
 *
 * ```ts
 * // Instead of:
 * const pattern = {
 * 	attributes: {
 * 		key1: 'value1',
 * 		key2: 'value2',
 * 		style: /^border.*$/
 * 	}
 * }
 *
 * // Use:
 * const pattern = {
 * 	attributes: {
 * 		key1: 'value1',
 * 		key2: 'value2'
 * 	},
 * 	styles: /^border.*$/
 * }
 * ```
 *
 * Refer to the {@glink updating/guides/update-to-29##update-to-ckeditor-5-v2910 Migration to v29.1.0} guide
 * and {@link module:engine/view/matcher~MatcherPattern} documentation.
 *
 * @param pattern Pattern with missing properties.
 * @error matcher-pattern-deprecated-attributes-style-key
 */  /**
 * The key-value matcher pattern for `attributes` option is using deprecated `class` key.
 *
 * Use `classes` matcher pattern option instead:
 *
 * ```ts
 * // Instead of:
 * const pattern = {
 * 	attributes: {
 * 		key1: 'value1',
 * 		key2: 'value2',
 * 		class: 'foobar'
 * 	}
 * }
 *
 * // Use:
 * const pattern = {
 * 	attributes: {
 * 		key1: 'value1',
 * 		key2: 'value2'
 * 	},
 * 	classes: 'foobar'
 * }
 * ```
 *
 * Refer to the {@glink updating/guides/update-to-29##update-to-ckeditor-5-v2910 Migration to v29.1.0} guide
 * and the {@link module:engine/view/matcher~MatcherPattern} documentation.
 *
 * @param pattern Pattern with missing properties.
 * @error matcher-pattern-deprecated-attributes-class-key
 */

/**
 * Styles map. Allows handling (adding, removing, retrieving) a set of style rules (usually, of an element).
 */ class StylesMap {
    /**
	 * Keeps an internal representation of styles map. Normalized styles are kept as object tree to allow unified modification and
	 * value access model using lodash's get, set, unset, etc methods.
	 *
	 * When no style processor rules are defined it acts as simple key-value storage.
	 */ _styles;
    /**
	 * Cached list of style names for faster access.
	 */ _cachedStyleNames = null;
    /**
	 * Cached list of expanded style names for faster access.
	 */ _cachedExpandedStyleNames = null;
    /**
	 * An instance of the {@link module:engine/view/stylesmap~StylesProcessor}.
	 */ _styleProcessor;
    /**
	 * Creates Styles instance.
	 */ constructor(styleProcessor){
        this._styles = {};
        this._styleProcessor = styleProcessor;
    }
    /**
	 * Returns true if style map has no styles set.
	 */ get isEmpty() {
        const entries = Object.entries(this._styles);
        return !entries.length;
    }
    /**
	 * Number of styles defined.
	 */ get size() {
        if (this.isEmpty) {
            return 0;
        }
        return this.getStyleNames().length;
    }
    /**
	 * Set styles map to a new value.
	 *
	 * ```ts
	 * styles.setTo( 'border:1px solid blue;margin-top:1px;' );
	 * ```
	 */ setTo(inlineStyle) {
        this.clear();
        const parsedStyles = parseInlineStyles(inlineStyle);
        for (const [key, value] of parsedStyles){
            this._styleProcessor.toNormalizedForm(key, value, this._styles);
        }
        return this;
    }
    /**
	 * Checks if a given style is set.
	 *
	 * ```ts
	 * styles.setTo( 'margin-left:1px;' );
	 *
	 * styles.has( 'margin-left' );    // -> true
	 * styles.has( 'padding' );        // -> false
	 * ```
	 *
	 * **Note**: This check supports normalized style names.
	 *
	 * ```ts
	 * // Enable 'margin' shorthand processing:
	 * editor.data.addStyleProcessorRules( addMarginStylesRules );
	 *
	 * styles.setTo( 'margin:2px;' );
	 *
	 * styles.has( 'margin' );         // -> true
	 * styles.has( 'margin-top' );     // -> true
	 * styles.has( 'margin-left' );    // -> true
	 *
	 * styles.remove( 'margin-top' );
	 *
	 * styles.has( 'margin' );         // -> false
	 * styles.has( 'margin-top' );     // -> false
	 * styles.has( 'margin-left' );    // -> true
	 * ```
	 *
	 * @param name Style name.
	 */ has(name) {
        if (this.isEmpty) {
            return false;
        }
        const styles = this._styleProcessor.getReducedForm(name, this._styles);
        const propertyDescriptor = styles.find(([property])=>property === name);
        // Only return a value if it is set;
        return Array.isArray(propertyDescriptor);
    }
    set(nameOrObject, valueOrObject) {
        this._cachedStyleNames = null;
        this._cachedExpandedStyleNames = null;
        if (isObject(nameOrObject)) {
            for (const [key, value] of Object.entries(nameOrObject)){
                this._styleProcessor.toNormalizedForm(key, value, this._styles);
            }
        } else {
            this._styleProcessor.toNormalizedForm(nameOrObject, valueOrObject, this._styles);
        }
    }
    /**
	 * Removes given style.
	 *
	 * ```ts
	 * styles.setTo( 'background:#f00;margin-right:2px;' );
	 *
	 * styles.remove( 'background' );
	 *
	 * styles.toString();   // -> 'margin-right:2px;'
	 * ```
	 *
	 * ***Note**:* This method uses {@link module:engine/controller/datacontroller~DataController#addStyleProcessorRules
	 * enabled style processor rules} to normalize passed values.
	 *
	 * ```ts
	 * // Enable 'margin' shorthand processing:
	 * editor.data.addStyleProcessorRules( addMarginStylesRules );
	 *
	 * styles.setTo( 'margin:1px' );
	 *
	 * styles.remove( 'margin-top' );
	 * styles.remove( 'margin-right' );
	 *
	 * styles.toString(); // -> 'margin-bottom:1px;margin-left:1px;'
	 * ```
	 *
	 * @param names Style name or an array of names.
	 */ remove(names) {
        const normalizedStylesToRemove = {};
        for (const name of toArray(names)){
            // First, try the easy path, when the path reflects normalized styles structure.
            const path = toPath(name);
            const pathValue = get(this._styles, path);
            if (pathValue) {
                appendStyleValue(normalizedStylesToRemove, path, pathValue);
            } else {
                // Easy path did not work, so try to get the value from the styles map.
                const value = this.getAsString(name);
                if (value !== undefined) {
                    this._styleProcessor.toNormalizedForm(name, value, normalizedStylesToRemove);
                }
            }
        }
        if (Object.keys(normalizedStylesToRemove).length) {
            removeStyles(this._styles, normalizedStylesToRemove);
            this._cachedStyleNames = null;
            this._cachedExpandedStyleNames = null;
        }
    }
    /**
	 * Returns a normalized style object or a single value.
	 *
	 * ```ts
	 * // Enable 'margin' shorthand processing:
	 * editor.data.addStyleProcessorRules( addMarginStylesRules );
	 *
	 * const styles = new Styles();
	 * styles.setTo( 'margin:1px 2px 3em;' );
	 *
	 * styles.getNormalized( 'margin' );
	 * // will log:
	 * // {
	 * //     top: '1px',
	 * //     right: '2px',
	 * //     bottom: '3em',
	 * //     left: '2px'     // normalized value from margin shorthand
	 * // }
	 *
	 * styles.getNormalized( 'margin-left' ); // -> '2px'
	 * ```
	 *
	 * **Note**: This method will only return normalized styles if a style processor was defined.
	 *
	 * @param name Style name.
	 */ getNormalized(name) {
        return this._styleProcessor.getNormalized(name, this._styles);
    }
    /**
	 * Returns a normalized style string. Styles are sorted by name.
	 *
	 * ```ts
	 * styles.set( 'margin' , '1px' );
	 * styles.set( 'background', '#f00' );
	 *
	 * styles.toString(); // -> 'background:#f00;margin:1px;'
	 * ```
	 *
	 * **Note**: This method supports normalized styles if defined.
	 *
	 * ```ts
	 * // Enable 'margin' shorthand processing:
	 * editor.data.addStyleProcessorRules( addMarginStylesRules );
	 *
	 * styles.set( 'margin' , '1px' );
	 * styles.set( 'background', '#f00' );
	 * styles.remove( 'margin-top' );
	 * styles.remove( 'margin-right' );
	 *
	 * styles.toString(); // -> 'background:#f00;margin-bottom:1px;margin-left:1px;'
	 * ```
	 */ toString() {
        if (this.isEmpty) {
            return '';
        }
        return this.getStylesEntries().map((arr)=>arr.join(':')).sort().join(';') + ';';
    }
    /**
	 * Returns property as a value string or undefined if property is not set.
	 *
	 * ```ts
	 * // Enable 'margin' shorthand processing:
	 * editor.data.addStyleProcessorRules( addMarginStylesRules );
	 *
	 * const styles = new Styles();
	 * styles.setTo( 'margin:1px;' );
	 * styles.set( 'margin-bottom', '3em' );
	 *
	 * styles.getAsString( 'margin' ); // -> 'margin: 1px 1px 3em;'
	 * ```
	 *
	 * Note, however, that all sub-values must be set for the longhand property name to return a value:
	 *
	 * ```ts
	 * const styles = new Styles();
	 * styles.setTo( 'margin:1px;' );
	 * styles.remove( 'margin-bottom' );
	 *
	 * styles.getAsString( 'margin' ); // -> undefined
	 * ```
	 *
	 * In the above scenario, it is not possible to return a `margin` value, so `undefined` is returned.
	 * Instead, you should use:
	 *
	 * ```ts
	 * const styles = new Styles();
	 * styles.setTo( 'margin:1px;' );
	 * styles.remove( 'margin-bottom' );
	 *
	 * for ( const styleName of styles.getStyleNames() ) {
	 * 	console.log( styleName, styles.getAsString( styleName ) );
	 * }
	 * // 'margin-top', '1px'
	 * // 'margin-right', '1px'
	 * // 'margin-left', '1px'
	 * ```
	 *
	 * In general, it is recommend to iterate over style names like in the example above. This way, you will always get all
	 * the currently set style values. So, if all the 4 margin values would be set
	 * the for-of loop above would yield only `'margin'`, `'1px'`:
	 *
	 * ```ts
	 * const styles = new Styles();
	 * styles.setTo( 'margin:1px;' );
	 *
	 * for ( const styleName of styles.getStyleNames() ) {
	 * 	console.log( styleName, styles.getAsString( styleName ) );
	 * }
	 * // 'margin', '1px'
	 * ```
	 *
	 * **Note**: To get a normalized version of a longhand property use the {@link #getNormalized `#getNormalized()`} method.
	 */ getAsString(propertyName) {
        if (this.isEmpty) {
            return;
        }
        if (this._styles[propertyName] && !isObject(this._styles[propertyName])) {
            // Try return styles set directly - values that are not parsed.
            return this._styles[propertyName];
        }
        const styles = this._styleProcessor.getReducedForm(propertyName, this._styles);
        const propertyDescriptor = styles.find(([property])=>property === propertyName);
        // Only return a value if it is set;
        if (Array.isArray(propertyDescriptor)) {
            return propertyDescriptor[1];
        }
    }
    /**
	 * Returns all style properties names as they would appear when using {@link #toString `#toString()`}.
	 *
	 * When `expand` is set to true and there's a shorthand style property set, it will also return all equivalent styles:
	 *
	 * ```ts
	 * stylesMap.setTo( 'margin: 1em' )
	 * ```
	 *
	 * will be expanded to:
	 *
	 * ```ts
	 * [ 'margin', 'margin-top', 'margin-right', 'margin-bottom', 'margin-left' ]
	 * ```
	 *
	 * @param expand Expand shorthand style properties and all return equivalent style representations.
	 */ getStyleNames(expand = false) {
        if (this.isEmpty) {
            return [];
        }
        if (expand) {
            this._cachedExpandedStyleNames ||= this._styleProcessor.getStyleNames(this._styles);
            return this._cachedExpandedStyleNames;
        }
        this._cachedStyleNames ||= this.getStylesEntries().map(([key])=>key);
        return this._cachedStyleNames;
    }
    /**
	 * Alias for {@link #getStyleNames}.
	 */ keys() {
        return this.getStyleNames();
    }
    /**
	 * Removes all styles.
	 */ clear() {
        this._styles = {};
        this._cachedStyleNames = null;
        this._cachedExpandedStyleNames = null;
    }
    /**
	 * Returns `true` if both attributes have the same styles.
	 */ isSimilar(other) {
        if (this.size !== other.size) {
            return false;
        }
        for (const property of this.getStyleNames()){
            if (!other.has(property) || other.getAsString(property) !== this.getAsString(property)) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Returns normalized styles entries for further processing.
	 */ getStylesEntries() {
        const parsed = [];
        const keys = Object.keys(this._styles);
        for (const key of keys){
            parsed.push(...this._styleProcessor.getReducedForm(key, this._styles));
        }
        return parsed;
    }
    /**
	 * Clones the attribute value.
	 *
	 * @internal
	 */ _clone() {
        const clone = new this.constructor(this._styleProcessor);
        clone.set(this.getNormalized());
        return clone;
    }
    /**
	 * Used by the {@link module:engine/view/matcher~Matcher Matcher} to collect matching styles.
	 *
	 * @internal
	 * @param tokenPattern The matched style name pattern.
	 * @param valuePattern The matched style value pattern.
	 * @returns An array of matching tokens (style names).
	 */ _getTokensMatch(tokenPattern, valuePattern) {
        const match = [];
        for (const styleName of this.getStyleNames(true)){
            if (isPatternMatched(tokenPattern, styleName)) {
                if (valuePattern === true) {
                    match.push(styleName);
                    continue;
                }
                // For now, the reducers are not returning the full tree of properties.
                // Casting to string preserves the old behavior until the root cause is fixed.
                // More can be found in https://github.com/ckeditor/ckeditor5/issues/10399.
                const value = this.getAsString(styleName);
                if (isPatternMatched(valuePattern, value)) {
                    match.push(styleName);
                }
            }
        }
        return match.length ? match : undefined;
    }
    /**
	 * Returns a list of consumables for the attribute. This includes related styles.
	 *
	 * Could be filtered by the given style name.
	 *
	 * @internal
	 */ _getConsumables(name) {
        const result = [];
        if (name) {
            result.push(name);
            for (const relatedName of this._styleProcessor.getRelatedStyles(name)){
                result.push(relatedName);
            }
        } else {
            for (const name of this.getStyleNames()){
                for (const relatedName of this._styleProcessor.getRelatedStyles(name)){
                    result.push(relatedName);
                }
                result.push(name);
            }
        }
        return result;
    }
    /**
	 * Used by {@link module:engine/view/element~ViewElement#_canMergeAttributesFrom} to verify if the given attribute can be merged without
	 * conflicts into the attribute.
	 *
	 * This method is indirectly used by the {@link module:engine/view/downcastwriter~ViewDowncastWriter} while down-casting
	 * an {@link module:engine/view/attributeelement~ViewAttributeElement} to merge it with other ViewAttributeElement.
	 *
	 * @internal
	 */ _canMergeFrom(other) {
        for (const key of other.getStyleNames()){
            if (this.has(key) && this.getAsString(key) !== other.getAsString(key)) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Used by {@link module:engine/view/element~ViewElement#_mergeAttributesFrom} to merge a given attribute into the attribute.
	 *
	 * This method is indirectly used by the {@link module:engine/view/downcastwriter~ViewDowncastWriter} while down-casting
	 * an {@link module:engine/view/attributeelement~ViewAttributeElement} to merge it with other ViewAttributeElement.
	 *
	 * @internal
	 */ _mergeFrom(other) {
        for (const prop of other.getStyleNames()){
            if (!this.has(prop)) {
                this.set(prop, other.getAsString(prop));
            }
        }
    }
    /**
	 * Used by {@link module:engine/view/element~ViewElement#_canSubtractAttributesOf} to verify if the given attribute can be fully
	 * subtracted from the attribute.
	 *
	 * This method is indirectly used by the {@link module:engine/view/downcastwriter~ViewDowncastWriter} while down-casting
	 * an {@link module:engine/view/attributeelement~ViewAttributeElement} to unwrap the ViewAttributeElement.
	 *
	 * @internal
	 */ _isMatching(other) {
        for (const key of other.getStyleNames()){
            if (!this.has(key) || this.getAsString(key) !== other.getAsString(key)) {
                return false;
            }
        }
        return true;
    }
}
/**
 * Style processor is responsible for writing and reading a normalized styles object.
 */ class StylesProcessor {
    _normalizers;
    _extractors;
    _reducers;
    _consumables;
    /**
	 * Creates StylesProcessor instance.
	 *
	 * @internal
	 */ constructor(){
        this._normalizers = new Map();
        this._extractors = new Map();
        this._reducers = new Map();
        this._consumables = new Map();
    }
    /**
	 * Parse style string value to a normalized object and appends it to styles object.
	 *
	 * ```ts
	 * const styles = {};
	 *
	 * stylesProcessor.toNormalizedForm( 'margin', '1px', styles );
	 *
	 * // styles will consist: { margin: { top: '1px', right: '1px', bottom: '1px', left: '1px; } }
	 * ```
	 *
	 * **Note**: To define normalizer callbacks use {@link #setNormalizer}.
	 *
	 * @param name Name of style property.
	 * @param propertyValue Value of style property.
	 * @param styles Object holding normalized styles.
	 */ toNormalizedForm(name, propertyValue, styles) {
        if (isObject(propertyValue)) {
            appendStyleValue(styles, toPath(name), propertyValue);
            return;
        }
        if (this._normalizers.has(name)) {
            const normalizer = this._normalizers.get(name);
            const { path, value } = normalizer(propertyValue);
            appendStyleValue(styles, path, value);
        } else {
            appendStyleValue(styles, name, propertyValue);
        }
    }
    /**
	 * Returns a normalized version of a style property.
	 *
	 * ```ts
	 * const styles = {
	 * 	margin: { top: '1px', right: '1px', bottom: '1px', left: '1px; },
	 * 	background: { color: '#f00' }
	 * };
	 *
	 * stylesProcessor.getNormalized( 'background' );
	 * // will return: { color: '#f00' }
	 *
	 * stylesProcessor.getNormalized( 'margin-top' );
	 * // will return: '1px'
	 * ```
	 *
	 * **Note**: In some cases extracting single value requires defining an extractor callback {@link #setExtractor}.
	 *
	 * @param name Name of style property.
	 * @param styles Object holding normalized styles.
	 */ getNormalized(name, styles) {
        if (!name) {
            return merge({}, styles);
        }
        // Might be empty string.
        if (styles[name] !== undefined) {
            return styles[name];
        }
        if (this._extractors.has(name)) {
            const extractor = this._extractors.get(name);
            if (typeof extractor === 'string') {
                return get(styles, extractor);
            }
            const value = extractor(name, styles);
            if (value) {
                return value;
            }
        }
        return get(styles, toPath(name));
    }
    /**
	 * Returns a reduced form of style property form normalized object.
	 *
	 * For default margin reducer, the below code:
	 *
	 * ```ts
	 * stylesProcessor.getReducedForm( 'margin', {
	 * 	margin: { top: '1px', right: '1px', bottom: '2px', left: '1px; }
	 * } );
	 * ```
	 *
	 * will return:
	 *
	 * ```ts
	 * [
	 * 	[ 'margin', '1px 1px 2px' ]
	 * ]
	 * ```
	 *
	 * because it might be represented as a shorthand 'margin' value. However if one of margin long hand values is missing it should return:
	 *
	 * ```ts
	 * [
	 * 	[ 'margin-top', '1px' ],
	 * 	[ 'margin-right', '1px' ],
	 * 	[ 'margin-bottom', '2px' ]
	 * 	// the 'left' value is missing - cannot use 'margin' shorthand.
	 * ]
	 * ```
	 *
	 * **Note**: To define reducer callbacks use {@link #setReducer}.
	 *
	 * @param name Name of style property.
	 */ getReducedForm(name, styles) {
        const normalizedValue = this.getNormalized(name, styles);
        // Might be empty string.
        if (normalizedValue === undefined) {
            return [];
        }
        if (this._reducers.has(name)) {
            const reducer = this._reducers.get(name);
            return reducer(normalizedValue);
        }
        return [
            [
                name,
                normalizedValue
            ]
        ];
    }
    /**
	 * Return all style properties. Also expand shorthand properties (e.g. `margin`, `background`) if respective extractor is available.
	 *
	 * @param styles Object holding normalized styles.
	 */ getStyleNames(styles) {
        const styleNamesKeysSet = new Set();
        // Find all extractable styles that have a value.
        for (const name of this._consumables.keys()){
            const style = this.getNormalized(name, styles);
            if (style && (typeof style != 'object' || Object.keys(style).length)) {
                styleNamesKeysSet.add(name);
            }
        }
        // For simple styles (for example `color`) we don't have a map of those styles
        // but they are 1 to 1 with normalized object keys.
        for (const name of Object.keys(styles)){
            styleNamesKeysSet.add(name);
        }
        return Array.from(styleNamesKeysSet);
    }
    /**
	 * Returns related style names.
	 *
	 * ```ts
	 * stylesProcessor.getRelatedStyles( 'margin' );
	 * // will return: [ 'margin-top', 'margin-right', 'margin-bottom', 'margin-left' ];
	 *
	 * stylesProcessor.getRelatedStyles( 'margin-top' );
	 * // will return: [ 'margin' ];
	 * ```
	 *
	 * **Note**: To define new style relations load an existing style processor or use
	 * {@link module:engine/view/stylesmap~StylesProcessor#setStyleRelation `StylesProcessor.setStyleRelation()`}.
	 */ getRelatedStyles(name) {
        return this._consumables.get(name) || [];
    }
    /**
	 * Adds a normalizer method for a style property.
	 *
	 * A normalizer returns describing how the value should be normalized.
	 *
	 * For instance 'margin' style is a shorthand for four margin values:
	 *
	 * - 'margin-top'
	 * - 'margin-right'
	 * - 'margin-bottom'
	 * - 'margin-left'
	 *
	 * and can be written in various ways if some values are equal to others. For instance `'margin: 1px 2em;'` is a shorthand for
	 * `'margin-top: 1px;margin-right: 2em;margin-bottom: 1px;margin-left: 2em'`.
	 *
	 * A normalizer should parse various margin notations as a single object:
	 *
	 * ```ts
	 * const styles = {
	 * 	margin: {
	 * 		top: '1px',
	 * 		right: '2em',
	 * 		bottom: '1px',
	 * 		left: '2em'
	 * 	}
	 * };
	 * ```
	 *
	 * Thus a normalizer for 'margin' style should return an object defining style path and value to store:
	 *
	 * ```ts
	 * const returnValue = {
	 * 	path: 'margin',
	 * 	value: {
	 * 		top: '1px',
	 * 		right: '2em',
	 * 		bottom: '1px',
	 * 		left: '2em'
	 * 	}
	 * };
	 * ```
	 *
	 * Additionally to fully support all margin notations there should be also defined 4 normalizers for longhand margin notations. Below
	 * is an example for 'margin-top' style property normalizer:
	 *
	 * ```ts
	 * stylesProcessor.setNormalizer( 'margin-top', valueString => {
	 * 	return {
	 * 		path: 'margin.top',
	 * 		value: valueString
	 * 	}
	 * } );
	 * ```
	 */ setNormalizer(name, callback) {
        this._normalizers.set(name, callback);
    }
    /**
	 * Adds a extractor callback for a style property.
	 *
	 * Most normalized style values are stored as one level objects. It is assumed that `'margin-top'` style will be stored as:
	 *
	 * ```ts
	 * const styles = {
	 * 	margin: {
	 * 		top: 'value'
	 * 	}
	 * }
	 * ```
	 *
	 * However, some styles can have conflicting notations and thus it might be harder to extract a style value from shorthand. For instance
	 * the 'border-top-style' can be defined using `'border-top:solid'`, `'border-style:solid none none none'` or by `'border:solid'`
	 * shorthands. The default border styles processors stores styles as:
	 *
	 * ```ts
	 * const styles = {
	 * 	border: {
	 * 		style: {
	 * 			top: 'solid'
	 * 		}
	 * 	}
	 * }
	 * ```
	 *
	 * as it is better to modify border style independently from other values. On the other part the output of the border might be
	 * desired as `border-top`, `border-left`, etc notation.
	 *
	 * In the above example an extractor should return a side border value that combines style, color and width:
	 *
	 * ```ts
	 * styleProcessor.setExtractor( 'border-top', styles => {
	 * 	return {
	 * 		color: styles.border.color.top,
	 * 		style: styles.border.style.top,
	 * 		width: styles.border.width.top
	 * 	}
	 * } );
	 * ```
	 *
	 * @param callbackOrPath Callback that return a requested value or path string for single values.
	 */ setExtractor(name, callbackOrPath) {
        this._extractors.set(name, callbackOrPath);
    }
    /**
	 * Adds a reducer callback for a style property.
	 *
	 * Reducer returns a minimal notation for given style name. For longhand properties it is not required to write a reducer as
	 * by default the direct value from style path is taken.
	 *
	 * For shorthand styles a reducer should return minimal style notation either by returning single name-value tuple or multiple tuples
	 * if a shorthand cannot be used. For instance for a margin shorthand a reducer might return:
	 *
	 * ```ts
	 * const marginShortHandTuple = [
	 * 	[ 'margin', '1px 1px 2px' ]
	 * ];
	 * ```
	 *
	 * or a longhand tuples for defined values:
	 *
	 * ```ts
	 * // Considering margin.bottom and margin.left are undefined.
	 * const marginLonghandsTuples = [
	 * 	[ 'margin-top', '1px' ],
	 * 	[ 'margin-right', '1px' ]
	 * ];
	 * ```
	 *
	 * A reducer obtains a normalized style value:
	 *
	 * ```ts
	 * // Simplified reducer that always outputs 4 values which are always present:
	 * stylesProcessor.setReducer( 'margin', margin => {
	 * 	return [
	 * 		[ 'margin', `${ margin.top } ${ margin.right } ${ margin.bottom } ${ margin.left }` ]
	 * 	]
	 * } );
	 * ```
	 */ setReducer(name, callback) {
        this._reducers.set(name, callback);
    }
    /**
	 * Defines a style shorthand relation to other style notations.
	 *
	 * ```ts
	 * stylesProcessor.setStyleRelation( 'margin', [
	 * 	'margin-top',
	 * 	'margin-right',
	 * 	'margin-bottom',
	 * 	'margin-left'
	 * ] );
	 * ```
	 *
	 * This enables expanding of style names for shorthands. For instance, if defined,
	 * {@link module:engine/conversion/viewconsumable~ViewConsumable view consumable} items are automatically created
	 * for long-hand margin style notation alongside the `'margin'` item.
	 *
	 * This means that when an element being converted has a style `margin`, a converter for `margin-left` will work just
	 * fine since the view consumable will contain a consumable `margin-left` item (thanks to the relation) and
	 * `element.getStyle( 'margin-left' )` will work as well assuming that the style processor was correctly configured.
	 * However, once `margin-left` is consumed, `margin` will not be consumable anymore.
	 */ setStyleRelation(shorthandName, styleNames) {
        this._mapStyleNames(shorthandName, styleNames);
        for (const alsoName of styleNames){
            this._mapStyleNames(alsoName, [
                shorthandName
            ]);
        }
    }
    /**
	 * Set two-way binding of style names.
	 */ _mapStyleNames(name, styleNames) {
        if (!this._consumables.has(name)) {
            this._consumables.set(name, []);
        }
        this._consumables.get(name).push(...styleNames);
    }
}
/**
 * Parses inline styles and puts property - value pairs into styles map.
 *
 * @param stylesString Styles to parse.
 * @returns Map of parsed properties and values.
 */ function parseInlineStyles(stylesString) {
    // `null` if no quote was found in input string or last found quote was a closing quote. See below.
    let quoteType = null;
    let propertyNameStart = 0;
    let propertyValueStart = 0;
    let propertyName = null;
    const stylesMap = new Map();
    // Do not set anything if input string is empty.
    if (stylesString === '') {
        return stylesMap;
    }
    // Fix inline styles that do not end with `;` so they are compatible with algorithm below.
    if (stylesString.charAt(stylesString.length - 1) != ';') {
        stylesString = stylesString + ';';
    }
    // Seek the whole string for "special characters".
    for(let i = 0; i < stylesString.length; i++){
        const char = stylesString.charAt(i);
        if (quoteType === null) {
            // No quote found yet or last found quote was a closing quote.
            switch(char){
                case ':':
                    // Most of time colon means that property name just ended.
                    // Sometimes however `:` is found inside property value (for example in background image url).
                    if (!propertyName) {
                        // Treat this as end of property only if property name is not already saved.
                        // Save property name.
                        propertyName = stylesString.substr(propertyNameStart, i - propertyNameStart);
                        // Save this point as the start of property value.
                        propertyValueStart = i + 1;
                    }
                    break;
                case '"':
                case '\'':
                    // Opening quote found (this is an opening quote, because `quoteType` is `null`).
                    quoteType = char;
                    break;
                case ';':
                    {
                        // Property value just ended.
                        // Use previously stored property value start to obtain property value.
                        const propertyValue = stylesString.substr(propertyValueStart, i - propertyValueStart);
                        if (propertyName) {
                            // Save parsed part.
                            stylesMap.set(propertyName.trim(), propertyValue.trim());
                        }
                        propertyName = null;
                        // Save this point as property name start. Property name starts immediately after previous property value ends.
                        propertyNameStart = i + 1;
                        break;
                    }
            }
        } else if (char === quoteType) {
            // If a quote char is found and it is a closing quote, mark this fact by `null`-ing `quoteType`.
            quoteType = null;
        }
    }
    return stylesMap;
}
/**
 * Return lodash compatible path from style name.
 */ function toPath(name) {
    return name.replace('-', '.');
}
/**
 * Appends style definition to the styles object.
 */ function appendStyleValue(stylesObject, nameOrPath, valueOrObject) {
    let valueToSet = valueOrObject;
    if (isObject(valueOrObject)) {
        valueToSet = merge({}, get(stylesObject, nameOrPath), valueOrObject);
    }
    set(stylesObject, nameOrPath, valueToSet);
}
/**
 * Modifies the `styles` deeply nested object by removing properties defined in `toRemove`.
 */ function removeStyles(styles, toRemove) {
    for (const key of Object.keys(toRemove)){
        if (styles[key] !== null && !Array.isArray(styles[key]) && typeof styles[key] == 'object' && typeof toRemove[key] == 'object') {
            removeStyles(styles[key], toRemove[key]);
            if (!Object.keys(styles[key]).length) {
                delete styles[key];
            }
        } else {
            delete styles[key];
        }
    }
}

/**
 * Token list. Allows handling (adding, removing, retrieving) a set of tokens (for example class names).
 */ class ViewTokenList {
    /**
	 * The set of tokens.
	 */ _set = new Set();
    /**
	 * Returns true if token list has no tokens set.
	 */ get isEmpty() {
        return this._set.size == 0;
    }
    /**
	 * Number of tokens.
	 */ get size() {
        return this._set.size;
    }
    /**
	 * Checks if a given token is set.
	 */ has(name) {
        return this._set.has(name);
    }
    /**
	 * Returns all tokens.
	 */ keys() {
        return Array.from(this._set.keys());
    }
    /**
	 * Resets the value to the given one.
	 */ setTo(value) {
        this.clear();
        for (const token of value.split(/\s+/)){
            if (token) {
                this._set.add(token);
            }
        }
        return this;
    }
    /**
	 * Sets a given token without affecting other tokens.
	 */ set(tokens) {
        for (const token of toArray(tokens)){
            if (token) {
                this._set.add(token);
            }
        }
    }
    /**
	 * Removes given token.
	 */ remove(tokens) {
        for (const token of toArray(tokens)){
            this._set.delete(token);
        }
    }
    /**
	 * Removes all tokens.
	 */ clear() {
        this._set.clear();
    }
    /**
	 * Returns a normalized tokens string.
	 */ toString() {
        return Array.from(this._set).join(' ');
    }
    /**
	 * Returns `true` if both attributes have the same tokens.
	 */ isSimilar(other) {
        if (this.size !== other.size) {
            return false;
        }
        for (const token of this.keys()){
            if (!other.has(token)) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Clones the attribute value.
	 *
	 * @internal
	 */ _clone() {
        const clone = new this.constructor();
        clone._set = new Set(this._set);
        return clone;
    }
    /**
	 * Used by the {@link module:engine/view/matcher~Matcher Matcher} to collect matching attribute tokens.
	 *
	 * @internal
	 * @param tokenPattern The matched token name pattern.
	 * @returns An array of matching tokens.
	 */ _getTokensMatch(tokenPattern) {
        const match = [];
        if (tokenPattern === true) {
            for (const token of this._set.keys()){
                match.push(token);
            }
            return match;
        }
        if (typeof tokenPattern == 'string') {
            for (const token of tokenPattern.split(/\s+/)){
                if (this._set.has(token)) {
                    match.push(token);
                } else {
                    return undefined;
                }
            }
            return match;
        }
        for (const token of this._set.keys()){
            if (token.match(tokenPattern)) {
                match.push(token);
            }
        }
        return match.length ? match : undefined;
    }
    /**
	 * Returns a list of consumables for the attribute.
	 *
	 * Could be filtered by the given token name.
	 *
	 * @internal
	 */ _getConsumables(name) {
        return name ? [
            name
        ] : this.keys();
    }
    /**
	 * Used by {@link module:engine/view/element~ViewElement#_canMergeAttributesFrom} to verify if the given attribute
	 * can be merged without conflicts into the attribute.
	 *
	 * This method is indirectly used by the {@link module:engine/view/downcastwriter~ViewDowncastWriter} while downcasting
	 * an {@link module:engine/view/attributeelement~ViewAttributeElement} to merge it with other `AttributeElement`.
	 *
	 * @internal
	 */ _canMergeFrom() {
        return true;
    }
    /**
	 * Used by {@link module:engine/view/element~ViewElement#_mergeAttributesFrom} to merge a given attribute into the attribute.
	 *
	 * This method is indirectly used by the {@link module:engine/view/downcastwriter~ViewDowncastWriter} while down-casting
	 * an {@link module:engine/view/attributeelement~ViewAttributeElement} to merge it with other ViewAttributeElement.
	 *
	 * @internal
	 */ _mergeFrom(other) {
        for (const token of other._set.keys()){
            if (!this._set.has(token)) {
                this._set.add(token);
            }
        }
    }
    /**
	 * Used by {@link module:engine/view/element~ViewElement#_canSubtractAttributesOf} to verify if the given attribute
	 * can be fully subtracted from the attribute.
	 *
	 * This method is indirectly used by the {@link module:engine/view/downcastwriter~ViewDowncastWriter} while down-casting
	 * an {@link module:engine/view/attributeelement~ViewAttributeElement} to unwrap the ViewAttributeElement.
	 *
	 * @internal
	 */ _isMatching(other) {
        for (const name of other._set.keys()){
            if (!this._set.has(name)) {
                return false;
            }
        }
        return true;
    }
}

// @if CK_DEBUG_ENGINE // const { convertMapToTags } = require( '../dev-utils/utils' );
/**
 * View element.
 *
 * The editing engine does not define a fixed semantics of its elements (it is "DTD-free").
 * This is why the type of the {@link module:engine/view/element~ViewElement} need to
 * be defined by the feature developer. When creating an element you should use one of the following methods:
 *
 * * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createContainerElement `downcastWriter#createContainerElement()`}
 * in order to create a {@link module:engine/view/containerelement~ViewContainerElement},
 * * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createAttributeElement `downcastWriter#createAttributeElement()`}
 * in order to create a {@link module:engine/view/attributeelement~ViewAttributeElement},
 * * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createEmptyElement `downcastWriter#createEmptyElement()`}
 * in order to create a {@link module:engine/view/emptyelement~ViewEmptyElement}.
 * * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createUIElement `downcastWriter#createUIElement()`}
 * in order to create a {@link module:engine/view/uielement~ViewUIElement}.
 * * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createEditableElement `downcastWriter#createEditableElement()`}
 * in order to create a {@link module:engine/view/editableelement~ViewEditableElement}.
 *
 * Note that for view elements which are not created from the model, like elements from mutations, paste or
 * {@link module:engine/controller/datacontroller~DataController#set data.set} it is not possible to define the type of the element.
 * In such cases the {@link module:engine/view/upcastwriter~ViewUpcastWriter#createElement `UpcastWriter#createElement()`} method
 * should be used to create generic view elements.
 */ class ViewElement extends ViewNode {
    /**
	 * Name of the element.
	 */ name;
    /**
	 * A list of attribute names that should be rendered in the editing pipeline even though filtering mechanisms
	 * implemented in the {@link module:engine/view/domconverter~ViewDomConverter} (for instance,
	 * {@link module:engine/view/domconverter~ViewDomConverter#shouldRenderAttribute}) would filter them out.
	 *
	 * These attributes can be specified as an option when the element is created by
	 * the {@link module:engine/view/downcastwriter~ViewDowncastWriter}. To check whether an unsafe an attribute should
	 * be permitted, use the {@link #shouldRenderUnsafeAttribute} method.
	 *
	 * @internal
	 */ _unsafeAttributesToRender = [];
    /**
	 * Map of attributes, where attributes names are keys and attributes values are values.
	 */ _attrs;
    /**
	 * Array of child nodes.
	 */ _children;
    /**
	 * Map of custom properties.
	 * Custom properties can be added to element instance, will be cloned but not rendered into DOM.
	 */ _customProperties = new Map();
    /**
	 * Set of classes associated with element instance.
	 *
	 * Note that this is just an alias for `this._attrs.get( 'class' );`
	 */ get _classes() {
        return this._attrs.get('class');
    }
    /**
	 * Normalized styles.
	 *
	 * Note that this is just an alias for `this._attrs.get( 'style' );`
	 */ get _styles() {
        return this._attrs.get('style');
    }
    /**
	 * Creates a view element.
	 *
	 * Attributes can be passed in various formats:
	 *
	 * ```ts
	 * new Element( viewDocument, 'div', { class: 'editor', contentEditable: 'true' } ); // object
	 * new Element( viewDocument, 'div', [ [ 'class', 'editor' ], [ 'contentEditable', 'true' ] ] ); // map-like iterator
	 * new Element( viewDocument, 'div', mapOfAttributes ); // map
	 * ```
	 *
	 * @internal
	 * @param document The document instance to which this element belongs.
	 * @param name Node name.
	 * @param attrs Collection of attributes.
	 * @param children A list of nodes to be inserted into created element.
	 */ constructor(document, name, attrs, children){
        super(document);
        this.name = name;
        this._attrs = this._parseAttributes(attrs);
        this._children = [];
        if (children) {
            this._insertChild(0, children);
        }
    }
    /**
	 * Number of element's children.
	 */ get childCount() {
        return this._children.length;
    }
    /**
	 * Is `true` if there are no nodes inside this element, `false` otherwise.
	 */ get isEmpty() {
        return this._children.length === 0;
    }
    /**
	 * Gets child at the given index.
	 *
	 * @param index Index of child.
	 * @returns Child node.
	 */ getChild(index) {
        return this._children[index];
    }
    /**
	 * Gets index of the given child node. Returns `-1` if child node is not found.
	 *
	 * @param node Child node.
	 * @returns Index of the child node.
	 */ getChildIndex(node) {
        return this._children.indexOf(node);
    }
    /**
	 * Gets child nodes iterator.
	 *
	 * @returns Child nodes iterator.
	 */ getChildren() {
        return this._children[Symbol.iterator]();
    }
    /**
	 * Returns an iterator that contains the keys for attributes. Order of inserting attributes is not preserved.
	 *
	 * @returns Keys for attributes.
	 */ *getAttributeKeys() {
        // This is yielded in this specific order to maintain backward compatibility of data.
        // Otherwise, we could simply just have the `for` loop only inside this method.
        if (this._classes) {
            yield 'class';
        }
        if (this._styles) {
            yield 'style';
        }
        for (const key of this._attrs.keys()){
            if (key != 'class' && key != 'style') {
                yield key;
            }
        }
    }
    /**
	 * Returns iterator that iterates over this element's attributes.
	 *
	 * Attributes are returned as arrays containing two items. First one is attribute key and second is attribute value.
	 * This format is accepted by native `Map` object and also can be passed in `Node` constructor.
	 */ *getAttributes() {
        for (const [name, value] of this._attrs.entries()){
            yield [
                name,
                String(value)
            ];
        }
    }
    /**
	 * Gets attribute by key. If attribute is not present - returns undefined.
	 *
	 * @param key Attribute key.
	 * @returns Attribute value.
	 */ getAttribute(key) {
        return this._attrs.has(key) ? String(this._attrs.get(key)) : undefined;
    }
    /**
	 * Returns a boolean indicating whether an attribute with the specified key exists in the element.
	 *
	 * @param key Attribute key.
	 * @returns `true` if attribute with the specified key exists in the element, `false` otherwise.
	 */ hasAttribute(key, token) {
        if (!this._attrs.has(key)) {
            return false;
        }
        if (token !== undefined) {
            if (usesStylesMap(this.name, key) || usesTokenList(this.name, key)) {
                return this._attrs.get(key).has(token);
            } else {
                return this._attrs.get(key) === token;
            }
        }
        return true;
    }
    /**
	 * Checks if this element is similar to other element.
	 * Both elements should have the same name and attributes to be considered as similar. Two similar elements
	 * can contain different set of children nodes.
	 */ isSimilar(otherElement) {
        if (!(otherElement instanceof ViewElement)) {
            return false;
        }
        // If exactly the same Element is provided - return true immediately.
        if (this === otherElement) {
            return true;
        }
        // Check element name.
        if (this.name != otherElement.name) {
            return false;
        }
        // Check number of attributes, classes and styles.
        if (this._attrs.size !== otherElement._attrs.size) {
            return false;
        }
        // Check if attributes are the same.
        for (const [key, value] of this._attrs){
            const otherValue = otherElement._attrs.get(key);
            if (otherValue === undefined) {
                return false;
            }
            if (typeof value == 'string' || typeof otherValue == 'string') {
                if (otherValue !== value) {
                    return false;
                }
            } else if (!value.isSimilar(otherValue)) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Returns true if class is present.
	 * If more then one class is provided - returns true only when all classes are present.
	 *
	 * ```ts
	 * element.hasClass( 'foo' ); // Returns true if 'foo' class is present.
	 * element.hasClass( 'foo', 'bar' ); // Returns true if 'foo' and 'bar' classes are both present.
	 * ```
	 */ hasClass(...className) {
        for (const name of className){
            if (!this._classes || !this._classes.has(name)) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Returns iterator that contains all class names.
	 */ getClassNames() {
        const array = this._classes ? this._classes.keys() : [];
        // This is overcomplicated because we need to be backward compatible for use cases when iterator is expected.
        const iterator = array[Symbol.iterator]();
        return Object.assign(array, {
            next: iterator.next.bind(iterator)
        });
    }
    /**
	 * Returns style value for the given property name.
	 * If the style does not exist `undefined` is returned.
	 *
	 * **Note**: This method can work with normalized style names if
	 * {@link module:engine/controller/datacontroller~DataController#addStyleProcessorRules a particular style processor rule is enabled}.
	 * See {@link module:engine/view/stylesmap~StylesMap#getAsString `StylesMap#getAsString()`} for details.
	 *
	 * For an element with style set to `'margin:1px'`:
	 *
	 * ```ts
	 * // Enable 'margin' shorthand processing:
	 * editor.data.addStyleProcessorRules( addMarginStylesRules );
	 *
	 * const element = view.change( writer => {
	 * 	const element = writer.createElement();
	 * 	writer.setStyle( 'margin', '1px' );
	 * 	writer.setStyle( 'margin-bottom', '3em' );
	 *
	 * 	return element;
	 * } );
	 *
	 * element.getStyle( 'margin' ); // -> 'margin: 1px 1px 3em;'
	 * ```
	 */ getStyle(property) {
        return this._styles && this._styles.getAsString(property);
    }
    /**
	 * Returns a normalized style object or single style value.
	 *
	 * For an element with style set to: margin:1px 2px 3em;
	 *
	 * ```ts
	 * element.getNormalizedStyle( 'margin' ) );
	 * ```
	 *
	 * will return:
	 *
	 * ```ts
	 * {
	 * 	top: '1px',
	 * 	right: '2px',
	 * 	bottom: '3em',
	 * 	left: '2px'    // a normalized value from margin shorthand
	 * }
	 * ```
	 *
	 * and reading for single style value:
	 *
	 * ```ts
	 * styles.getNormalizedStyle( 'margin-left' );
	 * ```
	 *
	 * Will return a `2px` string.
	 *
	 * **Note**: This method will return normalized values only if
	 * {@link module:engine/controller/datacontroller~DataController#addStyleProcessorRules a particular style processor rule is enabled}.
	 * See {@link module:engine/view/stylesmap~StylesMap#getNormalized `StylesMap#getNormalized()`} for details.
	 *
	 * @param property Name of CSS property
	 */ getNormalizedStyle(property) {
        return this._styles && this._styles.getNormalized(property);
    }
    /**
	 * Returns an array that contains all style names.
	 *
	 * @param expand Expand shorthand style properties and return all equivalent style representations.
	 */ getStyleNames(expand) {
        return this._styles ? this._styles.getStyleNames(expand) : [];
    }
    /**
	 * Returns true if style keys are present.
	 * If more then one style property is provided - returns true only when all properties are present.
	 *
	 * ```ts
	 * element.hasStyle( 'color' ); // Returns true if 'border-top' style is present.
	 * element.hasStyle( 'color', 'border-top' ); // Returns true if 'color' and 'border-top' styles are both present.
	 * ```
	 */ hasStyle(...property) {
        for (const name of property){
            if (!this._styles || !this._styles.has(name)) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Returns ancestor element that match specified pattern.
	 * Provided patterns should be compatible with {@link module:engine/view/matcher~Matcher Matcher} as it is used internally.
	 *
	 * @see module:engine/view/matcher~Matcher
	 * @param patterns Patterns used to match correct ancestor. See {@link module:engine/view/matcher~Matcher}.
	 * @returns Found element or `null` if no matching ancestor was found.
	 */ findAncestor(...patterns) {
        const matcher = new Matcher(...patterns);
        let parent = this.parent;
        while(parent && !parent.is('documentFragment')){
            if (matcher.match(parent)) {
                return parent;
            }
            parent = parent.parent;
        }
        return null;
    }
    /**
	 * Returns the custom property value for the given key.
	 */ getCustomProperty(key) {
        return this._customProperties.get(key);
    }
    /**
	 * Returns an iterator which iterates over this element's custom properties.
	 * Iterator provides `[ key, value ]` pairs for each stored property.
	 */ *getCustomProperties() {
        yield* this._customProperties.entries();
    }
    /**
	 * Returns identity string based on element's name, styles, classes and other attributes.
	 * Two elements that {@link #isSimilar are similar} will have same identity string.
	 * It has the following format:
	 *
	 * ```ts
	 * 'name class="class1,class2" style="style1:value1;style2:value2" attr1="val1" attr2="val2"'
	 * ```
 	 *
	 * For example:
	 *
	 * ```ts
	 * const element = writer.createContainerElement( 'foo', {
	 * 	banana: '10',
	 * 	apple: '20',
	 * 	style: 'color: red; border-color: white;',
	 * 	class: 'baz'
	 * } );
	 *
	 * // returns 'foo class="baz" style="border-color:white;color:red" apple="20" banana="10"'
	 * element.getIdentity();
	 * ```
	 *
	 * **Note**: Classes, styles and other attributes are sorted alphabetically.
	 */ getIdentity() {
        const classes = this._classes ? this._classes.keys().sort().join(',') : '';
        const styles = this._styles && String(this._styles);
        const attributes = Array.from(this._attrs).filter(([key])=>key != 'style' && key != 'class').map((i)=>`${i[0]}="${i[1]}"`).sort().join(' ');
        return this.name + (classes == '' ? '' : ` class="${classes}"`) + (!styles ? '' : ` style="${styles}"`) + (attributes == '' ? '' : ` ${attributes}`);
    }
    /**
	 * Decides whether an unsafe attribute is whitelisted and should be rendered in the editing pipeline even though filtering mechanisms
	 * like {@link module:engine/view/domconverter~ViewDomConverter#shouldRenderAttribute} say it should not.
	 *
	 * Unsafe attribute names can be specified when creating an element via {@link module:engine/view/downcastwriter~ViewDowncastWriter}.
	 *
	 * @param attributeName The name of the attribute to be checked.
	 */ shouldRenderUnsafeAttribute(attributeName) {
        return this._unsafeAttributesToRender.includes(attributeName);
    }
    /**
	 * Converts `ViewElement` to plain object and returns it.
	 *
	 * @returns `ViewElement` converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        json.name = this.name;
        json.type = 'Element';
        if (this._attrs.size) {
            json.attributes = Object.fromEntries(this.getAttributes());
        }
        if (this._children.length > 0) {
            json.children = [];
            for (const node of this._children){
                json.children.push(node.toJSON());
            }
        }
        return json;
    }
    /**
	 * Clones provided element.
	 *
	 * @internal
	 * @param deep If set to `true` clones element and all its children recursively. When set to `false`,
	 * element will be cloned without any children.
	 * @returns Clone of this element.
	 */ _clone(deep = false) {
        const childrenClone = [];
        if (deep) {
            for (const child of this.getChildren()){
                childrenClone.push(child._clone(deep));
            }
        }
        // ViewContainerElement and ViewAttributeElement should be also cloned properly.
        const cloned = new this.constructor(this.document, this.name, this._attrs, childrenClone);
        // Clone custom properties.
        cloned._customProperties = new Map(this._customProperties);
        // Clone filler offset method.
        // We can't define this method in a prototype because it's behavior which
        // is changed by e.g. toWidget() function from ckeditor5-widget. Perhaps this should be one of custom props.
        cloned.getFillerOffset = this.getFillerOffset;
        // Clone unsafe attributes list.
        cloned._unsafeAttributesToRender = this._unsafeAttributesToRender;
        return cloned;
    }
    /**
	 * {@link module:engine/view/element~ViewElement#_insertChild Insert} a child node or a list of child nodes at the end of this node
	 * and sets the parent of these nodes to this element.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#insert
	 * @internal
	 * @param items Items to be inserted.
	 * @fires change
	 * @returns Number of appended nodes.
	 */ _appendChild(items) {
        return this._insertChild(this.childCount, items);
    }
    /**
	 * Inserts a child node or a list of child nodes on the given index and sets the parent of these nodes to
	 * this element.
	 *
	 * @internal
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#insert
	 * @param index Position where nodes should be inserted.
	 * @param items Items to be inserted.
	 * @fires change
	 * @returns Number of inserted nodes.
	 */ _insertChild(index, items) {
        this._fireChange('children', this, {
            index
        });
        let count = 0;
        const nodes = normalize$3(this.document, items);
        for (const node of nodes){
            // If node that is being added to this element is already inside another element, first remove it from the old parent.
            if (node.parent !== null) {
                node._remove();
            }
            node.parent = this;
            node.document = this.document;
            this._children.splice(index, 0, node);
            index++;
            count++;
        }
        return count;
    }
    /**
	 * Removes number of child nodes starting at the given index and set the parent of these nodes to `null`.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#remove
	 * @internal
	 * @param index Number of the first node to remove.
	 * @param howMany Number of nodes to remove.
	 * @fires change
	 * @returns The array of removed nodes.
	 */ _removeChildren(index, howMany = 1) {
        this._fireChange('children', this, {
            index
        });
        for(let i = index; i < index + howMany; i++){
            this._children[i].parent = null;
        }
        return this._children.splice(index, howMany);
    }
    /**
	 * Adds or overwrite attribute with a specified key and value.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#setAttribute
	 * @internal
	 * @param key Attribute key.
	 * @param value Attribute value.
	 * @param overwrite Whether tokenized attribute should override the attribute value or just add a token.
	 * @fires change
	 */ _setAttribute(key, value, overwrite = true) {
        this._fireChange('attributes', this);
        if (usesStylesMap(this.name, key) || usesTokenList(this.name, key)) {
            let currentValue = this._attrs.get(key);
            if (!currentValue) {
                currentValue = usesStylesMap(this.name, key) ? new StylesMap(this.document.stylesProcessor) : new ViewTokenList();
                this._attrs.set(key, currentValue);
            }
            if (overwrite) {
                // If reset is set then value have to be a string to tokenize.
                currentValue.setTo(String(value));
            } else if (usesStylesMap(this.name, key)) {
                if (Array.isArray(value)) {
                    currentValue.set(value[0], value[1]);
                } else {
                    currentValue.set(value);
                }
            } else {
                currentValue.set(typeof value == 'string' ? value.split(/\s+/) : value);
            }
        } else {
            this._attrs.set(key, String(value));
        }
    }
    /**
	 * Removes attribute from the element.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#removeAttribute
	 * @internal
	 * @param key Attribute key.
	 * @param tokens Attribute value tokens to remove. The whole attribute is removed if not specified.
	 * @returns Returns true if an attribute existed and has been removed.
	 * @fires change
	 */ _removeAttribute(key, tokens) {
        this._fireChange('attributes', this);
        if (tokens !== undefined && (usesStylesMap(this.name, key) || usesTokenList(this.name, key))) {
            const currentValue = this._attrs.get(key);
            if (!currentValue) {
                return false;
            }
            if (usesTokenList(this.name, key) && typeof tokens == 'string') {
                tokens = tokens.split(/\s+/);
            }
            currentValue.remove(tokens);
            if (currentValue.isEmpty) {
                return this._attrs.delete(key);
            }
            return false;
        }
        return this._attrs.delete(key);
    }
    /**
	 * Adds specified class.
	 *
	 * ```ts
	 * element._addClass( 'foo' ); // Adds 'foo' class.
	 * element._addClass( [ 'foo', 'bar' ] ); // Adds 'foo' and 'bar' classes.
	 * ```
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#addClass
	 * @internal
	 * @fires change
	 */ _addClass(className) {
        this._setAttribute('class', className, false);
    }
    /**
	 * Removes specified class.
	 *
	 * ```ts
	 * element._removeClass( 'foo' );  // Removes 'foo' class.
	 * element._removeClass( [ 'foo', 'bar' ] ); // Removes both 'foo' and 'bar' classes.
	 * ```
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#removeClass
	 * @internal
	 * @fires change
	 */ _removeClass(className) {
        this._removeAttribute('class', className);
    }
    _setStyle(property, value) {
        if (typeof property != 'string') {
            this._setAttribute('style', property, false);
        } else {
            this._setAttribute('style', [
                property,
                value
            ], false);
        }
    }
    /**
	 * Removes specified style.
	 *
	 * ```ts
	 * element._removeStyle( 'color' );  // Removes 'color' style.
	 * element._removeStyle( [ 'color', 'border-top' ] ); // Removes both 'color' and 'border-top' styles.
	 * ```
	 *
	 * **Note**: This method can work with normalized style names if
	 * {@link module:engine/controller/datacontroller~DataController#addStyleProcessorRules a particular style processor rule is enabled}.
	 * See {@link module:engine/view/stylesmap~StylesMap#remove `StylesMap#remove()`} for details.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#removeStyle
	 * @internal
	 * @fires change
	 */ _removeStyle(property) {
        this._removeAttribute('style', property);
    }
    /**
	 * Used by the {@link module:engine/view/matcher~Matcher Matcher} to collect matching attribute tuples
	 * (attribute name and optional token).
	 *
	 * Normalized patterns can be used in following ways:
	 * - to match any attribute name with any or no value:
	 *
	 * ```ts
	 * patterns: [
	 * 	[ true, true ]
	 * ]
	 * ```
	 *
	 * - to match a specific attribute with any value:
	 *
	 * ```ts
	 * patterns: [
	 * 	[ 'required', true ]
	 * ]
	 * ```
	 *
	 * - to match an attribute name with a RegExp with any value:
	 *
	 * ```ts
	 * patterns: [
	 * 	[ /h[1-6]/, true ]
	 * ]
	 * ```
	 *
	 * 	- to match a specific attribute with the exact value:
	 *
	 * ```ts
	 * patterns: [
	 * 	[ 'rel', 'nofollow' ]
	 * ]
	 * ```
	 *
	 * 	- to match a specific attribute with a value matching a RegExp:
	 *
	 * ```ts
	 * patterns: [
	 * 	[ 'src', /^https/ ]
	 * ]
	 * ```
	 *
	 * 	- to match an attribute name with a RegExp and the exact value:
	 *
	 * ```ts
	 * patterns: [
	 * 	[ /^data-property-/, 'foobar' ],
	 * ]
	 * ```
	 *
	 * 	- to match an attribute name with a RegExp and match a value with another RegExp:
	 *
	 * ```ts
	 * patterns: [
	 * 	[ /^data-property-/, /^foo/ ]
	 * ]
	 * ```
	 *
	 * 	- to match a specific style property with the value matching a RegExp:
	 *
	 * ```ts
	 * patterns: [
	 * 	[ 'style', 'font-size', /px$/ ]
	 * ]
	 * ```
	 *
	 * 	- to match a specific class (class attribute is tokenized so it matches tokens individually):
	 *
	 * ```ts
	 * patterns: [
	 * 	[ 'class', 'foo' ]
	 * ]
	 * ```
	 *
	 * @internal
	 * @param patterns An array of normalized patterns (tuples of 2 or 3 items depending on if tokenized attribute value match is needed).
	 * @param match An array to populate with matching tuples.
	 * @param exclude Array of attribute names to exclude from match.
	 * @returns `true` if element matches all patterns. The matching tuples are pushed to the `match` array.
	 */ _collectAttributesMatch(patterns, match, exclude) {
        for (const [keyPattern, tokenPattern, valuePattern] of patterns){
            let hasKey = false;
            let hasValue = false;
            for (const [key, value] of this._attrs){
                if (exclude && exclude.includes(key) || !isPatternMatched(keyPattern, key)) {
                    continue;
                }
                hasKey = true;
                if (typeof value == 'string') {
                    if (isPatternMatched(tokenPattern, value)) {
                        match.push([
                            key
                        ]);
                        hasValue = true;
                    } else if (!(keyPattern instanceof RegExp)) {
                        return false;
                    }
                } else {
                    const tokenMatch = value._getTokensMatch(tokenPattern, valuePattern || true);
                    if (tokenMatch) {
                        hasValue = true;
                        for (const tokenMatchItem of tokenMatch){
                            match.push([
                                key,
                                tokenMatchItem
                            ]);
                        }
                    } else if (!(keyPattern instanceof RegExp)) {
                        return false;
                    }
                }
            }
            if (!hasKey || !hasValue) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Used by the {@link module:engine/conversion/viewconsumable~ViewConsumable} to collect the
	 * {@link module:engine/view/element~ViewNormalizedConsumables} for the element.
	 *
	 * When `key` and `token` parameters are provided the output is filtered for the specified attribute and it's tokens and related tokens.
	 *
	 * @internal
	 * @param key Attribute name.
	 * @param token Reference token to collect all related tokens.
	 */ _getConsumables(key, token) {
        const attributes = [];
        if (key) {
            const value = this._attrs.get(key);
            if (value !== undefined) {
                if (typeof value == 'string') {
                    attributes.push([
                        key
                    ]);
                } else {
                    for (const prop of value._getConsumables(token)){
                        attributes.push([
                            key,
                            prop
                        ]);
                    }
                }
            }
        } else {
            for (const [key, value] of this._attrs){
                if (typeof value == 'string') {
                    attributes.push([
                        key
                    ]);
                } else {
                    for (const prop of value._getConsumables()){
                        attributes.push([
                            key,
                            prop
                        ]);
                    }
                }
            }
        }
        return {
            name: !key,
            attributes
        };
    }
    /**
	 * Verify if the given element can be merged without conflicts into the element.
	 *
	 * Note that this method is extended by the {@link module:engine/view/attributeelement~ViewAttributeElement} implementation.
	 *
	 * This method is used by the {@link module:engine/view/downcastwriter~ViewDowncastWriter} while down-casting
	 * an {@link module:engine/view/attributeelement~ViewAttributeElement} to merge it with other ViewAttributeElement.
	 *
	 * @internal
	 * @returns Returns `true` if elements can be merged.
	 */ _canMergeAttributesFrom(otherElement) {
        if (this.name != otherElement.name) {
            return false;
        }
        for (const [key, otherValue] of otherElement._attrs){
            const value = this._attrs.get(key);
            if (value === undefined) {
                continue;
            }
            if (typeof value == 'string' || typeof otherValue == 'string') {
                if (value !== otherValue) {
                    return false;
                }
            } else if (!value._canMergeFrom(otherValue)) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Merges attributes of a given element into the element.
	 * This includes also tokenized attributes like style and class.
	 *
	 * Note that you should make sure there are no conflicts before merging (see {@link #_canMergeAttributesFrom}).
	 *
	 * This method is used by the {@link module:engine/view/downcastwriter~ViewDowncastWriter} while down-casting
	 * an {@link module:engine/view/attributeelement~ViewAttributeElement} to merge it with other ViewAttributeElement.
	 *
	 * @internal
	 */ _mergeAttributesFrom(otherElement) {
        this._fireChange('attributes', this);
        // Move all attributes/classes/styles from wrapper to wrapped ViewAttributeElement.
        for (const [key, otherValue] of otherElement._attrs){
            const value = this._attrs.get(key);
            if (value === undefined || typeof value == 'string' || typeof otherValue == 'string') {
                this._setAttribute(key, otherValue);
            } else {
                value._mergeFrom(otherValue);
            }
        }
    }
    /**
	 * Verify if the given element attributes can be fully subtracted from the element.
	 *
	 * Note that this method is extended by the {@link module:engine/view/attributeelement~ViewAttributeElement} implementation.
	 *
	 * This method is used by the {@link module:engine/view/downcastwriter~ViewDowncastWriter} while down-casting
	 * an {@link module:engine/view/attributeelement~ViewAttributeElement} to unwrap the ViewAttributeElement.
	 *
	 * @internal
	 * @returns Returns `true` if elements attributes can be fully subtracted.
	 */ _canSubtractAttributesOf(otherElement) {
        if (this.name != otherElement.name) {
            return false;
        }
        for (const [key, otherValue] of otherElement._attrs){
            const value = this._attrs.get(key);
            if (value === undefined) {
                return false;
            }
            if (typeof value == 'string' || typeof otherValue == 'string') {
                if (value !== otherValue) {
                    return false;
                }
            } else if (!value._isMatching(otherValue)) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Removes (subtracts) corresponding attributes of the given element from the element.
	 * This includes also tokenized attributes like style and class.
	 * All attributes, classes and styles from given element should be present inside the element being unwrapped.
	 *
	 * Note that you should make sure all attributes could be subtracted before subtracting them (see {@link #_canSubtractAttributesOf}).
	 *
	 * This method is used by the {@link module:engine/view/downcastwriter~ViewDowncastWriter} while down-casting
	 * an {@link module:engine/view/attributeelement~ViewAttributeElement} to unwrap the ViewAttributeElement.
	 *
	 * @internal
	 */ _subtractAttributesOf(otherElement) {
        this._fireChange('attributes', this);
        for (const [key, otherValue] of otherElement._attrs){
            const value = this._attrs.get(key);
            if (typeof value == 'string' || typeof otherValue == 'string') {
                this._attrs.delete(key);
            } else {
                value.remove(otherValue.keys());
                if (value.isEmpty) {
                    this._attrs.delete(key);
                }
            }
        }
    }
    /**
	 * Sets a custom property. Unlike attributes, custom properties are not rendered to the DOM,
	 * so they can be used to add special data to elements.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#setCustomProperty
	 * @internal
	 */ _setCustomProperty(key, value) {
        this._customProperties.set(key, value);
    }
    /**
	 * Removes the custom property stored under the given key.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#removeCustomProperty
	 * @internal
	 * @returns Returns true if property was removed.
	 */ _removeCustomProperty(key) {
        return this._customProperties.delete(key);
    }
    /**
	 * Parses attributes provided to the element constructor before they are applied to an element. If attributes are passed
	 * as an object (instead of `Iterable`), the object is transformed to the map. Attributes with `null` value are removed.
	 * Attributes with non-`String` value are converted to `String`.
	 *
	 * @param attrs Attributes to parse.
	 * @returns Parsed attributes.
	 */ _parseAttributes(attrs) {
        const attrsMap = toMap(attrs);
        for (const [key, value] of attrsMap){
            if (value === null) {
                attrsMap.delete(key);
            } else if (usesStylesMap(this.name, key)) {
                // This is either an element clone so we need to clone styles map, or a new instance which requires value to be parsed.
                const newValue = value instanceof StylesMap ? value._clone() : new StylesMap(this.document.stylesProcessor).setTo(String(value));
                attrsMap.set(key, newValue);
            } else if (usesTokenList(this.name, key)) {
                // This is either an element clone so we need to clone token list, or a new instance which requires value to be parsed.
                const newValue = value instanceof ViewTokenList ? value._clone() : new ViewTokenList().setTo(String(value));
                attrsMap.set(key, newValue);
            } else if (typeof value != 'string') {
                attrsMap.set(key, String(value));
            }
        }
        return attrsMap;
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewElement.prototype.is = function(type, name) {
    if (!name) {
        return type === 'element' || type === 'view:element' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'node' || type === 'view:node';
    } else {
        return name === this.name && (type === 'element' || type === 'view:element');
    }
};
/**
 * Converts strings to Text and non-iterables to arrays.
 */ function normalize$3(document, nodes) {
    // Separate condition because string is iterable.
    if (typeof nodes == 'string') {
        return [
            new ViewText(document, nodes)
        ];
    }
    if (!isIterable(nodes)) {
        nodes = [
            nodes
        ];
    }
    const normalizedNodes = [];
    for (const node of nodes){
        if (typeof node == 'string') {
            normalizedNodes.push(new ViewText(document, node));
        } else if (node instanceof ViewTextProxy) {
            normalizedNodes.push(new ViewText(document, node.data));
        } else {
            normalizedNodes.push(node);
        }
    }
    return normalizedNodes;
}
/**
 * Returns `true` if an attribute on a given element should be handled as a TokenList.
 */ function usesTokenList(elementName, key) {
    return key == 'class' || elementName == 'a' && key == 'rel';
}
/**
 * Returns `true` if an attribute on a given element should be handled as a StylesMap.
 */ function usesStylesMap(elementName, key) {
    return key == 'style';
}

/**
 * Containers are elements which define document structure. They define boundaries for
 * {@link module:engine/view/attributeelement~ViewAttributeElement attributes}.
 * They are mostly used for block elements like `<p>` or `<div>`.
 *
 * Editing engine does not define a fixed HTML DTD. This is why a feature developer needs to choose between various
 * types (container element, {@link module:engine/view/attributeelement~ViewAttributeElement attribute element},
 * {@link module:engine/view/emptyelement~ViewEmptyElement empty element}, etc) when developing a feature.
 *
 * The container element should be your default choice when writing a converter, unless:
 *
 * * this element represents a model text attribute (then use {@link module:engine/view/attributeelement~ViewAttributeElement}),
 * * this is an empty element like `<img>` (then use {@link module:engine/view/emptyelement~ViewEmptyElement}),
 * * this is a root element,
 * * this is a nested editable element (then use  {@link module:engine/view/editableelement~ViewEditableElement}).
 *
 * To create a new container element instance use the
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createContainerElement `ViewDowncastWriter#createContainerElement()`}
 * method.
 */ class ViewContainerElement extends ViewElement {
    /**
	 * Creates a container element.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#createContainerElement
	 * @see module:engine/view/element~ViewElement
	 * @internal
	 * @param document The document instance to which this element belongs.
	 * @param name Node name.
	 * @param attrs Collection of attributes.
	 * @param children A list of nodes to be inserted into created element.
	 */ constructor(document, name, attrs, children){
        super(document, name, attrs, children);
        this.getFillerOffset = getViewFillerOffset;
    }
    /**
	 * Converts `ViewContainerElement` instance to plain object and returns it.
	 *
	 * @returns `ViewContainerElement` instance converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        json.type = 'ContainerElement';
        return json;
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewContainerElement.prototype.is = function(type, name) {
    if (!name) {
        return type === 'containerElement' || type === 'view:containerElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'element' || type === 'view:element' || type === 'node' || type === 'view:node';
    } else {
        return name === this.name && (type === 'containerElement' || type === 'view:containerElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'element' || type === 'view:element');
    }
};
/**
 * Returns block {@link module:engine/view/filler filler} offset or `null` if block filler is not needed.
 *
 * @returns Block filler offset or `null` if block filler is not needed.
 */ function getViewFillerOffset() {
    const children = [
        ...this.getChildren()
    ];
    const lastChild = children[this.childCount - 1];
    // Block filler is required after a `<br>` if it's the last element in its container. See #1422.
    if (lastChild && lastChild.is('element', 'br')) {
        return this.childCount;
    }
    for (const child of children){
        // If there's any non-UI element – don't render the bogus.
        if (!child.is('uiElement')) {
            return null;
        }
    }
    // If there are only UI elements – render the bogus at the end of the element.
    return this.childCount;
}

/**
 * Editable element which can be a {@link module:engine/view/rooteditableelement~ViewRootEditableElement root}
 * or nested editable area in the editor.
 *
 * Editable is automatically read-only when its {@link module:engine/view/document~ViewDocument Document} is read-only.
 *
 * The constructor of this class shouldn't be used directly. To create new `ViewEditableElement` use the
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createEditableElement `downcastWriter#createEditableElement()`} method.
 */ class ViewEditableElement extends /* #__PURE__ */ ObservableMixin(ViewContainerElement) {
    /**
	 * Creates an editable element.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#createEditableElement
	 * @internal
	 * @param document The document instance to which this element belongs.
	 * @param name Node name.
	 * @param attributes Collection of attributes.
	 * @param children A list of nodes to be inserted into created element.
	 */ constructor(document, name, attributes, children){
        super(document, name, attributes, children);
        this.set('isReadOnly', false);
        this.set('isFocused', false);
        this.set('placeholder', undefined);
        this.bind('isReadOnly').to(document);
        this.bind('isFocused').to(document, 'isFocused', (isFocused)=>isFocused && document.selection.editableElement == this);
        // Update focus state based on selection changes.
        this.listenTo(document.selection, 'change', ()=>{
            this.isFocused = document.isFocused && document.selection.editableElement == this;
        });
    }
    destroy() {
        this.stopListening();
    }
    /**
	 * Converts `ViewEditableElement` instance to plain object and returns it.
	 *
	 * @returns `ViewEditableElement` instance converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        json.type = 'EditableElement';
        json.isReadOnly = this.isReadOnly;
        json.isFocused = this.isFocused;
        return json;
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewEditableElement.prototype.is = function(type, name) {
    if (!name) {
        return type === 'editableElement' || type === 'view:editableElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'containerElement' || type === 'view:containerElement' || type === 'element' || type === 'view:element' || type === 'node' || type === 'view:node';
    } else {
        return name === this.name && (type === 'editableElement' || type === 'view:editableElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'containerElement' || type === 'view:containerElement' || type === 'element' || type === 'view:element');
    }
};

const rootNameSymbol = Symbol('rootName');
/**
 * Class representing a single root in the data view. A root can be either
 * {@link ~ViewRootEditableElement#isReadOnly editable or read-only},
 * but in both cases it is called "an editable". Roots can contain other {@link module:engine/view/editableelement~ViewEditableElement
 * editable elements} making them "nested editables".
 */ class ViewRootEditableElement extends ViewEditableElement {
    /**
	 * Creates root editable element.
	 *
	 * @param document The document instance to which this element belongs.
	 * @param name Node name.
	 */ constructor(document, name){
        super(document, name);
        this.rootName = 'main';
    }
    /**
	 * Name of this root inside {@link module:engine/view/document~ViewDocument} that is an owner of this root. If no
	 * other name is set, `main` name is used.
	 *
	 * @readonly
	 */ get rootName() {
        return this.getCustomProperty(rootNameSymbol);
    }
    set rootName(rootName) {
        this._setCustomProperty(rootNameSymbol, rootName);
    }
    /**
	 * Converts `ViewRootEditableElement` instance to string and returns it.
	 *
	 * @returns `ViewRootEditableElement` instance converted to string.
	 */ toJSON() {
        return this.rootName;
    }
    /**
	 * Overrides old element name and sets new one.
	 * This is needed because view roots are created before they are attached to the DOM.
	 * The name of the root element is temporary at this stage. It has to be changed when the
	 * view root element is attached to the DOM element.
	 *
	 * @internal
	 * @param name The new name of element.
	 */ set _name(name) {
        this.name = name;
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewRootEditableElement.prototype.is = function(type, name) {
    if (!name) {
        return type === 'rootElement' || type === 'view:rootElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'editableElement' || type === 'view:editableElement' || type === 'containerElement' || type === 'view:containerElement' || type === 'element' || type === 'view:element' || type === 'node' || type === 'view:node';
    } else {
        return name === this.name && (type === 'rootElement' || type === 'view:rootElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'editableElement' || type === 'view:editableElement' || type === 'containerElement' || type === 'view:containerElement' || type === 'element' || type === 'view:element');
    }
};

/**
 * Position iterator class. It allows to iterate forward and backward over the document.
 */ class ViewTreeWalker {
    /**
	 * Walking direction. Defaults `'forward'`.
	 */ direction;
    /**
	 * Iterator boundaries.
	 *
	 * When the iterator is walking `'forward'` on the end of boundary or is walking `'backward'`
	 * on the start of boundary, then `{ done: true }` is returned.
	 *
	 * If boundaries are not defined they are set before first and after last child of the root node.
	 */ boundaries;
    /**
	 * Flag indicating whether all characters from {@link module:engine/view/text~ViewText} should be returned as one
	 * {@link module:engine/view/text~ViewText} or one by one as {@link module:engine/view/textproxy~ViewTextProxy}.
	 */ singleCharacters;
    /**
	 * Flag indicating whether iterator should enter elements or not. If the iterator is shallow child nodes of any
	 * iterated node will not be returned along with `elementEnd` tag.
	 */ shallow;
    /**
	 * Flag indicating whether iterator should ignore `elementEnd` tags. If set to `true`, walker will not
	 * return a parent node of the start position. Each {@link module:engine/view/element~ViewElement} will be returned once.
	 * When set to `false` each element might be returned twice: for `'elementStart'` and `'elementEnd'`.
	 */ ignoreElementEnd;
    /**
	 * Iterator position. If start position is not defined then position depends on {@link #direction}. If direction is
	 * `'forward'` position starts form the beginning, when direction is `'backward'` position starts from the end.
	 */ _position;
    /**
	 * Start boundary parent.
	 */ _boundaryStartParent;
    /**
	 * End boundary parent.
	 */ _boundaryEndParent;
    /**
	 * Creates a range iterator. All parameters are optional, but you have to specify either `boundaries` or `startPosition`.
	 *
	 * @param options Object with configuration.
	 */ constructor(options = {}){
        if (!options.boundaries && !options.startPosition) {
            /**
			 * Neither boundaries nor starting position have been defined.
			 *
			 * @error view-tree-walker-no-start-position
			 */ throw new CKEditorError('view-tree-walker-no-start-position', null);
        }
        if (options.direction && options.direction != 'forward' && options.direction != 'backward') {
            /**
			 * Only `backward` and `forward` direction allowed.
			 *
			 * @error view-tree-walker-unknown-direction
			 */ throw new CKEditorError('view-tree-walker-unknown-direction', options.startPosition, {
                direction: options.direction
            });
        }
        this.boundaries = options.boundaries || null;
        if (options.startPosition) {
            this._position = ViewPosition._createAt(options.startPosition);
        } else {
            this._position = ViewPosition._createAt(options.boundaries[options.direction == 'backward' ? 'end' : 'start']);
        }
        this.direction = options.direction || 'forward';
        this.singleCharacters = !!options.singleCharacters;
        this.shallow = !!options.shallow;
        this.ignoreElementEnd = !!options.ignoreElementEnd;
        this._boundaryStartParent = this.boundaries ? this.boundaries.start.parent : null;
        this._boundaryEndParent = this.boundaries ? this.boundaries.end.parent : null;
    }
    /**
	 * Iterable interface.
	 */ [Symbol.iterator]() {
        return this;
    }
    /**
	 * Iterator position. If start position is not defined then position depends on {@link #direction}. If direction is
	 * `'forward'` position starts form the beginning, when direction is `'backward'` position starts from the end.
	 */ get position() {
        return this._position;
    }
    /**
	 * Moves {@link #position} in the {@link #direction} skipping values as long as the callback function returns `true`.
	 *
	 * For example:
	 *
	 * ```ts
	 * walker.skip( value => value.type == 'text' ); // <p>{}foo</p> -> <p>foo[]</p>
	 * walker.skip( value => true ); // Move the position to the end: <p>{}foo</p> -> <p>foo</p>[]
	 * walker.skip( value => false ); // Do not move the position.
	 * ```
	 *
	 * @param skip Callback function. Gets {@link module:engine/view/treewalker~ViewTreeWalkerValue} and should
	 * return `true` if the value should be skipped or `false` if not.
	 */ skip(skip) {
        let nextResult;
        let prevPosition;
        do {
            prevPosition = this.position;
            nextResult = this.next();
        }while (!nextResult.done && skip(nextResult.value))
        if (!nextResult.done) {
            this._position = prevPosition;
        }
    }
    /**
	 * Moves tree walker {@link #position} to provided `position`. Tree walker will
	 * continue traversing from that position.
	 *
	 * Note: in contrary to {@link ~ViewTreeWalker#skip}, this method does not iterate over the nodes along the way.
	 * It simply sets the current tree walker position to a new one.
	 * From the performance standpoint, it is better to use {@link ~ViewTreeWalker#jumpTo} rather than {@link ~ViewTreeWalker#skip}.
	 *
	 * If the provided position is before the start boundary, the position will be
	 * set to the start boundary. If the provided position is after the end boundary,
	 * the position will be set to the end boundary.
	 * This is done to prevent the treewalker from traversing outside the boundaries.
	 *
	 * @param position Position to jump to.
	 */ jumpTo(position) {
        if (this._boundaryStartParent && position.isBefore(this.boundaries.start)) {
            position = this.boundaries.start;
        } else if (this._boundaryEndParent && position.isAfter(this.boundaries.end)) {
            position = this.boundaries.end;
        }
        this._position = position.clone();
    }
    /**
	 * Gets the next tree walker's value.
	 *
	 * @returns Object implementing iterator interface, returning
	 * information about taken step.
	 */ next() {
        if (this.direction == 'forward') {
            return this._next();
        } else {
            return this._previous();
        }
    }
    /**
	 * Makes a step forward in view. Moves the {@link #position} to the next position and returns the encountered value.
	 */ _next() {
        let position = this.position.clone();
        const previousPosition = this.position;
        const parent = position.parent;
        // We are at the end of the root.
        if (parent.parent === null && position.offset === parent.childCount) {
            return {
                done: true,
                value: undefined
            };
        }
        // We reached the walker boundary.
        if (parent === this._boundaryEndParent && position.offset == this.boundaries.end.offset) {
            return {
                done: true,
                value: undefined
            };
        }
        // Get node just after current position.
        let node;
        // Text is a specific parent because it contains string instead of child nodes.
        if (parent && parent.is('view:$text')) {
            if (position.isAtEnd) {
                // Prevent returning "elementEnd" for Text node. Skip that value and return the next walker step.
                this._position = ViewPosition._createAfter(parent);
                return this._next();
            }
            node = parent.data[position.offset];
        } else {
            node = parent.getChild(position.offset);
        }
        if (typeof node == 'string') {
            let textLength;
            if (this.singleCharacters) {
                textLength = 1;
            } else {
                // Check if text stick out of walker range.
                const endOffset = parent === this._boundaryEndParent ? this.boundaries.end.offset : parent.data.length;
                textLength = endOffset - position.offset;
            }
            const textProxy = new ViewTextProxy(parent, position.offset, textLength);
            position.offset += textLength;
            this._position = position;
            return this._formatReturnValue('text', textProxy, previousPosition, position, textLength);
        }
        if (node && node.is('view:element')) {
            if (!this.shallow) {
                position = new ViewPosition(node, 0);
            } else {
                // We are past the walker boundaries.
                if (this.boundaries && this.boundaries.end.isBefore(position)) {
                    return {
                        done: true,
                        value: undefined
                    };
                }
                position.offset++;
            }
            this._position = position;
            return this._formatReturnValue('elementStart', node, previousPosition, position, 1);
        }
        if (node && node.is('view:$text')) {
            if (this.singleCharacters) {
                position = new ViewPosition(node, 0);
                this._position = position;
                return this._next();
            }
            let charactersCount = node.data.length;
            let item;
            // If text stick out of walker range, we need to cut it and wrap in ViewTextProxy.
            if (node == this._boundaryEndParent) {
                charactersCount = this.boundaries.end.offset;
                item = new ViewTextProxy(node, 0, charactersCount);
                position = ViewPosition._createAfter(item);
            } else {
                item = new ViewTextProxy(node, 0, node.data.length);
                // If not just keep moving forward.
                position.offset++;
            }
            this._position = position;
            return this._formatReturnValue('text', item, previousPosition, position, charactersCount);
        }
        // `node` is not set, we reached the end of current `parent`.
        position = ViewPosition._createAfter(parent);
        this._position = position;
        if (this.ignoreElementEnd) {
            return this._next();
        }
        return this._formatReturnValue('elementEnd', parent, previousPosition, position);
    }
    /**
	 * Makes a step backward in view. Moves the {@link #position} to the previous position and returns the encountered value.
	 */ _previous() {
        let position = this.position.clone();
        const previousPosition = this.position;
        const parent = position.parent;
        // We are at the beginning of the root.
        if (parent.parent === null && position.offset === 0) {
            return {
                done: true,
                value: undefined
            };
        }
        // We reached the walker boundary.
        if (parent == this._boundaryStartParent && position.offset == this.boundaries.start.offset) {
            return {
                done: true,
                value: undefined
            };
        }
        // Get node just before current position.
        let node;
        // Text {@link module:engine/view/text~ViewText} element is a specific parent because contains string instead of child nodes.
        if (parent.is('view:$text')) {
            if (position.isAtStart) {
                // Prevent returning "elementStart" for Text node. Skip that value and return the next walker step.
                this._position = ViewPosition._createBefore(parent);
                return this._previous();
            }
            node = parent.data[position.offset - 1];
        } else {
            node = parent.getChild(position.offset - 1);
        }
        if (typeof node == 'string') {
            let textLength;
            if (!this.singleCharacters) {
                // Check if text stick out of walker range.
                const startOffset = parent === this._boundaryStartParent ? this.boundaries.start.offset : 0;
                textLength = position.offset - startOffset;
            } else {
                textLength = 1;
            }
            position.offset -= textLength;
            const textProxy = new ViewTextProxy(parent, position.offset, textLength);
            this._position = position;
            return this._formatReturnValue('text', textProxy, previousPosition, position, textLength);
        }
        if (node && node.is('view:element')) {
            if (this.shallow) {
                position.offset--;
                this._position = position;
                return this._formatReturnValue('elementStart', node, previousPosition, position, 1);
            }
            position = new ViewPosition(node, node.childCount);
            this._position = position;
            if (this.ignoreElementEnd) {
                return this._previous();
            }
            return this._formatReturnValue('elementEnd', node, previousPosition, position);
        }
        if (node && node.is('view:$text')) {
            if (this.singleCharacters) {
                position = new ViewPosition(node, node.data.length);
                this._position = position;
                return this._previous();
            }
            let charactersCount = node.data.length;
            let item;
            // If text stick out of walker range, we need to cut it and wrap in ViewTextProxy.
            if (node == this._boundaryStartParent) {
                const offset = this.boundaries.start.offset;
                item = new ViewTextProxy(node, offset, node.data.length - offset);
                charactersCount = item.data.length;
                position = ViewPosition._createBefore(item);
            } else {
                item = new ViewTextProxy(node, 0, node.data.length);
                // If not just keep moving backward.
                position.offset--;
            }
            this._position = position;
            return this._formatReturnValue('text', item, previousPosition, position, charactersCount);
        }
        // `node` is not set, we reached the beginning of current `parent`.
        position = ViewPosition._createBefore(parent);
        this._position = position;
        return this._formatReturnValue('elementStart', parent, previousPosition, position, 1);
    }
    /**
	 * Format returned data and adjust `previousPosition` and `nextPosition` if
	 * reach the bound of the {@link module:engine/view/text~ViewText}.
	 *
	 * @param type Type of step.
	 * @param item Item between old and new position.
	 * @param previousPosition Previous position of iterator.
	 * @param nextPosition Next position of iterator.
	 * @param length Length of the item.
	 */ _formatReturnValue(type, item, previousPosition, nextPosition, length) {
        // Text is a specific parent, because contains string instead of children.
        // Walker doesn't enter to the Text except situations when walker is iterating over every single character,
        // or the bound starts/ends inside the Text. So when the position is at the beginning or at the end of the Text
        // we move it just before or just after Text.
        if (item.is('view:$textProxy')) {
            // Position is at the end of Text.
            if (item.offsetInText + item.data.length == item.textNode.data.length) {
                if (this.direction == 'forward' && !(this.boundaries && this.boundaries.end.isEqual(this.position))) {
                    nextPosition = ViewPosition._createAfter(item.textNode);
                    // When we change nextPosition of returned value we need also update walker current position.
                    this._position = nextPosition;
                } else {
                    previousPosition = ViewPosition._createAfter(item.textNode);
                }
            }
            // Position is at the begining ot the text.
            if (item.offsetInText === 0) {
                if (this.direction == 'backward' && !(this.boundaries && this.boundaries.start.isEqual(this.position))) {
                    nextPosition = ViewPosition._createBefore(item.textNode);
                    // When we change nextPosition of returned value we need also update walker current position.
                    this._position = nextPosition;
                } else {
                    previousPosition = ViewPosition._createBefore(item.textNode);
                }
            }
        }
        return {
            done: false,
            value: {
                type,
                item,
                previousPosition,
                nextPosition,
                length
            }
        };
    }
}

/**
 * Position in the view tree. Position is represented by its parent node and an offset in this parent.
 *
 * In order to create a new position instance use the `createPosition*()` factory methods available in:
 *
 * * {@link module:engine/view/view~EditingView}
 * * {@link module:engine/view/downcastwriter~ViewDowncastWriter}
 * * {@link module:engine/view/upcastwriter~ViewUpcastWriter}
 */ class ViewPosition extends ViewTypeCheckable {
    /**
	 * Position parent.
	 */ parent;
    /**
	 * Position offset.
	 */ offset;
    /**
	 * Creates a position.
	 *
	 * @param parent Position parent.
	 * @param offset Position offset.
	 */ constructor(parent, offset){
        super();
        this.parent = parent;
        this.offset = offset;
    }
    /**
	 * Node directly after the position. Equals `null` when there is no node after position or position is located
	 * inside text node.
	 */ get nodeAfter() {
        if (this.parent.is('$text')) {
            return null;
        }
        return this.parent.getChild(this.offset) || null;
    }
    /**
	 * Node directly before the position. Equals `null` when there is no node before position or position is located
	 * inside text node.
	 */ get nodeBefore() {
        if (this.parent.is('$text')) {
            return null;
        }
        return this.parent.getChild(this.offset - 1) || null;
    }
    /**
	 * Is `true` if position is at the beginning of its {@link module:engine/view/position~ViewPosition#parent parent}, `false` otherwise.
	 */ get isAtStart() {
        return this.offset === 0;
    }
    /**
	 * Is `true` if position is at the end of its {@link module:engine/view/position~ViewPosition#parent parent}, `false` otherwise.
	 */ get isAtEnd() {
        const endOffset = this.parent.is('$text') ? this.parent.data.length : this.parent.childCount;
        return this.offset === endOffset;
    }
    /**
	 * Position's root, that is the root of the position's parent element.
	 */ get root() {
        return this.parent.root;
    }
    /**
	 * {@link module:engine/view/editableelement~ViewEditableElement ViewEditableElement} instance that contains this position, or `null` if
	 * position is not inside an editable element.
	 */ get editableElement() {
        let editable = this.parent;
        while(!(editable instanceof ViewEditableElement)){
            if (editable.parent) {
                editable = editable.parent;
            } else {
                return null;
            }
        }
        return editable;
    }
    /**
	 * Returns a new instance of Position with offset incremented by `shift` value.
	 *
	 * @param shift How position offset should get changed. Accepts negative values.
	 * @returns Shifted position.
	 */ getShiftedBy(shift) {
        const shifted = ViewPosition._createAt(this);
        const offset = shifted.offset + shift;
        shifted.offset = offset < 0 ? 0 : offset;
        return shifted;
    }
    /**
	 * Gets the farthest position which matches the callback using
	 * {@link module:engine/view/treewalker~ViewTreeWalker TreeWalker}.
	 *
	 * For example:
	 *
	 * ```ts
	 * getLastMatchingPosition( value => value.type == 'text' ); // <p>{}foo</p> -> <p>foo[]</p>
	 * getLastMatchingPosition( value => value.type == 'text', { direction: 'backward' } ); // <p>foo[]</p> -> <p>{}foo</p>
	 * getLastMatchingPosition( value => false ); // Do not move the position.
	 * ```
	 *
	 * @param skip Callback function. Gets {@link module:engine/view/treewalker~ViewTreeWalkerValue} and should
	 * return `true` if the value should be skipped or `false` if not.
	 * @param options Object with configuration options. See {@link module:engine/view/treewalker~ViewTreeWalker}.
	 * @returns The position after the last item which matches the `skip` callback test.
	 */ getLastMatchingPosition(skip, options = {}) {
        options.startPosition = this;
        const treeWalker = new ViewTreeWalker(options);
        treeWalker.skip(skip);
        return treeWalker.position;
    }
    /**
	 * Returns ancestors array of this position, that is this position's parent and it's ancestors.
	 *
	 * @returns Array with ancestors.
	 */ getAncestors() {
        if (this.parent.is('documentFragment')) {
            return [
                this.parent
            ];
        } else {
            return this.parent.getAncestors({
                includeSelf: true
            });
        }
    }
    /**
	 * Returns a {@link module:engine/view/node~ViewNode} or {@link module:engine/view/documentfragment~ViewDocumentFragment}
	 * which is a common ancestor of both positions.
	 */ getCommonAncestor(position) {
        const ancestorsA = this.getAncestors();
        const ancestorsB = position.getAncestors();
        let i = 0;
        while(ancestorsA[i] == ancestorsB[i] && ancestorsA[i]){
            i++;
        }
        return i === 0 ? null : ancestorsA[i - 1];
    }
    /**
	 * Checks whether this position equals given position.
	 *
	 * @param otherPosition Position to compare with.
	 * @returns True if positions are same.
	 */ isEqual(otherPosition) {
        return this.parent == otherPosition.parent && this.offset == otherPosition.offset;
    }
    /**
	 * Checks whether this position is located before given position. When method returns `false` it does not mean that
	 * this position is after give one. Two positions may be located inside separate roots and in that situation this
	 * method will still return `false`.
	 *
	 * @see module:engine/view/position~ViewPosition#isAfter
	 * @see module:engine/view/position~ViewPosition#compareWith
	 * @param otherPosition Position to compare with.
	 * @returns Returns `true` if this position is before given position.
	 */ isBefore(otherPosition) {
        return this.compareWith(otherPosition) == 'before';
    }
    /**
	 * Checks whether this position is located after given position. When method returns `false` it does not mean that
	 * this position is before give one. Two positions may be located inside separate roots and in that situation this
	 * method will still return `false`.
	 *
	 * @see module:engine/view/position~ViewPosition#isBefore
	 * @see module:engine/view/position~ViewPosition#compareWith
	 * @param otherPosition Position to compare with.
	 * @returns Returns `true` if this position is after given position.
	 */ isAfter(otherPosition) {
        return this.compareWith(otherPosition) == 'after';
    }
    /**
	 * Checks whether this position is before, after or in same position that other position. Two positions may be also
	 * different when they are located in separate roots.
	 *
	 * @param otherPosition Position to compare with.
	 */ compareWith(otherPosition) {
        if (this.root !== otherPosition.root) {
            return 'different';
        }
        if (this.isEqual(otherPosition)) {
            return 'same';
        }
        // Get path from root to position's parent element.
        const thisPath = this.parent.is('node') ? this.parent.getPath() : [];
        const otherPath = otherPosition.parent.is('node') ? otherPosition.parent.getPath() : [];
        // Add the positions' offsets to the parents offsets.
        thisPath.push(this.offset);
        otherPath.push(otherPosition.offset);
        // Compare both path arrays to find common ancestor.
        const result = compareArrays(thisPath, otherPath);
        switch(result){
            case 'prefix':
                return 'before';
            case 'extension':
                return 'after';
            default:
                // Cast to number to avoid having 'same' as a type of `result`.
                return thisPath[result] < otherPath[result] ? 'before' : 'after';
        }
    }
    /**
	 * Creates a {@link module:engine/view/treewalker~ViewTreeWalker TreeWalker} instance with this positions as a start position.
	 *
	 * @param options Object with configuration options. See {@link module:engine/view/treewalker~ViewTreeWalker}
	 */ getWalker(options = {}) {
        options.startPosition = this;
        return new ViewTreeWalker(options);
    }
    /**
	 * Clones this position.
	 */ clone() {
        return new ViewPosition(this.parent, this.offset);
    }
    /**
	 * Converts `ViewPosition` instance to plain object and returns it.
	 *
	 * @returns `ViewPosition` instance converted to plain object.
	 */ toJSON() {
        return {
            parent: this.parent.toJSON(),
            offset: this.offset
        };
    }
    /**
	 * Creates position at the given location. The location can be specified as:
	 *
	 * * a {@link module:engine/view/position~ViewPosition position},
	 * * parent element and offset (offset defaults to `0`),
	 * * parent element and `'end'` (sets position at the end of that element),
	 * * {@link module:engine/view/item~ViewItem view item} and `'before'` or `'after'` (sets position before or after given view item).
	 *
	 * This method is a shortcut to other constructors such as:
	 *
	 * * {@link module:engine/view/position~ViewPosition._createBefore},
	 * * {@link module:engine/view/position~ViewPosition._createAfter}.
	 *
	 * @internal
	 * @param offset Offset or one of the flags. Used only when first parameter is a {@link module:engine/view/item~ViewItem view item}.
	 */ static _createAt(itemOrPosition, offset) {
        if (itemOrPosition instanceof ViewPosition) {
            return new this(itemOrPosition.parent, itemOrPosition.offset);
        } else {
            const node = itemOrPosition;
            if (offset == 'end') {
                offset = node.is('$text') ? node.data.length : node.childCount;
            } else if (offset == 'before') {
                return this._createBefore(node);
            } else if (offset == 'after') {
                return this._createAfter(node);
            } else if (offset !== 0 && !offset) {
                /**
				 * {@link module:engine/view/view~EditingView#createPositionAt `View#createPositionAt()`}
				 * requires the offset to be specified when the first parameter is a view item.
				 *
				 * @error view-createpositionat-offset-required
				 */ throw new CKEditorError('view-createpositionat-offset-required', node);
            }
            return new ViewPosition(node, offset);
        }
    }
    /**
	 * Creates a new position after given view item.
	 *
	 * @internal
	 * @param item View item after which the position should be located.
	 */ static _createAfter(item) {
        // ViewTextProxy is not a instance of Node so we need do handle it in specific way.
        if (item.is('$textProxy')) {
            return new ViewPosition(item.textNode, item.offsetInText + item.data.length);
        }
        if (!item.parent) {
            /**
			 * You cannot make a position after a root.
			 *
			 * @error view-position-after-root
			 * @param {module:engine/view/node~ViewNode} root A root item.
			 */ throw new CKEditorError('view-position-after-root', item, {
                root: item
            });
        }
        return new ViewPosition(item.parent, item.index + 1);
    }
    /**
	 * Creates a new position before given view item.
	 *
	 * @internal
	 * @param item View item before which the position should be located.
	 */ static _createBefore(item) {
        // ViewTextProxy is not a instance of Node so we need do handle it in specific way.
        if (item.is('$textProxy')) {
            return new ViewPosition(item.textNode, item.offsetInText);
        }
        if (!item.parent) {
            /**
			 * You cannot make a position before a root.
			 *
			 * @error view-position-before-root
			 * @param {module:engine/view/node~ViewNode} root A root item.
			 */ throw new CKEditorError('view-position-before-root', item, {
                root: item
            });
        }
        return new ViewPosition(item.parent, item.index);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewPosition.prototype.is = function(type) {
    return type === 'position' || type === 'view:position';
};

/**
 * Range in the view tree. A range is represented by its start and end {@link module:engine/view/position~ViewPosition positions}.
 *
 * In order to create a new position instance use the `createPosition*()` factory methods available in:
 *
 * * {@link module:engine/view/view~EditingView}
 * * {@link module:engine/view/downcastwriter~ViewDowncastWriter}
 * * {@link module:engine/view/upcastwriter~ViewUpcastWriter}
 */ class ViewRange extends ViewTypeCheckable {
    /**
	 * Start position.
	 */ start;
    /**
	 * End position.
	 */ end;
    /**
	 * Creates a range spanning from `start` position to `end` position.
	 *
	 * **Note:** Constructor creates it's own {@link module:engine/view/position~ViewPosition} instances basing on passed values.
	 *
	 * @param start Start position.
	 * @param end End position. If not set, range will be collapsed at the `start` position.
	 */ constructor(start, end = null){
        super();
        this.start = start.clone();
        this.end = end ? end.clone() : start.clone();
    }
    /**
	 * Iterable interface.
	 *
	 * Iterates over all {@link module:engine/view/item~ViewItem view items} that are in this range and returns
	 * them together with additional information like length or {@link module:engine/view/position~ViewPosition positions},
	 * grouped as {@link module:engine/view/treewalker~ViewTreeWalkerValue}.
	 *
	 * This iterator uses {@link module:engine/view/treewalker~ViewTreeWalker TreeWalker} with `boundaries` set to this range and
	 * `ignoreElementEnd` option
	 * set to `true`.
	 */ *[Symbol.iterator]() {
        yield* new ViewTreeWalker({
            boundaries: this,
            ignoreElementEnd: true
        });
    }
    /**
	 * Returns whether the range is collapsed, that is it start and end positions are equal.
	 */ get isCollapsed() {
        return this.start.isEqual(this.end);
    }
    /**
	 * Returns whether this range is flat, that is if {@link module:engine/view/range~ViewRange#start start} position and
	 * {@link module:engine/view/range~ViewRange#end end} position are in the same
	 * {@link module:engine/view/position~ViewPosition#parent parent}.
	 */ get isFlat() {
        return this.start.parent === this.end.parent;
    }
    /**
	 * Range root element.
	 */ get root() {
        return this.start.root;
    }
    /**
	 * Creates a maximal range that has the same content as this range but is expanded in both ways (at the beginning
	 * and at the end).
	 *
	 * For example:
	 *
	 * ```html
	 * <p>Foo</p><p><b>{Bar}</b></p> -> <p>Foo</p>[<p><b>Bar</b>]</p>
	 * <p><b>foo</b>{bar}<span></span></p> -> <p><b>foo[</b>bar<span></span>]</p>
	 * ```
	 *
	 * Note that in the sample above:
	 *
	 * - `<p>` have type of {@link module:engine/view/containerelement~ViewContainerElement},
	 * - `<b>` have type of {@link module:engine/view/attributeelement~ViewAttributeElement},
	 * - `<span>` have type of {@link module:engine/view/uielement~ViewUIElement}.
	 *
	 * @returns Enlarged range.
	 */ getEnlarged() {
        let start = this.start.getLastMatchingPosition(enlargeTrimSkip, {
            direction: 'backward'
        });
        let end = this.end.getLastMatchingPosition(enlargeTrimSkip);
        // Fix positions, in case if they are in Text node.
        if (start.parent.is('$text') && start.isAtStart) {
            start = ViewPosition._createBefore(start.parent);
        }
        if (end.parent.is('$text') && end.isAtEnd) {
            end = ViewPosition._createAfter(end.parent);
        }
        return new ViewRange(start, end);
    }
    /**
	 * Creates a minimum range that has the same content as this range but is trimmed in both ways (at the beginning
	 * and at the end).
	 *
	 * For example:
	 *
	 * ```html
	 * <p>Foo</p>[<p><b>Bar</b>]</p> -> <p>Foo</p><p><b>{Bar}</b></p>
	 * <p><b>foo[</b>bar<span></span>]</p> -> <p><b>foo</b>{bar}<span></span></p>
	 * ```
	 *
	 * Note that in the sample above:
	 *
	 * - `<p>` have type of {@link module:engine/view/containerelement~ViewContainerElement},
	 * - `<b>` have type of {@link module:engine/view/attributeelement~ViewAttributeElement},
	 * - `<span>` have type of {@link module:engine/view/uielement~ViewUIElement}.
	 *
	 * @returns Shrunk range.
	 */ getTrimmed() {
        let start = this.start.getLastMatchingPosition(enlargeTrimSkip);
        if (start.isAfter(this.end) || start.isEqual(this.end)) {
            return new ViewRange(start, start);
        }
        let end = this.end.getLastMatchingPosition(enlargeTrimSkip, {
            direction: 'backward'
        });
        const nodeAfterStart = start.nodeAfter;
        const nodeBeforeEnd = end.nodeBefore;
        // Because TreeWalker prefers positions next to text node, we need to move them manually into these text nodes.
        if (nodeAfterStart && nodeAfterStart.is('$text')) {
            start = new ViewPosition(nodeAfterStart, 0);
        }
        if (nodeBeforeEnd && nodeBeforeEnd.is('$text')) {
            end = new ViewPosition(nodeBeforeEnd, nodeBeforeEnd.data.length);
        }
        return new ViewRange(start, end);
    }
    /**
	 * Two ranges are equal if their start and end positions are equal.
	 *
	 * @param otherRange Range to compare with.
	 * @returns `true` if ranges are equal, `false` otherwise
	 */ isEqual(otherRange) {
        return this == otherRange || this.start.isEqual(otherRange.start) && this.end.isEqual(otherRange.end);
    }
    /**
	 * Checks whether this range contains given {@link module:engine/view/position~ViewPosition position}.
	 *
	 * @param position Position to check.
	 * @returns `true` if given {@link module:engine/view/position~ViewPosition position} is contained in this range, `false` otherwise.
	 */ containsPosition(position) {
        return position.isAfter(this.start) && position.isBefore(this.end);
    }
    /**
	 * Checks whether this range contains given {@link module:engine/view/range~ViewRange range}.
	 *
	 * @param otherRange Range to check.
	 * @param loose Whether the check is loose or strict. If the check is strict (`false`), compared range cannot
	 * start or end at the same position as this range boundaries. If the check is loose (`true`), compared range can start, end or
	 * even be equal to this range. Note that collapsed ranges are always compared in strict mode.
	 * @returns `true` if given {@link module:engine/view/range~ViewRange range} boundaries are contained by this range, `false`
	 * otherwise.
	 */ containsRange(otherRange, loose = false) {
        if (otherRange.isCollapsed) {
            loose = false;
        }
        const containsStart = this.containsPosition(otherRange.start) || loose && this.start.isEqual(otherRange.start);
        const containsEnd = this.containsPosition(otherRange.end) || loose && this.end.isEqual(otherRange.end);
        return containsStart && containsEnd;
    }
    /**
	 * Computes which part(s) of this {@link module:engine/view/range~ViewRange range} is not a part of given
	 * {@link module:engine/view/range~ViewRange range}.
	 * Returned array contains zero, one or two {@link module:engine/view/range~ViewRange ranges}.
	 *
	 * Examples:
	 *
	 * ```ts
	 * let foo = downcastWriter.createText( 'foo' );
	 * let img = downcastWriter.createContainerElement( 'img' );
	 * let bar = downcastWriter.createText( 'bar' );
	 * let p = downcastWriter.createContainerElement( 'p', null, [ foo, img, bar ] );
	 *
	 * let range = view.createRange( view.createPositionAt( foo, 2 ), view.createPositionAt( bar, 1 ); // "o", img, "b" are in range.
	 * let otherRange = view.createRange( // "oo", img, "ba" are in range.
	 * 	view.createPositionAt( foo, 1 ),
	 * 	view.createPositionAt( bar, 2 )
	 * );
	 * let transformed = range.getDifference( otherRange );
	 * // transformed array has no ranges because `otherRange` contains `range`
	 *
	 * otherRange = view.createRange( view.createPositionAt( foo, 1 ), view.createPositionAt( p, 2 ); // "oo", img are in range.
	 * transformed = range.getDifference( otherRange );
	 * // transformed array has one range: from ( p, 2 ) to ( bar, 1 )
	 *
	 * otherRange = view.createRange( view.createPositionAt( p, 1 ), view.createPositionAt( p, 2 ) ); // img is in range.
	 * transformed = range.getDifference( otherRange );
	 * // transformed array has two ranges: from ( foo, 1 ) to ( p, 1 ) and from ( p, 2 ) to ( bar, 1 )
	 * ```
	 *
	 * @param otherRange Range to differentiate against.
	 * @returns The difference between ranges.
	 */ getDifference(otherRange) {
        const ranges = [];
        if (this.isIntersecting(otherRange)) {
            // Ranges intersect.
            if (this.containsPosition(otherRange.start)) {
                // Given range start is inside this range. This means that we have to
                // add shrunken range - from the start to the middle of this range.
                ranges.push(new ViewRange(this.start, otherRange.start));
            }
            if (this.containsPosition(otherRange.end)) {
                // Given range end is inside this range. This means that we have to
                // add shrunken range - from the middle of this range to the end.
                ranges.push(new ViewRange(otherRange.end, this.end));
            }
        } else {
            // Ranges do not intersect, return the original range.
            ranges.push(this.clone());
        }
        return ranges;
    }
    /**
	 * Returns an intersection of this {@link module:engine/view/range~ViewRange range}
	 * and given {@link module:engine/view/range~ViewRange range}.
	 * Intersection is a common part of both of those ranges. If ranges has no common part, returns `null`.
	 *
	 * Examples:
	 *
	 * ```ts
	 * let foo = downcastWriter.createText( 'foo' );
	 * let img = downcastWriter.createContainerElement( 'img' );
	 * let bar = downcastWriter.createText( 'bar' );
	 * let p = downcastWriter.createContainerElement( 'p', null, [ foo, img, bar ] );
	 *
	 * let range = view.createRange( view.createPositionAt( foo, 2 ), view.createPositionAt( bar, 1 ); // "o", img, "b" are in range.
	 * let otherRange = view.createRange( view.createPositionAt( foo, 1 ), view.createPositionAt( p, 2 ); // "oo", img are in range.
	 * let transformed = range.getIntersection( otherRange ); // range from ( foo, 1 ) to ( p, 2 ).
	 *
	 * otherRange = view.createRange( view.createPositionAt( bar, 1 ), view.createPositionAt( bar, 3 ); "ar" is in range.
	 * transformed = range.getIntersection( otherRange ); // null - no common part.
	 * ```
	 *
	 * @param otherRange Range to check for intersection.
	 * @returns A common part of given ranges or `null` if ranges have no common part.
	 */ getIntersection(otherRange) {
        if (this.isIntersecting(otherRange)) {
            // Ranges intersect, so a common range will be returned.
            // At most, it will be same as this range.
            let commonRangeStart = this.start;
            let commonRangeEnd = this.end;
            if (this.containsPosition(otherRange.start)) {
                // Given range start is inside this range. This means thaNt we have to
                // shrink common range to the given range start.
                commonRangeStart = otherRange.start;
            }
            if (this.containsPosition(otherRange.end)) {
                // Given range end is inside this range. This means that we have to
                // shrink common range to the given range end.
                commonRangeEnd = otherRange.end;
            }
            return new ViewRange(commonRangeStart, commonRangeEnd);
        }
        // Ranges do not intersect, so they do not have common part.
        return null;
    }
    /**
	 * Creates a {@link module:engine/view/treewalker~ViewTreeWalker TreeWalker} instance with this range as a boundary.
	 *
	 * @param options Object with configuration options. See {@link module:engine/view/treewalker~ViewTreeWalker}.
	 */ getWalker(options = {}) {
        options.boundaries = this;
        return new ViewTreeWalker(options);
    }
    /**
	 * Returns a {@link module:engine/view/node~ViewNode} or {@link module:engine/view/documentfragment~ViewDocumentFragment}
	 * which is a common ancestor of range's both ends (in which the entire range is contained).
	 */ getCommonAncestor() {
        return this.start.getCommonAncestor(this.end);
    }
    /**
	 * Returns an {@link module:engine/view/element~ViewElement Element} contained by the range.
	 * The element will be returned when it is the **only** node within the range and **fully–contained**
	 * at the same time.
	 */ getContainedElement() {
        if (this.isCollapsed) {
            return null;
        }
        let nodeAfterStart = this.start.nodeAfter;
        let nodeBeforeEnd = this.end.nodeBefore;
        // Handle the situation when the range position is at the beginning / at the end of a text node.
        // In such situation `.nodeAfter` and `.nodeBefore` are `null` but the range still might be spanning
        // over one element.
        //
        // <p>Foo{<span class="widget"></span>}bar</p> vs <p>Foo[<span class="widget"></span>]bar</p>
        //
        // These are basically the same range, only the difference is if the range position is at
        // at the end/at the beginning of a text node or just before/just after the text node.
        //
        if (this.start.parent.is('$text') && this.start.isAtEnd && this.start.parent.nextSibling) {
            nodeAfterStart = this.start.parent.nextSibling;
        }
        if (this.end.parent.is('$text') && this.end.isAtStart && this.end.parent.previousSibling) {
            nodeBeforeEnd = this.end.parent.previousSibling;
        }
        if (nodeAfterStart && nodeAfterStart.is('element') && nodeAfterStart === nodeBeforeEnd) {
            return nodeAfterStart;
        }
        return null;
    }
    /**
	 * Clones this range.
	 */ clone() {
        return new ViewRange(this.start, this.end);
    }
    /**
	 * Returns an iterator that iterates over all {@link module:engine/view/item~ViewItem view items} that are in this range and returns
	 * them.
	 *
	 * This method uses {@link module:engine/view/treewalker~ViewTreeWalker} with `boundaries` set to this range
	 * and `ignoreElementEnd` option set to `true`. However it returns only {@link module:engine/view/item~ViewItem items},
	 * not {@link module:engine/view/treewalker~ViewTreeWalkerValue}.
	 *
	 * You may specify additional options for the tree walker. See {@link module:engine/view/treewalker~ViewTreeWalker} for
	 * a full list of available options.
	 *
	 * @param options Object with configuration options. See {@link module:engine/view/treewalker~ViewTreeWalker}.
	 */ *getItems(options = {}) {
        options.boundaries = this;
        options.ignoreElementEnd = true;
        const treeWalker = new ViewTreeWalker(options);
        for (const value of treeWalker){
            yield value.item;
        }
    }
    /**
	 * Returns an iterator that iterates over all {@link module:engine/view/position~ViewPosition positions} that are boundaries or
	 * contained in this range.
	 *
	 * This method uses {@link module:engine/view/treewalker~ViewTreeWalker} with `boundaries` set to this range. However it returns only
	 * {@link module:engine/view/position~ViewPosition positions}, not {@link module:engine/view/treewalker~ViewTreeWalkerValue}.
	 *
	 * You may specify additional options for the tree walker. See {@link module:engine/view/treewalker~ViewTreeWalker} for
	 * a full list of available options.
	 *
	 * @param options Object with configuration options. See {@link module:engine/view/treewalker~ViewTreeWalker}.
	 */ *getPositions(options = {}) {
        options.boundaries = this;
        const treeWalker = new ViewTreeWalker(options);
        yield treeWalker.position;
        for (const value of treeWalker){
            yield value.nextPosition;
        }
    }
    /**
	 * Checks and returns whether this range intersects with the given range.
	 *
	 * @param otherRange Range to compare with.
	 * @returns True if ranges intersect.
	 */ isIntersecting(otherRange) {
        return this.start.isBefore(otherRange.end) && this.end.isAfter(otherRange.start);
    }
    /**
	 * Converts `ViewRange` instance to plain object and returns it.
	 *
	 * @returns `ViewRange` instance converted to plain object.
	 */ toJSON() {
        return {
            start: this.start.toJSON(),
            end: this.end.toJSON()
        };
    }
    /**
	 * Creates a range from the given parents and offsets.
	 *
	 * @internal
	 * @param startElement Start position parent element.
	 * @param startOffset Start position offset.
	 * @param endElement End position parent element.
	 * @param endOffset End position offset.
	 * @returns Created range.
	 */ static _createFromParentsAndOffsets(startElement, startOffset, endElement, endOffset) {
        return new this(new ViewPosition(startElement, startOffset), new ViewPosition(endElement, endOffset));
    }
    /**
	 * Creates a new range, spreading from specified {@link module:engine/view/position~ViewPosition position} to a position moved by
	 * given `shift`. If `shift` is a negative value, shifted position is treated as the beginning of the range.
	 *
	 * @internal
	 * @param position Beginning of the range.
	 * @param shift How long the range should be.
	 */ static _createFromPositionAndShift(position, shift) {
        const start = position;
        const end = position.getShiftedBy(shift);
        return shift > 0 ? new this(start, end) : new this(end, start);
    }
    /**
	 * Creates a range inside an {@link module:engine/view/element~ViewElement element} which starts before the first child of
	 * that element and ends after the last child of that element.
	 *
	 * @internal
	 * @param element Element which is a parent for the range.
	 */ static _createIn(element) {
        return this._createFromParentsAndOffsets(element, 0, element, element.childCount);
    }
    /**
	 * Creates a range that starts before given {@link module:engine/view/item~ViewItem view item} and ends after it.
	 *
	 * @internal
	 */ static _createOn(item) {
        const size = item.is('$textProxy') ? item.offsetSize : 1;
        return this._createFromPositionAndShift(ViewPosition._createBefore(item), size);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewRange.prototype.is = function(type) {
    return type === 'range' || type === 'view:range';
};
/**
 * Function used by getEnlarged and getTrimmed methods.
 */ function enlargeTrimSkip(value) {
    if (value.item.is('attributeElement') || value.item.is('uiElement')) {
        return true;
    }
    return false;
}

/**
 * Class representing an arbirtary selection in the view.
 * See also {@link module:engine/view/documentselection~ViewDocumentSelection}.
 *
 * New selection instances can be created via the constructor or one these methods:
 *
 * * {@link module:engine/view/view~EditingView#createSelection `View#createSelection()`},
 * * {@link module:engine/view/upcastwriter~ViewUpcastWriter#createSelection `UpcastWriter#createSelection()`}.
 *
 * A selection can consist of {@link module:engine/view/range~ViewRange ranges} that can be set by using
 * the {@link module:engine/view/selection~ViewSelection#setTo `Selection#setTo()`} method.
 */ class ViewSelection extends /* #__PURE__ */ EmitterMixin(ViewTypeCheckable) {
    /**
	 * Stores all ranges that are selected.
	 */ _ranges;
    /**
	 * Specifies whether the last added range was added as a backward or forward range.
	 */ _lastRangeBackward;
    /**
	 * Specifies whether selection instance is fake.
	 */ _isFake;
    /**
	 * Fake selection's label.
	 */ _fakeSelectionLabel;
    /**
	 * Creates new selection instance.
	 *
	 * **Note**: The selection constructor is available as a factory method:
	 *
	 * * {@link module:engine/view/view~EditingView#createSelection `View#createSelection()`},
	 * * {@link module:engine/view/upcastwriter~ViewUpcastWriter#createSelection `UpcastWriter#createSelection()`}.
	 *
	 * ```ts
	 * // Creates empty selection without ranges.
	 * const selection = writer.createSelection();
	 *
	 * // Creates selection at the given range.
	 * const range = writer.createRange( start, end );
	 * const selection = writer.createSelection( range );
	 *
	 * // Creates selection at the given ranges
	 * const ranges = [ writer.createRange( start1, end2 ), writer.createRange( star2, end2 ) ];
	 * const selection = writer.createSelection( ranges );
	 *
	 * // Creates selection from the other selection.
	 * const otherSelection = writer.createSelection();
	 * const selection = writer.createSelection( otherSelection );
	 *
	 * // Creates selection from the document selection.
	 * const selection = writer.createSelection( editor.editing.view.document.selection );
	 *
	 * // Creates selection at the given position.
	 * const position = writer.createPositionFromPath( root, path );
	 * const selection = writer.createSelection( position );
	 *
	 * // Creates collapsed selection at the position of given item and offset.
	 * const paragraph = writer.createContainerElement( 'paragraph' );
	 * const selection = writer.createSelection( paragraph, offset );
	 *
	 * // Creates a range inside an {@link module:engine/view/element~ViewElement element} which starts before the
	 * // first child of that element and ends after the last child of that element.
	 * const selection = writer.createSelection( paragraph, 'in' );
	 *
	 * // Creates a range on an {@link module:engine/view/item~ViewItem item} which starts before the item and ends
	 * // just after the item.
	 * const selection = writer.createSelection( paragraph, 'on' );
	 * ```
	 *
	 * `Selection`'s constructor allow passing additional options (`backward`, `fake` and `label`) as the last argument.
	 *
	 * ```ts
	 * // Creates backward selection.
	 * const selection = writer.createSelection( range, { backward: true } );
	 * ```
	 *
	 * Fake selection does not render as browser native selection over selected elements and is hidden to the user.
	 * This way, no native selection UI artifacts are displayed to the user and selection over elements can be
	 * represented in other way, for example by applying proper CSS class.
	 *
	 * Additionally fake's selection label can be provided. It will be used to describe fake selection in DOM
	 * (and be  properly handled by screen readers).
	 *
	 * ```ts
	 * // Creates fake selection with label.
	 * const selection = writer.createSelection( range, { fake: true, label: 'foo' } );
	 * ```
	 *
	 * @internal
	 */ constructor(...args){
        super();
        this._ranges = [];
        this._lastRangeBackward = false;
        this._isFake = false;
        this._fakeSelectionLabel = '';
        if (args.length) {
            this.setTo(...args);
        }
    }
    /**
	 * Returns true if selection instance is marked as `fake`.
	 *
	 * @see #setTo
	 */ get isFake() {
        return this._isFake;
    }
    /**
	 * Returns fake selection label.
	 *
	 * @see #setTo
	 */ get fakeSelectionLabel() {
        return this._fakeSelectionLabel;
    }
    /**
	 * Selection anchor. Anchor may be described as a position where the selection starts. Together with
	 * {@link #focus focus} they define the direction of selection, which is important
	 * when expanding/shrinking selection. Anchor is always the start or end of the most recent added range.
	 * It may be a bit unintuitive when there are multiple ranges in selection.
	 *
	 * @see #focus
	 */ get anchor() {
        if (!this._ranges.length) {
            return null;
        }
        const range = this._ranges[this._ranges.length - 1];
        const anchor = this._lastRangeBackward ? range.end : range.start;
        return anchor.clone();
    }
    /**
	 * Selection focus. Focus is a position where the selection ends.
	 *
	 * @see #anchor
	 */ get focus() {
        if (!this._ranges.length) {
            return null;
        }
        const range = this._ranges[this._ranges.length - 1];
        const focus = this._lastRangeBackward ? range.start : range.end;
        return focus.clone();
    }
    /**
	 * Returns whether the selection is collapsed. Selection is collapsed when there is exactly one range which is
	 * collapsed.
	 */ get isCollapsed() {
        return this.rangeCount === 1 && this._ranges[0].isCollapsed;
    }
    /**
	 * Returns number of ranges in selection.
	 */ get rangeCount() {
        return this._ranges.length;
    }
    /**
	 * Specifies whether the {@link #focus} precedes {@link #anchor}.
	 */ get isBackward() {
        return !this.isCollapsed && this._lastRangeBackward;
    }
    /**
	 * {@link module:engine/view/editableelement~ViewEditableElement ViewEditableElement} instance that contains this selection, or `null`
	 * if the selection is not inside an editable element.
	 */ get editableElement() {
        if (this.anchor) {
            return this.anchor.editableElement;
        }
        return null;
    }
    /**
	 * Returns an iterable that contains copies of all ranges added to the selection.
	 */ *getRanges() {
        for (const range of this._ranges){
            yield range.clone();
        }
    }
    /**
	 * Returns copy of the first range in the selection. First range is the one which
	 * {@link module:engine/view/range~ViewRange#start start} position
	 * {@link module:engine/view/position~ViewPosition#isBefore is before} start
	 * position of all other ranges (not to confuse with the first range added to the selection).
	 * Returns `null` if no ranges are added to selection.
	 */ getFirstRange() {
        let first = null;
        for (const range of this._ranges){
            if (!first || range.start.isBefore(first.start)) {
                first = range;
            }
        }
        return first ? first.clone() : null;
    }
    /**
	 * Returns copy of the last range in the selection. Last range is the one which {@link module:engine/view/range~ViewRange#end end}
	 * position {@link module:engine/view/position~ViewPosition#isAfter is after} end position of all other ranges (not to confuse
	 * with the last range added to the selection). Returns `null` if no ranges are added to selection.
	 */ getLastRange() {
        let last = null;
        for (const range of this._ranges){
            if (!last || range.end.isAfter(last.end)) {
                last = range;
            }
        }
        return last ? last.clone() : null;
    }
    /**
	 * Returns copy of the first position in the selection. First position is the position that
	 * {@link module:engine/view/position~ViewPosition#isBefore is before} any other position in the selection ranges.
	 * Returns `null` if no ranges are added to selection.
	 */ getFirstPosition() {
        const firstRange = this.getFirstRange();
        return firstRange ? firstRange.start.clone() : null;
    }
    /**
	 * Returns copy of the last position in the selection. Last position is the position that
	 * {@link module:engine/view/position~ViewPosition#isAfter is after} any other position in the selection ranges.
	 * Returns `null` if no ranges are added to selection.
	 */ getLastPosition() {
        const lastRange = this.getLastRange();
        return lastRange ? lastRange.end.clone() : null;
    }
    /**
	 * Checks whether, this selection is equal to given selection. Selections are equal if they have same directions,
	 * same number of ranges and all ranges from one selection equal to a range from other selection.
	 *
	 * @param otherSelection Selection to compare with.
	 * @returns `true` if selections are equal, `false` otherwise.
	 */ isEqual(otherSelection) {
        if (this.isFake != otherSelection.isFake) {
            return false;
        }
        if (this.isFake && this.fakeSelectionLabel != otherSelection.fakeSelectionLabel) {
            return false;
        }
        if (this.rangeCount != otherSelection.rangeCount) {
            return false;
        } else if (this.rangeCount === 0) {
            return true;
        }
        if (!this.anchor.isEqual(otherSelection.anchor) || !this.focus.isEqual(otherSelection.focus)) {
            return false;
        }
        for (const thisRange of this._ranges){
            let found = false;
            for (const otherRange of otherSelection._ranges){
                if (thisRange.isEqual(otherRange)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Checks whether this selection is similar to given selection. Selections are similar if they have same directions, same
	 * number of ranges, and all {@link module:engine/view/range~ViewRange#getTrimmed trimmed} ranges from one selection are
	 * equal to any trimmed range from other selection.
	 *
	 * @param otherSelection Selection to compare with.
	 * @returns `true` if selections are similar, `false` otherwise.
	 */ isSimilar(otherSelection) {
        if (this.isBackward != otherSelection.isBackward) {
            return false;
        }
        const numOfRangesA = count(this.getRanges());
        const numOfRangesB = count(otherSelection.getRanges());
        // If selections have different number of ranges, they cannot be similar.
        if (numOfRangesA != numOfRangesB) {
            return false;
        }
        // If both selections have no ranges, they are similar.
        if (numOfRangesA == 0) {
            return true;
        }
        // Check if each range in one selection has a similar range in other selection.
        for (let rangeA of this.getRanges()){
            rangeA = rangeA.getTrimmed();
            let found = false;
            for (let rangeB of otherSelection.getRanges()){
                rangeB = rangeB.getTrimmed();
                if (rangeA.start.isEqual(rangeB.start) && rangeA.end.isEqual(rangeB.end)) {
                    found = true;
                    break;
                }
            }
            // For `rangeA`, neither range in `otherSelection` was similar. So selections are not similar.
            if (!found) {
                return false;
            }
        }
        // There were no ranges that weren't matched. Selections are similar.
        return true;
    }
    /**
	 * Returns the selected element. {@link module:engine/view/element~ViewElement Element} is considered as selected if there is only
	 * one range in the selection, and that range contains exactly one element.
	 * Returns `null` if there is no selected element.
	 */ getSelectedElement() {
        if (this.rangeCount !== 1) {
            return null;
        }
        return this.getFirstRange().getContainedElement();
    }
    /**
	 * Sets this selection's ranges and direction to the specified location based on the given
	 * {@link module:engine/view/selection~ViewSelectable selectable}.
	 *
	 * ```ts
	 * // Sets selection to the given range.
	 * const range = writer.createRange( start, end );
	 * selection.setTo( range );
	 *
	 * // Sets selection to given ranges.
	 * const ranges = [ writer.createRange( start1, end2 ), writer.createRange( star2, end2 ) ];
	 * selection.setTo( range );
	 *
	 * // Sets selection to the other selection.
	 * const otherSelection = writer.createSelection();
	 * selection.setTo( otherSelection );
	 *
	 * // Sets selection to contents of ViewDocumentSelection.
	 * selection.setTo( editor.editing.view.document.selection );
	 *
	 * // Sets collapsed selection at the given position.
	 * const position = writer.createPositionAt( root, path );
	 * selection.setTo( position );
	 *
	 * // Sets collapsed selection at the position of given item and offset.
	 * selection.setTo( paragraph, offset );
	 * ```
	 *
	 * Creates a range inside an {@link module:engine/view/element~ViewElement element} which starts before the first child of
	 * that element and ends after the last child of that element.
	 *
	 * ```ts
	 * selection.setTo( paragraph, 'in' );
	 * ```
	 *
	 * Creates a range on an {@link module:engine/view/item~ViewItem item} which starts before the item and ends just after the item.
	 *
	 * ```ts
	 * selection.setTo( paragraph, 'on' );
	 *
	 * // Clears selection. Removes all ranges.
	 * selection.setTo( null );
	 * ```
	 *
	 * `Selection#setTo()` method allow passing additional options (`backward`, `fake` and `label`) as the last argument.
	 *
	 * ```ts
	 * // Sets selection as backward.
	 * selection.setTo( range, { backward: true } );
	 * ```
	 *
	 * Fake selection does not render as browser native selection over selected elements and is hidden to the user.
	 * This way, no native selection UI artifacts are displayed to the user and selection over elements can be
	 * represented in other way, for example by applying proper CSS class.
	 *
	 * Additionally fake's selection label can be provided. It will be used to describe fake selection in DOM
	 * (and be  properly handled by screen readers).
	 *
	 * ```ts
	 * // Creates fake selection with label.
	 * selection.setTo( range, { fake: true, label: 'foo' } );
	 * ```
	 *
	 * @fires change
	 */ setTo(...args) {
        let [selectable, placeOrOffset, options] = args;
        if (typeof placeOrOffset == 'object') {
            options = placeOrOffset;
            placeOrOffset = undefined;
        }
        if (selectable === null) {
            this._setRanges([]);
            this._setFakeOptions(options);
        } else if (selectable instanceof ViewSelection || selectable instanceof ViewDocumentSelection) {
            this._setRanges(selectable.getRanges(), selectable.isBackward);
            this._setFakeOptions({
                fake: selectable.isFake,
                label: selectable.fakeSelectionLabel
            });
        } else if (selectable instanceof ViewRange) {
            this._setRanges([
                selectable
            ], options && options.backward);
            this._setFakeOptions(options);
        } else if (selectable instanceof ViewPosition) {
            this._setRanges([
                new ViewRange(selectable)
            ]);
            this._setFakeOptions(options);
        } else if (selectable instanceof ViewNode) {
            const backward = !!options && !!options.backward;
            let range;
            if (placeOrOffset === undefined) {
                /**
				 * selection.setTo requires the second parameter when the first parameter is a node.
				 *
				 * @error view-selection-setto-required-second-parameter
				 */ throw new CKEditorError('view-selection-setto-required-second-parameter', this);
            } else if (placeOrOffset == 'in') {
                range = ViewRange._createIn(selectable);
            } else if (placeOrOffset == 'on') {
                range = ViewRange._createOn(selectable);
            } else {
                range = new ViewRange(ViewPosition._createAt(selectable, placeOrOffset));
            }
            this._setRanges([
                range
            ], backward);
            this._setFakeOptions(options);
        } else if (isIterable(selectable)) {
            // We assume that the selectable is an iterable of ranges.
            // Array.from() is used to prevent setting ranges to the old iterable
            this._setRanges(selectable, options && options.backward);
            this._setFakeOptions(options);
        } else {
            /**
			 * Cannot set selection to given place.
			 *
			 * @error view-selection-setto-not-selectable
			 */ throw new CKEditorError('view-selection-setto-not-selectable', this);
        }
        this.fire('change');
    }
    /**
	 * Moves {@link #focus} to the specified location.
	 *
	 * The location can be specified in the same form as
	 * {@link module:engine/view/view~EditingView#createPositionAt view.createPositionAt()}
	 * parameters.
	 *
	 * @fires change
	 * @param offset Offset or one of the flags. Used only when first parameter is a {@link module:engine/view/item~ViewItem view item}.
	 */ setFocus(itemOrPosition, offset) {
        if (this.anchor === null) {
            /**
			 * Cannot set selection focus if there are no ranges in selection.
			 *
			 * @error view-selection-setfocus-no-ranges
			 */ throw new CKEditorError('view-selection-setfocus-no-ranges', this);
        }
        const newFocus = ViewPosition._createAt(itemOrPosition, offset);
        if (newFocus.compareWith(this.focus) == 'same') {
            return;
        }
        const anchor = this.anchor;
        this._ranges.pop();
        if (newFocus.compareWith(anchor) == 'before') {
            this._addRange(new ViewRange(newFocus, anchor), true);
        } else {
            this._addRange(new ViewRange(anchor, newFocus));
        }
        this.fire('change');
    }
    /**
	 * Converts `ViewSelection` instance to plain object and returns it.
	 *
	 * @returns `ViewSelection` instance converted to plain object.
	 */ toJSON() {
        const json = {
            ranges: Array.from(this.getRanges()).map((range)=>range.toJSON())
        };
        if (this.isBackward) {
            json.isBackward = true;
        }
        if (this.isFake) {
            json.isFake = true;
        }
        return json;
    }
    /**
	 * Replaces all ranges that were added to the selection with given array of ranges. Last range of the array
	 * is treated like the last added range and is used to set {@link #anchor anchor} and {@link #focus focus}.
	 * Accepts a flag describing in which way the selection is made.
	 *
	 * @param newRanges Iterable object of ranges to set.
	 * @param isLastBackward Flag describing if last added range was selected forward - from start to end
	 * (`false`) or backward - from end to start (`true`). Defaults to `false`.
	 */ _setRanges(newRanges, isLastBackward = false) {
        // New ranges should be copied to prevent removing them by setting them to `[]` first.
        // Only applies to situations when selection is set to the same selection or same selection's ranges.
        newRanges = Array.from(newRanges);
        this._ranges = [];
        for (const range of newRanges){
            this._addRange(range);
        }
        this._lastRangeBackward = !!isLastBackward;
    }
    /**
	 * Sets this selection instance to be marked as `fake`. A fake selection does not render as browser native selection
	 * over selected elements and is hidden to the user. This way, no native selection UI artifacts are displayed to
	 * the user and selection over elements can be represented in other way, for example by applying proper CSS class.
	 *
	 * Additionally fake's selection label can be provided. It will be used to describe fake selection in DOM (and be
	 * properly handled by screen readers).
	 */ _setFakeOptions(options = {}) {
        this._isFake = !!options.fake;
        this._fakeSelectionLabel = options.fake ? options.label || '' : '';
    }
    /**
	 * Adds a range to the selection. Added range is copied. This means that passed range is not saved in the
	 * selection instance and you can safely operate on it.
	 *
	 * Accepts a flag describing in which way the selection is made - passed range might be selected from
	 * {@link module:engine/view/range~ViewRange#start start} to {@link module:engine/view/range~ViewRange#end end}
	 * or from {@link module:engine/view/range~ViewRange#end end} to {@link module:engine/view/range~ViewRange#start start}.
	 * The flag is used to set {@link #anchor anchor} and {@link #focus focus} properties.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-selection-range-intersects` if added range intersects
	 * with ranges already stored in Selection instance.
	 */ _addRange(range, isBackward = false) {
        if (!(range instanceof ViewRange)) {
            /**
			 * Selection range set to an object that is not an instance of {@link module:engine/view/range~ViewRange}.
			 *
			 * @error view-selection-add-range-not-range
			 */ throw new CKEditorError('view-selection-add-range-not-range', this);
        }
        this._pushRange(range);
        this._lastRangeBackward = !!isBackward;
    }
    /**
	 * Adds range to selection - creates copy of given range so it can be safely used and modified.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-selection-range-intersects` if added range intersects
	 * with ranges already stored in selection instance.
	 */ _pushRange(range) {
        for (const storedRange of this._ranges){
            if (range.isIntersecting(storedRange)) {
                /**
				 * Trying to add a range that intersects with another range from selection.
				 *
				 * @error view-selection-range-intersects
				 * @param {module:engine/view/range~ViewRange} addedRange Range that was added to the selection.
				 * @param {module:engine/view/range~ViewRange} intersectingRange Range from selection that intersects with `addedRange`.
				 */ throw new CKEditorError('view-selection-range-intersects', this, {
                    addedRange: range,
                    intersectingRange: storedRange
                });
            }
        }
        this._ranges.push(new ViewRange(range.start, range.end));
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewSelection.prototype.is = function(type) {
    return type === 'selection' || type === 'view:selection';
};

/**
 * Class representing the document selection in the view.
 *
 * Its instance is available in {@link module:engine/view/document~ViewDocument#selection `Document#selection`}.
 *
 * It is similar to {@link module:engine/view/selection~ViewSelection} but
 * it has a read-only API and can be modified only by the writer available in
 * the {@link module:engine/view/view~EditingView#change `View#change()`} block
 * (so via {@link module:engine/view/downcastwriter~ViewDowncastWriter#setSelection `ViewDowncastWriter#setSelection()`}).
 */ class ViewDocumentSelection extends /* #__PURE__ */ EmitterMixin(ViewTypeCheckable) {
    /**
	 * Selection is used internally (`ViewDocumentSelection` is a proxy to that selection).
	 */ _selection;
    constructor(...args){
        super();
        this._selection = new ViewSelection();
        // Delegate change event to be fired on ViewDocumentSelection instance.
        this._selection.delegate('change').to(this);
        // Set selection data.
        if (args.length) {
            this._selection.setTo(...args);
        }
    }
    /**
	 * Returns true if selection instance is marked as `fake`.
	 *
	 * @see #_setTo
	 */ get isFake() {
        return this._selection.isFake;
    }
    /**
	 * Returns fake selection label.
	 *
	 * @see #_setTo
	 */ get fakeSelectionLabel() {
        return this._selection.fakeSelectionLabel;
    }
    /**
	 * Selection anchor. Anchor may be described as a position where the selection starts. Together with
	 * {@link #focus focus} they define the direction of selection, which is important
	 * when expanding/shrinking selection. Anchor is always the start or end of the most recent added range.
	 * It may be a bit unintuitive when there are multiple ranges in selection.
	 *
	 * @see #focus
	 */ get anchor() {
        return this._selection.anchor;
    }
    /**
	 * Selection focus. Focus is a position where the selection ends.
	 *
	 * @see #anchor
	 */ get focus() {
        return this._selection.focus;
    }
    /**
	 * Returns whether the selection is collapsed. Selection is collapsed when there is exactly one range which is
	 * collapsed.
	 */ get isCollapsed() {
        return this._selection.isCollapsed;
    }
    /**
	 * Returns number of ranges in selection.
	 */ get rangeCount() {
        return this._selection.rangeCount;
    }
    /**
	 * Specifies whether the {@link #focus} precedes {@link #anchor}.
	 */ get isBackward() {
        return this._selection.isBackward;
    }
    /**
	 * {@link module:engine/view/editableelement~ViewEditableElement ViewEditableElement} instance that contains this selection, or `null`
	 * if the selection is not inside an editable element.
	 */ get editableElement() {
        return this._selection.editableElement;
    }
    /**
	 * Used for the compatibility with the {@link module:engine/view/selection~ViewSelection#isEqual} method.
	 *
	 * @internal
	 */ get _ranges() {
        return this._selection._ranges;
    }
    /**
	 * Returns an iterable that contains copies of all ranges added to the selection.
	 */ *getRanges() {
        yield* this._selection.getRanges();
    }
    /**
	 * Returns copy of the first range in the selection. First range is the one which
	 * {@link module:engine/view/range~ViewRange#start start} position
	 * {@link module:engine/view/position~ViewPosition#isBefore is before} start
	 * position of all other ranges (not to confuse with the first range added to the selection).
	 * Returns `null` if no ranges are added to selection.
	 */ getFirstRange() {
        return this._selection.getFirstRange();
    }
    /**
	 * Returns copy of the last range in the selection. Last range is the one which {@link module:engine/view/range~ViewRange#end end}
	 * position {@link module:engine/view/position~ViewPosition#isAfter is after} end position of all other ranges (not to confuse
	 * with the last range added to the selection). Returns `null` if no ranges are added to selection.
	 */ getLastRange() {
        return this._selection.getLastRange();
    }
    /**
	 * Returns copy of the first position in the selection. First position is the position that
	 * {@link module:engine/view/position~ViewPosition#isBefore is before} any other position in the selection ranges.
	 * Returns `null` if no ranges are added to selection.
	 */ getFirstPosition() {
        return this._selection.getFirstPosition();
    }
    /**
	 * Returns copy of the last position in the selection. Last position is the position that
	 * {@link module:engine/view/position~ViewPosition#isAfter is after} any other position in the selection ranges.
	 * Returns `null` if no ranges are added to selection.
	 */ getLastPosition() {
        return this._selection.getLastPosition();
    }
    /**
	 * Returns the selected element. {@link module:engine/view/element~ViewElement Element} is considered as selected if there is only
	 * one range in the selection, and that range contains exactly one element.
	 * Returns `null` if there is no selected element.
	 */ getSelectedElement() {
        return this._selection.getSelectedElement();
    }
    /**
	 * Checks whether, this selection is equal to given selection. Selections are equal if they have same directions,
	 * same number of ranges and all ranges from one selection equal to a range from other selection.
	 *
	 * @param otherSelection Selection to compare with.
	 * @returns `true` if selections are equal, `false` otherwise.
	 */ isEqual(otherSelection) {
        return this._selection.isEqual(otherSelection);
    }
    /**
	 * Checks whether this selection is similar to given selection. Selections are similar if they have same directions, same
	 * number of ranges, and all {@link module:engine/view/range~ViewRange#getTrimmed trimmed} ranges from one selection are
	 * equal to any trimmed range from other selection.
	 *
	 * @param otherSelection Selection to compare with.
	 * @returns `true` if selections are similar, `false` otherwise.
	 */ isSimilar(otherSelection) {
        return this._selection.isSimilar(otherSelection);
    }
    /**
	 * Converts `ViewDocumentSelection` instance to plain object and returns it.
	 *
	 * @returns `ViewDocumentSelection` instance converted to plain object.
	 */ toJSON() {
        return this._selection.toJSON();
    }
    /**
	 * Sets this selection's ranges and direction to the specified location based on the given
	 * {@link module:engine/view/selection~ViewSelectable selectable}.
	 *
	 * ```ts
	 * // Sets selection to the given range.
	 * const range = writer.createRange( start, end );
	 * documentSelection._setTo( range );
	 *
	 * // Sets selection to given ranges.
	 * const ranges = [ writer.createRange( start1, end2 ), writer.createRange( start2, end2 ) ];
	 * documentSelection._setTo( range );
	 *
	 * // Sets selection to the other selection.
	 * const otherSelection = writer.createSelection();
	 * documentSelection._setTo( otherSelection );
	 *
	 * // Sets collapsed selection at the given position.
	 * const position = writer.createPositionAt( root, offset );
	 * documentSelection._setTo( position );
	 *
	 * // Sets collapsed selection at the position of given item and offset.
	 * documentSelection._setTo( paragraph, offset );
	 * ```
	 *
	 * Creates a range inside an {@link module:engine/view/element~ViewElement element} which starts before the first child of
	 * that element and ends after the last child of that element.
	 *
	 * ```ts
	 * documentSelection._setTo( paragraph, 'in' );
	 * ```
	 *
	 * Creates a range on an {@link module:engine/view/item~ViewItem item} which starts before the item and ends just after the item.
	 *
	 * ```ts
	 * documentSelection._setTo( paragraph, 'on' );
	 *
	 * // Clears selection. Removes all ranges.
	 * documentSelection._setTo( null );
	 * ```
	 *
	 * `Selection#_setTo()` method allow passing additional options (`backward`, `fake` and `label`) as the last argument.
	 *
	 * ```ts
	 * // Sets selection as backward.
	 * documentSelection._setTo( range, { backward: true } );
	 * ```
	 *
	 * Fake selection does not render as browser native selection over selected elements and is hidden to the user.
	 * This way, no native selection UI artifacts are displayed to the user and selection over elements can be
	 * represented in other way, for example by applying proper CSS class.
	 *
	 * Additionally fake's selection label can be provided. It will be used to des cribe fake selection in DOM
	 * (and be  properly handled by screen readers).
	 *
	 * ```ts
	 * // Creates fake selection with label.
	 * documentSelection._setTo( range, { fake: true, label: 'foo' } );
	 * ```
	 *
	 * @internal
	 * @fires change
	 */ _setTo(...args) {
        this._selection.setTo(...args);
    }
    /**
	 * Moves {@link #focus} to the specified location.
	 *
	 * The location can be specified in the same form as
	 * {@link module:engine/view/view~EditingView#createPositionAt view.createPositionAt()}
	 * parameters.
	 *
	 * @internal
	 * @fires change
	 * @param offset Offset or one of the flags. Used only when first parameter is a {@link module:engine/view/item~ViewItem view item}.
	 */ _setFocus(itemOrPosition, offset) {
        this._selection.setFocus(itemOrPosition, offset);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewDocumentSelection.prototype.is = function(type) {
    return type === 'selection' || type == 'documentSelection' || type == 'view:selection' || type == 'view:documentSelection';
};

/**
 * The event object passed to bubbling event callbacks. It is used to provide information about the event as well as a tool to
 * manipulate it.
 */ class BubblingEventInfo extends EventInfo {
    /**
	 * The view range that the bubbling should start from.
	 */ startRange;
    /**
	 * The current event phase.
	 */ _eventPhase;
    /**
	 * The current bubbling target.
	 */ _currentTarget;
    /**
	 * @param source The emitter.
	 * @param name The event name.
	 * @param startRange The view range that the bubbling should start from.
	 */ constructor(source, name, startRange){
        super(source, name);
        this.startRange = startRange;
        this._eventPhase = 'none';
        this._currentTarget = null;
    }
    /**
	 * The current event phase.
	 */ get eventPhase() {
        return this._eventPhase;
    }
    /**
	 * The current bubbling target.
	 */ get currentTarget() {
        return this._currentTarget;
    }
}

const contextsSymbol = Symbol('bubbling contexts');
/**
 * Bubbling emitter mixin for the view document as described in the {@link ~BubblingEmitter} interface.
 *
 * This function creates a class that inherits from the provided `base` and implements `Emitter` interface.
 * The base class must implement {@link module:utils/emittermixin~Emitter} interface.
 *
 * ```ts
 * class BaseClass extends EmitterMixin() {
 * 	// ...
 * }
 *
 * class MyClass extends BubblingEmitterMixin( BaseClass ) {
 * 	// This class derives from `BaseClass` and implements the `BubblingEmitter` interface.
 * }
 * ```
 */ function BubblingEmitterMixin(base) {
    class Mixin extends base {
        fire(eventOrInfo, ...eventArgs) {
            try {
                const eventInfo = eventOrInfo instanceof EventInfo ? eventOrInfo : new EventInfo(this, eventOrInfo);
                const eventContexts = getBubblingContexts(this);
                if (!eventContexts.size) {
                    return;
                }
                updateEventInfo(eventInfo, 'capturing', this);
                // The capture phase of the event.
                if (fireListenerFor(eventContexts, '$capture', eventInfo, ...eventArgs)) {
                    return eventInfo.return;
                }
                const startRange = eventInfo.startRange || this.selection.getFirstRange();
                const selectedElement = startRange ? startRange.getContainedElement() : null;
                const isCustomContext = selectedElement ? Boolean(getCustomContext(eventContexts, selectedElement)) : false;
                let node = selectedElement || getDeeperRangeParent(startRange);
                updateEventInfo(eventInfo, 'atTarget', node);
                // For the not yet bubbling event trigger for $text node if selection can be there and it's not a custom context selected.
                if (!isCustomContext) {
                    if (fireListenerFor(eventContexts, '$text', eventInfo, ...eventArgs)) {
                        return eventInfo.return;
                    }
                    updateEventInfo(eventInfo, 'bubbling', node);
                }
                while(node){
                    // Root node handling.
                    if (node.is('rootElement')) {
                        if (fireListenerFor(eventContexts, '$root', eventInfo, ...eventArgs)) {
                            return eventInfo.return;
                        }
                    } else if (node.is('element')) {
                        if (fireListenerFor(eventContexts, node.name, eventInfo, ...eventArgs)) {
                            return eventInfo.return;
                        }
                    }
                    // Check custom contexts (i.e., a widget).
                    if (fireListenerFor(eventContexts, node, eventInfo, ...eventArgs)) {
                        return eventInfo.return;
                    }
                    node = node.parent;
                    updateEventInfo(eventInfo, 'bubbling', node);
                }
                updateEventInfo(eventInfo, 'bubbling', this);
                // Document context.
                fireListenerFor(eventContexts, '$document', eventInfo, ...eventArgs);
                return eventInfo.return;
            } catch (err) {
                // @if CK_DEBUG // throw err;
                /* istanbul ignore next -- @preserve */ CKEditorError.rethrowUnexpectedError(err, this);
            }
        }
        _addEventListener(event, callback, options) {
            const contexts = toArray(options.context || '$document');
            const eventContexts = getBubblingContexts(this);
            for (const context of contexts){
                let emitter = eventContexts.get(context);
                if (!emitter) {
                    emitter = new (EmitterMixin())();
                    eventContexts.set(context, emitter);
                }
                this.listenTo(emitter, event, callback, options);
            }
        }
        _removeEventListener(event, callback) {
            const eventContexts = getBubblingContexts(this);
            for (const emitter of eventContexts.values()){
                this.stopListening(emitter, event, callback);
            }
        }
    }
    return Mixin;
}
/**
 * Update the event info bubbling fields.
 *
 * @param eventInfo The event info object to update.
 * @param eventPhase The current event phase.
 * @param currentTarget The current bubbling target.
 */ function updateEventInfo(eventInfo, eventPhase, currentTarget) {
    if (eventInfo instanceof BubblingEventInfo) {
        eventInfo._eventPhase = eventPhase;
        eventInfo._currentTarget = currentTarget;
    }
}
/**
 * Fires the listener for the specified context. Returns `true` if event was stopped.
 *
 * @param eventInfo The `EventInfo` object.
 * @param eventArgs Additional arguments to be passed to the callbacks.
 * @returns True if event stop was called.
 */ function fireListenerFor(eventContexts, context, eventInfo, ...eventArgs) {
    const emitter = typeof context == 'string' ? eventContexts.get(context) : getCustomContext(eventContexts, context);
    if (!emitter) {
        return false;
    }
    emitter.fire(eventInfo, ...eventArgs);
    return eventInfo.stop.called;
}
/**
 * Returns an emitter for a specified view node.
 */ function getCustomContext(eventContexts, node) {
    for (const [context, emitter] of eventContexts){
        if (typeof context == 'function' && context(node)) {
            return emitter;
        }
    }
    return null;
}
/**
 * Returns bubbling contexts map for the source (emitter).
 */ function getBubblingContexts(source) {
    if (!source[contextsSymbol]) {
        source[contextsSymbol] = new Map();
    }
    return source[contextsSymbol];
}
/**
 * Returns the deeper parent element for the range.
 */ function getDeeperRangeParent(range) {
    if (!range) {
        return null;
    }
    const startParent = range.start.parent;
    const endParent = range.end.parent;
    const startPath = startParent.getPath();
    const endPath = endParent.getPath();
    return startPath.length > endPath.length ? startParent : endParent;
}

// @if CK_DEBUG_ENGINE // const { logDocument } = require( '../dev-utils/utils' );
/**
 * Document class creates an abstract layer over the content editable area, contains a tree of view elements and
 * {@link module:engine/view/documentselection~ViewDocumentSelection view selection} associated with this document.
 */ class ViewDocument extends /* #__PURE__ */ BubblingEmitterMixin(/* #__PURE__ */ ObservableMixin()) {
    /**
	 * Selection done on this document.
	 */ selection;
    /**
	 * Roots of the view tree. Collection of the {@link module:engine/view/element~ViewElement view elements}.
	 *
	 * View roots are created as a result of binding between {@link module:engine/view/document~ViewDocument#roots} and
	 * {@link module:engine/model/document~ModelDocument#roots} and this is handled by
	 * {@link module:engine/controller/editingcontroller~EditingController}, so to create view root we need to create
	 * model root using {@link module:engine/model/document~ModelDocument#createRoot}.
	 */ roots;
    /**
	 * The styles processor instance used by this document when normalizing styles.
	 */ stylesProcessor;
    /**
	 * Post-fixer callbacks registered to the view document.
	 */ _postFixers = new Set();
    /**
	 * Creates a Document instance.
	 *
	 * @param stylesProcessor The styles processor instance.
	 */ constructor(stylesProcessor){
        super();
        this.selection = new ViewDocumentSelection();
        this.roots = new Collection({
            idProperty: 'rootName'
        });
        this.stylesProcessor = stylesProcessor;
        this.set('isReadOnly', false);
        this.set('isFocused', false);
        this.set('isSelecting', false);
        this.set('isComposing', false);
    }
    /**
	 * Gets a {@link module:engine/view/document~ViewDocument#roots view root element} with the specified name. If the name is not
	 * specific "main" root is returned.
	 *
	 * @param name Name of the root.
	 * @returns The view root element with the specified name or null when there is no root of given name.
	 */ getRoot(name = 'main') {
        return this.roots.get(name);
    }
    /**
	 * Allows registering post-fixer callbacks. A post-fixers mechanism allows to update the view tree just before it is rendered
	 * to the DOM.
	 *
	 * Post-fixers are executed right after all changes from the outermost change block were applied but
	 * before the {@link module:engine/view/view~EditingView#event:render render event} is fired. If a post-fixer callback made
	 * a change, it should return `true`. When this happens, all post-fixers are fired again to check if something else should
	 * not be fixed in the new document tree state.
	 *
	 * View post-fixers are useful when you want to apply some fixes whenever the view structure changes. Keep in mind that
	 * changes executed in a view post-fixer should not break model-view mapping.
	 *
	 * The types of changes which should be safe:
	 *
	 * * adding or removing attribute from elements,
	 * * changes inside of {@link module:engine/view/uielement~ViewUIElement UI elements},
	 * * {@link module:engine/controller/editingcontroller~EditingController#reconvertItem marking some of the model elements to be
	 * re-converted}.
	 *
	 * Try to avoid changes which touch view structure:
	 *
	 * * you should not add or remove nor wrap or unwrap any view elements,
	 * * you should not change the editor data model in a view post-fixer.
	 *
	 * As a parameter, a post-fixer callback receives a {@link module:engine/view/downcastwriter~ViewDowncastWriter downcast writer}.
	 *
	 * Typically, a post-fixer will look like this:
	 *
	 * ```ts
	 * editor.editing.view.document.registerPostFixer( writer => {
	 * 	if ( checkSomeCondition() ) {
	 * 		writer.doSomething();
	 *
	 * 		// Let other post-fixers know that something changed.
	 * 		return true;
	 * 	}
	 *
	 * 	return false;
	 * } );
	 * ```
	 *
	 * Note that nothing happens right after you register a post-fixer (e.g. execute such a code in the console).
	 * That is because adding a post-fixer does not execute it.
	 * The post-fixer will be executed as soon as any change in the document needs to cause its rendering.
	 * If you want to re-render the editor's view after registering the post-fixer then you should do it manually by calling
	 * {@link module:engine/view/view~EditingView#forceRender `view.forceRender()`}.
	 *
	 * If you need to register a callback which is executed when DOM elements are already updated,
	 * use {@link module:engine/view/view~EditingView#event:render render event}.
	 */ registerPostFixer(postFixer) {
        this._postFixers.add(postFixer);
    }
    /**
	 * Destroys this instance. Makes sure that all observers are destroyed and listeners removed.
	 */ destroy() {
        this.roots.forEach((root)=>root.destroy());
        this.stopListening();
    }
    /**
	 * Performs post-fixer loops. Executes post-fixer callbacks as long as none of them has done any changes to the model.
	 *
	 * @internal
	 */ _callPostFixers(writer) {
        let wasFixed = false;
        do {
            for (const callback of this._postFixers){
                wasFixed = callback(writer);
                if (wasFixed) {
                    break;
                }
            }
        }while (wasFixed)
    }
}

// Default attribute priority.
const DEFAULT_PRIORITY = 10;
/**
 * Attribute elements are used to represent formatting elements in the view (think – `<b>`, `<span style="font-size: 2em">`, etc.).
 * Most often they are created when downcasting model text attributes.
 *
 * Editing engine does not define a fixed HTML DTD. This is why a feature developer needs to choose between various
 * types (container element, {@link module:engine/view/attributeelement~ViewAttributeElement attribute element},
 * {@link module:engine/view/emptyelement~ViewEmptyElement empty element}, etc) when developing a feature.
 *
 * To create a new attribute element instance use the
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createAttributeElement `ViewDowncastWriter#createAttributeElement()`} method.
 */ class ViewAttributeElement extends ViewElement {
    static DEFAULT_PRIORITY = DEFAULT_PRIORITY;
    /**
	 * Element priority. Decides in what order elements are wrapped by {@link module:engine/view/downcastwriter~ViewDowncastWriter}.
	 *
	 * @internal
	 * @readonly
	 */ _priority = DEFAULT_PRIORITY;
    /**
	 * Element identifier. If set, it is used by {@link module:engine/view/element~ViewElement#isSimilar},
	 * and then two elements are considered similar if, and only if they have the same `_id`.
	 *
	 * @internal
	 * @readonly
	 */ _id = null;
    /**
	 * Keeps all the attribute elements that have the same {@link module:engine/view/attributeelement~ViewAttributeElement#id ids}
	 * and still exist in the view tree.
	 *
	 * This property is managed by {@link module:engine/view/downcastwriter~ViewDowncastWriter}.
	 */ _clonesGroup = null;
    /**
	 * Creates an attribute element.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#createAttributeElement
	 * @see module:engine/view/element~ViewElement
	 * @protected
	 * @param document The document instance to which this element belongs.
	 * @param name Node name.
	 * @param attrs Collection of attributes.
	 * @param children A list of nodes to be inserted into created element.
	 */ constructor(document, name, attrs, children){
        super(document, name, attrs, children);
        this.getFillerOffset = getFillerOffset$3;
    }
    /**
	 * Element priority. Decides in what order elements are wrapped by {@link module:engine/view/downcastwriter~ViewDowncastWriter}.
	 */ get priority() {
        return this._priority;
    }
    /**
	 * Element identifier. If set, it is used by {@link module:engine/view/element~ViewElement#isSimilar},
	 * and then two elements are considered similar if, and only if they have the same `id`.
	 */ get id() {
        return this._id;
    }
    /**
	 * Returns all {@link module:engine/view/attributeelement~ViewAttributeElement attribute elements} that has the
	 * same {@link module:engine/view/attributeelement~ViewAttributeElement#id id} and are in the view tree (were not removed).
	 *
	 * Note: If this element has been removed from the tree, returned set will not include it.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError attribute-element-get-elements-with-same-id-no-id}
	 * if this element has no `id`.
	 *
	 * @returns Set containing all the attribute elements
	 * with the same `id` that were added and not removed from the view tree.
	 */ getElementsWithSameId() {
        if (this.id === null) {
            /**
			 * Cannot get elements with the same id for an attribute element without id.
			 *
			 * @error attribute-element-get-elements-with-same-id-no-id
			 */ throw new CKEditorError('attribute-element-get-elements-with-same-id-no-id', this);
        }
        return new Set(this._clonesGroup);
    }
    /**
	 * Checks if this element is similar to other element.
	 *
	 * If none of elements has set {@link module:engine/view/attributeelement~ViewAttributeElement#id}, then both elements
	 * should have the same name, attributes and priority to be considered as similar. Two similar elements can contain
	 * different set of children nodes.
	 *
	 * If at least one element has {@link module:engine/view/attributeelement~ViewAttributeElement#id} set, then both
	 * elements have to have the same {@link module:engine/view/attributeelement~ViewAttributeElement#id} value to be
	 * considered similar.
	 *
	 * Similarity is important for {@link module:engine/view/downcastwriter~ViewDowncastWriter}. For example:
	 *
	 * * two following similar elements can be merged together into one, longer element,
	 * * {@link module:engine/view/downcastwriter~ViewDowncastWriter#unwrap} checks similarity of passed element and processed element to
	 * decide whether processed element should be unwrapped,
	 * * etc.
	 */ isSimilar(otherElement) {
        // If any element has an `id` set, just compare the ids.
        if (this.id !== null || otherElement.id !== null) {
            return this.id === otherElement.id;
        }
        return super.isSimilar(otherElement) && this.priority == otherElement.priority;
    }
    /**
	 * Converts `ViewAttributeElement` instance to plain object and returns it.
	 *
	 * @returns `ViewAttributeElement` instance converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        json.type = 'AttributeElement';
        return json;
    }
    /**
	 * Clones provided element with priority.
	 *
	 * @internal
	 * @param deep If set to `true` clones element and all its children recursively. When set to `false`,
	 * element will be cloned without any children.
	 * @returns Clone of this element.
	 */ _clone(deep = false) {
        const cloned = super._clone(deep);
        // Clone priority too.
        cloned._priority = this._priority;
        // And id too.
        cloned._id = this._id;
        return cloned;
    }
    /**
	 * Used by {@link module:engine/view/element~ViewElement#_mergeAttributesFrom} to verify if the given element can be merged without
	 * conflicts into this element.
	 *
	 * @internal
	 */ _canMergeAttributesFrom(otherElement) {
        // Can't merge if any of elements have an id or a difference of priority.
        if (this.id !== null || otherElement.id !== null || this.priority !== otherElement.priority) {
            return false;
        }
        return super._canMergeAttributesFrom(otherElement);
    }
    /**
	 * Used by {@link module:engine/view/element~ViewElement#_subtractAttributesOf} to verify if the given element attributes
	 * can be fully subtracted from this element.
	 *
	 * @internal
	 */ _canSubtractAttributesOf(otherElement) {
        // Can't subtract if any of elements have an id or a difference of priority.
        if (this.id !== null || otherElement.id !== null || this.priority !== otherElement.priority) {
            return false;
        }
        return super._canSubtractAttributesOf(otherElement);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewAttributeElement.prototype.is = function(type, name) {
    if (!name) {
        return type === 'attributeElement' || type === 'view:attributeElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'element' || type === 'view:element' || type === 'node' || type === 'view:node';
    } else {
        return name === this.name && (type === 'attributeElement' || type === 'view:attributeElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'element' || type === 'view:element');
    }
};
/**
 * Returns block {@link module:engine/view/filler~Filler filler} offset or `null` if block filler is not needed.
 *
 * @returns Block filler offset or `null` if block filler is not needed.
 */ function getFillerOffset$3() {
    // <b>foo</b> does not need filler.
    if (nonUiChildrenCount(this)) {
        return null;
    }
    let element = this.parent;
    // <p><b></b></p> needs filler -> <p><b><br></b></p>
    while(element && element.is('attributeElement')){
        if (nonUiChildrenCount(element) > 1) {
            return null;
        }
        element = element.parent;
    }
    if (!element || nonUiChildrenCount(element) > 1) {
        return null;
    }
    // Render block filler at the end of element (after all ui elements).
    return this.childCount;
}
/**
 * Returns total count of children that are not {@link module:engine/view/uielement~ViewUIElement UIElements}.
 */ function nonUiChildrenCount(element) {
    return Array.from(element.getChildren()).filter((element)=>!element.is('uiElement')).length;
}

/**
 * Empty element class. It is used to represent elements that cannot contain any child nodes (for example `<img>` elements).
 *
 * To create a new empty element use the
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createEmptyElement `downcastWriter#createEmptyElement()`} method.
 */ class ViewEmptyElement extends ViewElement {
    /**
	 * Creates new instance of ViewEmptyElement.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-emptyelement-cannot-add` when third parameter is passed,
	 * to inform that usage of ViewEmptyElement is incorrect (adding child nodes to ViewEmptyElement is forbidden).
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#createEmptyElement
	 * @internal
	 * @param document The document instance to which this element belongs.
	 * @param name Node name.
	 * @param attributes Collection of attributes.
	 * @param children A list of nodes to be inserted into created element.
	 */ constructor(document, name, attributes, children){
        super(document, name, attributes, children);
        this.getFillerOffset = getFillerOffset$2;
    }
    /**
	 * Converts `ViewEmptyElement` instance to plain object and returns it.
	 *
	 * @returns `ViewEmptyElement` instance converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        json.type = 'EmptyElement';
        return json;
    }
    /**
	 * Overrides {@link module:engine/view/element~ViewElement#_insertChild} method.
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-emptyelement-cannot-add` to prevent
	 * adding any child nodes to ViewEmptyElement.
	 *
	 * @internal
	 */ _insertChild(index, items) {
        if (items && (items instanceof ViewNode || Array.from(items).length > 0)) {
            /**
			 * Cannot add children to {@link module:engine/view/emptyelement~ViewEmptyElement}.
			 *
			 * @error view-emptyelement-cannot-add
			 */ throw new CKEditorError('view-emptyelement-cannot-add', [
                this,
                items
            ]);
        }
        return 0;
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewEmptyElement.prototype.is = function(type, name) {
    if (!name) {
        return type === 'emptyElement' || type === 'view:emptyElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'element' || type === 'view:element' || type === 'node' || type === 'view:node';
    } else {
        return name === this.name && (type === 'emptyElement' || type === 'view:emptyElement' || type === 'element' || type === 'view:element');
    }
};
/**
 * Returns `null` because block filler is not needed for ViewEmptyElements.
 */ function getFillerOffset$2() {
    return null;
}

/**
 * UI element class. It should be used to represent editing UI which needs to be injected into the editing view
 * If possible, you should keep your UI outside the editing view. However, if that is not possible,
 * UI elements can be used.
 *
 * How a UI element is rendered is in your control (you pass a callback to
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createUIElement `downcastWriter#createUIElement()`}).
 * The editor will ignore your UI element – the selection cannot be placed in it, it is skipped (invisible) when
 * the user modifies the selection by using arrow keys and the editor does not listen to any mutations which
 * happen inside your UI elements.
 *
 * The limitation is that you cannot convert a model element to a UI element. UI elements need to be
 * created for {@link module:engine/model/markercollection~Marker markers} or as additinal elements
 * inside normal {@link module:engine/view/containerelement~ViewContainerElement container elements}.
 *
 * To create a new UI element use the
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createUIElement `downcastWriter#createUIElement()`} method.
 */ class ViewUIElement extends ViewElement {
    /**
	 * Creates new instance of UIElement.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-uielement-cannot-add` when third parameter is passed,
	 * to inform that usage of UIElement is incorrect (adding child nodes to UIElement is forbidden).
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#createUIElement
	 * @internal
	 * @param document The document instance to which this element belongs.
	 * @param name Node name.
	 * @param attrs Collection of attributes.
	 * @param children A list of nodes to be inserted into created element.
	 */ constructor(document, name, attrs, children){
        super(document, name, attrs, children);
        this.getFillerOffset = getFillerOffset$1;
    }
    /**
	 * Overrides {@link module:engine/view/element~ViewElement#_insertChild} method.
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-uielement-cannot-add` to prevent adding any child nodes
	 * to UIElement.
	 *
	 * @internal
	 */ _insertChild(index, items) {
        if (items && (items instanceof ViewNode || Array.from(items).length > 0)) {
            /**
			 * Cannot add children to {@link module:engine/view/uielement~ViewUIElement}.
			 *
			 * @error view-uielement-cannot-add
			 */ throw new CKEditorError('view-uielement-cannot-add', [
                this,
                items
            ]);
        }
        return 0;
    }
    /**
	 * Renders this {@link module:engine/view/uielement~ViewUIElement} to DOM. This method is called by
	 * {@link module:engine/view/domconverter~ViewDomConverter}.
	 * Do not use inheritance to create custom rendering method, replace `render()` method instead:
	 *
	 * ```ts
	 * const myUIElement = downcastWriter.createUIElement( 'span' );
	 * myUIElement.render = function( domDocument, domConverter ) {
	 * 	const domElement = this.toDomElement( domDocument );
	 *
	 * 	domConverter.setContentOf( domElement, '<b>this is ui element</b>' );
	 *
	 * 	return domElement;
	 * };
	 * ```
	 *
	 * If changes in your UI element should trigger some editor UI update you should call
	 * the {@link module:ui/editorui/editorui~EditorUI#update `editor.ui.update()`} method
	 * after rendering your UI element.
	 *
	 * @param domConverter Instance of the ViewDomConverter used to optimize the output.
	 */ render(domDocument, domConverter// eslint-disable-line @typescript-eslint/no-unused-vars
    ) {
        // Provide basic, default output.
        return this.toDomElement(domDocument);
    }
    /**
	 * Creates DOM element based on this view UIElement.
	 * Note that each time this method is called new DOM element is created.
	 */ toDomElement(domDocument) {
        const domElement = domDocument.createElement(this.name);
        for (const key of this.getAttributeKeys()){
            domElement.setAttribute(key, this.getAttribute(key));
        }
        return domElement;
    }
    /**
	 * Converts `ViewUIElement` instance to plain object and returns it.
	 *
	 * @returns `ViewUIElement` instance converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        json.type = 'UIElement';
        return json;
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewUIElement.prototype.is = function(type, name) {
    if (!name) {
        return type === 'uiElement' || type === 'view:uiElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'element' || type === 'view:element' || type === 'node' || type === 'view:node';
    } else {
        return name === this.name && (type === 'uiElement' || type === 'view:uiElement' || type === 'element' || type === 'view:element');
    }
};
/**
 * This function injects UI element handling to the given {@link module:engine/view/document~ViewDocument document}.
 *
 * A callback is added to {@link module:engine/view/document~ViewDocument#event:keydown document keydown event}.
 * The callback handles the situation when right arrow key is pressed and selection is collapsed before a UI element.
 * Without this handler, it would be impossible to "jump over" UI element using right arrow key.
 *
 * @param view View controller to which the quirks handling will be injected.
 * @internal
 */ function injectUiElementHandling(view) {
    view.document.on('arrowKey', (evt, data)=>jumpOverUiElement(evt, data, view.domConverter), {
        priority: 'low'
    });
}
/**
 * Returns `null` because block filler is not needed for UIElements.
 */ function getFillerOffset$1() {
    return null;
}
/**
 * Selection cannot be placed in a `UIElement`. Whenever it is placed there, it is moved before it. This
 * causes a situation when it is impossible to jump over `UIElement` using right arrow key, because the selection
 * ends up in ui element (in DOM) and is moved back to the left. This handler fixes this situation.
 */ function jumpOverUiElement(evt, data, domConverter) {
    if (data.keyCode == keyCodes.arrowright) {
        const domSelection = data.domTarget.ownerDocument.defaultView.getSelection();
        const domSelectionCollapsed = domSelection.rangeCount == 1 && domSelection.getRangeAt(0).collapsed;
        // Jump over UI element if selection is collapsed or shift key is pressed. These are the cases when selection would extend.
        if (domSelectionCollapsed || data.shiftKey) {
            const domParent = domSelection.focusNode;
            const domOffset = domSelection.focusOffset;
            const viewPosition = domConverter.domPositionToView(domParent, domOffset);
            // In case if dom element is not converted to view or is not mapped or something. Happens for example in some tests.
            if (viewPosition === null) {
                return;
            }
            // Skip all following ui elements.
            let jumpedOverAnyUiElement = false;
            const nextViewPosition = viewPosition.getLastMatchingPosition((value)=>{
                if (value.item.is('uiElement')) {
                    // Remember that there was at least one ui element.
                    jumpedOverAnyUiElement = true;
                }
                // Jump over ui elements, jump over empty attribute elements, move up from inside of attribute element.
                if (value.item.is('uiElement') || value.item.is('attributeElement')) {
                    return true;
                }
                // Don't jump over text or don't get out of container element.
                return false;
            });
            // If anything has been skipped, fix position.
            // This `if` could be possibly omitted but maybe it is better not to mess with DOM selection if not needed.
            if (jumpedOverAnyUiElement) {
                const newDomPosition = domConverter.viewPositionToDom(nextViewPosition);
                if (domSelectionCollapsed) {
                    // Selection was collapsed, so collapse it at further position.
                    domSelection.collapse(newDomPosition.parent, newDomPosition.offset);
                } else {
                    // Selection was not collapse, so extend it instead of collapsing.
                    domSelection.extend(newDomPosition.parent, newDomPosition.offset);
                }
            }
        }
    }
}

/**
 * The raw element class.
 *
 * The raw elements work as data containers ("wrappers", "sandboxes") but their children are not managed or
 * even recognized by the editor. This encapsulation allows integrations to maintain custom DOM structures
 * in the editor content without, for instance, worrying about compatibility with other editor features.
 * Raw elements are a perfect tool for integration with external frameworks and data sources.
 *
 * Unlike {@link module:engine/view/uielement~ViewUIElement UI elements}, raw elements act like real editor
 * content (similar to {@link module:engine/view/containerelement~ViewContainerElement} or
 * {@link module:engine/view/emptyelement~ViewEmptyElement}), they are considered by the editor selection and
 * {@link module:widget/utils~toWidget they can work as widgets}.
 *
 * To create a new raw element, use the
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createRawElement `downcastWriter#createRawElement()`} method.
 */ class ViewRawElement extends ViewElement {
    /**
	 * Creates a new instance of a raw element.
	 *
	 * Throws the `view-rawelement-cannot-add` {@link module:utils/ckeditorerror~CKEditorError CKEditorError} when the `children`
	 * parameter is passed to inform that the usage of `ViewRawElement` is incorrect (adding child nodes to `ViewRawElement` is forbidden).
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#createRawElement
	 * @internal
	 * @param document The document instance to which this element belongs.
	 * @param name Node name.
	 * @param attrs Collection of attributes.
	 * @param children A list of nodes to be inserted into created element.
	 */ constructor(document, name, attrs, children){
        super(document, name, attrs, children);
        // Returns `null` because filler is not needed for raw elements.
        this.getFillerOffset = getFillerOffset;
    }
    /**
	 * Converts `ViewRawElement` instance to plain object and returns it.
	 *
	 * @returns `ViewRawElement` instance converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        json.type = 'RawElement';
        return json;
    }
    /**
	 * Overrides the {@link module:engine/view/element~ViewElement#_insertChild} method.
	 * Throws the `view-rawelement-cannot-add` {@link module:utils/ckeditorerror~CKEditorError CKEditorError} to prevent
	 * adding any child nodes to a raw element.
	 *
	 * @internal
	 */ _insertChild(index, items) {
        if (items && (items instanceof ViewNode || Array.from(items).length > 0)) {
            /**
			 * Cannot add children to a {@link module:engine/view/rawelement~ViewRawElement} instance.
			 *
			 * @error view-rawelement-cannot-add
			 */ throw new CKEditorError('view-rawelement-cannot-add', [
                this,
                items
            ]);
        }
        return 0;
    }
    /**
	 * This allows rendering the children of a {@link module:engine/view/rawelement~ViewRawElement} on the DOM level.
	 * This method is called by the {@link module:engine/view/domconverter~ViewDomConverter} with the raw DOM element
	 * passed as an argument, leaving the number and shape of the children up to the integrator.
	 *
	 * This method **must be defined** for the raw element to work:
	 *
	 * ```ts
	 * const myRawElement = downcastWriter.createRawElement( 'div' );
	 *
	 * myRawElement.render = function( domElement, domConverter ) {
	 * 	domConverter.setContentOf( domElement, '<b>This is the raw content of myRawElement.</b>' );
	 * };
	 * ```
	 *
	 * @param domElement The native DOM element representing the raw view element.
	 * @param domConverter Instance of the ViewDomConverter used to optimize the output.
	 */ render(domElement, domConverter) {}
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewRawElement.prototype.is = function(type, name) {
    if (!name) {
        return type === 'rawElement' || type === 'view:rawElement' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === this.name || type === 'view:' + this.name || type === 'element' || type === 'view:element' || type === 'node' || type === 'view:node';
    } else {
        return name === this.name && (type === 'rawElement' || type === 'view:rawElement' || type === 'element' || type === 'view:element');
    }
};
/**
 * Returns `null` because block filler is not needed for raw elements.
 */ function getFillerOffset() {
    return null;
}

/**
 * Document fragment.
 *
 * To create a new document fragment instance use the
 * {@link module:engine/view/upcastwriter~ViewUpcastWriter#createDocumentFragment `ViewUpcastWriter#createDocumentFragment()`}
 * method.
 */ class ViewDocumentFragment extends /* #__PURE__ */ EmitterMixin(ViewTypeCheckable) {
    /**
	 * The document to which this document fragment belongs.
	 */ document;
    /**
	 * Array of child nodes.
	 */ _children = [];
    /**
	 * Map of custom properties.
	 * Custom properties can be added to document fragment instance.
	 */ _customProperties = new Map();
    /**
	 * Creates new DocumentFragment instance.
	 *
	 * @internal
	 * @param document The document to which this document fragment belongs.
	 * @param children A list of nodes to be inserted into the created document fragment.
	 */ constructor(document, children){
        super();
        this.document = document;
        if (children) {
            this._insertChild(0, children);
        }
    }
    /**
	 * Iterable interface.
	 *
	 * Iterates over nodes added to this document fragment.
	 */ [Symbol.iterator]() {
        return this._children[Symbol.iterator]();
    }
    /**
	 * Number of child nodes in this document fragment.
	 */ get childCount() {
        return this._children.length;
    }
    /**
	 * Is `true` if there are no nodes inside this document fragment, `false` otherwise.
	 */ get isEmpty() {
        return this.childCount === 0;
    }
    /**
	 * Artificial root of `DocumentFragment`. Returns itself. Added for compatibility reasons.
	 */ get root() {
        return this;
    }
    /**
	 * Artificial parent of `DocumentFragment`. Returns `null`. Added for compatibility reasons.
	 */ get parent() {
        return null;
    }
    /**
	 * Artificial element name. Returns `undefined`. Added for compatibility reasons.
	 */ get name() {
        return undefined;
    }
    /**
	 * Artificial element getFillerOffset. Returns `undefined`. Added for compatibility reasons.
	 */ get getFillerOffset() {
        return undefined;
    }
    /**
	 * Returns the custom property value for the given key.
	 */ getCustomProperty(key) {
        return this._customProperties.get(key);
    }
    /**
	 * Returns an iterator which iterates over this document fragment's custom properties.
	 * Iterator provides `[ key, value ]` pairs for each stored property.
	 */ *getCustomProperties() {
        yield* this._customProperties.entries();
    }
    /**
	 * Converts `ViewDocumentFragment` instance to plain object and returns it.
	 * Takes care of converting all of this document fragment's children.
	 *
	 * @returns `ViewDocumentFragment` instance converted to plain object.
	 */ toJSON() {
        const json = [];
        for (const node of this._children){
            json.push(node.toJSON());
        }
        return json;
    }
    /**
	 * {@link module:engine/view/documentfragment~ViewDocumentFragment#_insertChild Insert} a child node or a list of child nodes at the end
	 * and sets the parent of these nodes to this fragment.
	 *
	 * @internal
	 * @param items Items to be inserted.
	 * @returns Number of appended nodes.
	 */ _appendChild(items) {
        return this._insertChild(this.childCount, items);
    }
    /**
	 * Gets child at the given index.
	 *
	 * @param index Index of child.
	 * @returns Child node.
	 */ getChild(index) {
        return this._children[index];
    }
    /**
	 * Gets index of the given child node. Returns `-1` if child node is not found.
	 *
	 * @param node Child node.
	 * @returns Index of the child node.
	 */ getChildIndex(node) {
        return this._children.indexOf(node);
    }
    /**
	 * Gets child nodes iterator.
	 *
	 * @returns Child nodes iterator.
	 */ getChildren() {
        return this._children[Symbol.iterator]();
    }
    /**
	 * Inserts a child node or a list of child nodes on the given index and sets the parent of these nodes to
	 * this fragment.
	 *
	 * @internal
	 * @param index Position where nodes should be inserted.
	 * @param items Items to be inserted.
	 * @returns Number of inserted nodes.
	 */ _insertChild(index, items) {
        this._fireChange('children', this, {
            index
        });
        let count = 0;
        const nodes = normalize$2(this.document, items);
        for (const node of nodes){
            // If node that is being added to this element is already inside another element, first remove it from the old parent.
            if (node.parent !== null) {
                node._remove();
            }
            node.parent = this;
            this._children.splice(index, 0, node);
            index++;
            count++;
        }
        return count;
    }
    /**
	 * Removes number of child nodes starting at the given index and set the parent of these nodes to `null`.
	 *
	 * @internal
	 * @param index Number of the first node to remove.
	 * @param howMany Number of nodes to remove.
	 * @returns The array of removed nodes.
	 */ _removeChildren(index, howMany = 1) {
        this._fireChange('children', this, {
            index
        });
        for(let i = index; i < index + howMany; i++){
            this._children[i].parent = null;
        }
        return this._children.splice(index, howMany);
    }
    /**
	 * @internal
	 * @param type Type of the change.
	 * @param node Changed node.
	 * @param data Additional data.
	 * @fires module:engine/view/node~ViewNode#event:change
	 */ _fireChange(type, node, data) {
        this.fire(`change:${type}`, node, data);
    }
    /**
	 * Sets a custom property. They can be used to add special data to elements.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#setCustomProperty
	 * @internal
	 */ _setCustomProperty(key, value) {
        this._customProperties.set(key, value);
    }
    /**
	 * Removes the custom property stored under the given key.
	 *
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#removeCustomProperty
	 * @internal
	 * @returns Returns true if property was removed.
	 */ _removeCustomProperty(key) {
        return this._customProperties.delete(key);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ViewDocumentFragment.prototype.is = function(type) {
    return type === 'documentFragment' || type === 'view:documentFragment';
};
/**
 * Converts strings to Text and non-iterables to arrays.
 */ function normalize$2(document, nodes) {
    // Separate condition because string is iterable.
    if (typeof nodes == 'string') {
        return [
            new ViewText(document, nodes)
        ];
    }
    if (!isIterable(nodes)) {
        nodes = [
            nodes
        ];
    }
    // Array.from to enable .map() on non-arrays.
    return Array.from(nodes).map((node)=>{
        if (typeof node == 'string') {
            return new ViewText(document, node);
        }
        if (node instanceof ViewTextProxy) {
            return new ViewText(document, node.data);
        }
        return node;
    });
}

/**
 * View downcast writer.
 *
 * It provides a set of methods used to manipulate view nodes.
 *
 * Do not create an instance of this writer manually. To modify a view structure, use
 * the {@link module:engine/view/view~EditingView#change `View#change()`} block.
 *
 * The `ViewDowncastWriter` is designed to work with semantic views which are the views that were/are being downcasted from the model.
 * To work with ordinary views (e.g. parsed from a pasted content) use the
 * {@link module:engine/view/upcastwriter~ViewUpcastWriter upcast writer}.
 *
 * Read more about changing the view in the {@glink framework/architecture/editing-engine#changing-the-view Changing the view}
 * section of the {@glink framework/architecture/editing-engine Editing engine architecture} guide.
 */ class ViewDowncastWriter {
    /**
	 * The view document instance in which this writer operates.
	 */ document;
    /**
	 * Holds references to the attribute groups that share the same {@link module:engine/view/attributeelement~ViewAttributeElement#id id}.
	 * The keys are `id`s, the values are `Set`s holding {@link module:engine/view/attributeelement~ViewAttributeElement}s.
	 */ _cloneGroups = new Map();
    /**
	 * The slot factory used by the `elementToStructure` downcast helper.
	 */ _slotFactory = null;
    /**
	 * @param document The view document instance.
	 */ constructor(document){
        this.document = document;
    }
    setSelection(...args) {
        this.document.selection._setTo(...args);
    }
    /**
	 * Moves {@link module:engine/view/documentselection~ViewDocumentSelection#focus selection's focus} to the specified location.
	 *
	 * The location can be specified in the same form as
	 * {@link module:engine/view/view~EditingView#createPositionAt view.createPositionAt()}
	 * parameters.
	 *
	 * @param itemOrPosition
	 * @param offset Offset or one of the flags. Used only when the first parameter is a {@link module:engine/view/item~ViewItem view item}.
	 */ setSelectionFocus(itemOrPosition, offset) {
        this.document.selection._setFocus(itemOrPosition, offset);
    }
    /**
	 * Creates a new {@link module:engine/view/documentfragment~ViewDocumentFragment} instance.
	 *
	 * @param children A list of nodes to be inserted into the created document fragment.
	 * @returns The created document fragment.
	 */ createDocumentFragment(children) {
        return new ViewDocumentFragment(this.document, children);
    }
    /**
	 * Creates a new {@link module:engine/view/text~ViewText text node}.
	 *
	 * ```ts
	 * writer.createText( 'foo' );
	 * ```
	 *
	 * @param data The text's data.
	 * @returns The created text node.
	 */ createText(data) {
        return new ViewText(this.document, data);
    }
    /**
	 * Creates a new {@link module:engine/view/attributeelement~ViewAttributeElement}.
	 *
	 * ```ts
	 * writer.createAttributeElement( 'strong' );
	 * writer.createAttributeElement( 'a', { href: 'foo.bar' } );
	 *
	 * // Make `<a>` element contain other attributes element so the `<a>` element is not broken.
	 * writer.createAttributeElement( 'a', { href: 'foo.bar' }, { priority: 5 } );
	 *
	 * // Set `id` of a marker element so it is not joined or merged with "normal" elements.
	 * writer.createAttributeElement( 'span', { class: 'my-marker' }, { id: 'marker:my' } );
	 * ```
	 *
	 * @param name Name of the element.
	 * @param attributes Element's attributes.
	 * @param options Element's options.
	 * @param options.priority Element's {@link module:engine/view/attributeelement~ViewAttributeElement#priority priority}.
	 * @param options.id Element's {@link module:engine/view/attributeelement~ViewAttributeElement#id id}.
	 * @param options.renderUnsafeAttributes A list of attribute names that should be rendered in the editing
	 * pipeline even though they would normally be filtered out by unsafe attribute detection mechanisms.
	 * @returns Created element.
	 */ createAttributeElement(name, attributes, options = {}) {
        const attributeElement = new ViewAttributeElement(this.document, name, attributes);
        if (typeof options.priority === 'number') {
            attributeElement._priority = options.priority;
        }
        if (options.id) {
            attributeElement._id = options.id;
        }
        if (options.renderUnsafeAttributes) {
            attributeElement._unsafeAttributesToRender.push(...options.renderUnsafeAttributes);
        }
        return attributeElement;
    }
    createContainerElement(name, attributes, childrenOrOptions = {}, options = {}) {
        let children = undefined;
        if (isContainerOptions(childrenOrOptions)) {
            options = childrenOrOptions;
        } else {
            children = childrenOrOptions;
        }
        const containerElement = new ViewContainerElement(this.document, name, attributes, children);
        if (options.renderUnsafeAttributes) {
            containerElement._unsafeAttributesToRender.push(...options.renderUnsafeAttributes);
        }
        return containerElement;
    }
    /**
	 * Creates a new {@link module:engine/view/editableelement~ViewEditableElement}.
	 *
	 * ```ts
	 * writer.createEditableElement( 'div' );
	 * writer.createEditableElement( 'div', { id: 'foo-1234' } );
	 * ```
	 *
	 * Note: The editable element is to be used in the editing pipeline. Usually, together with
	 * {@link module:widget/utils~toWidgetEditable `toWidgetEditable()`}.
	 *
	 * @param name Name of the element.
	 * @param attributes Elements attributes.
	 * @param options Element's options.
	 * @param options.renderUnsafeAttributes A list of attribute names that should be rendered in the editing
	 * pipeline even though they would normally be filtered out by unsafe attribute detection mechanisms.
	 * @returns Created element.
	 */ createEditableElement(name, attributes, options = {}) {
        const editableElement = new ViewEditableElement(this.document, name, attributes);
        if (options.renderUnsafeAttributes) {
            editableElement._unsafeAttributesToRender.push(...options.renderUnsafeAttributes);
        }
        return editableElement;
    }
    /**
	 * Creates a new {@link module:engine/view/emptyelement~ViewEmptyElement}.
	 *
	 * ```ts
	 * writer.createEmptyElement( 'img' );
	 * writer.createEmptyElement( 'img', { id: 'foo-1234' } );
	 * ```
	 *
	 * @param name Name of the element.
	 * @param attributes Elements attributes.
	 * @param options Element's options.
	 * @param options.renderUnsafeAttributes A list of attribute names that should be rendered in the editing
	 * pipeline even though they would normally be filtered out by unsafe attribute detection mechanisms.
	 * @returns Created element.
	 */ createEmptyElement(name, attributes, options = {}) {
        const emptyElement = new ViewEmptyElement(this.document, name, attributes);
        if (options.renderUnsafeAttributes) {
            emptyElement._unsafeAttributesToRender.push(...options.renderUnsafeAttributes);
        }
        return emptyElement;
    }
    /**
	 * Creates a new {@link module:engine/view/uielement~ViewUIElement}.
	 *
	 * ```ts
	 * writer.createUIElement( 'span' );
	 * writer.createUIElement( 'span', { id: 'foo-1234' } );
	 * ```
	 *
	 * A custom render function can be provided as the third parameter:
	 *
	 * ```ts
	 * writer.createUIElement( 'span', null, function( domDocument ) {
	 * 	const domElement = this.toDomElement( domDocument );
	 * 	domElement.innerHTML = '<b>this is ui element</b>';
	 *
	 * 	return domElement;
	 * } );
	 * ```
	 *
	 * Unlike {@link #createRawElement raw elements}, UI elements are by no means editor content, for instance,
	 * they are ignored by the editor selection system.
	 *
	 * You should not use UI elements as data containers. Check out {@link #createRawElement} instead.
	 *
	 * @param name The name of the element.
	 * @param attributes Element attributes.
	 * @param renderFunction A custom render function.
	 * @returns The created element.
	 */ createUIElement(name, attributes, renderFunction) {
        const uiElement = new ViewUIElement(this.document, name, attributes);
        if (renderFunction) {
            uiElement.render = renderFunction;
        }
        return uiElement;
    }
    /**
	 * Creates a new {@link module:engine/view/rawelement~ViewRawElement}.
	 *
	 * ```ts
	 * writer.createRawElement( 'span', { id: 'foo-1234' }, function( domElement ) {
	 * 	domElement.innerHTML = '<b>This is the raw content of the raw element.</b>';
	 * } );
	 * ```
	 *
	 * Raw elements work as data containers ("wrappers", "sandboxes") but their children are not managed or
	 * even recognized by the editor. This encapsulation allows integrations to maintain custom DOM structures
	 * in the editor content without, for instance, worrying about compatibility with other editor features.
	 * Raw elements are a perfect tool for integration with external frameworks and data sources.
	 *
	 * Unlike {@link #createUIElement UI elements}, raw elements act like "real" editor content (similar to
	 * {@link module:engine/view/containerelement~ViewContainerElement} or {@link module:engine/view/emptyelement~ViewEmptyElement}),
	 * and they are considered by the editor selection.
	 *
	 * You should not use raw elements to render the UI in the editor content. Check out {@link #createUIElement `#createUIElement()`}
	 * instead.
	 *
	 * @param name The name of the element.
	 * @param attributes Element attributes.
	 * @param renderFunction A custom render function.
	 * @param options Element's options.
	 * @param options.renderUnsafeAttributes A list of attribute names that should be rendered in the editing
	 * pipeline even though they would normally be filtered out by unsafe attribute detection mechanisms.
	 * @returns The created element.
	 */ createRawElement(name, attributes, renderFunction, options = {}) {
        const rawElement = new ViewRawElement(this.document, name, attributes);
        if (renderFunction) {
            rawElement.render = renderFunction;
        }
        if (options.renderUnsafeAttributes) {
            rawElement._unsafeAttributesToRender.push(...options.renderUnsafeAttributes);
        }
        return rawElement;
    }
    setAttribute(key, value, elementOrOverwrite, element) {
        if (element !== undefined) {
            element._setAttribute(key, value, elementOrOverwrite);
        } else {
            elementOrOverwrite._setAttribute(key, value);
        }
    }
    removeAttribute(key, elementOrTokens, element) {
        if (element !== undefined) {
            element._removeAttribute(key, elementOrTokens);
        } else {
            elementOrTokens._removeAttribute(key);
        }
    }
    /**
	 * Adds specified class to the element.
	 *
	 * ```ts
	 * writer.addClass( 'foo', linkElement );
	 * writer.addClass( [ 'foo', 'bar' ], linkElement );
	 * ```
	 */ addClass(className, element) {
        element._addClass(className);
    }
    /**
	 * Removes specified class from the element.
	 *
	 * ```ts
	 * writer.removeClass( 'foo', linkElement );
	 * writer.removeClass( [ 'foo', 'bar' ], linkElement );
	 * ```
	 */ removeClass(className, element) {
        element._removeClass(className);
    }
    setStyle(property, value, element) {
        if (isPlainObject(property) && element === undefined) {
            value._setStyle(property);
        } else {
            element._setStyle(property, value);
        }
    }
    /**
	 * Removes specified style from the element.
	 *
	 * ```ts
	 * writer.removeStyle( 'color', element ); // Removes 'color' style.
	 * writer.removeStyle( [ 'color', 'border-top' ], element ); // Removes both 'color' and 'border-top' styles.
	 * ```
	 *
	 * **Note**: This method can work with normalized style names if
	 * {@link module:engine/controller/datacontroller~DataController#addStyleProcessorRules a particular style processor rule is enabled}.
	 * See {@link module:engine/view/stylesmap~StylesMap#remove `StylesMap#remove()`} for details.
	 */ removeStyle(property, element) {
        element._removeStyle(property);
    }
    /**
	 * Sets a custom property on element. Unlike attributes, custom properties are not rendered to the DOM,
	 * so they can be used to add special data to elements.
	 */ setCustomProperty(key, value, element) {
        element._setCustomProperty(key, value);
    }
    /**
	 * Removes a custom property stored under the given key.
	 *
	 * @returns Returns true if property was removed.
	 */ removeCustomProperty(key, element) {
        return element._removeCustomProperty(key);
    }
    /**
	 * Breaks attribute elements at the provided position or at the boundaries of a provided range. It breaks attribute elements
	 * up to their first ancestor that is a container element.
	 *
	 * In following examples `<p>` is a container, `<b>` and `<u>` are attribute elements:
	 *
	 * ```html
	 * <p>foo<b><u>bar{}</u></b></p> -> <p>foo<b><u>bar</u></b>[]</p>
	 * <p>foo<b><u>{}bar</u></b></p> -> <p>foo{}<b><u>bar</u></b></p>
	 * <p>foo<b><u>b{}ar</u></b></p> -> <p>foo<b><u>b</u></b>[]<b><u>ar</u></b></p>
	 * <p><b>fo{o</b><u>ba}r</u></p> -> <p><b>fo</b><b>o</b><u>ba</u><u>r</u></b></p>
	 * ```
	 *
	 * **Note:** {@link module:engine/view/documentfragment~ViewDocumentFragment DocumentFragment} is treated like a container.
	 *
	 * **Note:** The difference between {@link module:engine/view/downcastwriter~ViewDowncastWriter#breakAttributes breakAttributes()} and
	 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#breakContainer breakContainer()} is that `breakAttributes()` breaks all
	 * {@link module:engine/view/attributeelement~ViewAttributeElement attribute elements} that are ancestors of a given `position`,
	 * up to the first encountered {@link module:engine/view/containerelement~ViewContainerElement container element}.
	 * `breakContainer()` assumes that a given `position` is directly in the container element and breaks that container element.
	 *
	 * Throws the `view-writer-invalid-range-container` {@link module:utils/ckeditorerror~CKEditorError CKEditorError}
	 * when the {@link module:engine/view/range~ViewRange#start start}
	 * and {@link module:engine/view/range~ViewRange#end end} positions of a passed range are not placed inside same parent container.
	 *
	 * Throws the `view-writer-cannot-break-empty-element` {@link module:utils/ckeditorerror~CKEditorError CKEditorError}
	 * when trying to break attributes inside an {@link module:engine/view/emptyelement~ViewEmptyElement ViewEmptyElement}.
	 *
	 * Throws the `view-writer-cannot-break-ui-element` {@link module:utils/ckeditorerror~CKEditorError CKEditorError}
	 * when trying to break attributes inside a {@link module:engine/view/uielement~ViewUIElement UIElement}.
	 *
	 * @see module:engine/view/attributeelement~ViewAttributeElement
	 * @see module:engine/view/containerelement~ViewContainerElement
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#breakContainer
	 * @param positionOrRange The position where to break attribute elements.
	 * @returns The new position or range, after breaking the attribute elements.
	 */ breakAttributes(positionOrRange) {
        if (positionOrRange instanceof ViewPosition) {
            return this._breakAttributes(positionOrRange);
        } else {
            return this._breakAttributesRange(positionOrRange);
        }
    }
    /**
	 * Breaks a {@link module:engine/view/containerelement~ViewContainerElement container view element} into two, at the given position.
	 * The position has to be directly inside the container element and cannot be in the root. It does not break the conrainer view element
	 * if the position is at the beginning or at the end of its parent element.
	 *
	 * ```html
	 * <p>foo^bar</p> -> <p>foo</p><p>bar</p>
	 * <div><p>foo</p>^<p>bar</p></div> -> <div><p>foo</p></div><div><p>bar</p></div>
	 * <p>^foobar</p> -> ^<p>foobar</p>
	 * <p>foobar^</p> -> <p>foobar</p>^
	 * ```
	 *
	 * **Note:** The difference between {@link module:engine/view/downcastwriter~ViewDowncastWriter#breakAttributes breakAttributes()} and
	 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#breakContainer breakContainer()} is that `breakAttributes()` breaks all
	 * {@link module:engine/view/attributeelement~ViewAttributeElement attribute elements} that are ancestors of a given `position`,
	 * up to the first encountered {@link module:engine/view/containerelement~ViewContainerElement container element}.
	 * `breakContainer()` assumes that the given `position` is directly in the container element and breaks that container element.
	 *
	 * @see module:engine/view/attributeelement~ViewAttributeElement
	 * @see module:engine/view/containerelement~ViewContainerElement
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#breakAttributes
	 * @param position The position where to break the element.
	 * @returns The position between broken elements. If an element has not been broken,
	 * the returned position is placed either before or after it.
	 */ breakContainer(position) {
        const element = position.parent;
        if (!element.is('containerElement')) {
            /**
			 * Trying to break an element which is not a container element.
			 *
			 * @error view-writer-break-non-container-element
			 */ throw new CKEditorError('view-writer-break-non-container-element', this.document);
        }
        if (!element.parent) {
            /**
			 * Trying to break root element.
			 *
			 * @error view-writer-break-root
			 */ throw new CKEditorError('view-writer-break-root', this.document);
        }
        if (position.isAtStart) {
            return ViewPosition._createBefore(element);
        } else if (!position.isAtEnd) {
            const newElement = element._clone(false);
            this.insert(ViewPosition._createAfter(element), newElement);
            const sourceRange = new ViewRange(position, ViewPosition._createAt(element, 'end'));
            const targetPosition = new ViewPosition(newElement, 0);
            this.move(sourceRange, targetPosition);
        }
        return ViewPosition._createAfter(element);
    }
    /**
	 * Merges {@link module:engine/view/attributeelement~ViewAttributeElement attribute elements}. It also merges text nodes if needed.
	 * Only {@link module:engine/view/attributeelement~ViewAttributeElement#isSimilar similar} attribute elements can be merged.
	 *
	 * In following examples `<p>` is a container and `<b>` is an attribute element:
	 *
	 * ```html
	 * <p>foo[]bar</p> -> <p>foo{}bar</p>
	 * <p><b>foo</b>[]<b>bar</b></p> -> <p><b>foo{}bar</b></p>
	 * <p><b foo="bar">a</b>[]<b foo="baz">b</b></p> -> <p><b foo="bar">a</b>[]<b foo="baz">b</b></p>
	 * ```
	 *
	 * It will also take care about empty attributes when merging:
	 *
	 * ```html
	 * <p><b>[]</b></p> -> <p>[]</p>
	 * <p><b>foo</b><i>[]</i><b>bar</b></p> -> <p><b>foo{}bar</b></p>
	 * ```
	 *
	 * **Note:** Difference between {@link module:engine/view/downcastwriter~ViewDowncastWriter#mergeAttributes mergeAttributes} and
	 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#mergeContainers mergeContainers} is that `mergeAttributes` merges two
	 * {@link module:engine/view/attributeelement~ViewAttributeElement attribute elements} or
	 * {@link module:engine/view/text~ViewText text nodes} while `mergeContainer` merges two
	 * {@link module:engine/view/containerelement~ViewContainerElement container elements}.
	 *
	 * @see module:engine/view/attributeelement~ViewAttributeElement
	 * @see module:engine/view/containerelement~ViewContainerElement
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#mergeContainers
	 * @param position Merge position.
	 * @returns Position after merge.
	 */ mergeAttributes(position) {
        const positionOffset = position.offset;
        const positionParent = position.parent;
        // When inside text node - nothing to merge.
        if (positionParent.is('$text')) {
            return position;
        }
        // When inside empty attribute - remove it.
        if (positionParent.is('attributeElement') && positionParent.childCount === 0) {
            const parent = positionParent.parent;
            const offset = positionParent.index;
            positionParent._remove();
            this._removeFromClonedElementsGroup(positionParent);
            return this.mergeAttributes(new ViewPosition(parent, offset));
        }
        const nodeBefore = positionParent.getChild(positionOffset - 1);
        const nodeAfter = positionParent.getChild(positionOffset);
        // Position should be placed between two nodes.
        if (!nodeBefore || !nodeAfter) {
            return position;
        }
        // When position is between two text nodes.
        if (nodeBefore.is('$text') && nodeAfter.is('$text')) {
            return mergeTextNodes(nodeBefore, nodeAfter);
        } else if (nodeBefore.is('attributeElement') && nodeAfter.is('attributeElement') && nodeBefore.isSimilar(nodeAfter)) {
            // Move all children nodes from node placed after selection and remove that node.
            const count = nodeBefore.childCount;
            nodeBefore._appendChild(nodeAfter.getChildren());
            nodeAfter._remove();
            this._removeFromClonedElementsGroup(nodeAfter);
            // New position is located inside the first node, before new nodes.
            // Call this method recursively to merge again if needed.
            return this.mergeAttributes(new ViewPosition(nodeBefore, count));
        }
        return position;
    }
    /**
	 * Merges two {@link module:engine/view/containerelement~ViewContainerElement container elements} that are
	 * before and after given position. Precisely, the element after the position is removed and it's contents are
	 * moved to element before the position.
	 *
	 * ```html
	 * <p>foo</p>^<p>bar</p> -> <p>foo^bar</p>
	 * <div>foo</div>^<p>bar</p> -> <div>foo^bar</div>
	 * ```
	 *
	 * **Note:** Difference between {@link module:engine/view/downcastwriter~ViewDowncastWriter#mergeAttributes mergeAttributes} and
	 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#mergeContainers mergeContainers} is that `mergeAttributes` merges two
	 * {@link module:engine/view/attributeelement~ViewAttributeElement attribute elements} or
	 * {@link module:engine/view/text~ViewText text nodes} while `mergeContainer` merges two
	 * {@link module:engine/view/containerelement~ViewContainerElement container elements}.
	 *
	 * @see module:engine/view/attributeelement~ViewAttributeElement
	 * @see module:engine/view/containerelement~ViewContainerElement
	 * @see module:engine/view/downcastwriter~ViewDowncastWriter#mergeAttributes
	 * @param position Merge position.
	 * @returns Position after merge.
	 */ mergeContainers(position) {
        const prev = position.nodeBefore;
        const next = position.nodeAfter;
        if (!prev || !next || !prev.is('containerElement') || !next.is('containerElement')) {
            /**
			 * Element before and after given position cannot be merged.
			 *
			 * @error view-writer-merge-containers-invalid-position
			 */ throw new CKEditorError('view-writer-merge-containers-invalid-position', this.document);
        }
        const lastChild = prev.getChild(prev.childCount - 1);
        const newPosition = lastChild instanceof ViewText ? ViewPosition._createAt(lastChild, 'end') : ViewPosition._createAt(prev, 'end');
        this.move(ViewRange._createIn(next), ViewPosition._createAt(prev, 'end'));
        this.remove(ViewRange._createOn(next));
        return newPosition;
    }
    /**
	 * Inserts a node or nodes at specified position. Takes care about breaking attributes before insertion
	 * and merging them afterwards.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-writer-insert-invalid-node` when nodes to insert
	 * contains instances that are not {@link module:engine/view/text~ViewText Texts},
	 * {@link module:engine/view/attributeelement~ViewAttributeElement ViewAttributeElements},
	 * {@link module:engine/view/containerelement~ViewContainerElement ViewContainerElements},
	 * {@link module:engine/view/emptyelement~ViewEmptyElement ViewEmptyElements},
	 * {@link module:engine/view/rawelement~ViewRawElement RawElements} or
	 * {@link module:engine/view/uielement~ViewUIElement UIElements}.
	 *
	 * @param position Insertion position.
	 * @param nodes Node or nodes to insert.
	 * @returns Range around inserted nodes.
	 */ insert(position, nodes) {
        nodes = isIterable(nodes) ? [
            ...nodes
        ] : [
            nodes
        ];
        // Check if nodes to insert are instances of ViewAttributeElements, ViewContainerElements, ViewEmptyElements, UIElements or Text.
        validateNodesToInsert(nodes, this.document);
        // Group nodes in batches of nodes that require or do not require breaking an ViewAttributeElements.
        const nodeGroups = nodes.reduce((groups, node)=>{
            const lastGroup = groups[groups.length - 1];
            // Break attributes on nodes that do exist in the model tree so they can have attributes, other elements
            // can't have an attribute in model and won't get wrapped with an ViewAttributeElement while down-casted.
            const breakAttributes = !node.is('uiElement');
            if (!lastGroup || lastGroup.breakAttributes != breakAttributes) {
                groups.push({
                    breakAttributes,
                    nodes: [
                        node
                    ]
                });
            } else {
                lastGroup.nodes.push(node);
            }
            return groups;
        }, []);
        // Insert nodes in batches.
        let start = null;
        let end = position;
        for (const { nodes, breakAttributes } of nodeGroups){
            const range = this._insertNodes(end, nodes, breakAttributes);
            if (!start) {
                start = range.start;
            }
            end = range.end;
        }
        // When no nodes were inserted - return collapsed range.
        if (!start) {
            return new ViewRange(position);
        }
        return new ViewRange(start, end);
    }
    /**
	 * Removes provided range from the container.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-writer-invalid-range-container` when
	 * {@link module:engine/view/range~ViewRange#start start} and {@link module:engine/view/range~ViewRange#end end}
	 * positions are not placed inside same parent container.
	 *
	 * @param rangeOrItem Range to remove from container
	 * or an {@link module:engine/view/item~ViewItem item} to remove. If range is provided, after removing, it will be updated
	 * to a collapsed range showing the new position.
	 * @returns Document fragment containing removed nodes.
	 */ remove(rangeOrItem) {
        const range = rangeOrItem instanceof ViewRange ? rangeOrItem : ViewRange._createOn(rangeOrItem);
        validateRangeContainer(range, this.document);
        // If range is collapsed - nothing to remove.
        if (range.isCollapsed) {
            return new ViewDocumentFragment(this.document);
        }
        // Break attributes at range start and end.
        const { start: breakStart, end: breakEnd } = this._breakAttributesRange(range, true);
        const parentContainer = breakStart.parent;
        const count = breakEnd.offset - breakStart.offset;
        // Remove nodes in range.
        const removed = parentContainer._removeChildren(breakStart.offset, count);
        for (const node of removed){
            this._removeFromClonedElementsGroup(node);
        }
        // Merge after removing.
        const mergePosition = this.mergeAttributes(breakStart);
        range.start = mergePosition;
        range.end = mergePosition.clone();
        // Return removed nodes.
        return new ViewDocumentFragment(this.document, removed);
    }
    /**
	 * Removes matching elements from given range.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-writer-invalid-range-container` when
	 * {@link module:engine/view/range~ViewRange#start start} and {@link module:engine/view/range~ViewRange#end end}
	 * positions are not placed inside same parent container.
	 *
	 * @param range Range to clear.
	 * @param element Element to remove.
	 */ clear(range, element) {
        validateRangeContainer(range, this.document);
        // Create walker on given range.
        // We walk backward because when we remove element during walk it modifies range end position.
        const walker = range.getWalker({
            direction: 'backward',
            ignoreElementEnd: true
        });
        // Let's walk.
        for (const current of walker){
            const item = current.item;
            let rangeToRemove;
            // When current item matches to the given element.
            if (item.is('element') && element.isSimilar(item)) {
                // Create range on this element.
                rangeToRemove = ViewRange._createOn(item);
            // When range starts inside Text or TextProxy element.
            } else if (!current.nextPosition.isAfter(range.start) && item.is('$textProxy')) {
                // We need to check if parent of this text matches to given element.
                const parentElement = item.getAncestors().find((ancestor)=>{
                    return ancestor.is('element') && element.isSimilar(ancestor);
                });
                // If it is then create range inside this element.
                if (parentElement) {
                    rangeToRemove = ViewRange._createIn(parentElement);
                }
            }
            // If we have found element to remove.
            if (rangeToRemove) {
                // We need to check if element range stick out of the given range and truncate if it is.
                if (rangeToRemove.end.isAfter(range.end)) {
                    rangeToRemove.end = range.end;
                }
                if (rangeToRemove.start.isBefore(range.start)) {
                    rangeToRemove.start = range.start;
                }
                // At the end we remove range with found element.
                this.remove(rangeToRemove);
            }
        }
    }
    /**
	 * Moves nodes from provided range to target position.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-writer-invalid-range-container` when
	 * {@link module:engine/view/range~ViewRange#start start} and {@link module:engine/view/range~ViewRange#end end}
	 * positions are not placed inside same parent container.
	 *
	 * @param sourceRange Range containing nodes to move.
	 * @param targetPosition Position to insert.
	 * @returns Range in target container. Inserted nodes are placed between
	 * {@link module:engine/view/range~ViewRange#start start} and {@link module:engine/view/range~ViewRange#end end} positions.
	 */ move(sourceRange, targetPosition) {
        let nodes;
        if (targetPosition.isAfter(sourceRange.end)) {
            targetPosition = this._breakAttributes(targetPosition, true);
            const parent = targetPosition.parent;
            const countBefore = parent.childCount;
            sourceRange = this._breakAttributesRange(sourceRange, true);
            nodes = this.remove(sourceRange);
            targetPosition.offset += parent.childCount - countBefore;
        } else {
            nodes = this.remove(sourceRange);
        }
        return this.insert(targetPosition, nodes);
    }
    /**
	 * Wraps elements within range with provided {@link module:engine/view/attributeelement~ViewAttributeElement ViewAttributeElement}.
	 * If a collapsed range is provided, it will be wrapped only if it is equal to view selection.
	 *
	 * If a collapsed range was passed and is same as selection, the selection
	 * will be moved to the inside of the wrapped attribute element.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError} `view-writer-invalid-range-container`
	 * when {@link module:engine/view/range~ViewRange#start}
	 * and {@link module:engine/view/range~ViewRange#end} positions are not placed inside same parent container.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError} `view-writer-wrap-invalid-attribute` when passed attribute element is not
	 * an instance of {@link module:engine/view/attributeelement~ViewAttributeElement ViewAttributeElement}.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError} `view-writer-wrap-nonselection-collapsed-range` when passed range
	 * is collapsed and different than view selection.
	 *
	 * @param range Range to wrap.
	 * @param attribute Attribute element to use as wrapper.
	 * @returns range Range after wrapping, spanning over wrapping attribute element.
	 */ wrap(range, attribute) {
        if (!(attribute instanceof ViewAttributeElement)) {
            throw new CKEditorError('view-writer-wrap-invalid-attribute', this.document);
        }
        validateRangeContainer(range, this.document);
        if (!range.isCollapsed) {
            // Non-collapsed range. Wrap it with the attribute element.
            return this._wrapRange(range, attribute);
        } else {
            // Collapsed range. Wrap position.
            let position = range.start;
            if (position.parent.is('element') && !_hasNonUiChildren(position.parent)) {
                position = position.getLastMatchingPosition((value)=>value.item.is('uiElement'));
            }
            position = this._wrapPosition(position, attribute);
            const viewSelection = this.document.selection;
            // If wrapping position is equal to view selection, move view selection inside wrapping attribute element.
            if (viewSelection.isCollapsed && viewSelection.getFirstPosition().isEqual(range.start)) {
                this.setSelection(position);
            }
            return new ViewRange(position);
        }
    }
    /**
	 * Unwraps nodes within provided range from attribute element.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-writer-invalid-range-container` when
	 * {@link module:engine/view/range~ViewRange#start start} and {@link module:engine/view/range~ViewRange#end end}
	 * positions are not placed inside same parent container.
	 */ unwrap(range, attribute) {
        if (!(attribute instanceof ViewAttributeElement)) {
            /**
			 * The `attribute` passed to {@link module:engine/view/downcastwriter~ViewDowncastWriter#unwrap `ViewDowncastWriter#unwrap()`}
			 * must be an instance of {@link module:engine/view/attributeelement~ViewAttributeElement `AttributeElement`}.
			 *
			 * @error view-writer-unwrap-invalid-attribute
			 */ throw new CKEditorError('view-writer-unwrap-invalid-attribute', this.document);
        }
        validateRangeContainer(range, this.document);
        // If range is collapsed - nothing to unwrap.
        if (range.isCollapsed) {
            return range;
        }
        // Break attributes at range start and end.
        const { start: breakStart, end: breakEnd } = this._breakAttributesRange(range, true);
        const parentContainer = breakStart.parent;
        // Unwrap children located between break points.
        const newRange = this._unwrapChildren(parentContainer, breakStart.offset, breakEnd.offset, attribute);
        // Merge attributes at the both ends and return a new range.
        const start = this.mergeAttributes(newRange.start);
        // If start position was merged - move end position back.
        if (!start.isEqual(newRange.start)) {
            newRange.end.offset--;
        }
        const end = this.mergeAttributes(newRange.end);
        return new ViewRange(start, end);
    }
    /**
	 * Renames element by creating a copy of renamed element but with changed name and then moving contents of the
	 * old element to the new one. Keep in mind that this will invalidate all {@link module:engine/view/position~ViewPosition positions}
	 * which has renamed element as {@link module:engine/view/position~ViewPosition#parent a parent}.
	 *
	 * New element has to be created because `Element#tagName` property in DOM is readonly.
	 *
	 * Since this function creates a new element and removes the given one, the new element is returned to keep reference.
	 *
	 * @param newName New name for element.
	 * @param viewElement Element to be renamed.
	 * @returns Element created due to rename.
	 */ rename(newName, viewElement) {
        const newElement = new ViewContainerElement(this.document, newName, viewElement.getAttributes());
        this.insert(ViewPosition._createAfter(viewElement), newElement);
        this.move(ViewRange._createIn(viewElement), ViewPosition._createAt(newElement, 0));
        this.remove(ViewRange._createOn(viewElement));
        return newElement;
    }
    /**
	 * Cleans up memory by removing obsolete cloned elements group from the writer.
	 *
	 * Should be used whenever all {@link module:engine/view/attributeelement~ViewAttributeElement attribute elements}
	 * with the same {@link module:engine/view/attributeelement~ViewAttributeElement#id id} are going to be removed from the view and
	 * the group will no longer be needed.
	 *
	 * Cloned elements group are not removed automatically in case if the group is still needed after all its elements
	 * were removed from the view.
	 *
	 * Keep in mind that group names are equal to the `id` property of the attribute element.
	 *
	 * @param groupName Name of the group to clear.
	 */ clearClonedElementsGroup(groupName) {
        this._cloneGroups.delete(groupName);
    }
    /**
	 * Creates position at the given location. The location can be specified as:
	 *
	 * * a {@link module:engine/view/position~ViewPosition position},
	 * * parent element and offset (offset defaults to `0`),
	 * * parent element and `'end'` (sets position at the end of that element),
	 * * {@link module:engine/view/item~ViewItem view item} and `'before'` or `'after'` (sets position before or after given view item).
	 *
	 * This method is a shortcut to other constructors such as:
	 *
	 * * {@link #createPositionBefore},
	 * * {@link #createPositionAfter},
	 *
	 * @param offset Offset or one of the flags. Used only when the first parameter is a {@link module:engine/view/item~ViewItem view item}.
	 */ createPositionAt(itemOrPosition, offset) {
        return ViewPosition._createAt(itemOrPosition, offset);
    }
    /**
	 * Creates a new position after given view item.
	 *
	 * @param item View item after which the position should be located.
	 */ createPositionAfter(item) {
        return ViewPosition._createAfter(item);
    }
    /**
	 * Creates a new position before given view item.
	 *
	 * @param item View item before which the position should be located.
	 */ createPositionBefore(item) {
        return ViewPosition._createBefore(item);
    }
    /**
	 * Creates a range spanning from `start` position to `end` position.
	 *
	 * **Note:** This factory method creates its own {@link module:engine/view/position~ViewPosition} instances basing on passed values.
	 *
	 * @param start Start position.
	 * @param end End position. If not set, range will be collapsed at `start` position.
	 */ createRange(start, end) {
        return new ViewRange(start, end);
    }
    /**
	 * Creates a range that starts before given {@link module:engine/view/item~ViewItem view item} and ends after it.
	 */ createRangeOn(item) {
        return ViewRange._createOn(item);
    }
    /**
	 * Creates a range inside an {@link module:engine/view/element~ViewElement element} which starts before the first child of
	 * that element and ends after the last child of that element.
	 *
	 * @param element Element which is a parent for the range.
	 */ createRangeIn(element) {
        return ViewRange._createIn(element);
    }
    createSelection(...args) {
        return new ViewSelection(...args);
    }
    /**
	 * Creates placeholders for child elements of the {@link module:engine/conversion/downcasthelpers~DowncastHelpers#elementToStructure
	 * `elementToStructure()`} conversion helper.
	 *
	 * ```ts
	 * const viewSlot = conversionApi.writer.createSlot();
	 * const viewPosition = conversionApi.writer.createPositionAt( viewElement, 0 );
	 *
	 * conversionApi.writer.insert( viewPosition, viewSlot );
	 * ```
	 *
	 * It could be filtered down to a specific subset of children (only `<foo>` model elements in this case):
	 *
	 * ```ts
	 * const viewSlot = conversionApi.writer.createSlot( node => node.is( 'element', 'foo' ) );
	 * const viewPosition = conversionApi.writer.createPositionAt( viewElement, 0 );
	 *
	 * conversionApi.writer.insert( viewPosition, viewSlot );
	 * ```
	 *
	 * While providing a filtered slot, make sure to provide slots for all child nodes. A single node cannot be downcasted into
	 * multiple slots.
	 *
	 * **Note**: You should not change the order of nodes. View elements should be in the same order as model nodes.
	 *
	 * @param modeOrFilter The filter for child nodes.
	 * @returns The slot element to be placed in to the view structure while processing
	 * {@link module:engine/conversion/downcasthelpers~DowncastHelpers#elementToStructure `elementToStructure()`}.
	 */ createSlot(modeOrFilter = 'children') {
        if (!this._slotFactory) {
            /**
			 * The `createSlot()` method is only allowed inside the `elementToStructure` downcast helper callback.
			 *
			 * @error view-writer-invalid-create-slot-context
			 */ throw new CKEditorError('view-writer-invalid-create-slot-context', this.document);
        }
        return this._slotFactory(this, modeOrFilter);
    }
    /**
	 * Registers a slot factory.
	 *
	 * @internal
	 * @param slotFactory The slot factory.
	 */ _registerSlotFactory(slotFactory) {
        this._slotFactory = slotFactory;
    }
    /**
	 * Clears the registered slot factory.
	 *
	 * @internal
	 */ _clearSlotFactory() {
        this._slotFactory = null;
    }
    /**
	 * Inserts a node or nodes at the specified position. Takes care of breaking attributes before insertion
	 * and merging them afterwards if requested by the breakAttributes param.
	 *
	 * @param position Insertion position.
	 * @param nodes Node or nodes to insert.
	 * @param breakAttributes Whether attributes should be broken.
	 * @returns Range around inserted nodes.
	 */ _insertNodes(position, nodes, breakAttributes) {
        let parentElement;
        // Break attributes on nodes that do exist in the model tree so they can have attributes, other elements
        // can't have an attribute in model and won't get wrapped with an ViewAttributeElement while down-casted.
        if (breakAttributes) {
            parentElement = getParentContainer(position);
        } else {
            parentElement = position.parent.is('$text') ? position.parent.parent : position.parent;
        }
        if (!parentElement) {
            /**
			 * Position's parent container cannot be found.
			 *
			 * @error view-writer-invalid-position-container
			 */ throw new CKEditorError('view-writer-invalid-position-container', this.document);
        }
        let insertionPosition;
        if (breakAttributes) {
            insertionPosition = this._breakAttributes(position, true);
        } else {
            insertionPosition = position.parent.is('$text') ? breakTextNode(position) : position;
        }
        const length = parentElement._insertChild(insertionPosition.offset, nodes);
        for (const node of nodes){
            this._addToClonedElementsGroup(node);
        }
        const endPosition = insertionPosition.getShiftedBy(length);
        const start = this.mergeAttributes(insertionPosition);
        // If start position was merged - move end position.
        if (!start.isEqual(insertionPosition)) {
            endPosition.offset--;
        }
        const end = this.mergeAttributes(endPosition);
        return new ViewRange(start, end);
    }
    /**
	 * Wraps children with provided `wrapElement`. Only children contained in `parent` element between
	 * `startOffset` and `endOffset` will be wrapped.
	 */ _wrapChildren(parent, startOffset, endOffset, wrapElement) {
        let i = startOffset;
        const wrapPositions = [];
        while(i < endOffset){
            const child = parent.getChild(i);
            const isText = child.is('$text');
            const isAttribute = child.is('attributeElement');
            //
            // (In all examples, assume that `wrapElement` is `<span class="foo">` element.)
            //
            // Check if `wrapElement` can be joined with the wrapped element. One of requirements is having same name.
            // If possible, join elements.
            //
            // <p><span class="bar">abc</span></p>  -->  <p><span class="foo bar">abc</span></p>
            //
            if (isAttribute && child._canMergeAttributesFrom(wrapElement)) {
                child._mergeAttributesFrom(wrapElement);
                wrapPositions.push(new ViewPosition(parent, i));
            } else if (isText || !isAttribute || shouldABeOutsideB(wrapElement, child)) {
                // Clone attribute.
                const newAttribute = wrapElement._clone();
                // Wrap current node with new attribute.
                child._remove();
                newAttribute._appendChild(child);
                parent._insertChild(i, newAttribute);
                this._addToClonedElementsGroup(newAttribute);
                wrapPositions.push(new ViewPosition(parent, i));
            } else /* if ( isAttribute ) */ {
                this._wrapChildren(child, 0, child.childCount, wrapElement);
            }
            i++;
        }
        // Merge at each wrap.
        let offsetChange = 0;
        for (const position of wrapPositions){
            position.offset -= offsetChange;
            // Do not merge with elements outside selected children.
            if (position.offset == startOffset) {
                continue;
            }
            const newPosition = this.mergeAttributes(position);
            // If nodes were merged - other merge offsets will change.
            if (!newPosition.isEqual(position)) {
                offsetChange++;
                endOffset--;
            }
        }
        return ViewRange._createFromParentsAndOffsets(parent, startOffset, parent, endOffset);
    }
    /**
	 * Unwraps children from provided `unwrapElement`. Only children contained in `parent` element between
	 * `startOffset` and `endOffset` will be unwrapped.
	 */ _unwrapChildren(parent, startOffset, endOffset, unwrapElement) {
        let i = startOffset;
        const unwrapPositions = [];
        // Iterate over each element between provided offsets inside parent.
        // We don't use tree walker or range iterator because we will be removing and merging potentially multiple nodes,
        // so it could get messy. It is safer to it manually in this case.
        while(i < endOffset){
            const child = parent.getChild(i);
            // Skip all text nodes. There should be no container element's here either.
            if (!child.is('attributeElement')) {
                i++;
                continue;
            }
            //
            // (In all examples, assume that `unwrapElement` is `<span class="foo">` element.)
            //
            // If the child is similar to the given attribute element, unwrap it - it will be completely removed.
            //
            // <p><span class="foo">abc</span>xyz</p>  -->  <p>abcxyz</p>
            //
            if (child.isSimilar(unwrapElement)) {
                const unwrapped = child.getChildren();
                const count = child.childCount;
                // Replace wrapper element with its children
                child._remove();
                parent._insertChild(i, unwrapped);
                this._removeFromClonedElementsGroup(child);
                // Save start and end position of moved items.
                unwrapPositions.push(new ViewPosition(parent, i), new ViewPosition(parent, i + count));
                // Skip elements that were unwrapped. Assuming there won't be another element to unwrap in child elements.
                i += count;
                endOffset += count - 1;
                continue;
            }
            //
            // If the child is not similar but is an attribute element, try partial unwrapping - remove the same attributes/styles/classes.
            // Partial unwrapping will happen only if the elements have the same name.
            //
            // <p><span class="foo bar">abc</span>xyz</p>  -->  <p><span class="bar">abc</span>xyz</p>
            // <p><i class="foo">abc</i>xyz</p>            -->  <p><i class="foo">abc</i>xyz</p>
            //
            if (child._canSubtractAttributesOf(unwrapElement)) {
                child._subtractAttributesOf(unwrapElement);
                unwrapPositions.push(new ViewPosition(parent, i), new ViewPosition(parent, i + 1));
                i++;
                continue;
            }
            //
            // If other nested attribute is found, look through it's children for elements to unwrap.
            //
            // <p><i><span class="foo">abc</span></i><p>  -->  <p><i>abc</i><p>
            //
            this._unwrapChildren(child, 0, child.childCount, unwrapElement);
            i++;
        }
        // Merge at each unwrap.
        let offsetChange = 0;
        for (const position of unwrapPositions){
            position.offset -= offsetChange;
            // Do not merge with elements outside selected children.
            if (position.offset == startOffset || position.offset == endOffset) {
                continue;
            }
            const newPosition = this.mergeAttributes(position);
            // If nodes were merged - other merge offsets will change.
            if (!newPosition.isEqual(position)) {
                offsetChange++;
                endOffset--;
            }
        }
        return ViewRange._createFromParentsAndOffsets(parent, startOffset, parent, endOffset);
    }
    /**
	 * Helper function for `view.writer.wrap`. Wraps range with provided attribute element.
	 * This method will also merge newly added attribute element with its siblings whenever possible.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError} `view-writer-wrap-invalid-attribute` when passed attribute element is not
	 * an instance of {@link module:engine/view/attributeelement~ViewAttributeElement ViewAttributeElement}.
	 *
	 * @returns New range after wrapping, spanning over wrapping attribute element.
	 */ _wrapRange(range, attribute) {
        // Break attributes at range start and end.
        const { start: breakStart, end: breakEnd } = this._breakAttributesRange(range, true);
        const parentContainer = breakStart.parent;
        // Wrap all children with attribute.
        const newRange = this._wrapChildren(parentContainer, breakStart.offset, breakEnd.offset, attribute);
        // Merge attributes at the both ends and return a new range.
        const start = this.mergeAttributes(newRange.start);
        // If start position was merged - move end position back.
        if (!start.isEqual(newRange.start)) {
            newRange.end.offset--;
        }
        const end = this.mergeAttributes(newRange.end);
        return new ViewRange(start, end);
    }
    /**
	 * Helper function for {@link #wrap}. Wraps position with provided attribute element.
	 * This method will also merge newly added attribute element with its siblings whenever possible.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError} `view-writer-wrap-invalid-attribute` when passed attribute element is not
	 * an instance of {@link module:engine/view/attributeelement~ViewAttributeElement ViewAttributeElement}.
	 *
	 * @returns New position after wrapping.
	 */ _wrapPosition(position, attribute) {
        // Return same position when trying to wrap with attribute similar to position parent.
        if (attribute.isSimilar(position.parent)) {
            return movePositionToTextNode(position.clone());
        }
        // When position is inside text node - break it and place new position between two text nodes.
        if (position.parent.is('$text')) {
            position = breakTextNode(position);
        }
        // Create fake element that will represent position, and will not be merged with other attributes.
        const fakeElement = this.createAttributeElement('_wrapPosition-fake-element');
        fakeElement._priority = Number.POSITIVE_INFINITY;
        fakeElement.isSimilar = ()=>false;
        // Insert fake element in position location.
        position.parent._insertChild(position.offset, fakeElement);
        // Range around inserted fake attribute element.
        const wrapRange = new ViewRange(position, position.getShiftedBy(1));
        // Wrap fake element with attribute (it will also merge if possible).
        this.wrap(wrapRange, attribute);
        // Remove fake element and place new position there.
        const newPosition = new ViewPosition(fakeElement.parent, fakeElement.index);
        fakeElement._remove();
        // If position is placed between text nodes - merge them and return position inside.
        const nodeBefore = newPosition.nodeBefore;
        const nodeAfter = newPosition.nodeAfter;
        if (nodeBefore && nodeBefore.is('view:$text') && nodeAfter && nodeAfter.is('view:$text')) {
            return mergeTextNodes(nodeBefore, nodeAfter);
        }
        // If position is next to text node - move position inside.
        return movePositionToTextNode(newPosition);
    }
    /**
	 * Helper function used by other `ViewDowncastWriter` methods. Breaks attribute elements at the boundaries of given range.
	 *
	 * @param range Range which `start` and `end` positions will be used to break attributes.
	 * @param forceSplitText If set to `true`, will break text nodes even if they are directly in container element.
	 * This behavior will result in incorrect view state, but is needed by other view writing methods which then fixes view state.
	 * @returns New range with located at break positions.
	 */ _breakAttributesRange(range, forceSplitText = false) {
        const rangeStart = range.start;
        const rangeEnd = range.end;
        validateRangeContainer(range, this.document);
        // Break at the collapsed position. Return new collapsed range.
        if (range.isCollapsed) {
            const position = this._breakAttributes(range.start, forceSplitText);
            return new ViewRange(position, position);
        }
        const breakEnd = this._breakAttributes(rangeEnd, forceSplitText);
        const count = breakEnd.parent.childCount;
        const breakStart = this._breakAttributes(rangeStart, forceSplitText);
        // Calculate new break end offset.
        breakEnd.offset += breakEnd.parent.childCount - count;
        return new ViewRange(breakStart, breakEnd);
    }
    /**
	 * Helper function used by other `ViewDowncastWriter` methods. Breaks attribute elements at given position.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-writer-cannot-break-empty-element` when break position
	 * is placed inside {@link module:engine/view/emptyelement~ViewEmptyElement ViewEmptyElement}.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-writer-cannot-break-ui-element` when break position
	 * is placed inside {@link module:engine/view/uielement~ViewUIElement UIElement}.
	 *
	 * @param position Position where to break attributes.
	 * @param forceSplitText If set to `true`, will break text nodes even if they are directly in container element.
	 * This behavior will result in incorrect view state, but is needed by other view writing methods which then fixes view state.
	 * @returns New position after breaking the attributes.
	 */ _breakAttributes(position, forceSplitText = false) {
        const positionOffset = position.offset;
        const positionParent = position.parent;
        // If position is placed inside ViewEmptyElement - throw an exception as we cannot break inside.
        if (position.parent.is('emptyElement')) {
            /**
			 * Cannot break an `EmptyElement` instance.
			 *
			 * This error is thrown if
			 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#breakAttributes `ViewDowncastWriter#breakAttributes()`}
			 * was executed in an incorrect position.
			 *
			 * @error view-writer-cannot-break-empty-element
			 */ throw new CKEditorError('view-writer-cannot-break-empty-element', this.document);
        }
        // If position is placed inside UIElement - throw an exception as we cannot break inside.
        if (position.parent.is('uiElement')) {
            /**
			 * Cannot break a `UIElement` instance.
			 *
			 * This error is thrown if
			 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#breakAttributes `ViewDowncastWriter#breakAttributes()`}
			 * was executed in an incorrect position.
			 *
			 * @error view-writer-cannot-break-ui-element
			 */ throw new CKEditorError('view-writer-cannot-break-ui-element', this.document);
        }
        // If position is placed inside RawElement - throw an exception as we cannot break inside.
        if (position.parent.is('rawElement')) {
            /**
			 * Cannot break a `RawElement` instance.
			 *
			 * This error is thrown if
			 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#breakAttributes `ViewDowncastWriter#breakAttributes()`}
			 * was executed in an incorrect position.
			 *
			 * @error view-writer-cannot-break-raw-element
			 */ throw new CKEditorError('view-writer-cannot-break-raw-element', this.document);
        }
        // There are no attributes to break and text nodes breaking is not forced.
        if (!forceSplitText && positionParent.is('$text') && isContainerOrFragment(positionParent.parent)) {
            return position.clone();
        }
        // Position's parent is container, so no attributes to break.
        if (isContainerOrFragment(positionParent)) {
            return position.clone();
        }
        // Break text and start again in new position.
        if (positionParent.is('$text')) {
            return this._breakAttributes(breakTextNode(position), forceSplitText);
        }
        const length = positionParent.childCount;
        // <p>foo<b><u>bar{}</u></b></p>
        // <p>foo<b><u>bar</u>[]</b></p>
        // <p>foo<b><u>bar</u></b>[]</p>
        if (positionOffset == length) {
            const newPosition = new ViewPosition(positionParent.parent, positionParent.index + 1);
            return this._breakAttributes(newPosition, forceSplitText);
        } else {
            // <p>foo<b><u>{}bar</u></b></p>
            // <p>foo<b>[]<u>bar</u></b></p>
            // <p>foo{}<b><u>bar</u></b></p>
            if (positionOffset === 0) {
                const newPosition = new ViewPosition(positionParent.parent, positionParent.index);
                return this._breakAttributes(newPosition, forceSplitText);
            } else {
                const offsetAfter = positionParent.index + 1;
                // Break element.
                const clonedNode = positionParent._clone();
                // Insert cloned node to position's parent node.
                positionParent.parent._insertChild(offsetAfter, clonedNode);
                this._addToClonedElementsGroup(clonedNode);
                // Get nodes to move.
                const count = positionParent.childCount - positionOffset;
                const nodesToMove = positionParent._removeChildren(positionOffset, count);
                // Move nodes to cloned node.
                clonedNode._appendChild(nodesToMove);
                // Create new position to work on.
                const newPosition = new ViewPosition(positionParent.parent, offsetAfter);
                return this._breakAttributes(newPosition, forceSplitText);
            }
        }
    }
    /**
	 * Stores the information that an {@link module:engine/view/attributeelement~ViewAttributeElement attribute element} was
	 * added to the tree. Saves the reference to the group in the given element and updates the group, so other elements
	 * from the group now keep a reference to the given attribute element.
	 *
	 * The clones group can be obtained using {@link module:engine/view/attributeelement~ViewAttributeElement#getElementsWithSameId}.
	 *
	 * Does nothing if added element has no {@link module:engine/view/attributeelement~ViewAttributeElement#id id}.
	 *
	 * @param element Attribute element to save.
	 */ _addToClonedElementsGroup(element) {
        // Add only if the element is in document tree.
        if (!element.root.is('rootElement')) {
            return;
        }
        // Traverse the element's children recursively to find other attribute elements that also might got inserted.
        // The loop is at the beginning so we can make fast returns later in the code.
        if (element.is('element')) {
            for (const child of element.getChildren()){
                this._addToClonedElementsGroup(child);
            }
        }
        const id = element.id;
        if (!id) {
            return;
        }
        let group = this._cloneGroups.get(id);
        if (!group) {
            group = new Set();
            this._cloneGroups.set(id, group);
        }
        group.add(element);
        element._clonesGroup = group;
    }
    /**
	 * Removes all the information about the given {@link module:engine/view/attributeelement~ViewAttributeElement attribute element}
	 * from its clones group.
	 *
	 * Keep in mind, that the element will still keep a reference to the group (but the group will not keep a reference to it).
	 * This allows to reference the whole group even if the element was already removed from the tree.
	 *
	 * Does nothing if the element has no {@link module:engine/view/attributeelement~ViewAttributeElement#id id}.
	 *
	 * @param element Attribute element to remove.
	 */ _removeFromClonedElementsGroup(element) {
        // Traverse the element's children recursively to find other attribute elements that also got removed.
        // The loop is at the beginning so we can make fast returns later in the code.
        if (element.is('element')) {
            for (const child of element.getChildren()){
                this._removeFromClonedElementsGroup(child);
            }
        }
        const id = element.id;
        if (!id) {
            return;
        }
        const group = this._cloneGroups.get(id);
        if (!group) {
            return;
        }
        group.delete(element);
    // Not removing group from element on purpose!
    // If other parts of code have reference to this element, they will be able to get references to other elements from the group.
    }
}
// Helper function for `view.writer.wrap`. Checks if given element has any children that are not ui elements.
function _hasNonUiChildren(parent) {
    return Array.from(parent.getChildren()).some((child)=>!child.is('uiElement'));
}
/**
 * The `attribute` passed to {@link module:engine/view/downcastwriter~ViewDowncastWriter#wrap `ViewDowncastWriter#wrap()`}
 * must be an instance of {@link module:engine/view/attributeelement~ViewAttributeElement `AttributeElement`}.
 *
 * @error view-writer-wrap-invalid-attribute
 */ /**
 * Returns first parent container of specified {@link module:engine/view/position~ViewPosition Position}.
 * Position's parent node is checked as first, then next parents are checked.
 * Note that {@link module:engine/view/documentfragment~ViewDocumentFragment DocumentFragment} is treated like a container.
 *
 * @param position Position used as a start point to locate parent container.
 * @returns Parent container element or `undefined` if container is not found.
 */ function getParentContainer(position) {
    let parent = position.parent;
    while(!isContainerOrFragment(parent)){
        if (!parent) {
            return undefined;
        }
        parent = parent.parent;
    }
    return parent;
}
/**
 * Checks if first {@link module:engine/view/attributeelement~ViewAttributeElement ViewAttributeElement} provided to the function
 * can be wrapped outside second element. It is done by comparing elements'
 * {@link module:engine/view/attributeelement~ViewAttributeElement#priority priorities}, if both have same priority
 * {@link module:engine/view/element~ViewElement#getIdentity identities} are compared.
 */ function shouldABeOutsideB(a, b) {
    if (a.priority < b.priority) {
        return true;
    } else if (a.priority > b.priority) {
        return false;
    }
    // When priorities are equal and names are different - use identities.
    return a.getIdentity() < b.getIdentity();
}
/**
 * Returns new position that is moved to near text node. Returns same position if there is no text node before of after
 * specified position.
 *
 * ```html
 * <p>foo[]</p>  ->  <p>foo{}</p>
 * <p>[]foo</p>  ->  <p>{}foo</p>
 * ```
 *
 * @returns Position located inside text node or same position if there is no text nodes
 * before or after position location.
 */ function movePositionToTextNode(position) {
    const nodeBefore = position.nodeBefore;
    if (nodeBefore && nodeBefore.is('$text')) {
        return new ViewPosition(nodeBefore, nodeBefore.data.length);
    }
    const nodeAfter = position.nodeAfter;
    if (nodeAfter && nodeAfter.is('$text')) {
        return new ViewPosition(nodeAfter, 0);
    }
    return position;
}
/**
 * Breaks text node into two text nodes when possible.
 *
 * ```html
 * <p>foo{}bar</p> -> <p>foo[]bar</p>
 * <p>{}foobar</p> -> <p>[]foobar</p>
 * <p>foobar{}</p> -> <p>foobar[]</p>
 * ```
 *
 * @param position Position that need to be placed inside text node.
 * @returns New position after breaking text node.
 */ function breakTextNode(position) {
    if (position.offset == position.parent.data.length) {
        return new ViewPosition(position.parent.parent, position.parent.index + 1);
    }
    if (position.offset === 0) {
        return new ViewPosition(position.parent.parent, position.parent.index);
    }
    // Get part of the text that need to be moved.
    const textToMove = position.parent.data.slice(position.offset);
    // Leave rest of the text in position's parent.
    position.parent._data = position.parent.data.slice(0, position.offset);
    // Insert new text node after position's parent text node.
    position.parent.parent._insertChild(position.parent.index + 1, new ViewText(position.root.document, textToMove));
    // Return new position between two newly created text nodes.
    return new ViewPosition(position.parent.parent, position.parent.index + 1);
}
/**
 * Merges two text nodes into first node. Removes second node and returns merge position.
 *
 * @param t1 First text node to merge. Data from second text node will be moved at the end of this text node.
 * @param t2 Second text node to merge. This node will be removed after merging.
 * @returns Position after merging text nodes.
 */ function mergeTextNodes(t1, t2) {
    // Merge text data into first text node and remove second one.
    const nodeBeforeLength = t1.data.length;
    t1._data += t2.data;
    t2._remove();
    return new ViewPosition(t1, nodeBeforeLength);
}
const validNodesToInsert = [
    ViewText,
    ViewAttributeElement,
    ViewContainerElement,
    ViewEmptyElement,
    ViewRawElement,
    ViewUIElement
];
/**
 * Checks if provided nodes are valid to insert.
 *
 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-writer-insert-invalid-node` when nodes to insert
 * contains instances that are not supported ones (see error description for valid ones.
 */ function validateNodesToInsert(nodes, errorContext) {
    for (const node of nodes){
        if (!validNodesToInsert.some((validNode)=>node instanceof validNode)) {
            /**
			 * One of the nodes to be inserted is of an invalid type.
			 *
			 * Nodes to be inserted with {@link module:engine/view/downcastwriter~ViewDowncastWriter#insert `ViewDowncastWriter#insert()`}
			 * should be of the following types:
			 *
			 * * {@link module:engine/view/attributeelement~ViewAttributeElement ViewAttributeElement},
			 * * {@link module:engine/view/containerelement~ViewContainerElement ViewContainerElement},
			 * * {@link module:engine/view/emptyelement~ViewEmptyElement ViewEmptyElement},
			 * * {@link module:engine/view/uielement~ViewUIElement UIElement},
			 * * {@link module:engine/view/rawelement~ViewRawElement RawElement},
			 * * {@link module:engine/view/text~ViewText Text}.
			 *
			 * @error view-writer-insert-invalid-node-type
			 */ throw new CKEditorError('view-writer-insert-invalid-node-type', errorContext);
        }
        if (!node.is('$text')) {
            validateNodesToInsert(node.getChildren(), errorContext);
        }
    }
}
/**
 * Checks if node is ViewContainerElement or DocumentFragment, because in most cases they should be treated the same way.
 *
 * @returns Returns `true` if node is instance of ViewContainerElement or DocumentFragment.
 */ function isContainerOrFragment(node) {
    return node && (node.is('containerElement') || node.is('documentFragment'));
}
/**
 * Checks if {@link module:engine/view/range~ViewRange#start range start} and {@link module:engine/view/range~ViewRange#end range end}
 * are placed inside same {@link module:engine/view/containerelement~ViewContainerElement container element}.
 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `view-writer-invalid-range-container` when validation fails.
 */ function validateRangeContainer(range, errorContext) {
    const startContainer = getParentContainer(range.start);
    const endContainer = getParentContainer(range.end);
    if (!startContainer || !endContainer || startContainer !== endContainer) {
        /**
		 * The container of the given range is invalid.
		 *
		 * This may happen if {@link module:engine/view/range~ViewRange#start range start} and
		 * {@link module:engine/view/range~ViewRange#end range end} positions are not placed inside the same container element or
		 * a parent container for these positions cannot be found.
		 *
		 * Methods like {@link module:engine/view/downcastwriter~ViewDowncastWriter#wrap `ViewDowncastWriter#remove()`},
		 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#wrap `ViewDowncastWriter#clean()`},
		 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#wrap `ViewDowncastWriter#wrap()`},
		 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#wrap `ViewDowncastWriter#unwrap()`} need to be called
		 * on a range that has its start and end positions located in the same container element. Both positions can be
		 * nested within other elements (e.g. an attribute element) but the closest container ancestor must be the same.
		 *
		 * @error view-writer-invalid-range-container
		 */ throw new CKEditorError('view-writer-invalid-range-container', errorContext);
    }
}
/**
 * Checks if the provided argument is a plain object that can be used as options for container element.
 */ function isContainerOptions(childrenOrOptions) {
    return isPlainObject(childrenOrOptions);
}

/**
 * Set of utilities related to handling block and inline fillers.
 *
 * Browsers do not allow to put caret in elements which does not have height. Because of it, we need to fill all
 * empty elements which should be selectable with elements or characters called "fillers". Unfortunately there is no one
 * universal filler, this is why two types are uses:
 *
 * * Block filler is an element which fill block elements, like `<p>`. CKEditor uses `<br>` as a block filler during the editing,
 * as browsers do natively. So instead of an empty `<p>` there will be `<p><br></p>`. The advantage of block filler is that
 * it is transparent for the selection, so when the caret is before the `<br>` and user presses right arrow he will be
 * moved to the next paragraph, not after the `<br>`. The disadvantage is that it breaks a block, so it cannot be used
 * in the middle of a line of text. The {@link module:engine/view/filler~BR_FILLER `<br>` filler} can be replaced with any other
 * character in the data output, for instance {@link module:engine/view/filler~NBSP_FILLER non-breaking space} or
 * {@link module:engine/view/filler~MARKED_NBSP_FILLER marked non-breaking space}.
 *
 * * Inline filler is a filler which does not break a line of text, so it can be used inside the text, for instance in the empty
 * `<b>` surrendered by text: `foo<b></b>bar`, if we want to put the caret there. CKEditor uses a sequence of the zero-width
 * spaces as an {@link module:engine/view/filler~INLINE_FILLER inline filler} having the predetermined
 * {@link module:engine/view/filler~INLINE_FILLER_LENGTH length}. A sequence is used, instead of a single character to
 * avoid treating random zero-width spaces as the inline filler. Disadvantage of the inline filler is that it is not
 * transparent for the selection. The arrow key moves the caret between zero-width spaces characters, so the additional
 * code is needed to handle the caret.
 *
 * Both inline and block fillers are handled by the {@link module:engine/view/renderer~ViewRenderer renderer} and are not present in the
 * view.
 *
 * @module engine/view/filler
 */ /**
 * Non-breaking space filler creator. This function creates the `&nbsp;` text node.
 * It defines how the filler is created.
 *
 * @see module:engine/view/filler~MARKED_NBSP_FILLER
 * @see module:engine/view/filler~BR_FILLER
 * @internal
 */ const NBSP_FILLER = (domDocument)=>domDocument.createTextNode('\u00A0');
/**
 * Marked non-breaking space filler creator. This function creates the `<span data-cke-filler="true">&nbsp;</span>` element.
 * It defines how the filler is created.
 *
 * @see module:engine/view/filler~NBSP_FILLER
 * @see module:engine/view/filler~BR_FILLER
 * @internal
 */ const MARKED_NBSP_FILLER = (domDocument)=>{
    const span = domDocument.createElement('span');
    span.dataset.ckeFiller = 'true';
    span.innerText = '\u00A0';
    return span;
};
/**
 * `<br>` filler creator. This function creates the `<br data-cke-filler="true">` element.
 * It defines how the filler is created.
 *
 * @see module:engine/view/filler~NBSP_FILLER
 * @see module:engine/view/filler~MARKED_NBSP_FILLER
 * @internal
 */ const BR_FILLER = (domDocument)=>{
    const fillerBr = domDocument.createElement('br');
    fillerBr.dataset.ckeFiller = 'true';
    return fillerBr;
};
/**
 * Length of the {@link module:engine/view/filler~INLINE_FILLER INLINE_FILLER}.
 *
 * @internal
 */ const INLINE_FILLER_LENGTH = 7;
/**
 * Inline filler which is a sequence of the word joiners.
 *
 * @internal
 */ const INLINE_FILLER = '\u2060'.repeat(INLINE_FILLER_LENGTH);
/**
 * Checks if the node is a text node which starts with the {@link module:engine/view/filler~INLINE_FILLER inline filler}.
 *
 * ```ts
 * startsWithFiller( document.createTextNode( INLINE_FILLER ) ); // true
 * startsWithFiller( document.createTextNode( INLINE_FILLER + 'foo' ) ); // true
 * startsWithFiller( document.createTextNode( 'foo' ) ); // false
 * startsWithFiller( document.createElement( 'p' ) ); // false
 * ```
 *
 * @param domNode DOM node.
 * @returns True if the text node starts with the {@link module:engine/view/filler~INLINE_FILLER inline filler}.
 * @internal
 */ function startsWithFiller(domNode) {
    if (typeof domNode == 'string') {
        return domNode.substr(0, INLINE_FILLER_LENGTH) === INLINE_FILLER;
    }
    return isText(domNode) && domNode.data.substr(0, INLINE_FILLER_LENGTH) === INLINE_FILLER;
}
/**
 * Checks if the text node contains only the {@link module:engine/view/filler~INLINE_FILLER inline filler}.
 *
 * ```ts
 * isInlineFiller( document.createTextNode( INLINE_FILLER ) ); // true
 * isInlineFiller( document.createTextNode( INLINE_FILLER + 'foo' ) ); // false
 * ```
 *
 * @param domText DOM text node.
 * @returns True if the text node contains only the {@link module:engine/view/filler~INLINE_FILLER inline filler}.
 * @internal
 */ function isInlineFiller(domText) {
    return domText.data.length == INLINE_FILLER_LENGTH && startsWithFiller(domText);
}
/**
 * Get string data from the text node, removing an {@link module:engine/view/filler~INLINE_FILLER inline filler} from it,
 * if text node contains it.
 *
 * ```ts
 * getDataWithoutFiller( document.createTextNode( INLINE_FILLER + 'foo' ) ) == 'foo' // true
 * getDataWithoutFiller( document.createTextNode( 'foo' ) ) == 'foo' // true
 * ```
 *
 * @param domText DOM text node, possible with inline filler.
 * @returns Data without filler.
 * @internal
 */ function getDataWithoutFiller(domText) {
    const data = typeof domText == 'string' ? domText : domText.data;
    if (startsWithFiller(domText)) {
        return data.slice(INLINE_FILLER_LENGTH);
    }
    return data;
}
/**
 * Assign key observer which move cursor from the end of the inline filler to the beginning of it when
 * the left arrow is pressed, so the filler does not break navigation.
 *
 * @param view View controller instance we should inject quirks handling on.
 * @internal
 */ function injectQuirksHandling(view) {
    view.document.on('arrowKey', jumpOverInlineFiller, {
        priority: 'low'
    });
}
/**
 * Move cursor from the end of the inline filler to the beginning of it when, so the filler does not break navigation.
 */ function jumpOverInlineFiller(evt, data) {
    if (data.keyCode == keyCodes.arrowleft) {
        const domSelection = data.domTarget.ownerDocument.defaultView.getSelection();
        if (domSelection.rangeCount == 1 && domSelection.getRangeAt(0).collapsed) {
            const domParent = domSelection.getRangeAt(0).startContainer;
            const domOffset = domSelection.getRangeAt(0).startOffset;
            if (startsWithFiller(domParent) && domOffset <= INLINE_FILLER_LENGTH) {
                domSelection.collapse(domParent, 0);
            }
        }
    }
}

/**
 * Renderer is responsible for updating the DOM structure and the DOM selection based on
 * the {@link module:engine/view/renderer~ViewRenderer#markToSync information about updated view nodes}.
 * In other words, it renders the view to the DOM.
 *
 * Its main responsibility is to make only the necessary, minimal changes to the DOM. However, unlike in many
 * virtual DOM implementations, the primary reason for doing minimal changes is not the performance but ensuring
 * that native editing features such as text composition, autocompletion, spell checking, selection's x-index are
 * affected as little as possible.
 *
 * Renderer uses {@link module:engine/view/domconverter~ViewDomConverter} to transform view nodes and positions
 * to and from the DOM.
 */ class ViewRenderer extends /* #__PURE__ */ ObservableMixin() {
    /**
	 * Set of DOM Documents instances.
	 */ domDocuments = new Set();
    /**
	 * Converter instance.
	 */ domConverter;
    /**
	 * Set of nodes which attributes changed and may need to be rendered.
	 */ markedAttributes = new Set();
    /**
	 * Set of elements which child lists changed and may need to be rendered.
	 */ markedChildren = new Set();
    /**
	 * Set of text nodes which text data changed and may need to be rendered.
	 */ markedTexts = new Set();
    /**
	 * View selection. Renderer updates DOM selection based on the view selection.
	 */ selection;
    /**
	 * The text node in which the inline filler was rendered.
	 */ _inlineFiller = null;
    /**
	 * DOM element containing fake selection.
	 */ _fakeSelectionContainer = null;
    /**
	 * Creates a renderer instance.
	 *
	 * @param domConverter Converter instance.
	 * @param selection View selection.
	 */ constructor(domConverter, selection){
        super();
        this.domConverter = domConverter;
        this.selection = selection;
        this.set('isFocused', false);
        this.set('isSelecting', false);
        this.set('isComposing', false);
        // Rendering the selection and inline filler manipulation should be postponed in (non-Android) Blink until the user finishes
        // creating the selection in DOM to avoid accidental selection collapsing
        // (https://github.com/ckeditor/ckeditor5/issues/10562, https://github.com/ckeditor/ckeditor5/issues/10723).
        // When the user stops selecting, all pending changes should be rendered ASAP, though.
        if (env.isBlink && !env.isAndroid) {
            this.on('change:isSelecting', ()=>{
                if (!this.isSelecting) {
                    this.render();
                }
            });
        }
    }
    /**
	 * Marks a view node to be updated in the DOM by {@link #render `render()`}.
	 *
	 * Note that only view nodes whose parents have corresponding DOM elements need to be marked to be synchronized.
	 *
	 * @see #markedAttributes
	 * @see #markedChildren
	 * @see #markedTexts
	 *
	 * @param type Type of the change.
	 * @param node ViewNode to be marked.
	 */ markToSync(type, node) {
        if (type === 'text') {
            if (this.domConverter.mapViewToDom(node.parent)) {
                this.markedTexts.add(node);
            }
        } else {
            // If the node has no DOM element it is not rendered yet,
            // its children/attributes do not need to be marked to be sync.
            if (!this.domConverter.mapViewToDom(node)) {
                return;
            }
            if (type === 'attributes') {
                this.markedAttributes.add(node);
            } else if (type === 'children') {
                this.markedChildren.add(node);
            } else {
                /**
				 * Unknown type passed to Renderer.markToSync.
				 *
				 * @error view-renderer-unknown-type
				 */ throw new CKEditorError('view-renderer-unknown-type', this);
            }
        }
    }
    /**
	 * Renders all buffered changes ({@link #markedAttributes}, {@link #markedChildren} and {@link #markedTexts}) and
	 * the current view selection (if needed) to the DOM by applying a minimal set of changes to it.
	 *
	 * Renderer tries not to break the text composition (e.g. IME) and x-index of the selection,
	 * so it does as little as it is needed to update the DOM.
	 *
	 * Renderer also handles {@link module:engine/view/filler fillers}. Especially, it checks if the inline filler is needed
	 * at the selection position and adds or removes it. To prevent breaking text composition inline filler will not be
	 * removed as long as the selection is in the text node which needed it at first.
	 */ render() {
        // Ignore rendering while in the composition mode. Composition events are not cancellable and browser will modify the DOM tree.
        // All marked elements, attributes, etc. will wait until next render after the composition ends.
        // On Android composition events are immediately applied to the model, so we don't need to skip rendering,
        // and we should not do it because the difference between view and DOM could lead to position mapping problems.
        if (this.isComposing && !env.isAndroid) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
            // @if CK_DEBUG_TYPING // 		'%cRendering aborted while isComposing.',
            // @if CK_DEBUG_TYPING // 		'font-style: italic'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            return;
        }
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'Renderer',
        // @if CK_DEBUG_TYPING // 		'%cRendering',
        // @if CK_DEBUG_TYPING // 		'font-weight: bold'
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        let inlineFillerPosition = null;
        const isInlineFillerRenderingPossible = env.isBlink && !env.isAndroid ? !this.isSelecting : true;
        // Refresh mappings.
        for (const element of this.markedChildren){
            this._updateChildrenMappings(element);
        }
        // Don't manipulate inline fillers while the selection is being made in (non-Android) Blink to prevent accidental
        // DOM selection collapsing
        // (https://github.com/ckeditor/ckeditor5/issues/10562, https://github.com/ckeditor/ckeditor5/issues/10723).
        if (isInlineFillerRenderingPossible) {
            // There was inline filler rendered in the DOM but it's not
            // at the selection position any more, so we can remove it
            // (cause even if it's needed, it must be placed in another location).
            if (this._inlineFiller && !this._isSelectionInInlineFiller()) {
                this._removeInlineFiller();
            }
            // If we've got the filler, let's try to guess its position in the view.
            if (this._inlineFiller) {
                inlineFillerPosition = this._getInlineFillerPosition();
            } else if (this._needsInlineFillerAtSelection()) {
                inlineFillerPosition = this.selection.getFirstPosition();
                // Do not use `markToSync` so it will be added even if the parent is already added.
                this.markedChildren.add(inlineFillerPosition.parent);
            }
        } else if (this._inlineFiller && this._inlineFiller.parentNode) {
            // While the user is making selection, preserve the inline filler at its original position.
            inlineFillerPosition = this.domConverter.domPositionToView(this._inlineFiller);
            // While down-casting the document selection attributes, all existing empty
            // attribute elements (for selection position) are removed from the view and DOM,
            // so make sure that we were able to map filler position.
            // https://github.com/ckeditor/ckeditor5/issues/12026
            if (inlineFillerPosition && inlineFillerPosition.parent.is('$text')) {
                // The inline filler position is expected to be before the text node.
                inlineFillerPosition = ViewPosition._createBefore(inlineFillerPosition.parent);
            }
        }
        for (const element of this.markedAttributes){
            this._updateAttrs(element);
        }
        for (const element of this.markedChildren){
            this._updateChildren(element, {
                inlineFillerPosition
            });
        }
        for (const node of this.markedTexts){
            if (!this.markedChildren.has(node.parent) && this.domConverter.mapViewToDom(node.parent)) {
                this._updateText(node, {
                    inlineFillerPosition
                });
            }
        }
        // * Check whether the inline filler is required and where it really is in the DOM.
        //   At this point in most cases it will be in the DOM, but there are exceptions.
        //   For example, if the inline filler was deep in the created DOM structure, it will not be created.
        //   Similarly, if it was removed at the beginning of this function and then neither text nor children were updated,
        //   it will not be present. Fix those and similar scenarios.
        // * Don't manipulate inline fillers while the selection is being made in (non-Android) Blink to prevent accidental
        //   DOM selection collapsing
        //   (https://github.com/ckeditor/ckeditor5/issues/10562, https://github.com/ckeditor/ckeditor5/issues/10723).
        if (isInlineFillerRenderingPossible) {
            if (inlineFillerPosition) {
                const fillerDomPosition = this.domConverter.viewPositionToDom(inlineFillerPosition);
                const domDocument = fillerDomPosition.parent.ownerDocument;
                if (!startsWithFiller(fillerDomPosition.parent)) {
                    // Filler has not been created at filler position. Create it now.
                    this._inlineFiller = addInlineFiller(domDocument, fillerDomPosition.parent, fillerDomPosition.offset);
                } else {
                    // Filler has been found, save it.
                    this._inlineFiller = fillerDomPosition.parent;
                }
            } else {
                // There is no filler needed.
                this._inlineFiller = null;
            }
        }
        // First focus the new editing host, then update the selection.
        // Otherwise, FF may throw an error (https://github.com/ckeditor/ckeditor5/issues/721).
        this._updateFocus();
        this._updateSelection();
        this.domConverter._clearTemporaryCustomProperties();
        this.markedTexts.clear();
        this.markedAttributes.clear();
        this.markedChildren.clear();
    // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
    // @if CK_DEBUG_TYPING // 	console.groupEnd();
    // @if CK_DEBUG_TYPING // }
    }
    /**
	 * Updates mappings of view element's children.
	 *
	 * Children that were replaced in the view structure by similar elements (same tag name) are treated as 'replaced'.
	 * This means that their mappings can be updated so the new view elements are mapped to the existing DOM elements.
	 * Thanks to that these elements do not need to be re-rendered completely.
	 *
	 * @param viewElement The view element whose children mappings will be updated.
	 */ _updateChildrenMappings(viewElement) {
        const domElement = this.domConverter.mapViewToDom(viewElement);
        if (!domElement) {
            // If there is no `domElement` it means that it was already removed from DOM and there is no need to process it.
            return;
        }
        // Removing nodes from the DOM as we iterate can cause `actualDomChildren`
        // (which is a live-updating `NodeList`) to get out of sync with the
        // indices that we compute as we iterate over `actions`.
        // This would produce incorrect element mappings.
        //
        // Converting live list to an array to make the list static.
        const actualDomChildren = Array.from(domElement.childNodes);
        const expectedDomChildren = Array.from(this.domConverter.viewChildrenToDom(viewElement, {
            withChildren: false
        }));
        const diff = this._diffNodeLists(actualDomChildren, expectedDomChildren);
        const actions = this._findUpdateActions(diff, actualDomChildren, expectedDomChildren, areSimilarElements);
        if (actions.indexOf('update') !== -1) {
            const counter = {
                equal: 0,
                insert: 0,
                delete: 0
            };
            for (const action of actions){
                if (action === 'update') {
                    const insertIndex = counter.equal + counter.insert;
                    const deleteIndex = counter.equal + counter.delete;
                    const viewChild = viewElement.getChild(insertIndex);
                    // UIElement and RawElement are special cases. Their children are not stored in a view (#799)
                    // so we cannot use them with replacing flow (since they use view children during rendering
                    // which will always result in rendering empty elements).
                    if (viewChild && !viewChild.is('uiElement') && !viewChild.is('rawElement')) {
                        this._updateElementMappings(viewChild, actualDomChildren[deleteIndex]);
                    }
                    remove$1(expectedDomChildren[insertIndex]);
                    counter.equal++;
                } else {
                    counter[action]++;
                }
            }
        }
    }
    /**
	 * Updates mappings of a given view element.
	 *
	 * @param viewElement The view element whose mappings will be updated.
	 * @param domElement The DOM element representing the given view element.
	 */ _updateElementMappings(viewElement, domElement) {
        // Remap 'DomConverter' bindings.
        this.domConverter.unbindDomElement(domElement);
        this.domConverter.bindElements(domElement, viewElement);
        // View element may have children which needs to be updated, but are not marked, mark them to update.
        this.markedChildren.add(viewElement);
        // Because we replace new view element mapping with the existing one, the corresponding DOM element
        // will not be rerendered. The new view element may have different attributes than the previous one.
        // Since its corresponding DOM element will not be rerendered, new attributes will not be added
        // to the DOM, so we need to mark it here to make sure its attributes gets updated. See #1427 for more
        // detailed case study.
        // Also there are cases where replaced element is removed from the view structure and then has
        // its attributes changed or removed. In such cases the element will not be present in `markedAttributes`
        // and also may be the same (`element.isSimilar()`) as the reused element not having its attributes updated.
        // To prevent such situations we always mark reused element to have its attributes rerenderd (#1560).
        this.markedAttributes.add(viewElement);
    }
    /**
	 * Gets the position of the inline filler based on the current selection.
	 * Here, we assume that we know that the filler is needed and
	 * {@link #_isSelectionInInlineFiller is at the selection position}, and, since it is needed,
	 * it is somewhere at the selection position.
	 *
	 * Note: The filler position cannot be restored based on the filler's DOM text node, because
	 * when this method is called (before rendering), the bindings will often be broken. View-to-DOM
	 * bindings are only dependable after rendering.
	 */ _getInlineFillerPosition() {
        const firstPos = this.selection.getFirstPosition();
        if (firstPos.parent.is('$text')) {
            return ViewPosition._createBefore(firstPos.parent);
        } else {
            return firstPos;
        }
    }
    /**
	 * Returns `true` if the selection has not left the inline filler's text node.
	 * If it is `true`, it means that the filler had been added for a reason and the selection did not
	 * leave the filler's text node. For example, the user can be in the middle of a composition so it should not be touched.
	 *
	 * @returns `true` if the inline filler and selection are in the same place.
	 */ _isSelectionInInlineFiller() {
        if (this.selection.rangeCount != 1 || !this.selection.isCollapsed) {
            return false;
        }
        // Note, we can't check if selection's position equals position of the
        // this._inlineFiller node, because of #663. We may not be able to calculate
        // the filler's position in the view at this stage.
        // Instead, we check it the other way – whether selection is anchored in
        // that text node or next to it.
        // Possible options are:
        // "FILLER{}"
        // "FILLERadded-text{}"
        const selectionPosition = this.selection.getFirstPosition();
        const position = this.domConverter.viewPositionToDom(selectionPosition);
        if (position && isText(position.parent) && startsWithFiller(position.parent)) {
            return true;
        }
        return false;
    }
    /**
	 * Removes the inline filler.
	 */ _removeInlineFiller() {
        const domFillerNode = this._inlineFiller;
        // Something weird happened and the stored node doesn't contain the filler's text.
        if (!startsWithFiller(domFillerNode)) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.log( ..._buildLogMessage( this, 'Renderer',
            // @if CK_DEBUG_TYPING // 		'Inline filler node: ' +
            // @if CK_DEBUG_TYPING // 		`%c${ _escapeTextNodeData( domFillerNode.data ) }%c (${ domFillerNode.data.length })`,
            // @if CK_DEBUG_TYPING // 		'color: blue',
            // @if CK_DEBUG_TYPING // 		''
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            /**
			 * The inline filler node was lost. Most likely, something overwrote the filler text node
			 * in the DOM.
			 *
			 * @error view-renderer-filler-was-lost
			 */ throw new CKEditorError('view-renderer-filler-was-lost', this);
        }
        if (isInlineFiller(domFillerNode)) {
            domFillerNode.remove();
        } else {
            domFillerNode.data = domFillerNode.data.substr(INLINE_FILLER_LENGTH);
        }
        this._inlineFiller = null;
    }
    /**
	 * Checks if the inline {@link module:engine/view/filler filler} should be added.
	 *
	 * @returns `true` if the inline filler should be added.
	 */ _needsInlineFillerAtSelection() {
        if (this.selection.rangeCount != 1 || !this.selection.isCollapsed) {
            return false;
        }
        const selectionPosition = this.selection.getFirstPosition();
        const selectionParent = selectionPosition.parent;
        const selectionOffset = selectionPosition.offset;
        // If there is no DOM root we do not care about fillers.
        if (!this.domConverter.mapViewToDom(selectionParent.root)) {
            return false;
        }
        if (!selectionParent.is('element')) {
            return false;
        }
        // Prevent adding inline filler inside elements with contenteditable=false.
        // https://github.com/ckeditor/ckeditor5-engine/issues/1170
        if (!isEditable(selectionParent)) {
            return false;
        }
        const nodeBefore = selectionPosition.nodeBefore;
        const nodeAfter = selectionPosition.nodeAfter;
        if (nodeBefore instanceof ViewText || nodeAfter instanceof ViewText) {
            return false;
        }
        // We have block filler, we do not need inline one.
        if (selectionOffset === selectionParent.getFillerOffset() && (!nodeBefore || !nodeBefore.is('element', 'br'))) {
            return false;
        }
        // Do not use inline filler while typing outside inline elements on Android.
        // The deleteContentBackward would remove part of the inline filler instead of removing last letter in a link.
        if (env.isAndroid && (nodeBefore || nodeAfter)) {
            return false;
        }
        return true;
    }
    /**
	 * Checks if text needs to be updated and possibly updates it.
	 *
	 * @param viewText View text to update.
	 * @param options.inlineFillerPosition The position where the inline filler should be rendered.
	 */ _updateText(viewText, options) {
        const domText = this.domConverter.findCorrespondingDomText(viewText);
        const newDomText = this.domConverter.viewToDom(viewText);
        let expectedText = newDomText.data;
        const filler = options.inlineFillerPosition;
        if (filler && filler.parent == viewText.parent && filler.offset == viewText.index) {
            expectedText = INLINE_FILLER + expectedText;
        }
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'Renderer',
        // @if CK_DEBUG_TYPING // 		'%cUpdate text',
        // @if CK_DEBUG_TYPING // 		'font-weight: normal'
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        this._updateTextNode(domText, expectedText);
    // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
    // @if CK_DEBUG_TYPING // 	console.groupEnd();
    // @if CK_DEBUG_TYPING // }
    }
    /**
	 * Checks if attribute list needs to be updated and possibly updates it.
	 *
	 * @param viewElement The view element to update.
	 */ _updateAttrs(viewElement) {
        const domElement = this.domConverter.mapViewToDom(viewElement);
        if (!domElement) {
            // If there is no `domElement` it means that 'viewElement' is outdated as its mapping was updated
            // in 'this._updateChildrenMappings()'. There is no need to process it as new view element which
            // replaced old 'viewElement' mapping was also added to 'this.markedAttributes'
            // in 'this._updateChildrenMappings()' so it will be processed separately.
            return;
        }
        // Remove attributes from DOM elements if they do not exist in the view.
        //
        // Note: It is important to first remove DOM attributes and then set new ones, because some view attributes may be renamed
        // as they are set on DOM (due to unsafe attributes handling). If we set the view attribute first, and then remove
        // non-existing DOM attributes, then we would remove the attribute that we just set.
        //
        // Note: The domElement.attributes is a live collection, so we need to convert it to an array to avoid issues.
        for (const domAttr of Array.from(domElement.attributes)){
            const key = domAttr.name;
            // All other attributes not present in the DOM should be removed.
            if (!viewElement.hasAttribute(key)) {
                this.domConverter.removeDomElementAttribute(domElement, key);
            }
        }
        // Add or overwrite attributes.
        for (const key of viewElement.getAttributeKeys()){
            this.domConverter.setDomElementAttribute(domElement, key, viewElement.getAttribute(key), viewElement);
        }
    }
    /**
	 * Checks if elements child list needs to be updated and possibly updates it.
	 *
	 * Note that on Android, to reduce the risk of composition breaks, it tries to update data of an existing
	 * child text nodes instead of replacing them completely.
	 *
	 * @param viewElement View element to update.
	 * @param options.inlineFillerPosition The position where the inline filler should be rendered.
	 */ _updateChildren(viewElement, options) {
        const domElement = this.domConverter.mapViewToDom(viewElement);
        if (!domElement) {
            // If there is no `domElement` it means that it was already removed from DOM.
            // There is no need to process it. It will be processed when re-inserted.
            return;
        }
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'Renderer',
        // @if CK_DEBUG_TYPING // 		'%cUpdate children',
        // @if CK_DEBUG_TYPING // 		'font-weight: normal'
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        // IME on Android inserts a new text node while typing after a link
        // instead of updating an existing text node that follows the link.
        // We must normalize those text nodes so the diff won't get confused.
        // https://github.com/ckeditor/ckeditor5/issues/12574.
        if (env.isAndroid) {
            let previousDomNode = null;
            for (const domNode of Array.from(domElement.childNodes)){
                if (previousDomNode && isText(previousDomNode) && isText(domNode)) {
                    domElement.normalize();
                    break;
                }
                previousDomNode = domNode;
            }
        }
        const inlineFillerPosition = options.inlineFillerPosition;
        const actualDomChildren = domElement.childNodes;
        const expectedDomChildren = Array.from(this.domConverter.viewChildrenToDom(viewElement, {
            bind: true
        }));
        // Inline filler element has to be created as it is present in the DOM, but not in the view. It is required
        // during diffing so text nodes could be compared correctly and also during rendering to maintain
        // proper order and indexes while updating the DOM.
        if (inlineFillerPosition && inlineFillerPosition.parent === viewElement) {
            addInlineFiller(domElement.ownerDocument, expectedDomChildren, inlineFillerPosition.offset);
        }
        const diff = this._diffNodeLists(actualDomChildren, expectedDomChildren);
        // We need to make sure that we update the existing text node and not replace it with another one.
        // The composition and different "language" browser extensions are fragile to text node being completely replaced.
        const actions = this._findUpdateActions(diff, actualDomChildren, expectedDomChildren, areTextNodes);
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping && actions.every( a => a == 'equal' ) ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
        // @if CK_DEBUG_TYPING // 		'%cNothing to update.',
        // @if CK_DEBUG_TYPING // 		'font-style: italic'
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        let i = 0;
        const nodesToUnbind = new Set();
        // Handle deletions first.
        // This is to prevent a situation where an element that already exists in `actualDomChildren` is inserted at a different
        // index in `actualDomChildren`. Since `actualDomChildren` is a `NodeList`, this works like move, not like an insert,
        // and it disrupts the whole algorithm. See https://github.com/ckeditor/ckeditor5/issues/6367.
        //
        // It doesn't matter in what order we remove or add nodes, as long as we remove and add correct nodes at correct indexes.
        for (const action of actions){
            if (action === 'delete') {
                // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
                // @if CK_DEBUG_TYPING //	const node = actualDomChildren[ i ];
                // @if CK_DEBUG_TYPING // 	if ( isText( node ) ) {
                // @if CK_DEBUG_TYPING // 		console.info( ..._buildLogMessage( this, 'Renderer',
                // @if CK_DEBUG_TYPING // 			'%cRemove text node' +
                // @if CK_DEBUG_TYPING // 			`${ this.isComposing ? ' while composing (may break composition)' : '' }: ` +
                // @if CK_DEBUG_TYPING // 			`%c${ _escapeTextNodeData( node.data ) }%c (${ node.data.length })`,
                // @if CK_DEBUG_TYPING // 			this.isComposing ? 'color: red; font-weight: bold' : '',
                // @if CK_DEBUG_TYPING // 			'color: blue', ''
                // @if CK_DEBUG_TYPING // 		) );
                // @if CK_DEBUG_TYPING // 	} else {
                // @if CK_DEBUG_TYPING // 		console.info( ..._buildLogMessage( this, 'Renderer',
                // @if CK_DEBUG_TYPING // 			'%cRemove element' +
                // @if CK_DEBUG_TYPING // 			`${ this.isComposing ? ' while composing (may break composition)' : '' }: `,
                // @if CK_DEBUG_TYPING // 			this.isComposing ? 'color: red; font-weight: bold' : '',
                // @if CK_DEBUG_TYPING // 			node
                // @if CK_DEBUG_TYPING // 		) );
                // @if CK_DEBUG_TYPING // 	}
                // @if CK_DEBUG_TYPING // }
                nodesToUnbind.add(actualDomChildren[i]);
                remove$1(actualDomChildren[i]);
            } else if (action === 'equal' || action === 'update') {
                i++;
            }
        }
        i = 0;
        for (const action of actions){
            if (action === 'insert') {
                // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
                // @if CK_DEBUG_TYPING //	const node = expectedDomChildren[ i ];
                // @if CK_DEBUG_TYPING //	if ( isText( node ) ) {
                // @if CK_DEBUG_TYPING //		console.info( ..._buildLogMessage( this, 'Renderer',
                // @if CK_DEBUG_TYPING //			'%cInsert text node' +
                // @if CK_DEBUG_TYPING //			`${ this.isComposing ? ' while composing (may break composition)' : '' }: ` +
                // @if CK_DEBUG_TYPING //			`%c${ _escapeTextNodeData( node.data ) }%c (${ node.data.length })`,
                // @if CK_DEBUG_TYPING //			this.isComposing ? 'color: red; font-weight: bold' : '',
                // @if CK_DEBUG_TYPING //			'color: blue',
                // @if CK_DEBUG_TYPING //			''
                // @if CK_DEBUG_TYPING //		) );
                // @if CK_DEBUG_TYPING //	} else {
                // @if CK_DEBUG_TYPING //		console.info( ..._buildLogMessage( this, 'Renderer',
                // @if CK_DEBUG_TYPING //			'%cInsert element:',
                // @if CK_DEBUG_TYPING //			'font-weight: normal',
                // @if CK_DEBUG_TYPING //			node
                // @if CK_DEBUG_TYPING //		) );
                // @if CK_DEBUG_TYPING //	}
                // @if CK_DEBUG_TYPING // }
                insertAt(domElement, i, expectedDomChildren[i]);
                i++;
            } else if (action === 'update') {
                this._updateTextNode(actualDomChildren[i], expectedDomChildren[i].data);
                i++;
            } else if (action === 'equal') {
                // Force updating text nodes inside elements which did not change and do not need to be re-rendered (#1125).
                // Do it here (not in the loop above) because only after insertions the `i` index is correct.
                this._markDescendantTextToSync(this.domConverter.domToView(expectedDomChildren[i]));
                i++;
            }
        }
        // Unbind removed nodes. When node does not have a parent it means that it was removed from DOM tree during
        // comparison with the expected DOM. We don't need to check child nodes, because if child node was reinserted,
        // it was moved to DOM tree out of the removed node.
        for (const node of nodesToUnbind){
            if (!node.parentNode) {
                this.domConverter.unbindDomElement(node);
            }
        }
    // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
    // @if CK_DEBUG_TYPING // 	console.groupEnd();
    // @if CK_DEBUG_TYPING // }
    }
    /**
	 * Shorthand for diffing two arrays or node lists of DOM nodes.
	 *
	 * @param actualDomChildren Actual DOM children
	 * @param expectedDomChildren Expected DOM children.
	 * @returns The list of actions based on the {@link module:utils/diff~diff} function.
	 */ _diffNodeLists(actualDomChildren, expectedDomChildren) {
        actualDomChildren = filterOutFakeSelectionContainer(actualDomChildren, this._fakeSelectionContainer);
        return diff(actualDomChildren, expectedDomChildren, sameNodes$1.bind(null, this.domConverter));
    }
    /**
	 * Finds DOM nodes that were replaced with the similar nodes (same tag name) in the view. All nodes are compared
	 * within one `insert`/`delete` action group, for example:
	 *
	 * ```
	 * Actual DOM:		<p><b>Foo</b>Bar<i>Baz</i><b>Bax</b></p>
	 * Expected DOM:	<p>Bar<b>123</b><i>Baz</i><b>456</b></p>
	 * Input actions:	[ insert, insert, delete, delete, equal, insert, delete ]
	 * Output actions:	[ insert, replace, delete, equal, replace ]
	 * ```
	 *
	 * @param actions Actions array which is a result of the {@link module:utils/diff~diff} function.
	 * @param actualDom Actual DOM children
	 * @param expectedDom Expected DOM children.
	 * @param comparator A comparator function that should return `true` if the given node should be reused
	 * (either by the update of a text node data or an element children list for similar elements).
	 * @returns Actions array modified with the `update` actions.
	 */ _findUpdateActions(actions, actualDom, expectedDom, comparator) {
        // If there is no both 'insert' and 'delete' actions, no need to check for replaced elements.
        if (actions.indexOf('insert') === -1 || actions.indexOf('delete') === -1) {
            return actions;
        }
        let newActions = [];
        let actualSlice = [];
        let expectedSlice = [];
        const counter = {
            equal: 0,
            insert: 0,
            delete: 0
        };
        for (const action of actions){
            if (action === 'insert') {
                expectedSlice.push(expectedDom[counter.equal + counter.insert]);
            } else if (action === 'delete') {
                actualSlice.push(actualDom[counter.equal + counter.delete]);
            } else {
                newActions = newActions.concat(diff(actualSlice, expectedSlice, comparator).map((action)=>action === 'equal' ? 'update' : action));
                newActions.push('equal');
                // Reset stored elements on 'equal'.
                actualSlice = [];
                expectedSlice = [];
            }
            counter[action]++;
        }
        return newActions.concat(diff(actualSlice, expectedSlice, comparator).map((action)=>action === 'equal' ? 'update' : action));
    }
    /**
	 * Checks if text needs to be updated and possibly updates it by removing and inserting only parts
	 * of the data from the existing text node to reduce impact on the IME composition.
	 *
	 * @param domText DOM text node to update.
	 * @param expectedText The expected data of a text node.
	 */ _updateTextNode(domText, expectedText) {
        const actualText = domText.data;
        if (actualText == expectedText) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
            // @if CK_DEBUG_TYPING // 		'%cText node does not need update:%c ' +
            // @if CK_DEBUG_TYPING // 		`${ _escapeTextNodeData( actualText ) }%c (${ actualText.length })`,
            // @if CK_DEBUG_TYPING // 		'font-style: italic',
            // @if CK_DEBUG_TYPING // 		'color: blue',
            // @if CK_DEBUG_TYPING // 		''
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            return;
        }
        // Our approach to interleaving space character with NBSP might differ with the one implemented by the browser.
        // Avoid modifying the text node in the DOM if only NBSPs and spaces are interchanged.
        // We should avoid DOM modifications while composing to avoid breakage of composition.
        // See: https://github.com/ckeditor/ckeditor5/issues/13994.
        if (env.isAndroid && this.isComposing && actualText.replace(/\u00A0/g, ' ') == expectedText.replace(/\u00A0/g, ' ')) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
            // @if CK_DEBUG_TYPING // 		'%cText node ignore NBSP changes while composing: ' +
            // @if CK_DEBUG_TYPING // 		`%c${ _escapeTextNodeData( actualText ) }%c (${ actualText.length }) -> ` +
            // @if CK_DEBUG_TYPING // 		`%c${ _escapeTextNodeData( expectedText ) }%c (${ expectedText.length })`,
            // @if CK_DEBUG_TYPING // 		'font-style: italic',
            // @if CK_DEBUG_TYPING // 		'color: blue',
            // @if CK_DEBUG_TYPING // 		'',
            // @if CK_DEBUG_TYPING // 		'color: blue',
            // @if CK_DEBUG_TYPING // 		''
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            return;
        }
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
        // @if CK_DEBUG_TYPING // 		'%cUpdate text node' +
        // @if CK_DEBUG_TYPING // 		`${ this.isComposing ? ' while composing (may break composition)' : '' }: ` +
        // @if CK_DEBUG_TYPING // 		`%c${ _escapeTextNodeData( actualText ) }%c (${ actualText.length }) -> ` +
        // @if CK_DEBUG_TYPING // 		`%c${ _escapeTextNodeData( expectedText ) }%c (${ expectedText.length })`,
        // @if CK_DEBUG_TYPING // 		this.isComposing ? 'color: red; font-weight: bold' : '',
        // @if CK_DEBUG_TYPING // 		'color: blue',
        // @if CK_DEBUG_TYPING // 		'',
        // @if CK_DEBUG_TYPING // 		'color: blue',
        // @if CK_DEBUG_TYPING // 		''
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        this._updateTextNodeInternal(domText, expectedText);
    }
    /**
	 * Part of the `_updateTextNode` method extracted for easier testing.
	 */ _updateTextNodeInternal(domText, expectedText) {
        const actions = fastDiff(domText.data, expectedText);
        for (const action of actions){
            if (action.type === 'insert') {
                domText.insertData(action.index, action.values.join(''));
            } else {
                domText.deleteData(action.index, action.howMany);
            }
        }
    }
    /**
	 * Marks text nodes to be synchronized.
	 *
	 * If a text node is passed, it will be marked. If an element is passed, all descendant text nodes inside it will be marked.
	 *
	 * @param viewNode View node to sync.
	 */ _markDescendantTextToSync(viewNode) {
        if (!viewNode) {
            return;
        }
        if (viewNode.is('$text')) {
            this.markedTexts.add(viewNode);
        } else if (viewNode.is('element')) {
            for (const child of viewNode.getChildren()){
                this._markDescendantTextToSync(child);
            }
        }
    }
    /**
	 * Checks if the selection needs to be updated and possibly updates it.
	 */ _updateSelection() {
        // Block updating DOM selection in (non-Android) Blink while the user is selecting to prevent accidental selection collapsing.
        // Note: Structural changes in DOM must trigger selection rendering, though. Nodes the selection was anchored
        // to, may disappear in DOM which would break the selection (e.g. in real-time collaboration scenarios).
        // https://github.com/ckeditor/ckeditor5/issues/10562, https://github.com/ckeditor/ckeditor5/issues/10723
        if (env.isBlink && !env.isAndroid && this.isSelecting && !this.markedChildren.size) {
            return;
        }
        // If there is no selection - remove DOM and fake selections.
        if (this.selection.rangeCount === 0) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
            // @if CK_DEBUG_TYPING // 		'Update DOM selection: remove all ranges'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            this._removeDomSelection();
            this._removeFakeSelection();
            return;
        }
        const domEditable = this.domConverter.mapViewToDom(this.selection.editableElement);
        // Do not update DOM selection if there is no focus, or there is no DOM element corresponding to selection's editable element.
        if (!this.isFocused || !domEditable) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
            // @if CK_DEBUG_TYPING // 		'Skip updating DOM selection:',
            // @if CK_DEBUG_TYPING // 		`isFocused: ${ this.isFocused }, hasDomEditable: ${ !!domEditable }`
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            // But if there was a fake selection, and it is not fake anymore - remove it as it can map to no longer existing widget.
            // See https://github.com/ckeditor/ckeditor5/issues/18123.
            if (!this.selection.isFake && this._fakeSelectionContainer && this._fakeSelectionContainer.isConnected) {
                // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
                // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
                // @if CK_DEBUG_TYPING // 		'Remove fake selection (not focused editable)'
                // @if CK_DEBUG_TYPING // 	) );
                // @if CK_DEBUG_TYPING // }
                this._removeFakeSelection();
            }
            return;
        }
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
        // @if CK_DEBUG_TYPING // 		'Update DOM selection'
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        // Render fake selection - create the fake selection container (if needed) and move DOM selection to it.
        if (this.selection.isFake) {
            this._updateFakeSelection(domEditable);
        } else if (this._fakeSelectionContainer && this._fakeSelectionContainer.isConnected) {
            this._removeFakeSelection();
            this._updateDomSelection(domEditable);
        } else if (!(this.isComposing && env.isAndroid)) {
            this._updateDomSelection(domEditable);
        }
    }
    /**
	 * Updates the fake selection.
	 *
	 * @param domEditable A valid DOM editable where the fake selection container should be added.
	 */ _updateFakeSelection(domEditable) {
        const domDocument = domEditable.ownerDocument;
        if (!this._fakeSelectionContainer) {
            this._fakeSelectionContainer = createFakeSelectionContainer(domDocument);
        }
        const container = this._fakeSelectionContainer;
        // Bind fake selection container with the current selection *position*.
        this.domConverter.bindFakeSelection(container, this.selection);
        if (!this._fakeSelectionNeedsUpdate(domEditable)) {
            return;
        }
        if (!container.parentElement || container.parentElement != domEditable) {
            domEditable.appendChild(container);
        }
        container.textContent = this.selection.fakeSelectionLabel || '\u00A0';
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
        // @if CK_DEBUG_TYPING // 		'Set DOM fake selection'
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        const domSelection = domDocument.getSelection();
        const domRange = domDocument.createRange();
        domSelection.removeAllRanges();
        domRange.selectNodeContents(container);
        domSelection.addRange(domRange);
    }
    /**
	 * Updates the DOM selection.
	 *
	 * @param domEditable A valid DOM editable where the DOM selection should be rendered.
	 */ _updateDomSelection(domEditable) {
        const domSelection = domEditable.ownerDocument.defaultView.getSelection();
        // Let's check whether DOM selection needs updating at all.
        if (!this._domSelectionNeedsUpdate(domSelection)) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
            // @if CK_DEBUG_TYPING // 		'%cDOM selection is already correct',
            // @if CK_DEBUG_TYPING // 		'font-style: italic;'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            return;
        }
        // Multi-range selection is not available in most browsers, and, at least in Chrome, trying to
        // set such selection, that is not continuous, throws an error. Because of that, we will just use anchor
        // and focus of view selection.
        // Since we are not supporting multi-range selection, we also do not need to check if proper editable is
        // selected. If there is any editable selected, it is okay (editable is taken from selection anchor).
        const anchor = this.domConverter.viewPositionToDom(this.selection.anchor);
        const focus = this.domConverter.viewPositionToDom(this.selection.focus);
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
        // @if CK_DEBUG_TYPING // 		'Update DOM selection:',
        // @if CK_DEBUG_TYPING // 		anchor,
        // @if CK_DEBUG_TYPING // 		focus
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        domSelection.setBaseAndExtent(anchor.parent, anchor.offset, focus.parent, focus.offset);
        // Firefox–specific hack (https://github.com/ckeditor/ckeditor5-engine/issues/1439).
        if (env.isGecko) {
            fixGeckoSelectionAfterBr(focus, domSelection);
        }
    }
    /**
	 * Checks whether a given DOM selection needs to be updated.
	 *
	 * @param domSelection The DOM selection to check.
	 */ _domSelectionNeedsUpdate(domSelection) {
        if (!this.domConverter.isDomSelectionCorrect(domSelection)) {
            // Current DOM selection is in incorrect position. We need to update it.
            return true;
        }
        const oldViewSelection = domSelection && this.domConverter.domSelectionToView(domSelection);
        if (oldViewSelection && this.selection.isEqual(oldViewSelection)) {
            return false;
        }
        // If selection is not collapsed, it does not need to be updated if it is similar.
        if (!this.selection.isCollapsed && this.selection.isSimilar(oldViewSelection)) {
            // Selection did not changed and is correct, do not update.
            return false;
        }
        // Selections are not similar.
        return true;
    }
    /**
	 * Checks whether the fake selection needs to be updated.
	 *
	 * @param domEditable A valid DOM editable where a new fake selection container should be added.
	 */ _fakeSelectionNeedsUpdate(domEditable) {
        const container = this._fakeSelectionContainer;
        const domSelection = domEditable.ownerDocument.getSelection();
        // Fake selection needs to be updated if there's no fake selection container, or the container currently sits
        // in a different root.
        if (!container || container.parentElement !== domEditable) {
            return true;
        }
        // Make sure that the selection actually is within the fake selection.
        if (domSelection.anchorNode !== container && !container.contains(domSelection.anchorNode)) {
            return true;
        }
        return container.textContent !== this.selection.fakeSelectionLabel;
    }
    /**
	 * Removes the DOM selection.
	 */ _removeDomSelection() {
        for (const doc of this.domDocuments){
            const domSelection = doc.getSelection();
            if (domSelection.rangeCount) {
                const activeDomElement = doc.activeElement;
                const viewElement = this.domConverter.mapDomToView(activeDomElement);
                if (activeDomElement && viewElement) {
                    domSelection.removeAllRanges();
                }
            }
        }
    }
    /**
	 * Removes the fake selection.
	 */ _removeFakeSelection() {
        const container = this._fakeSelectionContainer;
        if (container) {
            container.remove();
        }
    }
    /**
	 * Checks if focus needs to be updated and possibly updates it.
	 */ _updateFocus() {
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'Renderer',
        // @if CK_DEBUG_TYPING // 		`update focus: ${ this.isFocused ? 'focused' : 'not focused' }`
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        if (this.isFocused) {
            const editable = this.selection.editableElement;
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'Renderer',
            // @if CK_DEBUG_TYPING // 		'focus editable:',
            // @if CK_DEBUG_TYPING // 		{ editable }
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            if (editable) {
                this.domConverter.focus(editable);
            }
        }
    // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
    // @if CK_DEBUG_TYPING // 	console.groupEnd();
    // @if CK_DEBUG_TYPING // }
    }
}
/**
 * Checks if provided element is editable.
 */ function isEditable(element) {
    if (element.getAttribute('contenteditable') == 'false') {
        return false;
    }
    const parent = element.findAncestor((element)=>element.hasAttribute('contenteditable'));
    return !parent || parent.getAttribute('contenteditable') == 'true';
}
/**
 * Adds inline filler at a given position.
 *
 * The position can be given as an array of DOM nodes and an offset in that array,
 * or a DOM parent element and an offset in that element.
 *
 * @returns The DOM text node that contains an inline filler.
 */ function addInlineFiller(domDocument, domParentOrArray, offset) {
    const childNodes = domParentOrArray instanceof Array ? domParentOrArray : domParentOrArray.childNodes;
    const nodeAfterFiller = childNodes[offset];
    if (isText(nodeAfterFiller)) {
        nodeAfterFiller.data = INLINE_FILLER + nodeAfterFiller.data;
        return nodeAfterFiller;
    } else {
        const fillerNode = domDocument.createTextNode(INLINE_FILLER);
        if (Array.isArray(domParentOrArray)) {
            childNodes.splice(offset, 0, fillerNode);
        } else {
            insertAt(domParentOrArray, offset, fillerNode);
        }
        return fillerNode;
    }
}
/**
 * Whether two DOM nodes should be considered as similar.
 * Nodes are considered similar if they have the same tag name.
 */ function areSimilarElements(node1, node2) {
    return isNode(node1) && isNode(node2) && !isText(node1) && !isText(node2) && !isComment(node1) && !isComment(node2) && node1.tagName.toLowerCase() === node2.tagName.toLowerCase();
}
/**
 * Whether two DOM nodes are text nodes.
 */ function areTextNodes(node1, node2) {
    return isNode(node1) && isNode(node2) && isText(node1) && isText(node2);
}
/**
 * Whether two dom nodes should be considered as the same.
 * Two nodes which are considered the same are:
 *
 * * Text nodes with the same text.
 * * Element nodes represented by the same object.
 * * Two block filler elements.
 *
 * @param blockFillerMode Block filler mode, see {@link module:engine/view/domconverter~ViewDomConverter#blockFillerMode}.
 */ function sameNodes$1(domConverter, actualDomChild, expectedDomChild) {
    // Elements.
    if (actualDomChild === expectedDomChild) {
        return true;
    } else if (isText(actualDomChild) && isText(expectedDomChild)) {
        return actualDomChild.data === expectedDomChild.data;
    } else if (domConverter.isBlockFiller(actualDomChild) && domConverter.isBlockFiller(expectedDomChild)) {
        return true;
    }
    // Not matching types.
    return false;
}
/**
 * The following is a Firefox–specific hack (https://github.com/ckeditor/ckeditor5-engine/issues/1439).
 * When the native DOM selection is at the end of the block and preceded by <br /> e.g.
 *
 * ```html
 * <p>foo<br/>[]</p>
 * ```
 *
 * which happens a lot when using the soft line break, the browser fails to (visually) move the
 * caret to the new line. A quick fix is as simple as force–refreshing the selection with the same range.
 */ function fixGeckoSelectionAfterBr(focus, domSelection) {
    let parent = focus.parent;
    let offset = focus.offset;
    if (isText(parent) && isInlineFiller(parent)) {
        offset = indexOf(parent) + 1;
        parent = parent.parentNode;
    }
    // This fix works only when the focus point is at the very end of an element.
    // There is no point in running it in cases unrelated to the browser bug.
    if (parent.nodeType != Node.ELEMENT_NODE || offset != parent.childNodes.length - 1) {
        return;
    }
    const childAtOffset = parent.childNodes[offset];
    // To stay on the safe side, the fix being as specific as possible, it targets only the
    // selection which is at the very end of the element and preceded by <br />.
    if (childAtOffset && childAtOffset.tagName == 'BR') {
        domSelection.addRange(domSelection.getRangeAt(0));
    }
}
function filterOutFakeSelectionContainer(domChildList, fakeSelectionContainer) {
    const childList = Array.from(domChildList);
    if (childList.length == 0 || !fakeSelectionContainer) {
        return childList;
    }
    const last = childList[childList.length - 1];
    if (last == fakeSelectionContainer) {
        childList.pop();
    }
    return childList;
}
/**
 * Creates a fake selection container for a given document.
 */ function createFakeSelectionContainer(domDocument) {
    const container = domDocument.createElement('div');
    container.className = 'ck-fake-selection-container';
    Object.assign(container.style, {
        position: 'fixed',
        top: 0,
        left: '-9999px',
        // See https://github.com/ckeditor/ckeditor5/issues/752.
        width: '42px'
    });
    // Fill it with a text node so we can update it later.
    container.textContent = '\u00A0';
    return container;
} // @if CK_DEBUG_TYPING // function _escapeTextNodeData( text ) {
 // @if CK_DEBUG_TYPING // 	const escapedText = text
 // @if CK_DEBUG_TYPING // 		.replace( /&/g, '&amp;' )
 // @if CK_DEBUG_TYPING // 		.replace( /\u00A0/g, '&nbsp;' )
 // @if CK_DEBUG_TYPING // 		.replace( /\u2060/g, '&NoBreak;' );
 // @if CK_DEBUG_TYPING //
 // @if CK_DEBUG_TYPING // 	return `"${ escapedText }"`;
 // @if CK_DEBUG_TYPING // }

const BR_FILLER_REF = BR_FILLER(global.document); // eslint-disable-line new-cap
const NBSP_FILLER_REF = NBSP_FILLER(global.document); // eslint-disable-line new-cap
const MARKED_NBSP_FILLER_REF = MARKED_NBSP_FILLER(global.document); // eslint-disable-line new-cap
const UNSAFE_ATTRIBUTE_NAME_PREFIX = 'data-ck-unsafe-attribute-';
const UNSAFE_ELEMENT_REPLACEMENT_ATTRIBUTE = 'data-ck-unsafe-element';
/**
 * `ViewDomConverter` is a set of tools to do transformations between DOM nodes and view nodes. It also handles
 * {@link module:engine/view/domconverter~ViewDomConverter#bindElements bindings} between these nodes.
 *
 * An instance of the DOM converter is available under
 * {@link module:engine/view/view~EditingView#domConverter `editor.editing.view.domConverter`}.
 *
 * The DOM converter does not check which nodes should be rendered (use {@link module:engine/view/renderer~ViewRenderer}), does not keep the
 * state of a tree nor keeps the synchronization between the tree view and
 * the DOM tree (use {@link module:engine/view/document~ViewDocument}).
 *
 * The DOM converter keeps DOM elements to view element bindings, so when the converter gets destroyed, the bindings are lost.
 * Two converters will keep separate binding maps, so one tree view can be bound with two DOM trees.
 */ class ViewDomConverter {
    document;
    /**
	 * Whether to leave the View-to-DOM conversion result unchanged or improve editing experience by filtering out interactive data.
	 */ renderingMode;
    /**
	 * The mode of a block filler used by the DOM converter.
	 */ blockFillerMode;
    /**
	 * Elements which are considered pre-formatted elements.
	 */ preElements;
    /**
	 * Elements which are considered block elements (and hence should be filled with a
	 * {@link #isBlockFiller block filler}).
	 *
	 * Whether an element is considered a block element also affects handling of trailing whitespaces.
	 *
	 * You can extend this array if you introduce support for block elements which are not yet recognized here.
	 */ blockElements;
    /**
	 * A list of elements that exist inline (in text) but their inner structure cannot be edited because
	 * of the way they are rendered by the browser. They are mostly HTML form elements but there are other
	 * elements such as `<img>` or `<iframe>` that also have non-editable children or no children whatsoever.
	 *
	 * Whether an element is considered an inline object has an impact on white space rendering (trimming)
	 * around (and inside of it). In short, white spaces in text nodes next to inline objects are not trimmed.
	 *
	 * You can extend this array if you introduce support for inline object elements which are not yet recognized here.
	 */ inlineObjectElements;
    /**
	 * A list of elements which may affect the editing experience. To avoid this, those elements are replaced with
	 * `<span data-ck-unsafe-element="[element name]"></span>` while rendering in the editing mode.
	 */ unsafeElements;
    /**
	 * The DOM Document used by `ViewDomConverter` to create DOM nodes.
	 */ _domDocument;
    /**
	 * The DOM-to-view mapping.
	 */ _domToViewMapping = new WeakMap();
    /**
	 * The view-to-DOM mapping.
	 */ _viewToDomMapping = new WeakMap();
    /**
	 * Holds the mapping between fake selection containers and corresponding view selections.
	 */ _fakeSelectionMapping = new WeakMap();
    /**
	 * Matcher for view elements whose content should be treated as raw data
	 * and not processed during the conversion from DOM nodes to view elements.
	 */ _rawContentElementMatcher = new Matcher();
    /**
	 * Matcher for inline object view elements. This is an extension of a simple {@link #inlineObjectElements} array of element names.
	 */ _inlineObjectElementMatcher = new Matcher();
    /**
	 * Set of elements with temporary custom properties that require clearing after render.
	 */ _elementsWithTemporaryCustomProperties = new Set();
    /**
	 * Creates a DOM converter.
	 *
	 * @param document The view document instance.
	 * @param options An object with configuration options.
	 * @param options.blockFillerMode The type of the block filler to use.
	 * Default value depends on the options.renderingMode:
	 *  'nbsp' when options.renderingMode == 'data',
	 *  'br' when options.renderingMode == 'editing'.
	 * @param options.renderingMode Whether to leave the View-to-DOM conversion result unchanged
	 * or improve editing experience by filtering out interactive data.
	 */ constructor(document, { blockFillerMode, renderingMode = 'editing' } = {}){
        this.document = document;
        this.renderingMode = renderingMode;
        this.blockFillerMode = blockFillerMode || (renderingMode === 'editing' ? 'br' : 'nbsp');
        this.preElements = [
            'pre',
            'textarea'
        ];
        this.blockElements = [
            'address',
            'article',
            'aside',
            'blockquote',
            'caption',
            'center',
            'dd',
            'details',
            'dir',
            'div',
            'dl',
            'dt',
            'fieldset',
            'figcaption',
            'figure',
            'footer',
            'form',
            'h1',
            'h2',
            'h3',
            'h4',
            'h5',
            'h6',
            'header',
            'hgroup',
            'legend',
            'li',
            'main',
            'menu',
            'nav',
            'ol',
            'p',
            'pre',
            'section',
            'summary',
            'table',
            'tbody',
            'td',
            'tfoot',
            'th',
            'thead',
            'tr',
            'ul'
        ];
        this.inlineObjectElements = [
            'object',
            'iframe',
            'input',
            'button',
            'textarea',
            'select',
            'option',
            'video',
            'embed',
            'audio',
            'img',
            'canvas'
        ];
        this.unsafeElements = [
            'script',
            'style'
        ];
        this._domDocument = this.renderingMode === 'editing' ? global.document : global.document.implementation.createHTMLDocument('');
    }
    /**
	 * The DOM Document used by `ViewDomConverter` to create DOM nodes.
	 */ get domDocument() {
        return this._domDocument;
    }
    /**
	 * Binds a given DOM element that represents fake selection to a **position** of a
	 * {@link module:engine/view/documentselection~ViewDocumentSelection document selection}.
	 * Document selection copy is stored and can be retrieved by the
	 * {@link module:engine/view/domconverter~ViewDomConverter#fakeSelectionToView} method.
	 */ bindFakeSelection(domElement, viewDocumentSelection) {
        this._fakeSelectionMapping.set(domElement, new ViewSelection(viewDocumentSelection));
    }
    /**
	 * Returns a {@link module:engine/view/selection~ViewSelection view selection} instance corresponding to a given
	 * DOM element that represents fake selection. Returns `undefined` if binding to the given DOM element does not exist.
	 */ fakeSelectionToView(domElement) {
        return this._fakeSelectionMapping.get(domElement);
    }
    /**
	 * Binds DOM and view elements, so it will be possible to get corresponding elements using
	 * {@link module:engine/view/domconverter~ViewDomConverter#mapDomToView} and
	 * {@link module:engine/view/domconverter~ViewDomConverter#mapViewToDom}.
	 *
	 * @param domElement The DOM element to bind.
	 * @param viewElement The view element to bind.
	 */ bindElements(domElement, viewElement) {
        this._domToViewMapping.set(domElement, viewElement);
        this._viewToDomMapping.set(viewElement, domElement);
    }
    /**
	 * Unbinds a given DOM element from the view element it was bound to. Unbinding is deep, meaning that all children of
	 * the DOM element will be unbound too.
	 *
	 * @param domElement The DOM element to unbind.
	 */ unbindDomElement(domElement) {
        const viewElement = this._domToViewMapping.get(domElement);
        if (viewElement) {
            this._domToViewMapping.delete(domElement);
            this._viewToDomMapping.delete(viewElement);
            for (const child of domElement.children){
                this.unbindDomElement(child);
            }
        }
    }
    /**
	 * Binds DOM and view document fragments, so it will be possible to get corresponding document fragments using
	 * {@link module:engine/view/domconverter~ViewDomConverter#mapDomToView} and
	 * {@link module:engine/view/domconverter~ViewDomConverter#mapViewToDom}.
	 *
	 * @param domFragment The DOM document fragment to bind.
	 * @param viewFragment The view document fragment to bind.
	 */ bindDocumentFragments(domFragment, viewFragment) {
        this._domToViewMapping.set(domFragment, viewFragment);
        this._viewToDomMapping.set(viewFragment, domFragment);
    }
    /**
	 * Decides whether a given pair of attribute key and value should be passed further down the pipeline.
	 *
	 * @param elementName Element name in lower case.
	 */ shouldRenderAttribute(attributeKey, attributeValue, elementName) {
        if (this.renderingMode === 'data') {
            return true;
        }
        attributeKey = attributeKey.toLowerCase();
        if (attributeKey.startsWith('on')) {
            return false;
        }
        if (attributeKey === 'srcdoc' && attributeValue.match(/\bon\S+\s*=|javascript:|<\s*\/*script/i)) {
            return false;
        }
        if (elementName === 'img' && (attributeKey === 'src' || attributeKey === 'srcset')) {
            return true;
        }
        if (elementName === 'source' && attributeKey === 'srcset') {
            return true;
        }
        if (attributeValue.match(/^\s*(javascript:|data:(image\/svg|text\/x?html))/i)) {
            return false;
        }
        return true;
    }
    /**
	 * Set `domElement`'s content using provided `html` argument. Apply necessary filtering for the editing pipeline.
	 *
	 * @param domElement DOM element that should have `html` set as its content.
	 * @param html Textual representation of the HTML that will be set on `domElement`.
	 */ setContentOf(domElement, html) {
        // For data pipeline we pass the HTML as-is.
        if (this.renderingMode === 'data') {
            domElement.innerHTML = html;
            return;
        }
        const document = new DOMParser().parseFromString(html, 'text/html');
        const fragment = document.createDocumentFragment();
        const bodyChildNodes = document.body.childNodes;
        while(bodyChildNodes.length > 0){
            fragment.appendChild(bodyChildNodes[0]);
        }
        const treeWalker = document.createTreeWalker(fragment, NodeFilter.SHOW_ELEMENT);
        const nodes = [];
        let currentNode;
        // eslint-disable-next-line no-cond-assign
        while(currentNode = treeWalker.nextNode()){
            nodes.push(currentNode);
        }
        for (const currentNode of nodes){
            // Go through nodes to remove those that are prohibited in editing pipeline.
            for (const attributeName of currentNode.getAttributeNames()){
                this.setDomElementAttribute(currentNode, attributeName, currentNode.getAttribute(attributeName));
            }
            const elementName = currentNode.tagName.toLowerCase();
            // There are certain nodes, that should be renamed to <span> in editing pipeline.
            if (this._shouldRenameElement(elementName)) {
                _logUnsafeElement(elementName);
                currentNode.replaceWith(this._createReplacementDomElement(elementName, currentNode));
            }
        }
        // Empty the target element.
        while(domElement.firstChild){
            domElement.firstChild.remove();
        }
        domElement.append(fragment);
    }
    /**
	 * Converts the view to the DOM. For all text nodes, not bound elements and document fragments new items will
	 * be created. For bound elements and document fragments the method will return corresponding items.
	 *
	 * @param viewNode View node or document fragment to transform.
	 * @param options Conversion options.
	 * @param options.bind Determines whether new elements will be bound.
	 * @param options.withChildren If `false`, node's and document fragment's children will not be converted.
	 * @returns Converted node or DocumentFragment.
	 */ viewToDom(viewNode, options = {}) {
        if (viewNode.is('$text')) {
            const textData = this._processDataFromViewText(viewNode);
            return this._domDocument.createTextNode(textData);
        } else {
            const viewElementOrFragment = viewNode;
            if (this.mapViewToDom(viewElementOrFragment)) {
                // Do not reuse element that is marked to not reuse (for example an IMG element
                // so it can immediately display a placeholder background instead of waiting for the new src to load).
                if (viewElementOrFragment.getCustomProperty('editingPipeline:doNotReuseOnce')) {
                    this._elementsWithTemporaryCustomProperties.add(viewElementOrFragment);
                } else {
                    return this.mapViewToDom(viewElementOrFragment);
                }
            }
            let domElement;
            if (viewElementOrFragment.is('documentFragment')) {
                // Create DOM document fragment.
                domElement = this._domDocument.createDocumentFragment();
                if (options.bind) {
                    this.bindDocumentFragments(domElement, viewElementOrFragment);
                }
            } else if (viewElementOrFragment.is('uiElement')) {
                if (viewElementOrFragment.name === '$comment') {
                    domElement = this._domDocument.createComment(viewElementOrFragment.getCustomProperty('$rawContent'));
                } else {
                    // UIElement has its own render() method (see #799).
                    domElement = viewElementOrFragment.render(this._domDocument, this);
                }
                if (options.bind) {
                    this.bindElements(domElement, viewElementOrFragment);
                }
                return domElement;
            } else {
                // Create DOM element.
                if (this._shouldRenameElement(viewElementOrFragment.name)) {
                    _logUnsafeElement(viewElementOrFragment.name);
                    domElement = this._createReplacementDomElement(viewElementOrFragment.name);
                } else if (viewElementOrFragment.hasAttribute('xmlns')) {
                    domElement = this._domDocument.createElementNS(viewElementOrFragment.getAttribute('xmlns'), viewElementOrFragment.name);
                } else {
                    domElement = this._domDocument.createElement(viewElementOrFragment.name);
                }
                // RawElement take care of their children in RawElement#render() method which can be customized
                // (see https://github.com/ckeditor/ckeditor5/issues/4469).
                if (viewElementOrFragment.is('rawElement')) {
                    viewElementOrFragment.render(domElement, this);
                }
                if (options.bind) {
                    this.bindElements(domElement, viewElementOrFragment);
                }
                // Copy element's attributes.
                for (const key of viewElementOrFragment.getAttributeKeys()){
                    this.setDomElementAttribute(domElement, key, viewElementOrFragment.getAttribute(key), viewElementOrFragment);
                }
            }
            if (options.withChildren !== false) {
                for (const child of this.viewChildrenToDom(viewElementOrFragment, options)){
                    if (domElement instanceof HTMLTemplateElement) {
                        domElement.content.appendChild(child);
                    } else {
                        domElement.appendChild(child);
                    }
                }
            }
            return domElement;
        }
    }
    /**
	 * Sets the attribute on a DOM element.
	 *
	 * **Note**: To remove the attribute, use {@link #removeDomElementAttribute}.
	 *
	 * @param domElement The DOM element the attribute should be set on.
	 * @param key The name of the attribute.
	 * @param value The value of the attribute.
	 * @param relatedViewElement The view element related to the `domElement` (if there is any).
	 * It helps decide whether the attribute set is unsafe. For instance, view elements created via the
	 * {@link module:engine/view/downcastwriter~ViewDowncastWriter} methods can allow certain attributes
	 * that would normally be filtered out.
	 */ setDomElementAttribute(domElement, key, value, relatedViewElement) {
        const shouldRenderAttribute = this.shouldRenderAttribute(key, value, domElement.tagName.toLowerCase()) || relatedViewElement && relatedViewElement.shouldRenderUnsafeAttribute(key);
        if (!shouldRenderAttribute) {
            logWarning('domconverter-unsafe-attribute-detected', {
                domElement,
                key,
                value
            });
        }
        if (!isValidAttributeName(key)) {
            /**
			 * Invalid attribute name was ignored during rendering.
			 *
			 * @error domconverter-invalid-attribute-detected
			 */ logWarning('domconverter-invalid-attribute-detected', {
                domElement,
                key,
                value
            });
            return;
        }
        // The old value was safe but the new value is unsafe.
        if (domElement.hasAttribute(key) && !shouldRenderAttribute) {
            domElement.removeAttribute(key);
        } else if (domElement.hasAttribute(UNSAFE_ATTRIBUTE_NAME_PREFIX + key) && shouldRenderAttribute) {
            domElement.removeAttribute(UNSAFE_ATTRIBUTE_NAME_PREFIX + key);
        }
        // If the attribute should not be rendered, rename it (instead of removing) to give developers some idea of what
        // is going on (https://github.com/ckeditor/ckeditor5/issues/10801).
        domElement.setAttribute(shouldRenderAttribute ? key : UNSAFE_ATTRIBUTE_NAME_PREFIX + key, value);
    }
    /**
	 * Removes an attribute from a DOM element.
	 *
	 * **Note**: To set the attribute, use {@link #setDomElementAttribute}.
	 *
	 * @param domElement The DOM element the attribute should be removed from.
	 * @param key The name of the attribute.
	 */ removeDomElementAttribute(domElement, key) {
        // See #_createReplacementDomElement() to learn what this is.
        if (key == UNSAFE_ELEMENT_REPLACEMENT_ATTRIBUTE) {
            return;
        }
        domElement.removeAttribute(key);
        // See setDomElementAttribute() to learn what this is.
        domElement.removeAttribute(UNSAFE_ATTRIBUTE_NAME_PREFIX + key);
    }
    /**
	 * Converts children of the view element to DOM using the
	 * {@link module:engine/view/domconverter~ViewDomConverter#viewToDom} method.
	 * Additionally, this method adds block {@link module:engine/view/filler filler} to the list of children, if needed.
	 *
	 * @param viewElement Parent view element.
	 * @param options See {@link module:engine/view/domconverter~ViewDomConverter#viewToDom} options parameter.
	 * @returns DOM nodes.
	 */ *viewChildrenToDom(viewElement, options = {}) {
        const fillerPositionOffset = viewElement.getFillerOffset && viewElement.getFillerOffset();
        let offset = 0;
        for (const childView of viewElement.getChildren()){
            if (fillerPositionOffset === offset) {
                yield this._getBlockFiller();
            }
            const transparentRendering = childView.is('element') && !!childView.getCustomProperty('dataPipeline:transparentRendering') && !first(childView.getAttributes());
            if (transparentRendering && this.renderingMode == 'data') {
                // `RawElement` doesn't have #children defined, so they need to be temporarily rendered
                // and extracted directly.
                if (childView.is('rawElement')) {
                    const tempElement = this._domDocument.createElement(childView.name);
                    childView.render(tempElement, this);
                    yield* [
                        ...tempElement.childNodes
                    ];
                } else {
                    yield* this.viewChildrenToDom(childView, options);
                }
            } else {
                if (transparentRendering) {
                    /**
					 * The `dataPipeline:transparentRendering` flag is supported only in the data pipeline.
					 *
					 * @error domconverter-transparent-rendering-unsupported-in-editing-pipeline
					 */ logWarning('domconverter-transparent-rendering-unsupported-in-editing-pipeline', {
                        viewElement: childView
                    });
                }
                yield this.viewToDom(childView, options);
            }
            offset++;
        }
        if (fillerPositionOffset === offset) {
            yield this._getBlockFiller();
        }
    }
    /**
	 * Converts view {@link module:engine/view/range~ViewRange} to DOM range.
	 * Inline and block {@link module:engine/view/filler fillers} are handled during the conversion.
	 *
	 * @param viewRange View range.
	 * @returns DOM range.
	 */ viewRangeToDom(viewRange) {
        const domStart = this.viewPositionToDom(viewRange.start);
        const domEnd = this.viewPositionToDom(viewRange.end);
        const domRange = this._domDocument.createRange();
        domRange.setStart(domStart.parent, domStart.offset);
        domRange.setEnd(domEnd.parent, domEnd.offset);
        return domRange;
    }
    /**
	 * Converts view {@link module:engine/view/position~ViewPosition} to DOM parent and offset.
	 *
	 * Inline and block {@link module:engine/view/filler fillers} are handled during the conversion.
	 * If the converted position is directly before inline filler it is moved inside the filler.
	 *
	 * @param viewPosition View position.
	 * @returns DOM position or `null` if view position could not be converted to DOM.
	 * DOM position has two properties:
	 * * `parent` - DOM position parent.
	 * * `offset` - DOM position offset.
	 */ viewPositionToDom(viewPosition) {
        const viewParent = viewPosition.parent;
        if (viewParent.is('$text')) {
            const domParent = this.findCorrespondingDomText(viewParent);
            if (!domParent) {
                // Position is in a view text node that has not been rendered to DOM yet.
                return null;
            }
            let offset = viewPosition.offset;
            if (startsWithFiller(domParent)) {
                offset += INLINE_FILLER_LENGTH;
            }
            // In case someone uses outdated view position, but DOM text node was already changed while typing.
            // See: https://github.com/ckeditor/ckeditor5/issues/18648.
            // Note that when checking Renderer#_isSelectionInInlineFiller() this might be other element
            // than a text node as it is triggered before applying view changes to the DOM.
            if (domParent.data && offset > domParent.data.length) {
                offset = domParent.data.length;
            }
            return {
                parent: domParent,
                offset
            };
        } else {
            // viewParent is instance of ViewElement.
            let domParent, domBefore, domAfter;
            if (viewPosition.offset === 0) {
                domParent = this.mapViewToDom(viewParent);
                if (!domParent) {
                    // Position is in a view element that has not been rendered to DOM yet.
                    return null;
                }
                domAfter = domParent.childNodes[0];
            } else {
                const nodeBefore = viewPosition.nodeBefore;
                domBefore = nodeBefore.is('$text') ? this.findCorrespondingDomText(nodeBefore) : this.mapViewToDom(nodeBefore);
                if (!domBefore) {
                    // Position is after a view element that has not been rendered to DOM yet.
                    return null;
                }
                domParent = domBefore.parentNode;
                domAfter = domBefore.nextSibling;
            }
            // If there is an inline filler at position return position inside the filler. We should never return
            // the position before the inline filler.
            if (isText(domAfter) && startsWithFiller(domAfter)) {
                return {
                    parent: domAfter,
                    offset: INLINE_FILLER_LENGTH
                };
            }
            const offset = domBefore ? indexOf(domBefore) + 1 : 0;
            return {
                parent: domParent,
                offset
            };
        }
    }
    /**
	 * Converts DOM to view. For all text nodes, not bound elements and document fragments new items will
	 * be created. For bound elements and document fragments function will return corresponding items. For
	 * {@link module:engine/view/filler fillers} `null` will be returned.
	 * For all DOM elements rendered by {@link module:engine/view/uielement~ViewUIElement} that UIElement will be returned.
	 *
	 * @param domNode DOM node or document fragment to transform.
	 * @param options Conversion options.
	 * @param options.bind Determines whether new elements will be bound. False by default.
	 * @param options.withChildren If `true`, node's and document fragment's children will be converted too. True by default.
	 * @param options.keepOriginalCase If `false`, node's tag name will be converted to lower case. False by default.
	 * @param options.skipComments If `false`, comment nodes will be converted to `$comment`
	 * {@link module:engine/view/uielement~ViewUIElement view UI elements}. False by default.
	 * @returns Converted node or document fragment or `null` if DOM node is a {@link module:engine/view/filler filler}
	 * or the given node is an empty text node.
	 */ domToView(domNode, options = {}) {
        const inlineNodes = [];
        const generator = this._domToView(domNode, options, inlineNodes);
        // Get the first yielded value or a returned value.
        const node = generator.next().value;
        if (!node) {
            return null;
        }
        // Trigger children handling.
        generator.next();
        // Whitespace cleaning.
        this._processDomInlineNodes(null, inlineNodes, options);
        // This was a single block filler so just remove it.
        if (this.blockFillerMode == 'br' && isViewBrFiller(node)) {
            return null;
        }
        // Text not got trimmed to an empty string so there is no result node.
        if (node.is('$text') && node.data.length == 0) {
            return null;
        }
        return node;
    }
    /**
	 * Converts children of the DOM element to view nodes using
	 * the {@link module:engine/view/domconverter~ViewDomConverter#domToView} method.
	 * Additionally this method omits block {@link module:engine/view/filler filler}, if it exists in the DOM parent.
	 *
	 * @param domElement Parent DOM element.
	 * @param options See {@link module:engine/view/domconverter~ViewDomConverter#domToView} options parameter.
	 * @param inlineNodes An array that will be populated with inline nodes. It's used internally for whitespace processing.
	 * @returns View nodes.
	 */ *domChildrenToView(domElement, options = {}, inlineNodes = []) {
        // Get child nodes from content document fragment if element is template
        let childNodes = [];
        if (domElement instanceof HTMLTemplateElement) {
            childNodes = [
                ...domElement.content.childNodes
            ];
        } else {
            childNodes = [
                ...domElement.childNodes
            ];
        }
        for(let i = 0; i < childNodes.length; i++){
            const domChild = childNodes[i];
            const generator = this._domToView(domChild, options, inlineNodes);
            // Get the first yielded value or a returned value.
            const viewChild = generator.next().value;
            if (viewChild !== null) {
                // Whitespace cleaning before entering a block element (between block elements).
                if (this._isBlockViewElement(viewChild)) {
                    this._processDomInlineNodes(domElement, inlineNodes, options);
                }
                // Yield only if this is not a block filler.
                if (!(this.blockFillerMode == 'br' && isViewBrFiller(viewChild))) {
                    yield viewChild;
                }
                // Trigger children handling.
                generator.next();
            }
        }
        // Whitespace cleaning before leaving a block element (content of block element).
        this._processDomInlineNodes(domElement, inlineNodes, options);
    }
    /**
	 * Converts DOM selection to view {@link module:engine/view/selection~ViewSelection}.
	 * Ranges which cannot be converted will be omitted.
	 *
	 * @param domSelection DOM selection.
	 * @returns View selection.
	 */ domSelectionToView(domSelection) {
        // See: https://github.com/ckeditor/ckeditor5/issues/9635.
        if (isGeckoRestrictedDomSelection(domSelection)) {
            return new ViewSelection([]);
        }
        // DOM selection might be placed in fake selection container.
        // If container contains fake selection - return corresponding view selection.
        if (domSelection.rangeCount === 1) {
            let container = domSelection.getRangeAt(0).startContainer;
            // The DOM selection might be moved to the text node inside the fake selection container.
            if (isText(container)) {
                container = container.parentNode;
            }
            const viewSelection = this.fakeSelectionToView(container);
            if (viewSelection) {
                return viewSelection;
            }
        }
        const isBackward = this.isDomSelectionBackward(domSelection);
        const viewRanges = [];
        for(let i = 0; i < domSelection.rangeCount; i++){
            // DOM Range have correct start and end, no matter what is the DOM Selection direction. So we don't have to fix anything.
            const domRange = domSelection.getRangeAt(i);
            const viewRange = this.domRangeToView(domRange);
            if (viewRange) {
                viewRanges.push(viewRange);
            }
        }
        return new ViewSelection(viewRanges, {
            backward: isBackward
        });
    }
    /**
	 * Converts DOM Range to view {@link module:engine/view/range~ViewRange}.
	 * If the start or end position cannot be converted `null` is returned.
	 *
	 * @param domRange DOM range.
	 * @returns View range.
	 */ domRangeToView(domRange) {
        const viewStart = this.domPositionToView(domRange.startContainer, domRange.startOffset);
        const viewEnd = this.domPositionToView(domRange.endContainer, domRange.endOffset);
        if (viewStart && viewEnd) {
            return new ViewRange(viewStart, viewEnd);
        }
        return null;
    }
    /**
	 * Converts DOM parent and offset to view {@link module:engine/view/position~ViewPosition}.
	 *
	 * If the position is inside a {@link module:engine/view/filler filler} which has no corresponding view node,
	 * position of the filler will be converted and returned.
	 *
	 * If the position is inside DOM element rendered by {@link module:engine/view/uielement~ViewUIElement}
	 * that position will be converted to view position before that UIElement.
	 *
	 * If structures are too different and it is not possible to find corresponding position then `null` will be returned.
	 *
	 * @param domParent DOM position parent.
	 * @param domOffset DOM position offset. You can skip it when converting the inline filler node.
	 * @returns View position.
	 */ domPositionToView(domParent, domOffset = 0) {
        if (this.isBlockFiller(domParent)) {
            return this.domPositionToView(domParent.parentNode, indexOf(domParent));
        }
        // If position is somewhere inside UIElement or a RawElement - return position before that element.
        const viewElement = this.mapDomToView(domParent);
        if (viewElement && (viewElement.is('uiElement') || viewElement.is('rawElement'))) {
            return ViewPosition._createBefore(viewElement);
        }
        if (isText(domParent)) {
            if (isInlineFiller(domParent)) {
                return this.domPositionToView(domParent.parentNode, indexOf(domParent));
            }
            const viewParent = this.findCorrespondingViewText(domParent);
            let offset = domOffset;
            if (!viewParent) {
                return null;
            }
            if (startsWithFiller(domParent)) {
                offset -= INLINE_FILLER_LENGTH;
                offset = offset < 0 ? 0 : offset;
            }
            return new ViewPosition(viewParent, offset);
        } else {
            if (domOffset === 0) {
                const viewParent = this.mapDomToView(domParent);
                if (viewParent) {
                    return new ViewPosition(viewParent, 0);
                }
            } else {
                const domBefore = domParent.childNodes[domOffset - 1];
                // Jump over an inline filler (and also on Firefox jump over a block filler while pressing backspace in an empty paragraph).
                if (isText(domBefore) && isInlineFiller(domBefore) || domBefore && this.isBlockFiller(domBefore)) {
                    return this.domPositionToView(domBefore.parentNode, indexOf(domBefore));
                }
                const viewBefore = isText(domBefore) ? this.findCorrespondingViewText(domBefore) : this.mapDomToView(domBefore);
                // TODO #663
                if (viewBefore && viewBefore.parent) {
                    return new ViewPosition(viewBefore.parent, viewBefore.index + 1);
                }
            }
            return null;
        }
    }
    /**
	 * Returns corresponding view {@link module:engine/view/element~ViewElement Element} or
	 * {@link module:engine/view/documentfragment~ViewDocumentFragment} for provided DOM element or
	 * document fragment. If there is no view item {@link module:engine/view/domconverter~ViewDomConverter#bindElements bound}
	 * to the given DOM - `undefined` is returned.
	 *
	 * For all DOM elements rendered by a {@link module:engine/view/uielement~ViewUIElement} or
	 * a {@link module:engine/view/rawelement~ViewRawElement}, the parent `UIElement` or `RawElement` will be returned.
	 *
	 * @param domElementOrDocumentFragment DOM element or document fragment.
	 * @returns Corresponding view element, document fragment or `undefined` if no element was bound.
	 */ mapDomToView(domElementOrDocumentFragment) {
        const hostElement = this.getHostViewElement(domElementOrDocumentFragment);
        return hostElement || this._domToViewMapping.get(domElementOrDocumentFragment);
    }
    /**
	 * Finds corresponding text node. Text nodes are not {@link module:engine/view/domconverter~ViewDomConverter#bindElements bound},
	 * corresponding text node is returned based on the sibling or parent.
	 *
	 * If the directly previous sibling is a {@link module:engine/view/domconverter~ViewDomConverter#bindElements bound} element, it is used
	 * to find the corresponding text node.
	 *
	 * If this is a first child in the parent and the parent is a
	 * {@link module:engine/view/domconverter~ViewDomConverter#bindElements bound}
	 * element, it is used to find the corresponding text node.
	 *
	 * For all text nodes rendered by a {@link module:engine/view/uielement~ViewUIElement} or
	 * a {@link module:engine/view/rawelement~ViewRawElement}, the parent `UIElement` or `RawElement` will be returned.
	 *
	 * Otherwise `null` is returned.
	 *
	 * Note that for the block or inline {@link module:engine/view/filler filler} this method returns `null`.
	 *
	 * @param domText DOM text node.
	 * @returns Corresponding view text node or `null`, if it was not possible to find a corresponding node.
	 */ findCorrespondingViewText(domText) {
        if (isInlineFiller(domText)) {
            return null;
        }
        // If DOM text was rendered by a UIElement or a RawElement - return this parent element.
        const hostElement = this.getHostViewElement(domText);
        if (hostElement) {
            return hostElement;
        }
        const previousSibling = domText.previousSibling;
        // Try to use previous sibling to find the corresponding text node.
        if (previousSibling) {
            if (!this.isElement(previousSibling)) {
                // The previous is text or comment.
                return null;
            }
            const viewElement = this.mapDomToView(previousSibling);
            if (viewElement) {
                const nextSibling = viewElement.nextSibling;
                // It might be filler which has no corresponding view node.
                if (nextSibling instanceof ViewText) {
                    return nextSibling;
                } else {
                    return null;
                }
            }
        } else {
            const viewElement = this.mapDomToView(domText.parentNode);
            if (viewElement) {
                const firstChild = viewElement.getChild(0);
                // It might be filler which has no corresponding view node.
                if (firstChild instanceof ViewText) {
                    return firstChild;
                } else {
                    return null;
                }
            }
        }
        return null;
    }
    mapViewToDom(documentFragmentOrElement) {
        return this._viewToDomMapping.get(documentFragmentOrElement);
    }
    /**
	 * Finds corresponding text node. Text nodes are not {@link module:engine/view/domconverter~ViewDomConverter#bindElements bound},
	 * corresponding text node is returned based on the sibling or parent.
	 *
	 * If the directly previous sibling is a {@link module:engine/view/domconverter~ViewDomConverter#bindElements bound} element, it is used
	 * to find the corresponding text node.
	 *
	 * If this is a first child in the parent and the parent is a
	 * {@link module:engine/view/domconverter~ViewDomConverter#bindElements bound}
	 * element, it is used to find the corresponding text node.
	 *
	 * Otherwise `null` is returned.
	 *
	 * @param viewText View text node.
	 * @returns Corresponding DOM text node or `null`, if it was not possible to find a corresponding node.
	 */ findCorrespondingDomText(viewText) {
        const previousSibling = viewText.previousSibling;
        // Try to use previous sibling to find the corresponding text node.
        if (previousSibling && this.mapViewToDom(previousSibling)) {
            return this.mapViewToDom(previousSibling).nextSibling;
        }
        // If this is a first node, try to use parent to find the corresponding text node.
        if (!previousSibling && viewText.parent && this.mapViewToDom(viewText.parent)) {
            return this.mapViewToDom(viewText.parent).childNodes[0];
        }
        return null;
    }
    /**
	 * Focuses DOM editable that is corresponding to provided {@link module:engine/view/editableelement~ViewEditableElement}.
	 */ focus(viewEditable) {
        const domEditable = this.mapViewToDom(viewEditable);
        if (!domEditable || domEditable.ownerDocument.activeElement === domEditable) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'ViewDomConverter',
            // @if CK_DEBUG_TYPING // 		'%cDOM editable is already active or does not exist',
            // @if CK_DEBUG_TYPING // 		'font-style: italic'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            return;
        }
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'ViewDomConverter',
        // @if CK_DEBUG_TYPING // 		'Focus DOM editable:',
        // @if CK_DEBUG_TYPING // 		{ domEditable }
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        // Save the scrollX and scrollY positions before the focus.
        const { scrollX, scrollY } = global.window;
        const scrollPositions = [];
        // Save all scrollLeft and scrollTop values starting from domEditable up to
        // document#documentElement.
        forEachDomElementAncestor(domEditable, (node)=>{
            const { scrollLeft, scrollTop } = node;
            scrollPositions.push([
                scrollLeft,
                scrollTop
            ]);
        });
        domEditable.focus({
            preventScroll: true
        });
        // Restore scrollLeft and scrollTop values starting from domEditable up to
        // document#documentElement.
        // https://github.com/ckeditor/ckeditor5-engine/issues/951
        // https://github.com/ckeditor/ckeditor5-engine/issues/957
        forEachDomElementAncestor(domEditable, (node)=>{
            const [scrollLeft, scrollTop] = scrollPositions.shift();
            node.scrollLeft = scrollLeft;
            node.scrollTop = scrollTop;
        });
        // Restore the scrollX and scrollY positions after the focus.
        // https://github.com/ckeditor/ckeditor5-engine/issues/951
        global.window.scrollTo(scrollX, scrollY);
    }
    /**
	 * Remove DOM selection from blurred editable, so it won't interfere with clicking on dropdowns (especially on iOS).
	 *
	 * @internal
	 */ _clearDomSelection() {
        const domEditable = this.mapViewToDom(this.document.selection.editableElement);
        if (!domEditable) {
            return;
        }
        // Check if DOM selection is inside editor editable element.
        const domSelection = domEditable.ownerDocument.defaultView.getSelection();
        const newViewSelection = this.domSelectionToView(domSelection);
        const selectionInEditable = newViewSelection && newViewSelection.rangeCount > 0;
        if (selectionInEditable) {
            domSelection.removeAllRanges();
        }
    }
    /**
	 * Returns `true` when `node.nodeType` equals `Node.ELEMENT_NODE`.
	 *
	 * @param node Node to check.
	 */ isElement(node) {
        return node && node.nodeType == Node.ELEMENT_NODE;
    }
    /**
	 * Returns `true` when `node.nodeType` equals `Node.DOCUMENT_FRAGMENT_NODE`.
	 *
	 * @param node Node to check.
	 */ isDocumentFragment(node) {
        return node && node.nodeType == Node.DOCUMENT_FRAGMENT_NODE;
    }
    /**
	 * Checks if the node is an instance of the block filler for this DOM converter.
	 *
	 * ```ts
	 * const converter = new ViewDomConverter( viewDocument, { blockFillerMode: 'br' } );
	 *
	 * converter.isBlockFiller( BR_FILLER( document ) ); // true
	 * converter.isBlockFiller( NBSP_FILLER( document ) ); // false
	 * ```
	 *
	 * **Note:**: For the `'nbsp'` mode the method also checks context of a node so it cannot be a detached node.
	 *
	 * **Note:** A special case in the `'nbsp'` mode exists where the `<br>` in `<p><br></p>` is treated as a block filler.
	 *
	 * @param domNode DOM node to check.
	 * @returns True if a node is considered a block filler for given mode.
	 */ isBlockFiller(domNode) {
        if (this.blockFillerMode == 'br') {
            return domNode.isEqualNode(BR_FILLER_REF);
        }
        // Special case for <p><br></p> in which <br> should be treated as filler even when we are not in the 'br' mode.
        // See https://github.com/ckeditor/ckeditor5/issues/5564.
        if (isOnlyBrInBlock(domNode, this.blockElements)) {
            return true;
        }
        // If not in 'br' mode, try recognizing both marked and regular nbsp block fillers.
        return domNode.isEqualNode(MARKED_NBSP_FILLER_REF) || isNbspBlockFiller(domNode, this.blockElements);
    }
    /**
	 * Returns `true` if given selection is a backward selection, that is, if it's `focus` is before `anchor`.
	 *
	 * @param selection Selection instance to check.
	 */ isDomSelectionBackward(selection) {
        if (selection.isCollapsed) {
            return false;
        }
        // Since it takes multiple lines of code to check whether a "DOM Position" is before/after another "DOM Position",
        // we will use the fact that range will collapse if it's end is before it's start.
        const range = this._domDocument.createRange();
        try {
            range.setStart(selection.anchorNode, selection.anchorOffset);
            range.setEnd(selection.focusNode, selection.focusOffset);
        } catch  {
            // Safari sometimes gives us a selection that makes Range.set{Start,End} throw.
            // See https://github.com/ckeditor/ckeditor5/issues/12375.
            return false;
        }
        const backward = range.collapsed;
        range.detach();
        return backward;
    }
    /**
	 * Returns a parent {@link module:engine/view/uielement~ViewUIElement} or {@link module:engine/view/rawelement~ViewRawElement}
	 * that hosts the provided DOM node. Returns `null` if there is no such parent.
	 */ getHostViewElement(domNode) {
        const ancestors = getAncestors(domNode);
        // Remove domNode from the list.
        ancestors.pop();
        while(ancestors.length){
            const domNode = ancestors.pop();
            const viewNode = this._domToViewMapping.get(domNode);
            if (viewNode && (viewNode.is('uiElement') || viewNode.is('rawElement'))) {
                return viewNode;
            }
        }
        return null;
    }
    /**
	 * Checks if the given selection's boundaries are at correct places.
	 *
	 * The following places are considered as incorrect for selection boundaries:
	 *
	 * * before or in the middle of an inline filler sequence,
	 * * inside a DOM element which represents {@link module:engine/view/uielement~ViewUIElement a view UI element},
	 * * inside a DOM element which represents {@link module:engine/view/rawelement~ViewRawElement a view raw element}.
	 *
	 * @param domSelection The DOM selection object to be checked.
	 * @returns `true` if the given selection is at a correct place, `false` otherwise.
	 */ isDomSelectionCorrect(domSelection) {
        return this._isDomSelectionPositionCorrect(domSelection.anchorNode, domSelection.anchorOffset) && this._isDomSelectionPositionCorrect(domSelection.focusNode, domSelection.focusOffset);
    }
    /**
	 * Registers a {@link module:engine/view/matcher~MatcherPattern} for view elements whose content should be treated as raw data
	 * and not processed during the conversion from DOM nodes to view elements.
	 *
	 * This is affecting how {@link module:engine/view/domconverter~ViewDomConverter#domToView} and
	 * {@link module:engine/view/domconverter~ViewDomConverter#domChildrenToView} process DOM nodes.
	 *
	 * The raw data can be later accessed by a
	 * {@link module:engine/view/element~ViewElement#getCustomProperty custom property of a view element} called `"$rawContent"`.
	 *
	 * @param pattern Pattern matching a view element whose content should
	 * be treated as raw data.
	 */ registerRawContentMatcher(pattern) {
        this._rawContentElementMatcher.add(pattern);
    }
    /**
	 * Registers a {@link module:engine/view/matcher~MatcherPattern} for inline object view elements.
	 *
	 * This is affecting how {@link module:engine/view/domconverter~ViewDomConverter#domToView} and
	 * {@link module:engine/view/domconverter~ViewDomConverter#domChildrenToView} process DOM nodes.
	 *
	 * This is an extension of a simple {@link #inlineObjectElements} array of element names.
	 *
	 * @param pattern Pattern matching a view element which should be treated as an inline object.
	 */ registerInlineObjectMatcher(pattern) {
        this._inlineObjectElementMatcher.add(pattern);
    }
    /**
	 * Clear temporary custom properties.
	 *
	 * @internal
	 */ _clearTemporaryCustomProperties() {
        for (const element of this._elementsWithTemporaryCustomProperties){
            element._removeCustomProperty('editingPipeline:doNotReuseOnce');
        }
        this._elementsWithTemporaryCustomProperties.clear();
    }
    /**
	 * Returns the block {@link module:engine/view/filler filler} node based on the current {@link #blockFillerMode} setting.
	 */ _getBlockFiller() {
        switch(this.blockFillerMode){
            case 'nbsp':
                return NBSP_FILLER(this._domDocument); // eslint-disable-line new-cap
            case 'markedNbsp':
                return MARKED_NBSP_FILLER(this._domDocument); // eslint-disable-line new-cap
            case 'br':
                return BR_FILLER(this._domDocument); // eslint-disable-line new-cap
        }
    }
    /**
	 * Checks if the given DOM position is a correct place for selection boundary. See {@link #isDomSelectionCorrect}.
	 *
	 * @param domParent Position parent.
	 * @param offset Position offset.
	 * @returns `true` if given position is at a correct place for selection boundary, `false` otherwise.
	 */ _isDomSelectionPositionCorrect(domParent, offset) {
        // If selection is before or in the middle of inline filler string, it is incorrect.
        if (isText(domParent) && startsWithFiller(domParent) && offset < INLINE_FILLER_LENGTH) {
            // Selection in a text node, at wrong position (before or in the middle of filler).
            return false;
        }
        if (this.isElement(domParent) && startsWithFiller(domParent.childNodes[offset])) {
            // Selection in an element node, before filler text node.
            return false;
        }
        const viewParent = this.mapDomToView(domParent);
        // The position is incorrect when anchored inside a UIElement or a RawElement.
        // Note: In case of UIElement and RawElement, mapDomToView() returns a parent element for any DOM child
        // so there's no need to perform any additional checks.
        if (viewParent && (viewParent.is('uiElement') || viewParent.is('rawElement'))) {
            return false;
        }
        return true;
    }
    /**
	 * Internal generator for {@link #domToView}. Also used by {@link #domChildrenToView}.
	 * Separates DOM nodes conversion from whitespaces processing.
	 *
	 * @param domNode DOM node or document fragment to transform.
	 * @param inlineNodes An array of recently encountered inline nodes truncated to the block element boundaries.
	 * Used later to process whitespaces.
	 */ *_domToView(domNode, options, inlineNodes) {
        // Special case for <p><br></p> in which <br> should be treated as filler even when we are not in the 'br' mode.
        // See https://github.com/ckeditor/ckeditor5/issues/5564.
        if (this.blockFillerMode != 'br' && isOnlyBrInBlock(domNode, this.blockElements)) {
            return null;
        }
        // When node is inside a UIElement or a RawElement return that parent as it's view representation.
        const hostElement = this.getHostViewElement(domNode);
        if (hostElement) {
            return hostElement;
        }
        if (isComment(domNode) && options.skipComments) {
            return null;
        }
        if (isText(domNode)) {
            if (isInlineFiller(domNode)) {
                return null;
            } else {
                const textData = domNode.data;
                if (textData === '') {
                    return null;
                }
                const textNode = new ViewText(this.document, textData);
                inlineNodes.push(textNode);
                return textNode;
            }
        } else {
            let viewElement = this.mapDomToView(domNode);
            if (viewElement) {
                if (this._isInlineObjectElement(viewElement)) {
                    inlineNodes.push(viewElement);
                }
                return viewElement;
            }
            if (this.isDocumentFragment(domNode)) {
                // Create view document fragment.
                viewElement = new ViewDocumentFragment(this.document);
                if (options.bind) {
                    this.bindDocumentFragments(domNode, viewElement);
                }
            } else {
                // Create view element.
                viewElement = this._createViewElement(domNode, options);
                if (options.bind) {
                    this.bindElements(domNode, viewElement);
                }
                // Copy element's attributes.
                const attrs = domNode.attributes;
                if (attrs) {
                    for(let l = attrs.length, i = 0; i < l; i++){
                        viewElement._setAttribute(attrs[i].name, attrs[i].value);
                    }
                }
                // Treat this element's content as a raw data if it was registered as such.
                if (this._isViewElementWithRawContent(viewElement, options)) {
                    viewElement._setCustomProperty('$rawContent', domNode.innerHTML);
                    if (!this._isBlockViewElement(viewElement)) {
                        inlineNodes.push(viewElement);
                    }
                    return viewElement;
                }
                // Comment node is also treated as an element with raw data.
                if (isComment(domNode)) {
                    viewElement._setCustomProperty('$rawContent', domNode.data);
                    return viewElement;
                }
            }
            // Yield the element first so the flow of nested inline nodes is not reversed inside elements.
            yield viewElement;
            const nestedInlineNodes = [];
            if (options.withChildren !== false) {
                for (const child of this.domChildrenToView(domNode, options, nestedInlineNodes)){
                    viewElement._appendChild(child);
                }
            }
            // Check if this is an inline object after processing child nodes so matcher
            // for inline objects can verify if the element is empty.
            if (this._isInlineObjectElement(viewElement)) {
                inlineNodes.push(viewElement);
                // Inline object content should be handled as a flow-root.
                this._processDomInlineNodes(null, nestedInlineNodes, options);
            } else {
                // It's an inline element that is not an object (like <b>, <i>) or a block element.
                for (const inlineNode of nestedInlineNodes){
                    inlineNodes.push(inlineNode);
                }
            }
        }
    }
    /**
	 * Internal helper that walks the list of inline view nodes already generated from DOM nodes
	 * and handles whitespaces and NBSPs.
	 *
	 * @param domParent The DOM parent of the given inline nodes. This should be a document fragment or
	 * a block element to whitespace processing start cleaning.
	 * @param inlineNodes An array of recently encountered inline nodes truncated to the block element boundaries.
	 */ _processDomInlineNodes(domParent, inlineNodes, options) {
        if (!inlineNodes.length) {
            return;
        }
        // Process text nodes only after reaching a block or document fragment,
        // do not alter whitespaces while processing an inline element like <b> or <i>.
        if (domParent && !this.isDocumentFragment(domParent) && !this._isBlockDomElement(domParent)) {
            return;
        }
        let prevNodeEndsWithSpace = false;
        for(let i = 0; i < inlineNodes.length; i++){
            const node = inlineNodes[i];
            if (!node.is('$text')) {
                prevNodeEndsWithSpace = false;
                continue;
            }
            let data;
            let nodeEndsWithSpace = false;
            if (this._isPreFormatted(node)) {
                data = getDataWithoutFiller(node.data);
            } else {
                // Change all consecutive whitespace characters (from the [ \n\t\r] set –
                // see https://github.com/ckeditor/ckeditor5-engine/issues/822#issuecomment-311670249) to a single space character.
                // That's how multiple whitespaces are treated when rendered, so we normalize those whitespaces.
                // We're replacing 1+ (and not 2+) to also normalize singular \n\t\r characters (#822).
                data = node.data.replace(/[ \n\t\r]{1,}/g, ' ');
                nodeEndsWithSpace = /[^\S\u00A0]/.test(data.charAt(data.length - 1));
                const prevNode = i > 0 ? inlineNodes[i - 1] : null;
                const nextNode = i + 1 < inlineNodes.length ? inlineNodes[i + 1] : null;
                const shouldLeftTrim = !prevNode || prevNode.is('element') && prevNode.name == 'br' || prevNodeEndsWithSpace;
                const shouldRightTrim = nextNode ? false : !startsWithFiller(node.data);
                // Do not try to clear whitespaces if this is flat mapping for the purpose of mutation observer and differ in rendering.
                if (options.withChildren !== false) {
                    // If the previous dom text node does not exist or it ends by whitespace character, remove space character from the
                    // beginning of this text node. Such space character is treated as a whitespace.
                    if (shouldLeftTrim) {
                        data = data.replace(/^ /, '');
                    }
                    // If the next text node does not exist remove space character from the end of this text node.
                    if (shouldRightTrim) {
                        data = data.replace(/ $/, '');
                    }
                }
                // At the beginning and end of a block element, Firefox inserts normal space + <br> instead of non-breaking space.
                // This means that the text node starts/end with normal space instead of non-breaking space.
                // This causes a problem because the normal space would be removed in `.replace` calls above. To prevent that,
                // the inline filler is removed only after the data is initially processed (by the `.replace` above). See ckeditor5#692.
                data = getDataWithoutFiller(data);
                // Block filler handling.
                if (this.blockFillerMode != 'br' && node.parent) {
                    if (isViewMarkedNbspFiller(node.parent, data)) {
                        data = '';
                        // Mark block element as it has a block filler and remove the `<span data-cke-filler="true">` element.
                        if (node.parent.parent) {
                            node.parent.parent._setCustomProperty('$hasBlockFiller', true);
                            node.parent._remove();
                        }
                    } else if (isViewNbspFiller(node.parent, data, this.blockElements)) {
                        data = '';
                        node.parent._setCustomProperty('$hasBlockFiller', true);
                    }
                }
                // At this point we should have removed all whitespaces from DOM text data.
                //
                // Now, We will reverse the process that happens in `_processDataFromViewText`.
                //
                // We have to change &nbsp; chars, that were in DOM text data because of rendering reasons, to spaces.
                // First, change all ` \u00A0` pairs (space + &nbsp;) to two spaces. DOM converter changes two spaces from model/view to
                // ` \u00A0` to ensure proper rendering. Since here we convert back, we recognize those pairs and change them back to `  `.
                data = data.replace(/ \u00A0/g, '  ');
                const isNextNodeInlineObjectElement = nextNode && nextNode.is('element') && nextNode.name != 'br';
                const isNextNodeStartingWithSpace = nextNode && nextNode.is('$text') && nextNode.data.charAt(0) == ' ';
                // Then, let's change the last nbsp to a space.
                if (/[ \u00A0]\u00A0$/.test(data) || !nextNode || isNextNodeInlineObjectElement || isNextNodeStartingWithSpace) {
                    data = data.replace(/\u00A0$/, ' ');
                }
                // Then, change &nbsp; character that is at the beginning of the text node to space character.
                // We do that replacement only if this is the first node or the previous node ends on whitespace character.
                if (shouldLeftTrim || prevNode && prevNode.is('element') && prevNode.name != 'br') {
                    data = data.replace(/^\u00A0/, ' ');
                }
            }
            // At this point, all whitespaces should be removed and all &nbsp; created for rendering reasons should be
            // changed to normal space. All left &nbsp; are &nbsp; inserted intentionally.
            if (data.length == 0 && node.parent) {
                node._remove();
                inlineNodes.splice(i, 1);
                i--;
            } else {
                node._data = data;
                prevNodeEndsWithSpace = nodeEndsWithSpace;
            }
        }
        inlineNodes.length = 0;
    }
    /**
	 * Takes text data from a given {@link module:engine/view/text~ViewText#data} and processes it so
	 * it is correctly displayed in the DOM.
	 *
	 * Following changes are done:
	 *
	 * * a space at the beginning is changed to `&nbsp;` if this is the first text node in its container
	 * element or if a previous text node ends with a space character,
	 * * space at the end of the text node is changed to `&nbsp;` if there are two spaces at the end of a node or if next node
	 * starts with a space or if it is the last text node in its container,
	 * * remaining spaces are replaced to a chain of spaces and `&nbsp;` (e.g. `'x   x'` becomes `'x &nbsp; x'`).
	 *
	 * Content of {@link #preElements} is not processed.
	 *
	 * @param node View text node to process.
	 * @returns Processed text data.
	 */ _processDataFromViewText(node) {
        let data = node.data;
        // If the currently processed view text node is preformatted, we should not change whitespaces.
        if (this._isPreFormatted(node)) {
            return data;
        }
        // 1. Replace the first space with a nbsp if the previous node ends with a space or there is no previous node
        // (container element boundary).
        if (data.charAt(0) == ' ') {
            const prevNode = this._getTouchingInlineViewNode(node, false);
            const prevEndsWithSpace = prevNode && prevNode.is('$textProxy') && this._nodeEndsWithSpace(prevNode);
            if (prevEndsWithSpace || !prevNode) {
                data = '\u00A0' + data.substr(1);
            }
        }
        // 2. Replace the last space with nbsp if there are two spaces at the end or if the next node starts with space or there is no
        // next node (container element boundary).
        //
        // Keep in mind that Firefox prefers $nbsp; before tag, not inside it:
        //
        // Foo <span>&nbsp;bar</span>  <-- bad.
        // Foo&nbsp;<span> bar</span>  <-- good.
        //
        // More here: https://github.com/ckeditor/ckeditor5-engine/issues/1747.
        if (data.charAt(data.length - 1) == ' ') {
            const nextNode = this._getTouchingInlineViewNode(node, true);
            const nextStartsWithSpace = nextNode && nextNode.is('$textProxy') && nextNode.data.charAt(0) == ' ';
            if (data.charAt(data.length - 2) == ' ' || !nextNode || nextStartsWithSpace) {
                data = data.substr(0, data.length - 1) + '\u00A0';
            }
        }
        // 3. Create space+nbsp pairs.
        return data.replace(/ {2}/g, ' \u00A0');
    }
    /**
	 * Checks whether given node ends with a space character after changing appropriate space characters to `&nbsp;`s.
	 *
	 * @param  node Node to check.
	 * @returns `true` if given `node` ends with space, `false` otherwise.
	 */ _nodeEndsWithSpace(node) {
        if (this._isPreFormatted(node)) {
            return false;
        }
        const data = this._processDataFromViewText(node);
        return data.charAt(data.length - 1) == ' ';
    }
    /**
	 * Checks whether given text contains preformatted white space. This is the case if
	 * * any of node ancestors has a name which is in `preElements` array, or
	 * * the closest ancestor that has the `white-space` CSS property sets it to a value that preserves spaces
	 *
	 * @param node Node to check
	 * @returns `true` if given node contains preformatted white space, `false` otherwise.
	 */ _isPreFormatted(node) {
        if (_hasViewParentOfType(node, this.preElements)) {
            return true;
        }
        for (const ancestor of node.getAncestors({
            parentFirst: true
        })){
            if (!ancestor.is('element') || !ancestor.hasStyle('white-space') || ancestor.getStyle('white-space') === 'inherit') {
                continue;
            }
            // If the node contains the `white-space` property with a value that does not preserve spaces, it will take
            // precedence over any white-space settings its ancestors contain, so no further parent checking needs to
            // be done.
            return [
                'pre',
                'pre-wrap',
                'break-spaces'
            ].includes(ancestor.getStyle('white-space'));
        }
        return false;
    }
    /**
	 * Helper function. For given {@link module:engine/view/text~ViewText view text node}, it finds previous or next sibling
	 * that is contained in the same container element. If there is no such sibling, `null` is returned.
	 *
	 * @param node Reference node.
	 * @returns Touching text node, an inline object
	 * or `null` if there is no next or previous touching text node.
	 */ _getTouchingInlineViewNode(node, getNext) {
        const treeWalker = new ViewTreeWalker({
            startPosition: getNext ? ViewPosition._createAfter(node) : ViewPosition._createBefore(node),
            direction: getNext ? 'forward' : 'backward'
        });
        for (const { item } of treeWalker){
            // Found a text node in the same container element.
            if (item.is('$textProxy')) {
                return item;
            } else if (item.is('element') && item.getCustomProperty('dataPipeline:transparentRendering')) {
                continue;
            } else if (item.is('element', 'br')) {
                return null;
            } else if (this._isInlineObjectElement(item)) {
                return item;
            } else if (item.is('containerElement') || this._isBlockViewElement(item)) {
                return null;
            }
        }
        return null;
    }
    /**
	 * Returns `true` if a DOM node belongs to {@link #blockElements}. `false` otherwise.
	 */ _isBlockDomElement(node) {
        return this.isElement(node) && this.blockElements.includes(node.tagName.toLowerCase());
    }
    /**
	 * Returns `true` if a view node belongs to {@link #blockElements}. `false` otherwise.
	 */ _isBlockViewElement(node) {
        return node.is('element') && this.blockElements.includes(node.name);
    }
    /**
	 * Returns `true` if a DOM node belongs to {@link #inlineObjectElements}. `false` otherwise.
	 */ _isInlineObjectElement(node) {
        if (!node.is('element')) {
            return false;
        }
        return node.name == 'br' || this.inlineObjectElements.includes(node.name) || !!this._inlineObjectElementMatcher.match(node);
    }
    /**
	 * Creates view element basing on the node type.
	 *
	 * @param node DOM node to check.
	 * @param options Conversion options. See {@link module:engine/view/domconverter~ViewDomConverter#domToView} options parameter.
	 */ _createViewElement(node, options) {
        if (isComment(node)) {
            return new ViewUIElement(this.document, '$comment');
        }
        const viewName = options.keepOriginalCase ? node.tagName : node.tagName.toLowerCase();
        return new ViewElement(this.document, viewName);
    }
    /**
	 * Checks if view element's content should be treated as a raw data.
	 *
	 * @param viewElement View element to check.
	 * @param options Conversion options. See {@link module:engine/view/domconverter~ViewDomConverter#domToView} options parameter.
	 */ _isViewElementWithRawContent(viewElement, options) {
        return options.withChildren !== false && viewElement.is('element') && !!this._rawContentElementMatcher.match(viewElement);
    }
    /**
	 * Checks whether a given element name should be renamed in a current rendering mode.
	 *
	 * @param elementName The name of view element.
	 */ _shouldRenameElement(elementName) {
        const name = elementName.toLowerCase();
        return this.renderingMode === 'editing' && this.unsafeElements.includes(name);
    }
    /**
	 * Return a <span> element with a special attribute holding the name of the original element.
	 * Optionally, copy all the attributes of the original element if that element is provided.
	 *
	 * @param elementName The name of view element.
	 * @param originalDomElement The original DOM element to copy attributes and content from.
	 */ _createReplacementDomElement(elementName, originalDomElement) {
        const newDomElement = this._domDocument.createElement('span');
        // Mark the span replacing a script as hidden.
        newDomElement.setAttribute(UNSAFE_ELEMENT_REPLACEMENT_ATTRIBUTE, elementName);
        if (originalDomElement) {
            while(originalDomElement.firstChild){
                newDomElement.appendChild(originalDomElement.firstChild);
            }
            for (const attributeName of originalDomElement.getAttributeNames()){
                newDomElement.setAttribute(attributeName, originalDomElement.getAttribute(attributeName));
            }
        }
        return newDomElement;
    }
}
/**
 * Helper function.
 * Used to check if given native `Element` or `Text` node has parent with tag name from `types` array.
 *
 * @returns`true` if such parent exists or `false` if it does not.
 */ function _hasViewParentOfType(node, types) {
    return node.getAncestors().some((parent)=>parent.is('element') && types.includes(parent.name));
}
/**
 * A helper that executes given callback for each DOM node's ancestor, starting from the given node
 * and ending in document#documentElement.
 *
 * @param callback A callback to be executed for each ancestor.
 */ function forEachDomElementAncestor(element, callback) {
    let node = element;
    while(node){
        callback(node);
        node = node.parentElement;
    }
}
/**
 * Checks if given DOM node is a nbsp block filler.
 *
 * A &nbsp; is a block filler only if it is a single child of a block element.
 *
 * @param domNode DOM node.
 */ function isNbspBlockFiller(domNode, blockElements) {
    const isNBSP = domNode.isEqualNode(NBSP_FILLER_REF);
    return isNBSP && hasBlockParent(domNode, blockElements) && domNode.parentNode.childNodes.length === 1;
}
/**
 * Checks if domNode has block parent.
 *
 * @param domNode DOM node.
 */ function hasBlockParent(domNode, blockElements) {
    const parent = domNode.parentNode;
    return !!parent && !!parent.tagName && blockElements.includes(parent.tagName.toLowerCase());
}
/**
 * Checks if given view node is a nbsp block filler.
 *
 * A &nbsp; is a block filler only if it is a single child of a block element.
 */ function isViewNbspFiller(parent, data, blockElements) {
    return data == '\u00A0' && parent && parent.is('element') && parent.childCount == 1 && blockElements.includes(parent.name);
}
/**
 * Checks if given view node is a marked-nbsp block filler.
 *
 * A &nbsp; is a block filler only if it is wrapped in `<span data-cke-filler="true">` element.
 */ function isViewMarkedNbspFiller(parent, data) {
    return data == '\u00A0' && parent && parent.is('element', 'span') && parent.childCount == 1 && parent.hasAttribute('data-cke-filler');
}
/**
 * Checks if given view node is a br block filler.
 *
 * A <br> is a block filler only if it has data-cke-filler attribute set.
 */ function isViewBrFiller(node) {
    return node.is('element', 'br') && node.hasAttribute('data-cke-filler');
}
/**
 * Special case for `<p><br></p>` in which `<br>` should be treated as filler even when we are not in the 'br' mode.
 */ function isOnlyBrInBlock(domNode, blockElements) {
    // See https://github.com/ckeditor/ckeditor5/issues/5564.
    return domNode.tagName === 'BR' && hasBlockParent(domNode, blockElements) && domNode.parentNode.childNodes.length === 1;
}
/**
 * Log to console the information about element that was replaced.
 * Check UNSAFE_ELEMENTS for all recognized unsafe elements.
 *
 * @param elementName The name of the view element.
 */ function _logUnsafeElement(elementName) {
    if (elementName === 'script') {
        logWarning('domconverter-unsafe-script-element-detected');
    }
    if (elementName === 'style') {
        logWarning('domconverter-unsafe-style-element-detected');
    }
}
/**
 * In certain cases, Firefox mysteriously assigns so called "restricted objects" to native DOM Range properties.
 * Any attempt at accessing restricted object's properties causes errors.
 * See: https://github.com/ckeditor/ckeditor5/issues/9635.
 */ function isGeckoRestrictedDomSelection(domSelection) {
    if (!env.isGecko) {
        return false;
    }
    if (!domSelection.rangeCount) {
        return false;
    }
    const container = domSelection.getRangeAt(0).startContainer;
    try {
        Object.prototype.toString.call(container);
    } catch  {
        return true;
    }
    return false;
}
 /**
 * While rendering the editor content, the {@link module:engine/view/domconverter~ViewDomConverter} detected a `<script>` element that may
 * disrupt the editing experience. To avoid this, the `<script>` element was replaced with `<span data-ck-unsafe-element="script"></span>`.
 *
 * @error domconverter-unsafe-script-element-detected
 */  /**
 * While rendering the editor content, the
 * {@link module:engine/view/domconverter~ViewDomConverter} detected a `<style>` element that may affect
 * the editing experience. To avoid this, the `<style>` element was replaced with `<span data-ck-unsafe-element="style"></span>`.
 *
 * @error domconverter-unsafe-style-element-detected
 */  /**
 * The {@link module:engine/view/domconverter~ViewDomConverter} detected an interactive attribute in the
 * {@glink framework/architecture/editing-engine#editing-pipeline editing pipeline}. For the best
 * editing experience, the attribute was renamed to `data-ck-unsafe-attribute-[original attribute name]`.
 *
 * If you are the author of the plugin that generated this attribute and you want it to be preserved
 * in the editing pipeline, you can configure this when creating the element
 * using {@link module:engine/view/downcastwriter~ViewDowncastWriter} during the
 * {@glink framework/architecture/editing-engine#conversion model–view conversion}. Methods such as
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createContainerElement},
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createAttributeElement}, or
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createEmptyElement}
 * accept an option that will disable filtering of specific attributes:
 *
 * ```ts
 * const paragraph = writer.createContainerElement( 'p',
 * 	{
 * 		class: 'clickable-paragraph',
 * 		onclick: 'alert( "Paragraph clicked!" )'
 * 	},
 * 	{
 * 		// Make sure the "onclick" attribute will pass through.
 * 		renderUnsafeAttributes: [ 'onclick' ]
 * 	}
 * );
 * ```
 *
 * @error domconverter-unsafe-attribute-detected
 * @param {HTMLElement} domElement The DOM element the attribute was set on.
 * @param {string} key The original name of the attribute
 * @param {string} value The value of the original attribute
 */

/**
 * Abstract base observer class. Observers are classes which listen to DOM events, do the preliminary
 * processing and fire events on the {@link module:engine/view/document~ViewDocument} objects.
 * Observers can also add features to the view, for instance by updating its status or marking elements
 * which need a refresh on DOM events.
 */ class Observer extends /* #__PURE__ */ DomEmitterMixin() {
    /**
	 * An instance of the view controller.
	 */ view;
    /**
	 * A reference to the {@link module:engine/view/document~ViewDocument} object.
	 */ document;
    /**
	 * The state of the observer. If it is disabled, no events will be fired.
	 */ _isEnabled = false;
    /**
	 * Creates an instance of the observer.
	 */ constructor(view){
        super();
        this.view = view;
        this.document = view.document;
    }
    /**
	 * The state of the observer. If it is disabled, no events will be fired.
	 */ get isEnabled() {
        return this._isEnabled;
    }
    /**
	 * Enables the observer. This method is called when the observer is registered to the
	 * {@link module:engine/view/view~EditingView} and after {@link module:engine/view/view~EditingView#forceRender rendering}
	 * (all observers are {@link #disable disabled} before rendering).
	 *
	 * A typical use case for disabling observers is that mutation observers need to be disabled for the rendering.
	 * However, a child class may not need to be disabled, so it can implement an empty method.
	 *
	 * @see module:engine/view/observer/observer~Observer#disable
	 */ enable() {
        this._isEnabled = true;
    }
    /**
	 * Disables the observer. This method is called before
	 * {@link module:engine/view/view~EditingView#forceRender rendering} to prevent firing events during rendering.
	 *
	 * @see module:engine/view/observer/observer~Observer#enable
	 */ disable() {
        this._isEnabled = false;
    }
    /**
	 * Disables and destroys the observer, among others removes event listeners created by the observer.
	 */ destroy() {
        this.disable();
        this.stopListening();
    }
    /**
	 * Checks whether a given DOM event should be ignored (should not be turned into a synthetic view document event).
	 *
	 * Currently, an event will be ignored only if its target or any of its ancestors has the `data-cke-ignore-events` attribute.
	 * This attribute can be used inside the structures generated by
	 * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createUIElement `ViewDowncastWriter#createUIElement()`} to ignore events
	 * fired within a UI that should be excluded from CKEditor 5's realms.
	 *
	 * @param domTarget The DOM event target to check (usually an element, sometimes a text node and
	 * potentially sometimes a document, too).
	 * @returns Whether this event should be ignored by the observer.
	 */ checkShouldIgnoreEventFromTarget(domTarget) {
        if (domTarget && domTarget.nodeType === 3) {
            domTarget = domTarget.parentNode;
        }
        if (!domTarget || domTarget.nodeType !== 1) {
            return false;
        }
        return domTarget.matches('[data-cke-ignore-events], [data-cke-ignore-events] *');
    }
}

/**
 * Information about a DOM event in context of the {@link module:engine/view/document~ViewDocument}.
 * It wraps the native event, which usually should not be used as the wrapper contains
 * additional data (like key code for keyboard events).
 *
 * @typeParam TEvent The type of DOM Event that this class represents.
 */ class ViewDocumentDomEventData {
    /**
	 * Instance of the view controller.
	 */ view;
    /**
	 * The instance of the document.
	 */ document;
    /**
	 * The DOM event.
	 */ domEvent;
    /**
	 * The DOM target.
	 */ domTarget;
    /**
	 * @param view The instance of the view controller.
	 * @param domEvent The DOM event.
	 * @param additionalData Additional properties that the instance should contain.
	 */ constructor(view, domEvent, additionalData){
        this.view = view;
        this.document = view.document;
        this.domEvent = domEvent;
        this.domTarget = domEvent.target;
        extend(this, additionalData);
    }
    /**
	 * The tree view element representing the target.
	 */ get target() {
        return this.view.domConverter.mapDomToView(this.domTarget);
    }
    /**
	 * Prevents the native's event default action.
	 */ preventDefault() {
        this.domEvent.preventDefault();
    }
    /**
	 * Stops native event propagation.
	 */ stopPropagation() {
        this.domEvent.stopPropagation();
    }
}

/**
 * Base class for DOM event observers. This class handles
 * {@link module:engine/view/observer/observer~Observer#observe adding} listeners to DOM elements,
 * {@link module:engine/view/observer/observer~Observer#disable disabling} and
 * {@link module:engine/view/observer/observer~Observer#enable re-enabling} events.
 * Child class needs to define
 * {@link module:engine/view/observer/domeventobserver~DomEventObserver#domEventType DOM event type} and
 * {@link module:engine/view/observer/domeventobserver~DomEventObserver#onDomEvent callback}.
 *
 * For instance:
 *
 * ```ts
 * class ClickObserver extends DomEventObserver<'click'> {
 * 	// It can also be defined as a normal property in the constructor.
 * 	get domEventType(): 'click' {
 * 		return 'click';
 * 	}
 *
 * 	onDomEvent( domEvent: MouseEvent ): void {
 * 		this.fire( 'click', domEvent );
 * 	}
 * }
 * ```
 *
 * @typeParam EventType DOM Event type name or an union of those.
 * @typeParam AdditionalData Additional data passed along with the event.
 */ class DomEventObserver extends Observer {
    /**
	 * If set to `true` DOM events will be listened on the capturing phase.
	 * Default value is `false`.
	 */ useCapture = false;
    /**
	 * If set to `true`, indicates that the function specified by listener will never call `preventDefault()`.
	 * Default value is `false`.
	 */ usePassive = false;
    /**
	 * @inheritDoc
	 */ observe(domElement) {
        const types = typeof this.domEventType == 'string' ? [
            this.domEventType
        ] : this.domEventType;
        types.forEach((type)=>{
            this.listenTo(domElement, type, (eventInfo, domEvent)=>{
                if (this.isEnabled && !this.checkShouldIgnoreEventFromTarget(domEvent.target)) {
                    this.onDomEvent(domEvent);
                }
            }, {
                useCapture: this.useCapture,
                usePassive: this.usePassive
            });
        });
    }
    /**
	 * @inheritDoc
	 */ stopObserving(domElement) {
        this.stopListening(domElement);
    }
    /**
	 * Calls `Document#fire()` if observer {@link #isEnabled is enabled}.
	 *
	 * @see module:utils/emittermixin~Emitter#fire
	 * @param eventType The event type (name).
	 * @param domEvent The DOM event.
	 * @param additionalData The additional data which should extend the
	 * {@link module:engine/view/observer/domeventdata~ViewDocumentDomEventData event data} object.
	 */ fire(eventType, domEvent, additionalData) {
        if (this.isEnabled) {
            this.document.fire(eventType, new ViewDocumentDomEventData(this.view, domEvent, additionalData));
        }
    }
}

/**
 * Observer for events connected with pressing keyboard keys.
 *
 * Note that this observer is attached by the {@link module:engine/view/view~EditingView} and is available by default.
 */ class KeyObserver extends DomEventObserver {
    /**
	 * @inheritDoc
	 */ domEventType = [
        'keydown',
        'keyup'
    ];
    /**
	 * @inheritDoc
	 */ onDomEvent(domEvt) {
        const data = {
            keyCode: domEvt.keyCode,
            altKey: domEvt.altKey,
            ctrlKey: domEvt.ctrlKey,
            shiftKey: domEvt.shiftKey,
            metaKey: domEvt.metaKey,
            get keystroke () {
                return getCode(this);
            }
        };
        this.fire(domEvt.type, domEvt, data);
    }
}

/**
 * Fake selection observer class. If view selection is fake it is placed in dummy DOM container. This observer listens
 * on {@link module:engine/view/document~ViewDocument#event:keydown keydown} events and handles moving
 * fake view selection to the correct place if arrow keys are pressed.
 * Fires {@link module:engine/view/document~ViewDocument#event:selectionChange selectionChange event} simulating natural behaviour of
 * {@link module:engine/view/observer/selectionobserver~SelectionObserver SelectionObserver}.
 */ class FakeSelectionObserver extends Observer {
    /**
	 * Fires debounced event `selectionChangeDone`. It uses `es-toolkit#debounce` method to delay function call.
	 */ _fireSelectionChangeDoneDebounced;
    /**
	 * Creates new FakeSelectionObserver instance.
	 */ constructor(view){
        super(view);
        this._fireSelectionChangeDoneDebounced = debounce((data)=>{
            this.document.fire('selectionChangeDone', data);
        }, 200);
    }
    /**
	 * @inheritDoc
	 */ observe() {
        const document = this.document;
        document.on('arrowKey', (eventInfo, data)=>{
            const selection = document.selection;
            if (selection.isFake && this.isEnabled) {
                // Prevents default key down handling - no selection change will occur.
                data.preventDefault();
            }
        }, {
            context: '$capture'
        });
        document.on('arrowKey', (eventInfo, data)=>{
            const selection = document.selection;
            if (selection.isFake && this.isEnabled) {
                this._handleSelectionMove(data.keyCode);
            }
        }, {
            priority: 'lowest'
        });
    }
    /**
	 * @inheritDoc
	 */ stopObserving() {}
    /**
	 * @inheritDoc
	 */ destroy() {
        super.destroy();
        this._fireSelectionChangeDoneDebounced.cancel();
    }
    /**
	 * Handles collapsing view selection according to given key code. If left or up key is provided - new selection will be
	 * collapsed to left. If right or down key is pressed - new selection will be collapsed to right.
	 *
	 * This method fires {@link module:engine/view/document~ViewDocument#event:selectionChange} and
	 * {@link module:engine/view/document~ViewDocument#event:selectionChangeDone} events imitating behaviour of
	 * {@link module:engine/view/observer/selectionobserver~SelectionObserver}.
	 */ _handleSelectionMove(keyCode) {
        const selection = this.document.selection;
        const newSelection = new ViewSelection(selection.getRanges(), {
            backward: selection.isBackward,
            fake: false
        });
        // Left or up arrow pressed - move selection to start.
        if (keyCode == keyCodes.arrowleft || keyCode == keyCodes.arrowup) {
            newSelection.setTo(newSelection.getFirstPosition());
        }
        // Right or down arrow pressed - move selection to end.
        if (keyCode == keyCodes.arrowright || keyCode == keyCodes.arrowdown) {
            newSelection.setTo(newSelection.getLastPosition());
        }
        const data = {
            oldSelection: selection,
            newSelection,
            domSelection: null
        };
        // Fire dummy selection change event.
        this.document.fire('selectionChange', data);
        // Call` #_fireSelectionChangeDoneDebounced` every time when `selectionChange` event is fired.
        // This function is debounced what means that `selectionChangeDone` event will be fired only when
        // defined int the function time will elapse since the last time the function was called.
        // So `selectionChangeDone` will be fired when selection will stop changing.
        this._fireSelectionChangeDoneDebounced(data);
    }
}

// @if CK_DEBUG_TYPING // const { _debouncedLine, _buildLogMessage } = require( '../../dev-utils/utils.js' );
/**
 * Mutation observer's role is to watch for any DOM changes inside the editor that weren't
 * done by the editor's {@link module:engine/view/renderer~ViewRenderer} itself and reverting these changes.
 *
 * It does this by observing all mutations in the DOM, marking related view elements as changed and calling
 * {@link module:engine/view/renderer~ViewRenderer#render}. Because all mutated nodes are marked as
 * "to be rendered" and the {@link module:engine/view/renderer~ViewRenderer#render `render()`} method is called,
 * all changes are reverted in the DOM (the DOM is synced with the editor's view structure).
 *
 * Note that this observer is attached by the {@link module:engine/view/view~EditingView} and is available by default.
 */ class MutationObserver extends Observer {
    /**
	 * Reference to the {@link module:engine/view/view~EditingView#domConverter}.
	 */ domConverter;
    /**
	 * Native mutation observer config.
	 */ _config;
    /**
	 * Observed DOM elements.
	 */ _domElements;
    /**
	 * Native mutation observer.
	 */ _mutationObserver;
    /**
	 * @inheritDoc
	 */ constructor(view){
        super(view);
        this._config = {
            childList: true,
            characterData: true,
            subtree: true
        };
        this.domConverter = view.domConverter;
        this._domElements = new Set();
        this._mutationObserver = new window.MutationObserver(this._onMutations.bind(this));
    }
    /**
	 * Synchronously handles mutations and empties the queue.
	 */ flush() {
        this._onMutations(this._mutationObserver.takeRecords());
    }
    /**
	 * @inheritDoc
	 */ observe(domElement) {
        this._domElements.add(domElement);
        if (this.isEnabled) {
            this._mutationObserver.observe(domElement, this._config);
        }
    }
    /**
	 * @inheritDoc
	 */ stopObserving(domElement) {
        this._domElements.delete(domElement);
        if (this.isEnabled) {
            // Unfortunately, it is not possible to stop observing particular DOM element.
            // In order to stop observing one of multiple DOM elements, we need to re-connect the mutation observer.
            this._mutationObserver.disconnect();
            for (const domElement of this._domElements){
                this._mutationObserver.observe(domElement, this._config);
            }
        }
    }
    /**
	 * @inheritDoc
	 */ enable() {
        super.enable();
        for (const domElement of this._domElements){
            this._mutationObserver.observe(domElement, this._config);
        }
    }
    /**
	 * @inheritDoc
	 */ disable() {
        super.disable();
        this._mutationObserver.disconnect();
    }
    /**
	 * @inheritDoc
	 */ destroy() {
        super.destroy();
        this._mutationObserver.disconnect();
    }
    /**
	 * Handles mutations. Mark view elements to sync and call render.
	 *
	 * @param domMutations Array of native mutations.
	 */ _onMutations(domMutations) {
        // As a result of this.flush() we can have an empty collection.
        if (domMutations.length === 0) {
            return;
        }
        const domConverter = this.domConverter;
        // Use map and set for deduplication.
        const mutatedTextNodes = new Set();
        const elementsWithMutatedChildren = new Set();
        // Handle `childList` mutations first, so we will be able to check if the `characterData` mutation is in the
        // element with changed structure anyway.
        for (const mutation of domMutations){
            const element = domConverter.mapDomToView(mutation.target);
            if (!element) {
                continue;
            }
            // Do not collect mutations from UIElements and RawElements.
            if (element.is('uiElement') || element.is('rawElement')) {
                continue;
            }
            if (mutation.type === 'childList' && !this._isBogusBrMutation(mutation)) {
                elementsWithMutatedChildren.add(element);
            }
        }
        // Handle `characterData` mutations later, when we have the full list of nodes which changed structure.
        for (const mutation of domMutations){
            const element = domConverter.mapDomToView(mutation.target);
            // Do not collect mutations from UIElements and RawElements.
            if (element && (element.is('uiElement') || element.is('rawElement'))) {
                continue;
            }
            if (mutation.type === 'characterData') {
                const text = domConverter.findCorrespondingViewText(mutation.target);
                if (text && !elementsWithMutatedChildren.has(text.parent)) {
                    mutatedTextNodes.add(text);
                } else if (!text && startsWithFiller(mutation.target)) {
                    elementsWithMutatedChildren.add(domConverter.mapDomToView(mutation.target.parentNode));
                }
            }
        }
        // Now we build the list of mutations to mark elements. We did not do it earlier to avoid marking the
        // same node multiple times in case of duplication.
        const mutations = [];
        for (const textNode of mutatedTextNodes){
            mutations.push({
                type: 'text',
                node: textNode
            });
        }
        for (const viewElement of elementsWithMutatedChildren){
            const domElement = domConverter.mapViewToDom(viewElement);
            const viewChildren = Array.from(viewElement.getChildren());
            const newViewChildren = Array.from(domConverter.domChildrenToView(domElement, {
                withChildren: false
            }));
            // It may happen that as a result of many changes (sth was inserted and then removed),
            // both elements haven't really changed. #1031
            if (!isEqualWith(viewChildren, newViewChildren, sameNodes)) {
                mutations.push({
                    type: 'children',
                    node: viewElement
                });
            }
        }
        // In case only non-relevant mutations were recorded it skips the event and force render (#5600).
        if (mutations.length) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	_debouncedLine();
            // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'MutationObserver',
            // @if CK_DEBUG_TYPING // 		'%cMutations detected',
            // @if CK_DEBUG_TYPING // 		'font-weight: bold'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            this.document.fire('mutations', {
                mutations
            });
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.groupEnd();
        // @if CK_DEBUG_TYPING // }
        }
    }
    /**
	 * Checks if mutation was generated by the browser inserting bogus br on the end of the block element.
	 * Such mutations are generated while pressing space or performing native spellchecker correction
	 * on the end of the block element in Firefox browser.
	 *
	 * @param mutation Native mutation object.
	 */ _isBogusBrMutation(mutation) {
        let addedNode = null;
        // Check if mutation added only one node on the end of its parent.
        if (mutation.nextSibling === null && mutation.removedNodes.length === 0 && mutation.addedNodes.length == 1) {
            addedNode = this.domConverter.domToView(mutation.addedNodes[0], {
                withChildren: false
            });
        }
        return addedNode && addedNode.is('element', 'br');
    }
}
function sameNodes(child1, child2) {
    // First level of comparison (array of children vs array of children) – use the es-toolkit's default behavior.
    if (Array.isArray(child1)) {
        return;
    }
    // Elements.
    if (child1 === child2) {
        return true;
    } else if (child1.is('$text') && child2.is('$text')) {
        return child1.data === child2.data;
    }
    // Not matching types.
    return false;
}

// @if CK_DEBUG_TYPING // const { _debouncedLine, _buildLogMessage } = require( '../../dev-utils/utils.js' );
/**
 * {@link module:engine/view/document~ViewDocument#event:focus Focus}
 * and {@link module:engine/view/document~ViewDocument#event:blur blur} events observer.
 * Focus observer handle also {@link module:engine/view/rooteditableelement~ViewRootEditableElement#isFocused isFocused} property of the
 * {@link module:engine/view/rooteditableelement~ViewRootEditableElement root elements}.
 *
 * Note that this observer is attached by the {@link module:engine/view/view~EditingView} and is available by default.
 */ class FocusObserver extends DomEventObserver {
    /**
	 * Identifier of the timeout currently used by focus listener to delay rendering execution.
	 */ _renderTimeoutId = null;
    /**
	 * Set to `true` if the document is in the process of setting the focus.
	 *
	 * The flag is used to indicate that setting the focus is in progress.
	 */ _isFocusChanging = false;
    /**
	 * @inheritDoc
	 */ domEventType = [
        'focus',
        'blur'
    ];
    /**
	 * @inheritDoc
	 */ constructor(view){
        super(view);
        this.useCapture = true;
        const document = this.document;
        document.on('focus', ()=>this._handleFocus());
        document.on('blur', (evt, data)=>this._handleBlur(data));
        // Focus the editor in cases where browser dispatches `beforeinput` event to a not-focused editable element.
        // This is flushed by the beforeinput listener in the `InsertTextObserver`.
        // Note that focus is set only if the document is not focused yet.
        // See https://github.com/ckeditor/ckeditor5/issues/14702.
        document.on('beforeinput', ()=>{
            if (!document.isFocused) {
                this._handleFocus();
            }
        }, {
            priority: 'highest'
        });
    }
    /**
	 * Finishes setting the document focus state.
	 */ flush() {
        if (this._isFocusChanging) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	_debouncedLine();
            // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'FocusObserver',
            // @if CK_DEBUG_TYPING // 		'flush focus'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            this._isFocusChanging = false;
            this.document.isFocused = true;
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.groupEnd();
        // @if CK_DEBUG_TYPING // }
        }
    }
    /**
	 * @inheritDoc
	 */ onDomEvent(domEvent) {
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	_debouncedLine();
        // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'FocusObserver',
        // @if CK_DEBUG_TYPING // 		`${ domEvent.type } event`
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'FocusObserver',
        // @if CK_DEBUG_TYPING // 		'DOM target:',
        // @if CK_DEBUG_TYPING // 		{ target: domEvent.target, relatedTarget: domEvent.relatedTarget }
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // 	const domSelection = window.getSelection();
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'FocusObserver',
        // @if CK_DEBUG_TYPING // 		'DOM Selection:',
        // @if CK_DEBUG_TYPING // 		{ node: domSelection!.anchorNode, offset: domSelection!.anchorOffset },
        // @if CK_DEBUG_TYPING // 		{ node: domSelection!.focusNode, offset: domSelection!.focusOffset }
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        this.fire(domEvent.type, domEvent);
    // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
    // @if CK_DEBUG_TYPING // 	console.groupEnd();
    // @if CK_DEBUG_TYPING // }
    }
    /**
	 * @inheritDoc
	 */ destroy() {
        this._clearTimeout();
        super.destroy();
    }
    /**
	 * The `focus` event handler.
	 */ _handleFocus() {
        this._clearTimeout();
        this._isFocusChanging = true;
        // Unfortunately native `selectionchange` event is fired asynchronously.
        // We need to wait until `SelectionObserver` handle the event and then render. Otherwise rendering will
        // overwrite new DOM selection with selection from the view.
        // See https://github.com/ckeditor/ckeditor5-engine/issues/795 for more details.
        // Long timeout is needed to solve #676 and https://github.com/ckeditor/ckeditor5-engine/issues/1157 issues.
        //
        // Using `view.change()` instead of `view.forceRender()` to prevent double rendering
        // in a situation where `selectionchange` already caused selection change.
        this._renderTimeoutId = setTimeout(()=>{
            this._renderTimeoutId = null;
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'FocusObserver',
            // @if CK_DEBUG_TYPING // 		'flush on timeout'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            this.flush();
            this.view.change(()=>{});
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.groupEnd();
        // @if CK_DEBUG_TYPING // }
        }, 50);
    }
    /**
	 * The `blur` event handler.
	 */ _handleBlur(data) {
        const selectedEditable = this.document.selection.editableElement;
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'FocusObserver',
        // @if CK_DEBUG_TYPING // 		'selectedEditable:',
        // @if CK_DEBUG_TYPING // 		{ selectedEditable }
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        if (selectedEditable === null || selectedEditable === data.target) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'FocusObserver',
            // @if CK_DEBUG_TYPING // 		'document no longer focused'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            this.document.isFocused = false;
            this._isFocusChanging = false;
            // Re-render the document to update view elements
            // (changing document.isFocused already marked view as changed since last rendering).
            this.view.change(()=>{});
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.groupEnd();
        // @if CK_DEBUG_TYPING // }
        }
    }
    /**
	 * Clears timeout.
	 */ _clearTimeout() {
        if (this._renderTimeoutId) {
            clearTimeout(this._renderTimeoutId);
            this._renderTimeoutId = null;
        }
    }
}

/**
 * Selection observer class observes selection changes in the document. If a selection changes on the document this
 * observer checks if the DOM selection is different from the {@link module:engine/view/document~ViewDocument#selection view selection}.
 * The selection observer fires {@link module:engine/view/document~ViewDocument#event:selectionChange} event only if
 * a selection change was the only change in the document and the DOM selection is different from the view selection.
 *
 * This observer also manages the {@link module:engine/view/document~ViewDocument#isSelecting} property of the view document.
 *
 * Note that this observer is attached by the {@link module:engine/view/view~EditingView} and is available by default.
 */ class SelectionObserver extends Observer {
    /**
	 * Instance of the mutation observer. Selection observer calls
	 * {@link module:engine/view/observer/mutationobserver~MutationObserver#flush} to ensure that the mutations will be handled
	 * before the {@link module:engine/view/document~ViewDocument#event:selectionChange} event is fired.
	 */ mutationObserver;
    /**
	 * Instance of the focus observer. Focus observer calls
	 * {@link module:engine/view/observer/focusobserver~FocusObserver#flush} to mark the latest focus change as complete.
	 */ focusObserver;
    /**
	 * Reference to the view {@link module:engine/view/documentselection~ViewDocumentSelection} object used to compare
	 * new selection with it.
	 */ selection;
    /**
	 * Reference to the {@link module:engine/view/view~EditingView#domConverter}.
	 */ domConverter;
    /**
	 * A set of documents which have added `selectionchange` listener to avoid adding a listener twice to the same
	 * document.
	 */ _documents = new WeakSet();
    /**
	 * Fires debounced event `selectionChangeDone`. It uses `es-toolkit#debounce` method to delay function call.
	 */ _fireSelectionChangeDoneDebounced;
    /**
	 * When called, starts clearing the {@link #_loopbackCounter} counter in time intervals. When the number of selection
	 * changes exceeds a certain limit within the interval of time, the observer will not fire `selectionChange` but warn about
	 * possible infinite selection loop.
	 */ _clearInfiniteLoopInterval;
    /**
	 * Unlocks the `isSelecting` state of the view document in case the selection observer did not record this fact
	 * correctly (for whatever reason). It is a safeguard (paranoid check), that returns document to the normal state
	 * after a certain period of time (debounced, postponed by each selectionchange event).
	 */ _documentIsSelectingInactivityTimeoutDebounced;
    /**
	 * Private property to check if the code does not enter infinite loop.
	 */ _loopbackCounter = 0;
    /**
	 * A set of DOM documents that have a pending selection change.
	 * Pending selection change is recorded while selection change event is detected on non focused editable.
	 */ _pendingSelectionChange = new Set();
    constructor(view){
        super(view);
        this.mutationObserver = view.getObserver(MutationObserver);
        this.focusObserver = view.getObserver(FocusObserver);
        this.selection = this.document.selection;
        this.domConverter = view.domConverter;
        this._fireSelectionChangeDoneDebounced = debounce((data)=>{
            this.document.fire('selectionChangeDone', data);
        }, 200);
        this._clearInfiniteLoopInterval = setInterval(()=>this._clearInfiniteLoop(), 1000);
        this._documentIsSelectingInactivityTimeoutDebounced = debounce(()=>this.document.isSelecting = false, 5000);
        this.view.document.on('change:isFocused', (evt, name, isFocused)=>{
            if (isFocused && this._pendingSelectionChange.size) {
                // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
                // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'SelectionObserver',
                // @if CK_DEBUG_TYPING // 		'Flush pending selection change'
                // @if CK_DEBUG_TYPING // 	) );
                // @if CK_DEBUG_TYPING // }
                // Iterate over a copy of set because it is modified in selection change handler.
                for (const domDocument of Array.from(this._pendingSelectionChange)){
                    this._handleSelectionChange(domDocument);
                }
                this._pendingSelectionChange.clear();
            }
        });
    }
    /**
	 * @inheritDoc
	 */ observe(domElement) {
        const domDocument = domElement.ownerDocument;
        const startDocumentIsSelecting = ()=>{
            this.document.isSelecting = true;
            // Let's activate the safety timeout each time the document enters the "is selecting" state.
            this._documentIsSelectingInactivityTimeoutDebounced();
        };
        const endDocumentIsSelecting = ()=>{
            if (!this.document.isSelecting) {
                return;
            }
            // Make sure that model selection is up-to-date at the end of selecting process.
            // Sometimes `selectionchange` events could arrive after the `mouseup` event and that selection could be already outdated.
            this._handleSelectionChange(domDocument);
            this.document.isSelecting = false;
            // The safety timeout can be canceled when the document leaves the "is selecting" state.
            this._documentIsSelectingInactivityTimeoutDebounced.cancel();
        };
        // The document has the "is selecting" state while the user keeps making (extending) the selection
        // (e.g. by holding the mouse button and moving the cursor). The state resets when they either released
        // the mouse button or interrupted the process by pressing or releasing any key.
        this.listenTo(domElement, 'selectstart', startDocumentIsSelecting, {
            priority: 'highest'
        });
        this.listenTo(domElement, 'keydown', endDocumentIsSelecting, {
            priority: 'highest',
            useCapture: true
        });
        this.listenTo(domElement, 'keyup', endDocumentIsSelecting, {
            priority: 'highest',
            useCapture: true
        });
        // Add document-wide listeners only once. This method could be called for multiple editing roots.
        if (this._documents.has(domDocument)) {
            return;
        }
        // This listener is using capture mode to make sure that selection is upcasted before any other
        // handler would like to check it and update (for example table multi cell selection).
        this.listenTo(domDocument, 'mouseup', endDocumentIsSelecting, {
            priority: 'highest',
            useCapture: true
        });
        this.listenTo(domDocument, 'selectionchange', ()=>{
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	_debouncedLine();
            // @if CK_DEBUG_TYPING // 	const domSelection = domDocument.defaultView!.getSelection();
            // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'SelectionObserver',
            // @if CK_DEBUG_TYPING // 		'selectionchange'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'SelectionObserver',
            // @if CK_DEBUG_TYPING // 		'DOM Selection:',
            // @if CK_DEBUG_TYPING // 		{ node: domSelection!.anchorNode, offset: domSelection!.anchorOffset },
            // @if CK_DEBUG_TYPING // 		{ node: domSelection!.focusNode, offset: domSelection!.focusOffset }
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            // The Renderer is disabled while composing on non-android browsers, so we can't update the view selection
            // because the DOM and view tree drifted apart. Position mapping could fail because of it.
            if (this.document.isComposing && !env.isAndroid) {
                // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
                // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'SelectionObserver',
                // @if CK_DEBUG_TYPING //		'Selection change ignored (isComposing)'
                // @if CK_DEBUG_TYPING //	) );
                // @if CK_DEBUG_TYPING // 	console.groupEnd();
                // @if CK_DEBUG_TYPING // }
                return;
            }
            this._handleSelectionChange(domDocument);
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.groupEnd();
            // @if CK_DEBUG_TYPING // }
            // Defer the safety timeout when the selection changes (e.g. the user keeps extending the selection
            // using their mouse).
            this._documentIsSelectingInactivityTimeoutDebounced();
        });
        // Update the model ViewDocumentSelection just after the Renderer and the SelectionObserver are locked.
        // We do this synchronously (without waiting for the `selectionchange` DOM event) as browser updates
        // the DOM selection (but not visually) to span the text that is under composition and could be replaced.
        this.listenTo(this.view.document, 'compositionstart', ()=>{
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	const domSelection = domDocument.defaultView!.getSelection();
            // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'SelectionObserver',
            // @if CK_DEBUG_TYPING // 		'update selection on compositionstart'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'SelectionObserver',
            // @if CK_DEBUG_TYPING // 		'DOM Selection:',
            // @if CK_DEBUG_TYPING // 		{ node: domSelection!.anchorNode, offset: domSelection!.anchorOffset },
            // @if CK_DEBUG_TYPING // 		{ node: domSelection!.focusNode, offset: domSelection!.focusOffset }
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            this._handleSelectionChange(domDocument);
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.groupEnd();
        // @if CK_DEBUG_TYPING // }
        }, {
            priority: 'lowest'
        });
        this._documents.add(domDocument);
    }
    /**
	 * @inheritDoc
	 */ stopObserving(domElement) {
        this.stopListening(domElement);
    }
    /**
	 * @inheritDoc
	 */ destroy() {
        super.destroy();
        clearInterval(this._clearInfiniteLoopInterval);
        this._fireSelectionChangeDoneDebounced.cancel();
        this._documentIsSelectingInactivityTimeoutDebounced.cancel();
    }
    /* istanbul ignore next -- @preserve */ _reportInfiniteLoop() {
    // @if CK_DEBUG //		throw new Error(
    // @if CK_DEBUG //			'Selection change observer detected an infinite rendering loop.\n\n' +
    // @if CK_DEBUG //	 		'⚠️⚠️ Report this error on https://github.com/ckeditor/ckeditor5/issues/11658.'
    // @if CK_DEBUG //		);
    }
    /**
	 * Selection change listener. {@link module:engine/view/observer/mutationobserver~MutationObserver#flush Flush} mutations, check if
	 * a selection changes and fires {@link module:engine/view/document~ViewDocument#event:selectionChange} event on every change
	 * and {@link module:engine/view/document~ViewDocument#event:selectionChangeDone} when a selection stop changing.
	 *
	 * @param domDocument DOM document.
	 */ _handleSelectionChange(domDocument) {
        if (!this.isEnabled) {
            return;
        }
        const domSelection = domDocument.defaultView.getSelection();
        if (this.checkShouldIgnoreEventFromTarget(domSelection.anchorNode)) {
            return;
        }
        // Ensure the mutation event will be before selection event on all browsers.
        this.mutationObserver.flush();
        const newViewSelection = this.domConverter.domSelectionToView(domSelection);
        // Do not convert selection change if the new view selection has no ranges in it.
        //
        // It means that the DOM selection is in some way incorrect. Ranges that were in the DOM selection could not be
        // converted to the view. This happens when the DOM selection was moved outside of the editable element.
        if (newViewSelection.rangeCount == 0) {
            this.view.hasDomSelection = false;
            return;
        }
        this.view.hasDomSelection = true;
        // Mark the latest focus change as complete (we got new selection after the focus so the selection is in the focused element).
        this.focusObserver.flush();
        // Ignore selection change as the editable is not focused. Note that in read-only mode, we have to update
        // the model selection as there won't be any focus change to flush the pending selection changes.
        if (!this.view.document.isFocused && !this.view.document.isReadOnly) {
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'SelectionObserver',
            // @if CK_DEBUG_TYPING // 		'Ignore selection change while editable is not focused'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            this._pendingSelectionChange.add(domDocument);
            return;
        }
        this._pendingSelectionChange.delete(domDocument);
        if (this.selection.isEqual(newViewSelection) && this.domConverter.isDomSelectionCorrect(domSelection)) {
            return;
        }
        // Ensure we are not in the infinite loop (#400).
        // This counter is reset each second. 60 selection changes in 1 second is enough high number
        // to be very difficult (impossible) to achieve using just keyboard keys (during normal editor use).
        if (++this._loopbackCounter > 60) {
            // Selection change observer detected an infinite rendering loop.
            // Most probably you try to put the selection in the position which is not allowed
            // by the browser and browser fixes it automatically what causes `selectionchange` event on
            // which a loopback through a model tries to re-render the wrong selection and again.
            this._reportInfiniteLoop();
            return;
        }
        if (!isSelectionWithinRootElements(newViewSelection)) {
            // Occasionally, such as when removing markers during a focus change, the selection may end up inside a view element
            // whose parent has already been detached from the DOM. In most cases this is harmless, but if the selectionchange
            // event fires before the view is fully synchronized with the DOM converter, some elements in the selection may become orphans.
            // This can result in the view being out of sync with the actual DOM structure.
            // See: https://github.com/ckeditor/ckeditor5/issues/18744
            this.view.forceRender();
        } else if (this.selection.isSimilar(newViewSelection)) {
            // If selection was equal and we are at this point of algorithm, it means that it was incorrect.
            // Just re-render it, no need to fire any events, etc.
            this.view.forceRender();
        } else {
            const data = {
                oldSelection: this.selection,
                newSelection: newViewSelection,
                domSelection
            };
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'SelectionObserver',
            // @if CK_DEBUG_TYPING // 		'Fire selection change:',
            // @if CK_DEBUG_TYPING // 		newViewSelection.getFirstRange()
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            // Prepare data for new selection and fire appropriate events.
            this.document.fire('selectionChange', data);
            // Call `#_fireSelectionChangeDoneDebounced` every time when `selectionChange` event is fired.
            // This function is debounced what means that `selectionChangeDone` event will be fired only when
            // defined int the function time will elapse since the last time the function was called.
            // So `selectionChangeDone` will be fired when selection will stop changing.
            this._fireSelectionChangeDoneDebounced(data);
        }
    }
    /**
	 * Clears `SelectionObserver` internal properties connected with preventing infinite loop.
	 */ _clearInfiniteLoop() {
        this._loopbackCounter = 0;
    }
}
/**
 * Checks if all roots of the selection's range boundaries are root elements.
 * Returns true if the selection is fully contained within root elements (not orphaned).
 *
 * @param selection The view selection to check.
 * @returns True if all range boundaries are within root elements, false otherwise.
 */ function isSelectionWithinRootElements(selection) {
    return Array.from(selection.getRanges()).flatMap((range)=>[
            range.start.root,
            range.end.root
        ]).every((root)=>root && root.is('rootElement'));
}

// @if CK_DEBUG_TYPING // const { _debouncedLine, _buildLogMessage } = require( '../../dev-utils/utils.js' );
/**
 * {@link module:engine/view/document~ViewDocument#event:compositionstart Compositionstart},
 * {@link module:engine/view/document~ViewDocument#event:compositionupdate compositionupdate} and
 * {@link module:engine/view/document~ViewDocument#event:compositionend compositionend} events observer.
 *
 * Note that this observer is attached by the {@link module:engine/view/view~EditingView} and is available by default.
 */ class CompositionObserver extends DomEventObserver {
    /**
	 * @inheritDoc
	 */ domEventType = [
        'compositionstart',
        'compositionupdate',
        'compositionend'
    ];
    /**
	 * @inheritDoc
	 */ constructor(view){
        super(view);
        const document = this.document;
        document.on('compositionstart', ()=>{
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.log( ..._buildLogMessage( this, 'CompositionObserver',
            // @if CK_DEBUG_TYPING // 		'%c┌───────────────────────────── isComposing = true ─────────────────────────────┐',
            // @if CK_DEBUG_TYPING // 		'font-weight: bold; color: green'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            document.isComposing = true;
        });
        document.on('compositionend', ()=>{
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.log( ..._buildLogMessage( this, 'CompositionObserver',
            // @if CK_DEBUG_TYPING // 		'%c└───────────────────────────── isComposing = false ─────────────────────────────┘',
            // @if CK_DEBUG_TYPING // 		'font-weight: bold; color: green'
            // @if CK_DEBUG_TYPING // 	) );
            // @if CK_DEBUG_TYPING // }
            document.isComposing = false;
        });
    }
    /**
	 * @inheritDoc
	 */ onDomEvent(domEvent) {
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	_debouncedLine();
        // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'CompositionObserver',
        // @if CK_DEBUG_TYPING // 		`${ domEvent.type }`
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        this.fire(domEvent.type, domEvent, {
            data: domEvent.data
        });
    // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
    // @if CK_DEBUG_TYPING // 	console.groupEnd();
    // @if CK_DEBUG_TYPING // }
    }
}

/**
 * @license Copyright (c) 2003-2025, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-licensing-options
 */ /**
 * @module engine/view/datatransfer
 */ /**
 * A facade over the native [`DataTransfer`](https://developer.mozilla.org/en-US/docs/Web/API/DataTransfer) object.
 */ class ViewDataTransfer {
    /**
	 * The array of files created from the native `DataTransfer#files` or `DataTransfer#items`.
	 */ _files;
    /**
	 * The native DataTransfer object.
	 */ _native;
    /**
	 * @param nativeDataTransfer The native [`DataTransfer`](https://developer.mozilla.org/en-US/docs/Web/API/DataTransfer) object.
	 * @param options.cacheFiles Whether `files` list should be initialized in the constructor.
	 */ constructor(nativeDataTransfer, options = {}){
        // We should store references to the File instances in case someone would like to process this files
        // outside the event handler. Files are stored only for `drop` and `paste` events because they are not usable
        // in other events and are generating a huge delay on Firefox while dragging.
        // See https://github.com/ckeditor/ckeditor5/issues/13366.
        this._files = options.cacheFiles ? getFiles(nativeDataTransfer) : null;
        this._native = nativeDataTransfer;
    }
    /**
	 * The array of files created from the native `DataTransfer#files` or `DataTransfer#items`.
	 */ get files() {
        if (!this._files) {
            this._files = getFiles(this._native);
        }
        return this._files;
    }
    /**
	 * Returns an array of available native content types.
	 */ get types() {
        return this._native.types;
    }
    /**
	 * Gets the data from the data transfer by its MIME type.
	 *
	 * ```ts
	 * dataTransfer.getData( 'text/plain' );
	 * ```
	 *
	 * @param type The MIME type. E.g. `text/html` or `text/plain`.
	 */ getData(type) {
        return this._native.getData(type);
    }
    /**
	 * Sets the data in the data transfer.
	 *
	 * @param type The MIME type. E.g. `text/html` or `text/plain`.
	 */ setData(type, data) {
        this._native.setData(type, data);
    }
    /**
	 * The effect that is allowed for a drag operation.
	 */ set effectAllowed(value) {
        this._native.effectAllowed = value;
    }
    get effectAllowed() {
        return this._native.effectAllowed;
    }
    /**
	 * The actual drop effect.
	 */ set dropEffect(value) {
        this._native.dropEffect = value;
    }
    get dropEffect() {
        return this._native.dropEffect;
    }
    /**
	 * Set a preview image of the dragged content.
	 */ setDragImage(image, x, y) {
        this._native.setDragImage(image, x, y);
    }
    /**
	 * Whether the dragging operation was canceled.
	 */ get isCanceled() {
        return this._native.dropEffect == 'none' || !!this._native.mozUserCancelled;
    }
}
function getFiles(nativeDataTransfer) {
    // DataTransfer.files and items are array-like and might not have an iterable interface.
    const files = Array.from(nativeDataTransfer.files || []);
    const items = Array.from(nativeDataTransfer.items || []);
    if (files.length) {
        return files;
    }
    // Chrome has empty DataTransfer.files, but allows getting files through the items interface.
    return items.filter((item)=>item.kind === 'file').map((item)=>item.getAsFile());
}

// @if CK_DEBUG_TYPING // const { _debouncedLine, _buildLogMessage } = require( '../../dev-utils/utils.js' );
/**
 * Observer for events connected with data input.
 *
 * **Note**: This observer is attached by {@link module:engine/view/view~EditingView} and available by default in all
 * editor instances.
 */ class InputObserver extends DomEventObserver {
    /**
	 * @inheritDoc
	 */ domEventType = 'beforeinput';
    /**
	 * @inheritDoc
	 */ onDomEvent(domEvent) {
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	_debouncedLine();
        // @if CK_DEBUG_TYPING // 	console.group( ..._buildLogMessage( this, 'InputObserver',
        // @if CK_DEBUG_TYPING // 		`${ domEvent.type }: ${ domEvent.inputType } - ${ domEvent.isComposing ? 'is' : 'not' } composing`,
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        const domTargetRanges = domEvent.getTargetRanges();
        const view = this.view;
        const viewDocument = view.document;
        let dataTransfer = null;
        let data = null;
        let targetRanges = [];
        if (domEvent.dataTransfer) {
            dataTransfer = new ViewDataTransfer(domEvent.dataTransfer);
        }
        if (domEvent.data !== null) {
            data = domEvent.data;
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'InputObserver',
        // @if CK_DEBUG_TYPING // 		`%cevent data: %c${ JSON.stringify( data ) }`,
        // @if CK_DEBUG_TYPING // 		'font-weight: bold',
        // @if CK_DEBUG_TYPING // 		'color: blue;'
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        } else if (dataTransfer) {
            data = dataTransfer.getData('text/plain');
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'InputObserver',
        // @if CK_DEBUG_TYPING // 		`%cevent data transfer: %c${ JSON.stringify( data ) }`,
        // @if CK_DEBUG_TYPING // 		'font-weight: bold',
        // @if CK_DEBUG_TYPING // 		'color: blue;'
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        }
        // If the editor selection is fake (an object is selected), the DOM range does not make sense because it is anchored
        // in the fake selection container.
        if (viewDocument.selection.isFake) {
            // Future-proof: in case of multi-range fake selections being possible.
            targetRanges = Array.from(viewDocument.selection.getRanges());
            // Do not allow typing inside a fake selection container, we will handle it manually.
            domEvent.preventDefault();
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'InputObserver',
        // @if CK_DEBUG_TYPING // 		'%cusing fake selection:',
        // @if CK_DEBUG_TYPING // 		'font-weight: bold',
        // @if CK_DEBUG_TYPING // 		targetRanges,
        // @if CK_DEBUG_TYPING // 		viewDocument.selection.isFake ? 'fake view selection' : 'fake DOM parent'
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        } else if (domTargetRanges.length) {
            targetRanges = domTargetRanges.map((domRange)=>{
                // Sometimes browser provides range that starts before editable node.
                // We try to fall back to collapsed range at the valid end position.
                // See https://github.com/ckeditor/ckeditor5/issues/14411.
                // See https://github.com/ckeditor/ckeditor5/issues/14050.
                let viewStart = view.domConverter.domPositionToView(domRange.startContainer, domRange.startOffset);
                const viewEnd = view.domConverter.domPositionToView(domRange.endContainer, domRange.endOffset);
                // When text replacement is enabled and browser tries to replace double space with dot, and space,
                // but the first space is no longer where browser put it (it was moved to an attribute element),
                // then we must extend the target range so it does not include a part of an inline filler.
                if (viewStart && startsWithFiller(domRange.startContainer) && domRange.startOffset < INLINE_FILLER_LENGTH) {
                    // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
                    // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'InputObserver',
                    // @if CK_DEBUG_TYPING // 		'%cTarget range starts in an inline filler - adjusting it',
                    // @if CK_DEBUG_TYPING // 		'font-style: italic'
                    // @if CK_DEBUG_TYPING // 	) );
                    // @if CK_DEBUG_TYPING // }
                    domEvent.preventDefault();
                    let count = INLINE_FILLER_LENGTH - domRange.startOffset;
                    viewStart = viewStart.getLastMatchingPosition((value)=>{
                        // Ignore attribute and UI elements but stop on container elements.
                        if (value.item.is('attributeElement') || value.item.is('uiElement')) {
                            return true;
                        }
                        // Skip as many characters as inline filler was overlapped.
                        if (value.item.is('$textProxy') && count--) {
                            return true;
                        }
                        return false;
                    }, {
                        direction: 'backward',
                        singleCharacters: true
                    });
                }
                // Check if there is no an inline filler just after the target range.
                if (isFollowedByInlineFiller(domRange.endContainer, domRange.endOffset)) {
                    // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
                    // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'InputObserver',
                    // @if CK_DEBUG_TYPING // 		'%cTarget range ends just before an inline filler - prevent default behavior',
                    // @if CK_DEBUG_TYPING // 		'font-style: italic'
                    // @if CK_DEBUG_TYPING // 	) );
                    // @if CK_DEBUG_TYPING // }
                    domEvent.preventDefault();
                }
                if (viewStart) {
                    return view.createRange(viewStart, viewEnd);
                } else if (viewEnd) {
                    return view.createRange(viewEnd);
                }
            }).filter((range)=>!!range);
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'InputObserver',
        // @if CK_DEBUG_TYPING // 		'%cusing target ranges:',
        // @if CK_DEBUG_TYPING // 		'font-weight: bold',
        // @if CK_DEBUG_TYPING // 		targetRanges
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        } else if (env.isAndroid) {
            const domSelection = domEvent.target.ownerDocument.defaultView.getSelection();
            targetRanges = Array.from(view.domConverter.domSelectionToView(domSelection).getRanges());
        // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
        // @if CK_DEBUG_TYPING // 	console.info( ..._buildLogMessage( this, 'InputObserver',
        // @if CK_DEBUG_TYPING // 		'%cusing selection ranges:',
        // @if CK_DEBUG_TYPING // 		'font-weight: bold',
        // @if CK_DEBUG_TYPING // 		targetRanges
        // @if CK_DEBUG_TYPING // 	) );
        // @if CK_DEBUG_TYPING // }
        }
        // Android sometimes fires insertCompositionText with a new-line character at the end of the data
        // instead of firing insertParagraph beforeInput event.
        // Fire the correct type of beforeInput event and ignore the replaced fragment of text because
        // it wants to replace "test" with "test\n".
        // https://github.com/ckeditor/ckeditor5/issues/12368.
        if (env.isAndroid && domEvent.inputType == 'insertCompositionText' && data && data.endsWith('\n')) {
            this.fire(domEvent.type, domEvent, {
                inputType: 'insertParagraph',
                targetRanges: [
                    view.createRange(targetRanges[0].end)
                ]
            });
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.groupEnd();
            // @if CK_DEBUG_TYPING // }
            return;
        }
        // Normalize the insertText data that includes new-line characters.
        // https://github.com/ckeditor/ckeditor5/issues/2045.
        if ([
            'insertText',
            'insertReplacementText'
        ].includes(domEvent.inputType) && data && data.includes('\n')) {
            // There might be a single new-line or double for new paragraph, but we translate
            // it to paragraphs as it is our default action for enter handling.
            const parts = data.split(/\n{1,2}/g);
            let partTargetRanges = targetRanges;
            // Handle all parts on our side as we rely on paragraph inserting and synchronously updated view selection.
            domEvent.preventDefault();
            for(let i = 0; i < parts.length; i++){
                const dataPart = parts[i];
                if (dataPart != '') {
                    this.fire(domEvent.type, domEvent, {
                        data: dataPart,
                        dataTransfer,
                        targetRanges: partTargetRanges,
                        inputType: domEvent.inputType,
                        isComposing: domEvent.isComposing
                    });
                    // Use the result view selection so following events will be added one after another.
                    partTargetRanges = [
                        viewDocument.selection.getFirstRange()
                    ];
                }
                if (i + 1 < parts.length) {
                    this.fire(domEvent.type, domEvent, {
                        inputType: 'insertParagraph',
                        targetRanges: partTargetRanges
                    });
                    // Use the result view selection so following events will be added one after another.
                    partTargetRanges = [
                        viewDocument.selection.getFirstRange()
                    ];
                }
            }
            // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
            // @if CK_DEBUG_TYPING // 	console.groupEnd();
            // @if CK_DEBUG_TYPING // }
            return;
        }
        // Fire the normalized beforeInput event.
        this.fire(domEvent.type, domEvent, {
            data,
            dataTransfer,
            targetRanges,
            inputType: domEvent.inputType,
            isComposing: domEvent.isComposing
        });
    // @if CK_DEBUG_TYPING // if ( ( window as any ).logCKETyping ) {
    // @if CK_DEBUG_TYPING // 	console.groupEnd();
    // @if CK_DEBUG_TYPING // }
    }
}
/**
 * Returns `true` if there is an inline filler just after the position in DOM.
 * It walks up the DOM tree if the offset is at the end of the node.
 */ function isFollowedByInlineFiller(node, offset) {
    while(node.parentNode){
        if (isText(node)) {
            if (offset != node.data.length) {
                return false;
            }
        } else {
            if (offset != node.childNodes.length) {
                return false;
            }
        }
        offset = indexOf(node) + 1;
        node = node.parentNode;
        if (offset < node.childNodes.length && startsWithFiller(node.childNodes[offset])) {
            return true;
        }
    }
    return false;
}

/**
 * Arrow keys observer introduces the {@link module:engine/view/document~ViewDocument#event:arrowKey `Document#arrowKey`} event.
 *
 * Note that this observer is attached by the {@link module:engine/view/view~EditingView} and is available by default.
 */ class ArrowKeysObserver extends Observer {
    /**
	 * @inheritDoc
	 */ constructor(view){
        super(view);
        this.document.on('keydown', (event, data)=>{
            if (this.isEnabled && isArrowKeyCode(data.keyCode)) {
                const eventInfo = new BubblingEventInfo(this.document, 'arrowKey', this.document.selection.getFirstRange());
                this.document.fire(eventInfo, data);
                if (eventInfo.stop.called) {
                    event.stop();
                }
            }
        });
    }
    /**
	 * @inheritDoc
	 */ observe() {}
    /**
	 * @inheritDoc
	 */ stopObserving() {}
}

/**
 * Tab observer introduces the {@link module:engine/view/document~ViewDocument#event:tab `Document#tab`} event.
 *
 * Note that because {@link module:engine/view/observer/tabobserver~TabObserver} is attached by the
 * {@link module:engine/view/view~EditingView}, this event is available by default.
 */ class TabObserver extends Observer {
    /**
	 * @inheritDoc
	 */ constructor(view){
        super(view);
        const doc = this.document;
        doc.on('keydown', (evt, data)=>{
            if (!this.isEnabled || data.keyCode != keyCodes.tab || data.ctrlKey) {
                return;
            }
            const event = new BubblingEventInfo(doc, 'tab', doc.selection.getFirstRange());
            doc.fire(event, data);
            if (event.stop.called) {
                evt.stop();
            }
        });
    }
    /**
	 * @inheritDoc
	 */ observe() {}
    /**
	 * @inheritDoc
	 */ stopObserving() {}
}

/**
 * Editor's view controller class. Its main responsibility is DOM - View management for editing purposes, to provide
 * abstraction over the DOM structure and events and hide all browsers quirks.
 *
 * View controller renders view document to DOM whenever view structure changes. To determine when view can be rendered,
 * all changes need to be done using the {@link module:engine/view/view~EditingView#change} method, using
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter}:
 *
 * ```ts
 * view.change( writer => {
 * 	writer.insert( position, writer.createText( 'foo' ) );
 * } );
 * ```
 *
 * View controller also register {@link module:engine/view/observer/observer~Observer observers} which observes changes
 * on DOM and fire events on the {@link module:engine/view/document~ViewDocument Document}.
 * Note that the following observers are added by the class constructor and are always available:
 *
 * * {@link module:engine/view/observer/selectionobserver~SelectionObserver},
 * * {@link module:engine/view/observer/focusobserver~FocusObserver},
 * * {@link module:engine/view/observer/keyobserver~KeyObserver},
 * * {@link module:engine/view/observer/fakeselectionobserver~FakeSelectionObserver}.
 * * {@link module:engine/view/observer/compositionobserver~CompositionObserver}.
 * * {@link module:engine/view/observer/inputobserver~InputObserver}.
 * * {@link module:engine/view/observer/arrowkeysobserver~ArrowKeysObserver}.
 * * {@link module:engine/view/observer/tabobserver~TabObserver}.
 *
 * This class also {@link module:engine/view/view~EditingView#attachDomRoot binds the DOM and the view elements}.
 *
 * If you do not need full a DOM - view management, and only want to transform a tree of view elements to a tree of DOM
 * elements you do not need this controller. You can use the {@link module:engine/view/domconverter~ViewDomConverter ViewDomConverter}
 * instead.
 */ class EditingView extends /* #__PURE__ */ ObservableMixin() {
    /**
	 * Instance of the {@link module:engine/view/document~ViewDocument} associated with this view controller.
	 */ document;
    /**
	 * Instance of the {@link module:engine/view/domconverter~ViewDomConverter domConverter} used by
	 * {@link module:engine/view/view~EditingView#_renderer renderer}
	 * and {@link module:engine/view/observer/observer~Observer observers}.
	 */ domConverter;
    /**
	 * Roots of the DOM tree. Map on the `HTMLElement`s with roots names as keys.
	 */ domRoots = new Map();
    /**
	 * Instance of the {@link module:engine/view/renderer~ViewRenderer renderer}.
	 */ _renderer;
    /**
	 * A DOM root attributes cache. It saves the initial values of DOM root attributes before the DOM element
	 * is {@link module:engine/view/view~EditingView#attachDomRoot attached} to the view so later on, when
	 * the view is destroyed ({@link module:engine/view/view~EditingView#detachDomRoot}), they can be easily restored.
	 * This way, the DOM element can go back to the (clean) state as if the editing view never used it.
	 */ _initialDomRootAttributes = new WeakMap();
    /**
	 * Map of registered {@link module:engine/view/observer/observer~Observer observers}.
	 */ _observers = new Map();
    /**
	 * ViewDowncastWriter instance used in {@link #change change method} callbacks.
	 */ _writer;
    /**
	 * Is set to `true` when {@link #change view changes} are currently in progress.
	 */ _ongoingChange = false;
    /**
	 * Used to prevent calling {@link #forceRender} and {@link #change} during rendering view to the DOM.
	 */ _postFixersInProgress = false;
    /**
	 * Internal flag to temporary disable rendering. See the usage in the {@link #_disableRendering}.
	 */ _renderingDisabled = false;
    /**
	 * Internal flag that disables rendering when there are no changes since the last rendering.
	 * It stores information about changed selection and changed elements from attached document roots.
	 */ _hasChangedSinceTheLastRendering = false;
    /**
	 * @param stylesProcessor The styles processor instance.
	 */ constructor(stylesProcessor){
        super();
        this.document = new ViewDocument(stylesProcessor);
        this.domConverter = new ViewDomConverter(this.document);
        this.set('isRenderingInProgress', false);
        this.set('hasDomSelection', false);
        this._renderer = new ViewRenderer(this.domConverter, this.document.selection);
        this._renderer.bind('isFocused', 'isSelecting', 'isComposing').to(this.document, 'isFocused', 'isSelecting', 'isComposing');
        this._writer = new ViewDowncastWriter(this.document);
        // Add default observers.
        // Make sure that this list matches AlwaysRegisteredViewObservers type.
        this.addObserver(MutationObserver);
        this.addObserver(FocusObserver);
        this.addObserver(SelectionObserver);
        this.addObserver(KeyObserver);
        this.addObserver(FakeSelectionObserver);
        this.addObserver(CompositionObserver);
        this.addObserver(ArrowKeysObserver);
        this.addObserver(InputObserver);
        this.addObserver(TabObserver);
        // Inject quirks handlers.
        injectQuirksHandling(this);
        injectUiElementHandling(this);
        // Use 'normal' priority so that rendering is performed as first when using that priority.
        this.on('render', ()=>{
            this._render();
            // Informs that layout has changed after render.
            this.document.fire('layoutChanged');
            // Reset the `_hasChangedSinceTheLastRendering` flag after rendering.
            this._hasChangedSinceTheLastRendering = false;
        });
        // Listen to the document selection changes directly.
        this.listenTo(this.document.selection, 'change', ()=>{
            this._hasChangedSinceTheLastRendering = true;
        });
        // Trigger re-render if only the focus changed.
        this.listenTo(this.document, 'change:isFocused', ()=>{
            this._hasChangedSinceTheLastRendering = true;
        });
        // Remove ranges from DOM selection if editor is blurred.
        // See https://github.com/ckeditor/ckeditor5/issues/5753.
        if (env.isiOS) {
            this.listenTo(this.document, 'blur', (evt, data)=>{
                const relatedViewElement = this.domConverter.mapDomToView(data.domEvent.relatedTarget);
                // Do not modify DOM selection if focus is moved to other editable of the same editor.
                if (!relatedViewElement) {
                    this.domConverter._clearDomSelection();
                }
            });
        }
        // Listen to external content mutations (directly in the DOM) and mark them to get verified by the renderer.
        this.listenTo(this.document, 'mutations', (evt, { mutations })=>{
            mutations.forEach((mutation)=>this._renderer.markToSync(mutation.type, mutation.node));
        }, {
            priority: 'low'
        });
        // After all mutated nodes were marked to sync we can trigger view to DOM synchronization
        // to make sure the DOM structure matches the view.
        this.listenTo(this.document, 'mutations', ()=>{
            this.forceRender();
        }, {
            priority: 'lowest'
        });
    }
    /**
	 * Attaches a DOM root element to the view element and enable all observers on that element.
	 * Also {@link module:engine/view/renderer~ViewRenderer#markToSync mark element} to be synchronized
	 * with the view what means that all child nodes will be removed and replaced with content of the view root.
	 *
	 * This method also will change view element name as the same as tag name of given dom root.
	 * Name is always transformed to lower case.
	 *
	 * **Note:** Use {@link #detachDomRoot `detachDomRoot()`} to revert this action.
	 *
	 * @param domRoot DOM root element.
	 * @param name Name of the root.
	 */ attachDomRoot(domRoot, name = 'main') {
        const viewRoot = this.document.getRoot(name);
        // Set view root name the same as DOM root tag name.
        viewRoot._name = domRoot.tagName.toLowerCase();
        const initialDomRootAttributes = {};
        // 1. Copy and cache the attributes to remember the state of the element before attaching.
        //    The cached attributes will be restored in detachDomRoot() so the element goes to the
        //    clean state as if the editing view never used it.
        // 2. Apply the attributes using the view writer, so they all go under the control of the engine.
        //    The editing view takes over the attribute management completely because various
        //    features (e.g. addPlaceholder()) require dynamic changes of those attributes and they
        //    cannot be managed by the engine and the UI library at the same time.
        for (const { name, value } of Array.from(domRoot.attributes)){
            initialDomRootAttributes[name] = value;
            // Do not use writer.setAttribute() for the class attribute. The EditableUIView class
            // and its descendants could have already set some using the writer.addClass() on the view
            // document root. They haven't been rendered yet so they are not present in the DOM root.
            // Using writer.setAttribute( 'class', ... ) would override them completely.
            if (name === 'class') {
                this._writer.addClass(value.split(' '), viewRoot);
            } else {
                // There is a chance that some attributes have already been set on the view root before attaching
                // the DOM root and should be preserved. This is a similar case to the "class" attribute except
                // this time there is no workaround using a some low-level API.
                if (!viewRoot.hasAttribute(name)) {
                    this._writer.setAttribute(name, value, viewRoot);
                }
            }
        }
        this._initialDomRootAttributes.set(domRoot, initialDomRootAttributes);
        const updateContenteditableAttribute = ()=>{
            this._writer.setAttribute('contenteditable', (!viewRoot.isReadOnly).toString(), viewRoot);
            if (viewRoot.isReadOnly) {
                this._writer.addClass('ck-read-only', viewRoot);
            } else {
                this._writer.removeClass('ck-read-only', viewRoot);
            }
        };
        // Set initial value.
        updateContenteditableAttribute();
        this.domRoots.set(name, domRoot);
        this.domConverter.bindElements(domRoot, viewRoot);
        this._renderer.markToSync('children', viewRoot);
        this._renderer.markToSync('attributes', viewRoot);
        this._renderer.domDocuments.add(domRoot.ownerDocument);
        viewRoot.on('change:children', (evt, node)=>this._renderer.markToSync('children', node));
        viewRoot.on('change:attributes', (evt, node)=>this._renderer.markToSync('attributes', node));
        viewRoot.on('change:text', (evt, node)=>this._renderer.markToSync('text', node));
        viewRoot.on('change:isReadOnly', ()=>this.change(updateContenteditableAttribute));
        viewRoot.on('change', ()=>{
            this._hasChangedSinceTheLastRendering = true;
        });
        for (const observer of this._observers.values()){
            observer.observe(domRoot, name);
        }
    }
    /**
	 * Detaches a DOM root element from the view element and restores its attributes to the state before
	 * {@link #attachDomRoot `attachDomRoot()`}.
	 *
	 * @param name Name of the root to detach.
	 */ detachDomRoot(name) {
        const domRoot = this.domRoots.get(name);
        // Remove all root attributes so the DOM element is "bare".
        Array.from(domRoot.attributes).forEach(({ name })=>domRoot.removeAttribute(name));
        const initialDomRootAttributes = this._initialDomRootAttributes.get(domRoot);
        // Revert all view root attributes back to the state before attachDomRoot was called.
        for(const attribute in initialDomRootAttributes){
            domRoot.setAttribute(attribute, initialDomRootAttributes[attribute]);
        }
        this.domRoots.delete(name);
        this.domConverter.unbindDomElement(domRoot);
        for (const observer of this._observers.values()){
            observer.stopObserving(domRoot);
        }
    }
    /**
	 * Gets DOM root element.
	 *
	 * @param name  Name of the root.
	 * @returns DOM root element instance.
	 */ getDomRoot(name = 'main') {
        return this.domRoots.get(name);
    }
    /**
	 * Creates observer of the given type if not yet created, {@link module:engine/view/observer/observer~Observer#enable enables} it
	 * and {@link module:engine/view/observer/observer~Observer#observe attaches} to all existing and future
	 * {@link #domRoots DOM roots}.
	 *
	 * Note: Observers are recognized by their constructor (classes). A single observer will be instantiated and used only
	 * when registered for the first time. This means that features and other components can register a single observer
	 * multiple times without caring whether it has been already added or not.
	 *
	 * @param ObserverConstructor The constructor of an observer to add.
	 * Should create an instance inheriting from {@link module:engine/view/observer/observer~Observer}.
	 * @returns Added observer instance.
	 */ addObserver(ObserverConstructor) {
        let observer = this._observers.get(ObserverConstructor);
        if (observer) {
            return observer;
        }
        observer = new ObserverConstructor(this);
        this._observers.set(ObserverConstructor, observer);
        for (const [name, domElement] of this.domRoots){
            observer.observe(domElement, name);
        }
        observer.enable();
        return observer;
    }
    /**
	 * Returns observer of the given type or `undefined` if such observer has not been added yet.
	 *
	 * @param ObserverConstructor The constructor of an observer to get.
	 * @returns Observer instance or undefined.
	 */ getObserver(ObserverConstructor) {
        return this._observers.get(ObserverConstructor);
    }
    /**
	 * Disables all added observers.
	 */ disableObservers() {
        for (const observer of this._observers.values()){
            observer.disable();
        }
    }
    /**
	 * Enables all added observers.
	 */ enableObservers() {
        for (const observer of this._observers.values()){
            observer.enable();
        }
    }
    /**
	 * Scrolls the page viewport and {@link #domRoots} with their ancestors to reveal the
	 * caret, **if not already visible to the user**.
	 *
	 * **Note**: Calling this method fires the {@link module:engine/view/view~ViewScrollToTheSelectionEvent} event that
	 * allows custom behaviors.
	 *
	 * @param options Additional configuration of the scrolling behavior.
	 * @param options.viewportOffset A distance between the DOM selection and the viewport boundary to be maintained
	 * while scrolling to the selection (default is 20px). Setting this value to `0` will reveal the selection precisely at
	 * the viewport boundary.
	 * @param options.ancestorOffset A distance between the DOM selection and scrollable DOM root ancestor(s) to be maintained
	 * while scrolling to the selection (default is 20px). Setting this value to `0` will reveal the selection precisely at
	 * the scrollable ancestor(s) boundary.
	 * @param options.alignToTop When set `true`, the DOM selection will be aligned to the top of the viewport if not already visible
	 * (see `forceScroll` to learn more).
	 * @param options.forceScroll When set `true`, the DOM selection will be aligned to the top of the viewport and scrollable ancestors
	 * whether it is already visible or not. This option will only work when `alignToTop` is `true`.
	 */ scrollToTheSelection({ alignToTop, forceScroll, viewportOffset = 20, ancestorOffset = 20 } = {}) {
        const range = this.document.selection.getFirstRange();
        if (!range) {
            return;
        }
        // Clone to make sure properties like `viewportOffset` are not mutated in the event listeners.
        const originalArgs = cloneDeep({
            alignToTop,
            forceScroll,
            viewportOffset,
            ancestorOffset
        });
        if (typeof viewportOffset === 'number') {
            viewportOffset = {
                top: viewportOffset,
                bottom: viewportOffset,
                left: viewportOffset,
                right: viewportOffset
            };
        }
        const options = {
            target: this.domConverter.viewRangeToDom(range),
            viewportOffset,
            ancestorOffset,
            alignToTop,
            forceScroll
        };
        this.fire('scrollToTheSelection', options, originalArgs);
        scrollViewportToShowTarget(options);
    }
    /**
	 * It will focus DOM element representing {@link module:engine/view/editableelement~ViewEditableElement ViewEditableElement}
	 * that is currently having selection inside.
	 */ focus() {
        if (!this.document.isFocused) {
            const editable = this.document.selection.editableElement;
            if (editable) {
                this.domConverter.focus(editable);
                this.forceRender();
            }
        }
    }
    /**
	 * The `change()` method is the primary way of changing the view. You should use it to modify any node in the view tree.
	 * It makes sure that after all changes are made the view is rendered to the DOM (assuming that the view will be changed
	 * inside the callback). It prevents situations when the DOM is updated when the view state is not yet correct. It allows
	 * to nest calls one inside another and still performs a single rendering after all those changes are made.
	 * It also returns the return value of its callback.
	 *
	 * ```ts
	 * const text = view.change( writer => {
	 * 	const newText = writer.createText( 'foo' );
	 * 	writer.insert( position1, newText );
	 *
	 * 	view.change( writer => {
	 * 		writer.insert( position2, writer.createText( 'bar' ) );
	 * 	} );
	 *
	 * 	writer.remove( range );
	 *
	 * 	return newText;
	 * } );
	 * ```
	 *
	 * When the outermost change block is done and rendering to the DOM is over the
	 * {@link module:engine/view/view~EditingView#event:render `View#render`} event is fired.
	 *
	 * This method throws a `applying-view-changes-on-rendering` error when
	 * the change block is used after rendering to the DOM has started.
	 *
	 * @param callback Callback function which may modify the view.
	 * @returns Value returned by the callback.
	 */ change(callback) {
        if (this.isRenderingInProgress || this._postFixersInProgress) {
            /**
			 * Thrown when there is an attempt to make changes to the view tree when it is in incorrect state. This may
			 * cause some unexpected behaviour and inconsistency between the DOM and the view.
			 * This may be caused by:
			 *
			 * * calling {@link module:engine/view/view~EditingView#change} or {@link module:engine/view/view~EditingView#forceRender}
			 * during rendering process,
			 * * calling {@link module:engine/view/view~EditingView#change} or {@link module:engine/view/view~EditingView#forceRender}
			 * inside of {@link module:engine/view/document~ViewDocument#registerPostFixer post-fixer function}.
			 *
			 * @error cannot-change-view-tree
			 */ throw new CKEditorError('cannot-change-view-tree', this);
        }
        try {
            // Recursive call to view.change() method - execute listener immediately.
            if (this._ongoingChange) {
                return callback(this._writer);
            }
            // This lock will assure that all recursive calls to view.change() will end up in same block - one "render"
            // event for all nested calls.
            this._ongoingChange = true;
            const callbackResult = callback(this._writer);
            this._ongoingChange = false;
            // This lock is used by editing controller to render changes from outer most model.change() once. As plugins might call
            // view.change() inside model.change() block - this will ensures that postfixers and rendering are called once after all
            // changes. Also, we don't need to render anything if there're no changes since last rendering.
            if (!this._renderingDisabled && this._hasChangedSinceTheLastRendering) {
                this._postFixersInProgress = true;
                this.document._callPostFixers(this._writer);
                this._postFixersInProgress = false;
                this.fire('render');
            }
            return callbackResult;
        } catch (err) {
            // @if CK_DEBUG // throw err;
            /* istanbul ignore next -- @preserve */ CKEditorError.rethrowUnexpectedError(err, this);
        }
    }
    /**
	 * Forces rendering {@link module:engine/view/document~ViewDocument view document} to DOM. If any view changes are
	 * currently in progress, rendering will start after all {@link #change change blocks} are processed.
	 *
	 * Note that this method is dedicated for special cases. All view changes should be wrapped in the {@link #change}
	 * block and the view will automatically check whether it needs to render DOM or not.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `applying-view-changes-on-rendering` when
	 * trying to re-render when rendering to DOM has already started.
	 */ forceRender() {
        this._hasChangedSinceTheLastRendering = true;
        this.getObserver(FocusObserver).flush();
        this.change(()=>{});
    }
    /**
	 * Destroys this instance. Makes sure that all observers are destroyed and listeners removed.
	 */ destroy() {
        for (const observer of this._observers.values()){
            observer.destroy();
        }
        this.document.destroy();
        this.stopListening();
    }
    /**
	 * Creates position at the given location. The location can be specified as:
	 *
	 * * a {@link module:engine/view/position~ViewPosition position},
	 * * parent element and offset (offset defaults to `0`),
	 * * parent element and `'end'` (sets position at the end of that element),
	 * * {@link module:engine/view/item~ViewItem view item} and `'before'` or `'after'` (sets position before or after given view item).
	 *
	 * This method is a shortcut to other constructors such as:
	 *
	 * * {@link #createPositionBefore},
	 * * {@link #createPositionAfter},
	 *
	 * @param offset Offset or one of the flags. Used only when first parameter is a {@link module:engine/view/item~ViewItem view item}.
	 */ createPositionAt(itemOrPosition, offset) {
        return ViewPosition._createAt(itemOrPosition, offset);
    }
    /**
	 * Creates a new position after given view item.
	 *
	 * @param item View item after which the position should be located.
	 */ createPositionAfter(item) {
        return ViewPosition._createAfter(item);
    }
    /**
	 * Creates a new position before given view item.
	 *
	 * @param item View item before which the position should be located.
	 */ createPositionBefore(item) {
        return ViewPosition._createBefore(item);
    }
    /**
	 * Creates a range spanning from `start` position to `end` position.
	 *
	 * **Note:** This factory method creates it's own {@link module:engine/view/position~ViewPosition} instances basing on passed values.
	 *
	 * @param start Start position.
	 * @param end End position. If not set, range will be collapsed at `start` position.
	 */ createRange(start, end) {
        return new ViewRange(start, end);
    }
    /**
	 * Creates a range that starts before given {@link module:engine/view/item~ViewItem view item} and ends after it.
	 */ createRangeOn(item) {
        return ViewRange._createOn(item);
    }
    /**
	 * Creates a range inside an {@link module:engine/view/element~ViewElement element} which starts before the first child of
	 * that element and ends after the last child of that element.
	 *
	 * @param element Element which is a parent for the range.
	 */ createRangeIn(element) {
        return ViewRange._createIn(element);
    }
    createSelection(...args) {
        return new ViewSelection(...args);
    }
    /**
	 * Disables or enables rendering. If the flag is set to `true` then the rendering will be disabled.
	 * If the flag is set to `false` and if there was some change in the meantime, then the rendering action will be performed.
	 *
	 * @internal
	 * @param flag A flag indicates whether the rendering should be disabled.
	 */ _disableRendering(flag) {
        this._renderingDisabled = flag;
        if (flag == false) {
            // Render when you stop blocking rendering.
            this.change(()=>{});
        }
    }
    /**
	 * Renders all changes. In order to avoid triggering the observers (e.g. selection) all observers are disabled
	 * before rendering and re-enabled after that.
	 */ _render() {
        this.isRenderingInProgress = true;
        this.disableObservers();
        this._renderer.render();
        this.enableObservers();
        this.isRenderingInProgress = false;
    }
}

/**
 * @license Copyright (c) 2003-2025, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-licensing-options
 */ /**
 * @module engine/model/typecheckable
 */ class ModelTypeCheckable {
    /* istanbul ignore next -- @preserve */ is() {
        // There are a lot of overloads above.
        // Overriding method in derived classes remove them and only `is( type: string ): boolean` is visible which we don't want.
        // One option would be to copy them all to all classes, but that's ugly.
        // It's best when TypeScript compiler doesn't see those overloads, except the one in the top base class.
        // To overload a method, but not let the compiler see it, do after class definition:
        // `MyClass.prototype.is = function( type: string ) {...}`
        throw new Error('is() method is abstract');
    }
}

// @if CK_DEBUG_ENGINE // const { convertMapToStringifiedObject } = require( '../dev-utils/utils' );
/**
 * `ModelTextProxy` represents a part of {@link module:engine/model/text~ModelText text node}.
 *
 * Since {@link module:engine/model/position~ModelPosition positions} can be placed between characters of a text node,
 * {@link module:engine/model/range~ModelRange ranges} may contain only parts of text nodes.
 * When {@link module:engine/model/range~ModelRange#getItems getting items}
 * contained in such range, we need to represent a part of that text node, since returning the whole text node would be incorrect.
 * `ModelTextProxy` solves this issue.
 *
 * `ModelTextProxy` has an API similar to {@link module:engine/model/text~ModelText Text} and allows to do
 * most of the common tasks performed on model nodes.
 *
 * **Note:** Some `ModelTextProxy` instances may represent whole text node, not just a part of it.
 * See {@link module:engine/model/textproxy~ModelTextProxy#isPartial}.
 *
 * **Note:** `ModelTextProxy` is not an instance of {@link module:engine/model/node~ModelNode node}. Keep this in mind when using it as a
 * parameter of methods.
 *
 * **Note:** `ModelTextProxy` is a readonly interface. If you want to perform changes on model data represented by a `ModelTextProxy`
 * use {@link module:engine/model/writer~ModelWriter model writer API}.
 *
 * **Note:** `ModelTextProxy` instances are created on the fly, basing on the current state of model. Because of this, it is
 * highly unrecommended to store references to `ModelTextProxy` instances. `ModelTextProxy` instances are not refreshed when
 * model changes, so they might get invalidated. Instead, consider creating {@link module:engine/model/liveposition~ModelLivePosition live
 * position}.
 *
 * `ModelTextProxy` instances are created by {@link module:engine/model/treewalker~ModelTreeWalker model tree walker}.
 * You should not need to create an instance of this class by your own.
 */ class ModelTextProxy extends ModelTypeCheckable {
    /**
	 * Text node which part is represented by this text proxy.
	 */ textNode;
    /**
	 * Text data represented by this text proxy.
	 */ data;
    /**
	 * Offset in {@link module:engine/model/textproxy~ModelTextProxy#textNode text node} from which the text proxy starts.
	 */ offsetInText;
    /**
	 * Creates a text proxy.
	 *
	 * @internal
	 * @param textNode Text node which part is represented by this text proxy.
	 * @param offsetInText Offset in {@link module:engine/model/textproxy~ModelTextProxy#textNode text node} from which the text proxy
	 * starts.
	 * @param length Text proxy length, that is how many text node's characters, starting from `offsetInText` it represents.
	 */ constructor(textNode, offsetInText, length){
        super();
        this.textNode = textNode;
        if (offsetInText < 0 || offsetInText > textNode.offsetSize) {
            /**
			 * Given `offsetInText` value is incorrect.
			 *
			 * @error model-textproxy-wrong-offsetintext
			 */ throw new CKEditorError('model-textproxy-wrong-offsetintext', this);
        }
        if (length < 0 || offsetInText + length > textNode.offsetSize) {
            /**
			 * Given `length` value is incorrect.
			 *
			 * @error model-textproxy-wrong-length
			 */ throw new CKEditorError('model-textproxy-wrong-length', this);
        }
        this.data = textNode.data.substring(offsetInText, offsetInText + length);
        this.offsetInText = offsetInText;
    }
    /**
	 * Offset at which this text proxy starts in it's parent.
	 *
	 * @see module:engine/model/node~ModelNode#startOffset
	 */ get startOffset() {
        return this.textNode.startOffset !== null ? this.textNode.startOffset + this.offsetInText : null;
    }
    /**
	 * Offset size of this text proxy. Equal to the number of characters represented by the text proxy.
	 *
	 * @see module:engine/model/node~ModelNode#offsetSize
	 */ get offsetSize() {
        return this.data.length;
    }
    /**
	 * Offset at which this text proxy ends in it's parent.
	 *
	 * @see module:engine/model/node~ModelNode#endOffset
	 */ get endOffset() {
        return this.startOffset !== null ? this.startOffset + this.offsetSize : null;
    }
    /**
	 * Flag indicating whether `ModelTextProxy` instance covers only part of the original
	 * {@link module:engine/model/text~ModelText text node} (`true`) or the whole text node (`false`).
	 *
	 * This is `false` when text proxy starts at the very beginning of
	 * {@link module:engine/model/textproxy~ModelTextProxy#textNode textNode}
	 * ({@link module:engine/model/textproxy~ModelTextProxy#offsetInText offsetInText} equals `0`) and text proxy sizes is equal to
	 * text node size.
	 */ get isPartial() {
        return this.offsetSize !== this.textNode.offsetSize;
    }
    /**
	 * Parent of this text proxy, which is same as parent of text node represented by this text proxy.
	 */ get parent() {
        return this.textNode.parent;
    }
    /**
	 * Root of this text proxy, which is same as root of text node represented by this text proxy.
	 */ get root() {
        return this.textNode.root;
    }
    /**
	 * Gets path to this text proxy.
	 *
	 * @see module:engine/model/node~ModelNode#getPath
	 */ getPath() {
        const path = this.textNode.getPath();
        if (path.length > 0) {
            path[path.length - 1] += this.offsetInText;
        }
        return path;
    }
    /**
	 * Returns ancestors array of this text proxy.
	 *
	 * @param options Options object.
	 * @param options.includeSelf When set to `true` this text proxy will be also included in parent's array.
	 * @param options.parentFirst When set to `true`, array will be sorted from text proxy parent to root element,
	 * otherwise root element will be the first item in the array.
	 * @returns Array with ancestors.
	 */ getAncestors(options = {}) {
        const ancestors = [];
        let parent = options.includeSelf ? this : this.parent;
        while(parent){
            ancestors[options.parentFirst ? 'push' : 'unshift'](parent);
            parent = parent.parent;
        }
        return ancestors;
    }
    /**
	 * Checks if this text proxy has an attribute for given key.
	 *
	 * @param key Key of attribute to check.
	 * @returns `true` if attribute with given key is set on text proxy, `false` otherwise.
	 */ hasAttribute(key) {
        return this.textNode.hasAttribute(key);
    }
    /**
	 * Gets an attribute value for given key or `undefined` if that attribute is not set on text proxy.
	 *
	 * @param key Key of attribute to look for.
	 * @returns Attribute value or `undefined`.
	 */ getAttribute(key) {
        return this.textNode.getAttribute(key);
    }
    /**
	 * Returns iterator that iterates over this node's attributes. Attributes are returned as arrays containing two
	 * items. First one is attribute key and second is attribute value.
	 *
	 * This format is accepted by native `Map` object and also can be passed in `Node` constructor.
	 */ getAttributes() {
        return this.textNode.getAttributes();
    }
    /**
	 * Returns iterator that iterates over this node's attribute keys.
	 */ getAttributeKeys() {
        return this.textNode.getAttributeKeys();
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ModelTextProxy.prototype.is = function(type) {
    return type === '$textProxy' || type === 'model:$textProxy' || // This are legacy values kept for backward compatibility.
    type === 'textProxy' || type === 'model:textProxy';
};

/**
 * Position iterator class. It allows to iterate forward and backward over the document.
 */ class ModelTreeWalker {
    /**
	 * Walking direction. Defaults `'forward'`.
	 */ direction;
    /**
	 * Iterator boundaries.
	 *
	 * When the iterator is walking `'forward'` on the end of boundary or is walking `'backward'`
	 * on the start of boundary, then `{ done: true }` is returned.
	 *
	 * If boundaries are not defined they are set before first and after last child of the root node.
	 */ boundaries;
    /**
	 * Flag indicating whether all consecutive characters with the same attributes should be
	 * returned as one {@link module:engine/model/textproxy~ModelTextProxy} (`true`) or one by one (`false`).
	 */ singleCharacters;
    /**
	 * Flag indicating whether iterator should enter elements or not. If the iterator is shallow child nodes of any
	 * iterated node will not be returned along with `elementEnd` tag.
	 */ shallow;
    /**
	 * Flag indicating whether iterator should ignore `elementEnd` tags. If the option is true walker will not
	 * return a parent node of the start position. If this option is `true` each {@link module:engine/model/element~ModelElement} will
	 * be returned once, while if the option is `false` they might be returned twice:
	 * for `'elementStart'` and `'elementEnd'`.
	 */ ignoreElementEnd;
    /**
	 * Iterator position. This is always static position, even if the initial position was a
	 * {@link module:engine/model/liveposition~ModelLivePosition live position}. If start position is not defined then position depends
	 * on {@link #direction}. If direction is `'forward'` position starts form the beginning, when direction
	 * is `'backward'` position starts from the end.
	 */ _position;
    /**
	 * Start boundary cached for optimization purposes.
	 */ _boundaryStartParent;
    /**
	 * End boundary cached for optimization purposes.
	 */ _boundaryEndParent;
    /**
	 * Parent of the most recently visited node. Cached for optimization purposes.
	 */ _visitedParent;
    /**
	 * Creates a range iterator. All parameters are optional, but you have to specify either `boundaries` or `startPosition`.
	 *
	 * @param options Object with configuration.
	 */ constructor(options){
        if (!options || !options.boundaries && !options.startPosition) {
            /**
			 * Neither boundaries nor starting position of a `TreeWalker` have been defined.
			 *
			 * @error model-tree-walker-no-start-position
			 */ throw new CKEditorError('model-tree-walker-no-start-position', null);
        }
        const direction = options.direction || 'forward';
        if (direction != 'forward' && direction != 'backward') {
            /**
			 * Only `backward` and `forward` direction allowed.
			 *
			 * @error model-tree-walker-unknown-direction
			 */ throw new CKEditorError('model-tree-walker-unknown-direction', options, {
                direction
            });
        }
        this.direction = direction;
        this.boundaries = options.boundaries || null;
        if (options.startPosition) {
            this._position = options.startPosition.clone();
        } else {
            this._position = ModelPosition._createAt(this.boundaries[this.direction == 'backward' ? 'end' : 'start']);
        }
        // Reset position stickiness in case it was set to other value, as the stickiness is kept after cloning.
        this.position.stickiness = 'toNone';
        this.singleCharacters = !!options.singleCharacters;
        this.shallow = !!options.shallow;
        this.ignoreElementEnd = !!options.ignoreElementEnd;
        this._boundaryStartParent = this.boundaries ? this.boundaries.start.parent : null;
        this._boundaryEndParent = this.boundaries ? this.boundaries.end.parent : null;
        this._visitedParent = this.position.parent;
    }
    /**
	 * Iterable interface.
	 *
	 * @returns {Iterable.<module:engine/model/treewalker~ModelTreeWalkerValue>}
	 */ [Symbol.iterator]() {
        return this;
    }
    /**
	 * Iterator position. This is always static position, even if the initial position was a
	 * {@link module:engine/model/liveposition~ModelLivePosition live position}. If start position is not defined then position depends
	 * on {@link #direction}. If direction is `'forward'` position starts form the beginning, when direction
	 * is `'backward'` position starts from the end.
	 */ get position() {
        return this._position;
    }
    /**
	 * Moves {@link #position} in the {@link #direction} skipping values as long as the callback function returns `true`.
	 *
	 * For example:
	 *
	 * ```ts
	 * walker.skip( value => value.type == 'text' ); // <paragraph>[]foo</paragraph> -> <paragraph>foo[]</paragraph>
	 * walker.skip( () => true ); // Move the position to the end: <paragraph>[]foo</paragraph> -> <paragraph>foo</paragraph>[]
	 * walker.skip( () => false ); // Do not move the position.
	 * ```
	 *
	 * @param skip Callback function. Gets {@link module:engine/model/treewalker~ModelTreeWalkerValue} and should
	 * return `true` if the value should be skipped or `false` if not.
	 */ skip(skip) {
        let done, value, prevPosition, prevVisitedParent;
        do {
            prevPosition = this.position;
            prevVisitedParent = this._visitedParent;
            ({ done, value } = this.next());
        }while (!done && skip(value))
        if (!done) {
            this._position = prevPosition;
            this._visitedParent = prevVisitedParent;
        }
    }
    /**
	 * Moves tree walker {@link #position} to provided `position`. Tree walker will
	 * continue traversing from that position.
	 *
	 * Note: in contrary to {@link ~ModelTreeWalker#skip}, this method does not iterate over the nodes along the way.
	 * It simply sets the current tree walker position to a new one.
	 * From the performance standpoint, it is better to use {@link ~ModelTreeWalker#jumpTo} rather than {@link ~ModelTreeWalker#skip}.
	 *
	 * If the provided position is before the start boundary, the position will be
	 * set to the start boundary. If the provided position is after the end boundary,
	 * the position will be set to the end boundary.
	 * This is done to prevent the treewalker from traversing outside the boundaries.
	 *
	 * @param position Position to jump to.
	 */ jumpTo(position) {
        if (this._boundaryStartParent && position.isBefore(this.boundaries.start)) {
            position = this.boundaries.start;
        } else if (this._boundaryEndParent && position.isAfter(this.boundaries.end)) {
            position = this.boundaries.end;
        }
        this._position = position.clone();
        this._visitedParent = position.parent;
    }
    /**
	 * Gets the next tree walker's value.
	 */ next() {
        if (this.direction == 'forward') {
            return this._next();
        } else {
            return this._previous();
        }
    }
    /**
	 * Makes a step forward in model. Moves the {@link #position} to the next position and returns the encountered value.
	 */ _next() {
        const previousPosition = this.position;
        const position = this.position.clone();
        const parent = this._visitedParent;
        // We are at the end of the root.
        if (parent.parent === null && position.offset === parent.maxOffset) {
            return {
                done: true,
                value: undefined
            };
        }
        // We reached the walker boundary.
        if (parent === this._boundaryEndParent && position.offset == this.boundaries.end.offset) {
            return {
                done: true,
                value: undefined
            };
        }
        // Get node just after the current position.
        // Use a highly optimized version instead of checking the text node first and then getting the node after. See #6582.
        const textNodeAtPosition = getTextNodeAtPosition(position, parent);
        const node = textNodeAtPosition || getNodeAfterPosition(position, parent, textNodeAtPosition);
        if (node && node.is('model:element')) {
            if (!this.shallow) {
                // Manual operations on path internals for optimization purposes. Here and in the rest of the method.
                position.path.push(0);
                this._visitedParent = node;
            } else {
                // We are past the walker boundaries.
                if (this.boundaries && this.boundaries.end.isBefore(position)) {
                    return {
                        done: true,
                        value: undefined
                    };
                }
                position.offset++;
            }
            this._position = position;
            return formatReturnValue('elementStart', node, previousPosition, position, 1);
        }
        if (node && node.is('model:$text')) {
            let charactersCount;
            if (this.singleCharacters) {
                charactersCount = 1;
            } else {
                let offset = node.endOffset;
                if (this._boundaryEndParent == parent && this.boundaries.end.offset < offset) {
                    offset = this.boundaries.end.offset;
                }
                charactersCount = offset - position.offset;
            }
            const offsetInTextNode = position.offset - node.startOffset;
            const item = new ModelTextProxy(node, offsetInTextNode, charactersCount);
            position.offset += charactersCount;
            this._position = position;
            return formatReturnValue('text', item, previousPosition, position, charactersCount);
        }
        // `node` is not set, we reached the end of current `parent`.
        position.path.pop();
        position.offset++;
        this._position = position;
        this._visitedParent = parent.parent;
        if (this.ignoreElementEnd) {
            return this._next();
        }
        return formatReturnValue('elementEnd', parent, previousPosition, position);
    }
    /**
	 * Makes a step backward in model. Moves the {@link #position} to the previous position and returns the encountered value.
	 */ _previous() {
        const previousPosition = this.position;
        const position = this.position.clone();
        const parent = this._visitedParent;
        // We are at the beginning of the root.
        if (parent.parent === null && position.offset === 0) {
            return {
                done: true,
                value: undefined
            };
        }
        // We reached the walker boundary.
        if (parent == this._boundaryStartParent && position.offset == this.boundaries.start.offset) {
            return {
                done: true,
                value: undefined
            };
        }
        // Get node just before the current position.
        // Use a highly optimized version instead of checking the text node first and then getting the node before. See #6582.
        const positionParent = position.parent;
        const textNodeAtPosition = getTextNodeAtPosition(position, positionParent);
        const node = textNodeAtPosition || getNodeBeforePosition(position, positionParent, textNodeAtPosition);
        if (node && node.is('model:element')) {
            position.offset--;
            if (this.shallow) {
                this._position = position;
                return formatReturnValue('elementStart', node, previousPosition, position, 1);
            }
            position.path.push(node.maxOffset);
            this._position = position;
            this._visitedParent = node;
            if (this.ignoreElementEnd) {
                return this._previous();
            }
            return formatReturnValue('elementEnd', node, previousPosition, position);
        }
        if (node && node.is('model:$text')) {
            let charactersCount;
            if (this.singleCharacters) {
                charactersCount = 1;
            } else {
                let offset = node.startOffset;
                if (this._boundaryStartParent == parent && this.boundaries.start.offset > offset) {
                    offset = this.boundaries.start.offset;
                }
                charactersCount = position.offset - offset;
            }
            const offsetInTextNode = position.offset - node.startOffset;
            const item = new ModelTextProxy(node, offsetInTextNode - charactersCount, charactersCount);
            position.offset -= charactersCount;
            this._position = position;
            return formatReturnValue('text', item, previousPosition, position, charactersCount);
        }
        // `node` is not set, we reached the beginning of current `parent`.
        position.path.pop();
        this._position = position;
        this._visitedParent = parent.parent;
        return formatReturnValue('elementStart', parent, previousPosition, position, 1);
    }
}
function formatReturnValue(type, item, previousPosition, nextPosition, length) {
    return {
        done: false,
        value: {
            type,
            item,
            previousPosition,
            nextPosition,
            length
        }
    };
}

/**
 * Represents a position in the model tree.
 *
 * A position is represented by its {@link module:engine/model/position~ModelPosition#root} and
 * a {@link module:engine/model/position~ModelPosition#path} in that root.
 *
 * You can create position instances via its constructor or the `createPosition*()` factory methods of
 * {@link module:engine/model/model~Model} and {@link module:engine/model/writer~ModelWriter}.
 *
 * **Note:** Position is based on offsets, not indexes. This means that a position between two text nodes
 * `foo` and `bar` has offset `3`, not `1`. See {@link module:engine/model/position~ModelPosition#path} for more information.
 *
 * Since a position in the model is represented by a {@link module:engine/model/position~ModelPosition#root position root} and
 * {@link module:engine/model/position~ModelPosition#path position path} it is possible to create positions placed in non-existing places.
 * This requirement is important for operational transformation algorithms.
 *
 * Also, {@link module:engine/model/operation/operation~Operation operations}
 * kept in the {@link module:engine/model/document~ModelDocument#history document history}
 * are storing positions (and ranges) which were correct when those operations were applied, but may not be correct
 * after the document has changed.
 *
 * When changes are applied to the model, it may also happen that {@link module:engine/model/position~ModelPosition#parent position parent}
 * will change even if position path has not changed. Keep in mind, that if a position leads to non-existing element,
 * {@link module:engine/model/position~ModelPosition#parent} and some other properties and methods will throw errors.
 *
 * In most cases, position with wrong path is caused by an error in code, but it is sometimes needed, as described above.
 */ class ModelPosition extends ModelTypeCheckable {
    /**
	 * Root of the position path.
	 */ root;
    /**
	 * Position of the node in the tree. **Path contains offsets, not indexes.**
	 *
	 * Position can be placed before, after or in a {@link module:engine/model/node~ModelNode node} if that node has
	 * {@link module:engine/model/node~ModelNode#offsetSize} greater than `1`. Items in position path are
	 * {@link module:engine/model/node~ModelNode#startOffset starting offsets} of position ancestors, starting from direct root children,
	 * down to the position offset in it's parent.
	 *
	 * ```
	 * ROOT
	 *  |- P            before: [ 0 ]         after: [ 1 ]
	 *  |- UL           before: [ 1 ]         after: [ 2 ]
	 *     |- LI        before: [ 1, 0 ]      after: [ 1, 1 ]
	 *     |  |- foo    before: [ 1, 0, 0 ]   after: [ 1, 0, 3 ]
	 *     |- LI        before: [ 1, 1 ]      after: [ 1, 2 ]
	 *        |- bar    before: [ 1, 1, 0 ]   after: [ 1, 1, 3 ]
	 * ```
	 *
	 * `foo` and `bar` are representing {@link module:engine/model/text~ModelText text nodes}. Since text nodes has offset size
	 * greater than `1` you can place position offset between their start and end:
	 *
	 * ```
	 * ROOT
	 *  |- P
	 *  |- UL
	 *     |- LI
	 *     |  |- f^o|o  ^ has path: [ 1, 0, 1 ]   | has path: [ 1, 0, 2 ]
	 *     |- LI
	 *        |- b^a|r  ^ has path: [ 1, 1, 1 ]   | has path: [ 1, 1, 2 ]
	 * ```
	 */ path;
    /**
	 * Position stickiness. See {@link module:engine/model/position~ModelPositionStickiness}.
	 */ stickiness;
    /**
	 * Creates a position.
	 *
	 * @param root Root of the position.
	 * @param path Position path. See {@link module:engine/model/position~ModelPosition#path}.
	 * @param stickiness Position stickiness. See {@link module:engine/model/position~ModelPositionStickiness}.
	 */ constructor(root, path, stickiness = 'toNone'){
        super();
        if (!root.is('element') && !root.is('documentFragment')) {
            /**
			 * Position root is invalid.
			 *
			 * Positions can only be anchored in elements or document fragments.
			 *
			 * @error model-position-root-invalid
			 */ throw new CKEditorError('model-position-root-invalid', root);
        }
        if (!Array.isArray(path) || path.length === 0) {
            /**
			 * Position path must be an array with at least one item.
			 *
			 * @error model-position-path-incorrect-format
			 * @param {Array.<number>} path A path to the position.
			 */ throw new CKEditorError('model-position-path-incorrect-format', root, {
                path
            });
        }
        // Normalize the root and path when element (not root) is passed.
        if (root.is('rootElement')) {
            path = path.slice();
        } else {
            path = [
                ...root.getPath(),
                ...path
            ];
            root = root.root;
        }
        this.root = root;
        this.path = path;
        this.stickiness = stickiness;
    }
    /**
	 * Offset at which this position is located in its {@link module:engine/model/position~ModelPosition#parent parent}. It is equal
	 * to the last item in position {@link module:engine/model/position~ModelPosition#path path}.
	 *
	 * @type {Number}
	 */ get offset() {
        return this.path[this.path.length - 1];
    }
    set offset(newOffset) {
        this.path[this.path.length - 1] = newOffset;
    }
    /**
	 * Parent element of this position.
	 *
	 * Keep in mind that `parent` value is calculated when the property is accessed.
	 * If {@link module:engine/model/position~ModelPosition#path position path}
	 * leads to a non-existing element, `parent` property will throw error.
	 *
	 * Also it is a good idea to cache `parent` property if it is used frequently in an algorithm (i.e. in a long loop).
	 */ get parent() {
        let parent = this.root;
        for(let i = 0; i < this.path.length - 1; i++){
            parent = parent.getChildAtOffset(this.path[i]);
            if (!parent) {
                /**
				 * The position's path is incorrect. This means that a position does not point to
				 * a correct place in the tree and hence, some of its methods and getters cannot work correctly.
				 *
				 * **Note**: Unlike DOM and view positions, in the model, the
				 * {@link module:engine/model/position~ModelPosition#parent position's parent} is always an element or a document fragment.
				 * The last offset in the
				 * {@link module:engine/model/position~ModelPosition#path position's path} is the point in this element
				 * where this position points.
				 *
				 * Read more about model positions and offsets in
				 * the {@glink framework/architecture/editing-engine#indexes-and-offsets Editing engine architecture} guide.
				 *
				 * @error model-position-path-incorrect
				 * @param {module:engine/model/position~ModelPosition} position The incorrect position.
				 */ throw new CKEditorError('model-position-path-incorrect', this, {
                    position: this
                });
            }
        }
        if (parent.is('$text')) {
            throw new CKEditorError('model-position-path-incorrect', this, {
                position: this
            });
        }
        return parent;
    }
    /**
	 * Position {@link module:engine/model/position~ModelPosition#offset offset} converted to an index in position's parent node. It is
	 * equal to the {@link module:engine/model/node~ModelNode#index index} of a node after this position. If position is placed
	 * in text node, position index is equal to the index of that text node.
	 */ get index() {
        return this.parent.offsetToIndex(this.offset);
    }
    /**
	 * Returns {@link module:engine/model/text~ModelText text node} instance in which this position is placed or `null` if this
	 * position is not in a text node.
	 */ get textNode() {
        return getTextNodeAtPosition(this, this.parent);
    }
    /**
	 * Node directly after this position. Returns `null` if this position is at the end of its parent, or if it is in a text node.
	 */ get nodeAfter() {
        // Cache the parent and reuse for performance reasons. See #6579 and #6582.
        const parent = this.parent;
        return getNodeAfterPosition(this, parent, getTextNodeAtPosition(this, parent));
    }
    /**
	 * Node directly before this position. Returns `null` if this position is at the start of its parent, or if it is in a text node.
	 */ get nodeBefore() {
        // Cache the parent and reuse for performance reasons. See #6579 and #6582.
        const parent = this.parent;
        return getNodeBeforePosition(this, parent, getTextNodeAtPosition(this, parent));
    }
    /**
	 * Is `true` if position is at the beginning of its {@link module:engine/model/position~ModelPosition#parent parent}, `false` otherwise.
	 */ get isAtStart() {
        return this.offset === 0;
    }
    /**
	 * Is `true` if position is at the end of its {@link module:engine/model/position~ModelPosition#parent parent}, `false` otherwise.
	 */ get isAtEnd() {
        return this.offset == this.parent.maxOffset;
    }
    /**
	 * Checks whether the position is valid in current model tree, that is whether it points to an existing place in the model.
	 */ isValid() {
        if (this.offset < 0) {
            return false;
        }
        let parent = this.root;
        for(let i = 0; i < this.path.length - 1; i++){
            parent = parent.getChildAtOffset(this.path[i]);
            if (!parent) {
                return false;
            }
        }
        return this.offset <= parent.maxOffset;
    }
    /**
	 * Checks whether this position is before or after given position.
	 *
	 * This method is safe to use it on non-existing positions (for example during operational transformation).
	 */ compareWith(otherPosition) {
        if (this.root != otherPosition.root) {
            return 'different';
        }
        const result = compareArrays(this.path, otherPosition.path);
        switch(result){
            case 'same':
                return 'same';
            case 'prefix':
                return 'before';
            case 'extension':
                return 'after';
            default:
                return this.path[result] < otherPosition.path[result] ? 'before' : 'after';
        }
    }
    /**
	 * Gets the farthest position which matches the callback using
	 * {@link module:engine/model/treewalker~ModelTreeWalker TreeWalker}.
	 *
	 * For example:
	 *
	 * ```ts
	 * getLastMatchingPosition( value => value.type == 'text' );
	 * // <paragraph>[]foo</paragraph> -> <paragraph>foo[]</paragraph>
	 *
	 * getLastMatchingPosition( value => value.type == 'text', { direction: 'backward' } );
	 * // <paragraph>foo[]</paragraph> -> <paragraph>[]foo</paragraph>
	 *
	 * getLastMatchingPosition( value => false );
	 * // Do not move the position.
	 * ```
	 *
	 * @param skip Callback function. Gets {@link module:engine/model/treewalker~ModelTreeWalkerValue} and should
	 * return `true` if the value should be skipped or `false` if not.
	 * @param options Object with configuration options. See {@link module:engine/model/treewalker~ModelTreeWalker}.
	 *
	 * @returns The position after the last item which matches the `skip` callback test.
	 */ getLastMatchingPosition(skip, options = {}) {
        options.startPosition = this;
        const treeWalker = new ModelTreeWalker(options);
        treeWalker.skip(skip);
        return treeWalker.position;
    }
    /**
	 * Returns a path to this position's parent. Parent path is equal to position
	 * {@link module:engine/model/position~ModelPosition#path path} but without the last item.
	 *
	 * This method is safe to use it on non-existing positions (for example during operational transformation).
	 *
	 * @returns Path to the parent.
	 */ getParentPath() {
        return this.path.slice(0, -1);
    }
    /**
	 * Returns ancestors array of this position, that is this position's parent and its ancestors.
	 *
	 * @returns Array with ancestors.
	 */ getAncestors() {
        const parent = this.parent;
        if (parent.is('documentFragment')) {
            return [
                parent
            ];
        } else {
            return parent.getAncestors({
                includeSelf: true
            });
        }
    }
    /**
	 * Returns the parent element of the given name. Returns null if the position is not inside the desired parent.
	 *
	 * @param parentName The name of the parent element to find.
	 */ findAncestor(parentName) {
        const parent = this.parent;
        if (parent.is('element')) {
            return parent.findAncestor(parentName, {
                includeSelf: true
            });
        }
        return null;
    }
    /**
	 * Returns the slice of two position {@link #path paths} which is identical. The {@link #root roots}
	 * of these two paths must be identical.
	 *
	 * This method is safe to use it on non-existing positions (for example during operational transformation).
	 *
	 * @param position The second position.
	 * @returns The common path.
	 */ getCommonPath(position) {
        if (this.root != position.root) {
            return [];
        }
        // We find on which tree-level start and end have the lowest common ancestor
        const cmp = compareArrays(this.path, position.path);
        // If comparison returned string it means that arrays are same.
        const diffAt = typeof cmp == 'string' ? Math.min(this.path.length, position.path.length) : cmp;
        return this.path.slice(0, diffAt);
    }
    /**
	 * Returns an {@link module:engine/model/element~ModelElement} or {@link module:engine/model/documentfragment~ModelDocumentFragment}
	 * which is a common ancestor of both positions. The {@link #root roots} of these two positions must be identical.
	 *
	 * @param position The second position.
	 */ getCommonAncestor(position) {
        const ancestorsA = this.getAncestors();
        const ancestorsB = position.getAncestors();
        let i = 0;
        while(ancestorsA[i] == ancestorsB[i] && ancestorsA[i]){
            i++;
        }
        return i === 0 ? null : ancestorsA[i - 1];
    }
    /**
	 * Returns a new instance of `Position`, that has same {@link #parent parent} but it's offset
	 * is shifted by `shift` value (can be a negative value).
	 *
	 * This method is safe to use it on non-existing positions (for example during operational transformation).
	 *
	 * @param shift Offset shift. Can be a negative value.
	 * @returns Shifted position.
	 */ getShiftedBy(shift) {
        const shifted = this.clone();
        const offset = shifted.offset + shift;
        shifted.offset = offset < 0 ? 0 : offset;
        return shifted;
    }
    /**
	 * Checks whether this position is after given position.
	 *
	 * This method is safe to use it on non-existing positions (for example during operational transformation).
	 *
	 * @see module:engine/model/position~ModelPosition#isBefore
	 * @param  otherPosition Position to compare with.
	 * @returns True if this position is after given position.
	 */ isAfter(otherPosition) {
        return this.compareWith(otherPosition) == 'after';
    }
    /**
	 * Checks whether this position is before given position.
	 *
	 * **Note:** watch out when using negation of the value returned by this method, because the negation will also
	 * be `true` if positions are in different roots and you might not expect this. You should probably use
	 * `a.isAfter( b ) || a.isEqual( b )` or `!a.isBefore( p ) && a.root == b.root` in most scenarios. If your
	 * condition uses multiple `isAfter` and `isBefore` checks, build them so they do not use negated values, i.e.:
	 *
	 * ```ts
	 * if ( a.isBefore( b ) && c.isAfter( d ) ) {
	 * 	// do A.
	 * } else {
	 * 	// do B.
	 * }
	 * ```
	 *
	 * or, if you have only one if-branch:
	 *
	 * ```ts
	 * if ( !( a.isBefore( b ) && c.isAfter( d ) ) {
	 * 	// do B.
	 * }
	 * ```
	 *
	 * rather than:
	 *
	 * ```ts
	 * if ( !a.isBefore( b ) || && !c.isAfter( d ) ) {
	 * 	// do B.
	 * } else {
	 * 	// do A.
	 * }
	 * ```
	 *
	 * This method is safe to use it on non-existing positions (for example during operational transformation).
	 *
	 * @param otherPosition Position to compare with.
	 * @returns True if this position is before given position.
	 */ isBefore(otherPosition) {
        return this.compareWith(otherPosition) == 'before';
    }
    /**
	 * Checks whether this position is equal to given position.
	 *
	 * This method is safe to use it on non-existing positions (for example during operational transformation).
	 *
	 * @param otherPosition Position to compare with.
	 * @returns True if positions are same.
	 */ isEqual(otherPosition) {
        return this.compareWith(otherPosition) == 'same';
    }
    /**
	 * Checks whether this position is touching given position. Positions touch when there are no text nodes
	 * or empty nodes in a range between them. Technically, those positions are not equal but in many cases
	 * they are very similar or even indistinguishable.
	 *
	 * @param otherPosition Position to compare with.
	 * @returns True if positions touch.
	 */ isTouching(otherPosition) {
        if (this.root !== otherPosition.root) {
            return false;
        }
        const commonLevel = Math.min(this.path.length, otherPosition.path.length);
        for(let level = 0; level < commonLevel; level++){
            const diff = this.path[level] - otherPosition.path[level];
            // Positions are spread by a node, so they are not touching.
            if (diff < -1 || diff > 1) {
                return false;
            } else if (diff === 1) {
                // `otherPosition` is on the left.
                // `this` is on the right.
                return checkTouchingBranch(otherPosition, this, level);
            } else if (diff === -1) {
                // `this` is on the left.
                // `otherPosition` is on the right.
                return checkTouchingBranch(this, otherPosition, level);
            }
        // `diff === 0`.
        // Positions are inside the same element on this level, compare deeper.
        }
        // If we ended up here, it means that positions paths have the same beginning.
        // If the paths have the same length, then it means that they are identical, so the positions are same.
        if (this.path.length === otherPosition.path.length) {
            return true;
        } else if (this.path.length > otherPosition.path.length) {
            return checkOnlyZeroes(this.path, commonLevel);
        } else {
            return checkOnlyZeroes(otherPosition.path, commonLevel);
        }
    }
    /**
	 * Checks if two positions are in the same parent.
	 *
	 * This method is safe to use it on non-existing positions (for example during operational transformation).
	 *
	 * @param position Position to compare with.
	 * @returns `true` if positions have the same parent, `false` otherwise.
	 */ hasSameParentAs(position) {
        if (this.root !== position.root) {
            return false;
        }
        const thisParentPath = this.getParentPath();
        const posParentPath = position.getParentPath();
        return compareArrays(thisParentPath, posParentPath) == 'same';
    }
    /**
	 * Returns a copy of this position that is transformed by given `operation`.
	 *
	 * The new position's parameters are updated accordingly to the effect of the `operation`.
	 *
	 * For example, if `n` nodes are inserted before the position, the returned position {@link ~ModelPosition#offset} will be
	 * increased by `n`. If the position was in a merged element, it will be accordingly moved to the new element, etc.
	 *
	 * This method is safe to use it on non-existing positions (for example during operational transformation).
	 *
	 * @param operation Operation to transform by.
	 * @returns Transformed position.
	 */ getTransformedByOperation(operation) {
        let result;
        switch(operation.type){
            case 'insert':
                result = this._getTransformedByInsertOperation(operation);
                break;
            case 'move':
            case 'remove':
            case 'reinsert':
                result = this._getTransformedByMoveOperation(operation);
                break;
            case 'split':
                result = this._getTransformedBySplitOperation(operation);
                break;
            case 'merge':
                result = this._getTransformedByMergeOperation(operation);
                break;
            default:
                result = ModelPosition._createAt(this);
                break;
        }
        return result;
    }
    /**
	 * Returns a copy of this position transformed by an insert operation.
	 *
	 * @internal
	 */ _getTransformedByInsertOperation(operation) {
        return this._getTransformedByInsertion(operation.position, operation.howMany);
    }
    /**
	 * Returns a copy of this position transformed by a move operation.
	 *
	 * @internal
	 */ _getTransformedByMoveOperation(operation) {
        return this._getTransformedByMove(operation.sourcePosition, operation.targetPosition, operation.howMany);
    }
    /**
	 * Returns a copy of this position transformed by a split operation.
	 *
	 * @internal
	 */ _getTransformedBySplitOperation(operation) {
        const movedRange = operation.movedRange;
        const isContained = movedRange.containsPosition(this) || movedRange.start.isEqual(this) && this.stickiness == 'toNext';
        if (isContained) {
            return this._getCombined(operation.splitPosition, operation.moveTargetPosition);
        } else {
            if (operation.graveyardPosition) {
                return this._getTransformedByMove(operation.graveyardPosition, operation.insertionPosition, 1);
            } else {
                return this._getTransformedByInsertion(operation.insertionPosition, 1);
            }
        }
    }
    /**
	 * Returns a copy of this position transformed by merge operation.
	 *
	 * @internal
	 */ _getTransformedByMergeOperation(operation) {
        const movedRange = operation.movedRange;
        const isContained = movedRange.containsPosition(this) || movedRange.start.isEqual(this);
        let pos;
        if (isContained) {
            pos = this._getCombined(operation.sourcePosition, operation.targetPosition);
            if (operation.sourcePosition.isBefore(operation.targetPosition)) {
                // Above happens during OT when the merged element is moved before the merged-to element.
                pos = pos._getTransformedByDeletion(operation.deletionPosition, 1);
            }
        } else if (this.isEqual(operation.deletionPosition)) {
            pos = ModelPosition._createAt(operation.deletionPosition);
        } else {
            pos = this._getTransformedByMove(operation.deletionPosition, operation.graveyardPosition, 1);
        }
        return pos;
    }
    /**
	 * Returns a copy of this position that is updated by removing `howMany` nodes starting from `deletePosition`.
	 * It may happen that this position is in a removed node. If that is the case, `null` is returned instead.
	 *
	 * @internal
	 * @param deletePosition Position before the first removed node.
	 * @param howMany How many nodes are removed.
	 * @returns Transformed position or `null`.
	 */ _getTransformedByDeletion(deletePosition, howMany) {
        const transformed = ModelPosition._createAt(this);
        // This position can't be affected if deletion was in a different root.
        if (this.root != deletePosition.root) {
            return transformed;
        }
        if (compareArrays(deletePosition.getParentPath(), this.getParentPath()) == 'same') {
            // If nodes are removed from the node that is pointed by this position...
            if (deletePosition.offset < this.offset) {
                // And are removed from before an offset of that position...
                if (deletePosition.offset + howMany > this.offset) {
                    // Position is in removed range, it's no longer in the tree.
                    return null;
                } else {
                    // Decrement the offset accordingly.
                    transformed.offset -= howMany;
                }
            }
        } else if (compareArrays(deletePosition.getParentPath(), this.getParentPath()) == 'prefix') {
            // If nodes are removed from a node that is on a path to this position...
            const i = deletePosition.path.length - 1;
            if (deletePosition.offset <= this.path[i]) {
                // And are removed from before next node of that path...
                if (deletePosition.offset + howMany > this.path[i]) {
                    // If the next node of that path is removed return null
                    // because the node containing this position got removed.
                    return null;
                } else {
                    // Otherwise, decrement index on that path.
                    transformed.path[i] -= howMany;
                }
            }
        }
        return transformed;
    }
    /**
	 * Returns a copy of this position that is updated by inserting `howMany` nodes at `insertPosition`.
	 *
	 * @internal
	 * @param insertPosition Position where nodes are inserted.
	 * @param howMany How many nodes are inserted.
	 * @returns Transformed position.
	 */ _getTransformedByInsertion(insertPosition, howMany) {
        const transformed = ModelPosition._createAt(this);
        // This position can't be affected if insertion was in a different root.
        if (this.root != insertPosition.root) {
            return transformed;
        }
        if (compareArrays(insertPosition.getParentPath(), this.getParentPath()) == 'same') {
            // If nodes are inserted in the node that is pointed by this position...
            if (insertPosition.offset < this.offset || insertPosition.offset == this.offset && this.stickiness != 'toPrevious') {
                // And are inserted before an offset of that position...
                // "Push" this positions offset.
                transformed.offset += howMany;
            }
        } else if (compareArrays(insertPosition.getParentPath(), this.getParentPath()) == 'prefix') {
            // If nodes are inserted in a node that is on a path to this position...
            const i = insertPosition.path.length - 1;
            if (insertPosition.offset <= this.path[i]) {
                // And are inserted before next node of that path...
                // "Push" the index on that path.
                transformed.path[i] += howMany;
            }
        }
        return transformed;
    }
    /**
	 * Returns a copy of this position that is updated by moving `howMany` nodes from `sourcePosition` to `targetPosition`.
	 *
	 * @internal
	 * @param sourcePosition Position before the first element to move.
	 * @param targetPosition Position where moved elements will be inserted.
	 * @param howMany How many consecutive nodes to move, starting from `sourcePosition`.
	 * @returns Transformed position.
	 */ _getTransformedByMove(sourcePosition, targetPosition, howMany) {
        // Update target position, as it could be affected by nodes removal.
        targetPosition = targetPosition._getTransformedByDeletion(sourcePosition, howMany);
        if (sourcePosition.isEqual(targetPosition)) {
            // If `targetPosition` is equal to `sourcePosition` this isn't really any move. Just return position as it is.
            return ModelPosition._createAt(this);
        }
        // Moving a range removes nodes from their original position. We acknowledge this by proper transformation.
        const transformed = this._getTransformedByDeletion(sourcePosition, howMany);
        const isMoved = transformed === null || sourcePosition.isEqual(this) && this.stickiness == 'toNext' || sourcePosition.getShiftedBy(howMany).isEqual(this) && this.stickiness == 'toPrevious';
        if (isMoved) {
            // This position is inside moved range (or sticks to it).
            // In this case, we calculate a combination of this position, move source position and target position.
            return this._getCombined(sourcePosition, targetPosition);
        } else {
            // This position is not inside a removed range.
            //
            // In next step, we simply reflect inserting `howMany` nodes, which might further affect the position.
            return transformed._getTransformedByInsertion(targetPosition, howMany);
        }
    }
    /**
	 * Returns a new position that is a combination of this position and given positions.
	 *
	 * The combined position is a copy of this position transformed by moving a range starting at `source` position
	 * to the `target` position. It is expected that this position is inside the moved range.
	 *
	 * Example:
	 *
	 * ```ts
	 * let original = model.createPositionFromPath( root, [ 2, 3, 1 ] );
	 * let source = model.createPositionFromPath( root, [ 2, 2 ] );
	 * let target = model.createPositionFromPath( otherRoot, [ 1, 1, 3 ] );
	 * original._getCombined( source, target ); // path is [ 1, 1, 4, 1 ], root is `otherRoot`
	 * ```
	 *
	 * Explanation:
	 *
	 * We have a position `[ 2, 3, 1 ]` and move some nodes from `[ 2, 2 ]` to `[ 1, 1, 3 ]`. The original position
	 * was inside moved nodes and now should point to the new place. The moved nodes will be after
	 * positions `[ 1, 1, 3 ]`, `[ 1, 1, 4 ]`, `[ 1, 1, 5 ]`. Since our position was in the second moved node,
	 * the transformed position will be in a sub-tree of a node at `[ 1, 1, 4 ]`. Looking at original path, we
	 * took care of `[ 2, 3 ]` part of it. Now we have to add the rest of the original path to the transformed path.
	 * Finally, the transformed position will point to `[ 1, 1, 4, 1 ]`.
	 *
	 * @internal
	 * @param source Beginning of the moved range.
	 * @param target Position where the range is moved.
	 * @returns Combined position.
	 */ _getCombined(source, target) {
        const i = source.path.length - 1;
        // The first part of a path to combined position is a path to the place where nodes were moved.
        const combined = ModelPosition._createAt(target);
        combined.stickiness = this.stickiness;
        // Then we have to update the rest of the path.
        // Fix the offset because this position might be after `from` position and we have to reflect that.
        combined.offset = combined.offset + this.path[i] - source.offset;
        // Then, add the rest of the path.
        // If this position is at the same level as `from` position nothing will get added.
        combined.path = [
            ...combined.path,
            ...this.path.slice(i + 1)
        ];
        return combined;
    }
    /**
	 * @inheritDoc
	 */ toJSON() {
        return {
            root: this.root.toJSON(),
            path: Array.from(this.path),
            stickiness: this.stickiness
        };
    }
    /**
	 * Returns a new position that is equal to current position.
	 */ clone() {
        return new this.constructor(this.root, this.path, this.stickiness);
    }
    /**
	 * Creates position at the given location. The location can be specified as:
	 *
	 * * a {@link module:engine/model/position~ModelPosition position},
	 * * parent element and offset (offset defaults to `0`),
	 * * parent element and `'end'` (sets position at the end of that element),
	 * * {@link module:engine/model/item~ModelItem model item} and `'before'` or `'after'` (sets position before or after given model item).
	 *
	 * This method is a shortcut to other factory methods such as:
	 *
	 * * {@link module:engine/model/position~ModelPosition._createBefore},
	 * * {@link module:engine/model/position~ModelPosition._createAfter}.
	 *
	 * @internal
	 * @param offset Offset or one of the flags. Used only when the first parameter
	 * is a {@link module:engine/model/item~ModelItem model item}.
	 * @param stickiness Position stickiness. Used only when the first parameter is a {@link module:engine/model/item~ModelItem model item}.
	 */ static _createAt(itemOrPosition, offset, stickiness = 'toNone') {
        if (itemOrPosition.is('model:position')) {
            return new ModelPosition(itemOrPosition.root, itemOrPosition.path, itemOrPosition.stickiness);
        } else {
            const node = itemOrPosition;
            if (offset == 'end') {
                offset = node.maxOffset;
            } else if (offset == 'before') {
                return this._createBefore(node, stickiness);
            } else if (offset == 'after') {
                return this._createAfter(node, stickiness);
            } else if (offset !== 0 && !offset) {
                /**
				 * {@link module:engine/model/model~Model#createPositionAt `Model#createPositionAt()`}
				 * requires the offset to be specified when the first parameter is a model item.
				 *
				 * @error model-createpositionat-offset-required
				 */ throw new CKEditorError('model-createpositionat-offset-required', [
                    this,
                    itemOrPosition
                ]);
            }
            if (!node.is('element') && !node.is('documentFragment')) {
                /**
				 * Position parent have to be a model element or model document fragment.
				 *
				 * @error model-position-parent-incorrect
				 */ throw new CKEditorError('model-position-parent-incorrect', [
                    this,
                    itemOrPosition
                ]);
            }
            const path = node.getPath();
            path.push(offset);
            return new this(node.root, path, stickiness);
        }
    }
    /**
	 * Creates a new position, after given {@link module:engine/model/item~ModelItem model item}.
	 *
	 * @internal
	 * @param item Item after which the position should be placed.
	 * @param stickiness Position stickiness.
	 */ static _createAfter(item, stickiness) {
        if (!item.parent) {
            /**
			 * You cannot make a position after a root element.
			 *
			 * @error model-position-after-root
			 * @param {module:engine/model/rootelement~ModelRootElement} root The root element..
			 */ throw new CKEditorError('model-position-after-root', [
                this,
                item
            ], {
                root: item
            });
        }
        return this._createAt(item.parent, item.endOffset, stickiness);
    }
    /**
	 * Creates a new position, before the given {@link module:engine/model/item~ModelItem model item}.
	 *
	 * @internal
	 * @param item Item before which the position should be placed.
	 * @param stickiness Position stickiness.
	 */ static _createBefore(item, stickiness) {
        if (!item.parent) {
            /**
			 * You cannot make a position before a root element.
			 *
			 * @error model-position-before-root
			 * @param {module:engine/model/rootelement~ModelRootElement} root The root element..
			 */ throw new CKEditorError('model-position-before-root', item, {
                root: item
            });
        }
        return this._createAt(item.parent, item.startOffset, stickiness);
    }
    /**
	 * Creates a `Position` instance from given plain object (i.e. parsed JSON string).
	 *
	 * @param json Plain object to be converted to `Position`.
	 * @param doc Document object that will be position owner.
	 * @returns `Position` instance created using given plain object.
	 */ static fromJSON(json, doc) {
        if (json.root === '$graveyard') {
            const pos = new ModelPosition(doc.graveyard, json.path);
            pos.stickiness = json.stickiness;
            return pos;
        }
        if (!doc.getRoot(json.root)) {
            /**
			 * Cannot create position for document. Root with specified name does not exist.
			 *
			 * @error model-position-fromjson-no-root
			 * @param {string} rootName The root name.
			 */ throw new CKEditorError('model-position-fromjson-no-root', doc, {
                rootName: json.root
            });
        }
        return new ModelPosition(doc.getRoot(json.root), json.path, json.stickiness);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ModelPosition.prototype.is = function(type) {
    return type === 'position' || type === 'model:position';
};
/**
 * Returns a text node at the given position.
 *
 * This is a helper function optimized to reuse the position parent instance for performance reasons.
 *
 * Normally, you should use {@link module:engine/model/position~ModelPosition#textNode `Position#textNode`}.
 * If you start hitting performance issues with {@link module:engine/model/position~ModelPosition#parent `Position#parent`}
 * check if your algorithm does not access it multiple times (which can happen directly or indirectly via other position properties).
 *
 * See https://github.com/ckeditor/ckeditor5/issues/6579.
 *
 * See also:
 *
 * * {@link module:engine/model/position~getNodeAfterPosition}
 * * {@link module:engine/model/position~getNodeBeforePosition}
 *
 * @param position
 * @param positionParent The parent of the given position.
 * @internal
 */ function getTextNodeAtPosition(position, positionParent) {
    const node = positionParent.getChildAtOffset(position.offset);
    if (node && node.is('$text') && node.startOffset < position.offset) {
        return node;
    }
    return null;
}
/**
 * Returns the node after the given position.
 *
 * This is a helper function optimized to reuse the position parent instance and the calculation of the text node at the
 * specific position for performance reasons.
 *
 * Normally, you should use {@link module:engine/model/position~ModelPosition#nodeAfter `Position#nodeAfter`}.
 * If you start hitting performance issues with {@link module:engine/model/position~ModelPosition#parent `Position#parent`} and/or
 * {@link module:engine/model/position~ModelPosition#textNode `Position#textNode`}
 * check if your algorithm does not access those properties multiple times
 * (which can happen directly or indirectly via other position properties).
 *
 * See https://github.com/ckeditor/ckeditor5/issues/6579 and https://github.com/ckeditor/ckeditor5/issues/6582.
 *
 * See also:
 *
 * * {@link module:engine/model/position~getTextNodeAtPosition}
 * * {@link module:engine/model/position~getNodeBeforePosition}
 *
 * @param position Position to check.
 * @param positionParent The parent of the given position.
 * @param textNode Text node at the given position.
 * @internal
 */ function getNodeAfterPosition(position, positionParent, textNode) {
    if (textNode !== null) {
        return null;
    }
    return positionParent.getChildAtOffset(position.offset);
}
/**
 * Returns the node before the given position.
 *
 * Refer to {@link module:engine/model/position~getNodeBeforePosition} for documentation on when to use this util method.
 *
 * See also:
 *
 * * {@link module:engine/model/position~getTextNodeAtPosition}
 * * {@link module:engine/model/position~getNodeAfterPosition}
 *
 * @param position Position to check.
 * @param positionParent The parent of the given position.
 * @param textNode Text node at the given position.
 * @internal
 */ function getNodeBeforePosition(position, positionParent, textNode) {
    if (textNode !== null) {
        return null;
    }
    return positionParent.getChild(positionParent.offsetToIndex(position.offset) - 1);
}
/**
 * This is a helper function for `Position#isTouching()`.
 *
 * It checks whether to given positions are touching, considering that they have the same root and paths
 * until given level, and at given level they differ by 1 (so they are branching at `level` point).
 *
 * The exact requirements for touching positions are described in `Position#isTouching()` and also
 * in the body of this function.
 *
 * @param left Position "on the left" (it is before `right`).
 * @param right Position "on the right" (it is after `left`).
 * @param level Level on which the positions are different.
 */ function checkTouchingBranch(left, right, level) {
    if (level + 1 === left.path.length) {
        // Left position does not have any more entries after the point where the positions differ.
        // [ 2 ] vs [ 3 ]
        // [ 2 ] vs [ 3, 0, 0 ]
        // The positions are spread by node at [ 2 ].
        return false;
    }
    if (!checkOnlyZeroes(right.path, level + 1)) {
        // Right position does not have only zeroes, so we have situation like:
        // [ 2, maxOffset ] vs [ 3, 1 ]
        // [ 2, maxOffset ] vs [ 3, 1, 0, 0 ]
        // The positions are spread by node at [ 3, 0 ].
        return false;
    }
    if (!checkOnlyMaxOffset(left, level + 1)) {
        // Left position does not have only max offsets, so we have situation like:
        // [ 2, 4 ] vs [ 3 ]
        // [ 2, 4 ] vs [ 3, 0, 0 ]
        // The positions are spread by node at [ 2, 5 ].
        return false;
    }
    // Left position has only max offsets and right position has only zeroes or nothing.
    // [ 2, maxOffset ] vs [ 3 ]
    // [ 2, maxOffset, maxOffset ] vs [ 3, 0 ]
    // There are not elements between positions. The positions are touching.
    return true;
}
/**
 * Checks whether for given array, starting from given index until the end of the array, all items are `0`s.
 *
 * This is a helper function for `Position#isTouching()`.
 */ function checkOnlyZeroes(arr, idx) {
    while(idx < arr.length){
        if (arr[idx] !== 0) {
            return false;
        }
        idx++;
    }
    return true;
}
/**
 * Checks whether for given position, starting from given path level, whether the position is at the end of
 * its parent and whether each element on the path to the position is also at at the end of its parent.
 *
 * This is a helper function for `Position#isTouching()`.
 */ function checkOnlyMaxOffset(pos, level) {
    let parent = pos.parent;
    let idx = pos.path.length - 1;
    let add = 0;
    while(idx >= level){
        if (pos.path[idx] + add !== parent.maxOffset) {
            return false;
        }
        // After the first check, we "go up", and check whether the position's parent-parent is the last element.
        // However, we need to add 1 to the value in the path to "simulate" moving the path after the parent.
        // It happens just once.
        add = 1;
        idx--;
        parent = parent.parent;
    }
    return true;
}

/**
 * Represents a range in the model tree.
 *
 * A range is defined by its {@link module:engine/model/range~ModelRange#start} and {@link module:engine/model/range~ModelRange#end}
 * positions.
 *
 * You can create range instances via its constructor or the `createRange*()` factory methods of
 * {@link module:engine/model/model~Model} and {@link module:engine/model/writer~ModelWriter}.
 */ class ModelRange extends ModelTypeCheckable {
    /**
	 * Start position.
	 */ start;
    /**
	 * End position.
	 */ end;
    /**
	 * Creates a range spanning from `start` position to `end` position.
	 *
	 * @param start The start position.
	 * @param end The end position. If not set, the range will be collapsed at the `start` position.
	 */ constructor(start, end){
        super();
        this.start = ModelPosition._createAt(start);
        this.end = end ? ModelPosition._createAt(end) : ModelPosition._createAt(start);
        // If the range is collapsed, treat in a similar way as a position and set its boundaries stickiness to 'toNone'.
        // In other case, make the boundaries stick to the "inside" of the range.
        this.start.stickiness = this.isCollapsed ? 'toNone' : 'toNext';
        this.end.stickiness = this.isCollapsed ? 'toNone' : 'toPrevious';
    }
    /**
	 * Iterable interface.
	 *
	 * Iterates over all {@link module:engine/model/item~ModelItem items} that are in this range and returns
	 * them together with additional information like length or {@link module:engine/model/position~ModelPosition positions},
	 * grouped as {@link module:engine/model/treewalker~ModelTreeWalkerValue}.
	 * It iterates over all {@link module:engine/model/textproxy~ModelTextProxy text contents} that are inside the range
	 * and all the {@link module:engine/model/element~ModelElement}s that are entered into when iterating over this range.
	 *
	 * This iterator uses {@link module:engine/model/treewalker~ModelTreeWalker} with `boundaries` set to this range
	 * and `ignoreElementEnd` option set to `true`.
	 */ *[Symbol.iterator]() {
        yield* new ModelTreeWalker({
            boundaries: this,
            ignoreElementEnd: true
        });
    }
    /**
	 * Describes whether the range is collapsed, that is if {@link #start} and
	 * {@link #end} positions are equal.
	 */ get isCollapsed() {
        return this.start.isEqual(this.end);
    }
    /**
	 * Describes whether this range is flat, that is if {@link #start} position and
	 * {@link #end} position are in the same {@link module:engine/model/position~ModelPosition#parent}.
	 */ get isFlat() {
        const startParentPath = this.start.getParentPath();
        const endParentPath = this.end.getParentPath();
        return compareArrays(startParentPath, endParentPath) == 'same';
    }
    /**
	 * Range root element.
	 */ get root() {
        return this.start.root;
    }
    /**
	 * Checks whether this range contains given {@link module:engine/model/position~ModelPosition position}.
	 *
	 * @param position Position to check.
	 * @returns `true` if given {@link module:engine/model/position~ModelPosition position} is contained
	 * in this range,`false` otherwise.
	 */ containsPosition(position) {
        return position.isAfter(this.start) && position.isBefore(this.end);
    }
    /**
	 * Checks whether this range contains given {@link ~ModelRange range}.
	 *
	 * @param otherRange Range to check.
	 * @param loose Whether the check is loose or strict. If the check is strict (`false`), compared range cannot
	 * start or end at the same position as this range boundaries. If the check is loose (`true`), compared range can start, end or
	 * even be equal to this range. Note that collapsed ranges are always compared in strict mode.
	 * @returns {Boolean} `true` if given {@link ~ModelRange range} boundaries are contained by this range, `false` otherwise.
	 */ containsRange(otherRange, loose = false) {
        if (otherRange.isCollapsed) {
            loose = false;
        }
        const containsStart = this.containsPosition(otherRange.start) || loose && this.start.isEqual(otherRange.start);
        const containsEnd = this.containsPosition(otherRange.end) || loose && this.end.isEqual(otherRange.end);
        return containsStart && containsEnd;
    }
    /**
	 * Checks whether given {@link module:engine/model/item~ModelItem} is inside this range.
	 */ containsItem(item) {
        const pos = ModelPosition._createBefore(item);
        return this.containsPosition(pos) || this.start.isEqual(pos);
    }
    /**
	 * Two ranges are equal if their {@link #start} and {@link #end} positions are equal.
	 *
	 * @param otherRange Range to compare with.
	 * @returns `true` if ranges are equal, `false` otherwise.
	 */ isEqual(otherRange) {
        return this.start.isEqual(otherRange.start) && this.end.isEqual(otherRange.end);
    }
    /**
	 * Checks and returns whether this range intersects with given range.
	 *
	 * @param otherRange Range to compare with.
	 * @returns `true` if ranges intersect, `false` otherwise.
	 */ isIntersecting(otherRange) {
        return this.start.isBefore(otherRange.end) && this.end.isAfter(otherRange.start);
    }
    /**
	 * Computes which part(s) of this {@link ~ModelRange range} is not a part of given {@link ~ModelRange range}.
	 * Returned array contains zero, one or two {@link ~ModelRange ranges}.
	 *
	 * Examples:
	 *
	 * ```ts
	 * let range = model.createRange(
	 * 	model.createPositionFromPath( root, [ 2, 7 ] ),
	 * 	model.createPositionFromPath( root, [ 4, 0, 1 ] )
	 * );
	 * let otherRange = model.createRange( model.createPositionFromPath( root, [ 1 ] ), model.createPositionFromPath( root, [ 5 ] ) );
	 * let transformed = range.getDifference( otherRange );
	 * // transformed array has no ranges because `otherRange` contains `range`
	 *
	 * otherRange = model.createRange( model.createPositionFromPath( root, [ 1 ] ), model.createPositionFromPath( root, [ 3 ] ) );
	 * transformed = range.getDifference( otherRange );
	 * // transformed array has one range: from [ 3 ] to [ 4, 0, 1 ]
	 *
	 * otherRange = model.createRange( model.createPositionFromPath( root, [ 3 ] ), model.createPositionFromPath( root, [ 4 ] ) );
	 * transformed = range.getDifference( otherRange );
	 * // transformed array has two ranges: from [ 2, 7 ] to [ 3 ] and from [ 4 ] to [ 4, 0, 1 ]
	 * ```
	 *
	 * @param otherRange Range to differentiate against.
	 * @returns The difference between ranges.
	 */ getDifference(otherRange) {
        const ranges = [];
        if (this.isIntersecting(otherRange)) {
            // Ranges intersect.
            if (this.containsPosition(otherRange.start)) {
                // Given range start is inside this range. This means that we have to
                // add shrunken range - from the start to the middle of this range.
                ranges.push(new ModelRange(this.start, otherRange.start));
            }
            if (this.containsPosition(otherRange.end)) {
                // Given range end is inside this range. This means that we have to
                // add shrunken range - from the middle of this range to the end.
                ranges.push(new ModelRange(otherRange.end, this.end));
            }
        } else {
            // Ranges do not intersect, return the original range.
            ranges.push(new ModelRange(this.start, this.end));
        }
        return ranges;
    }
    /**
	 * Returns an intersection of this {@link ~ModelRange range} and given {@link ~ModelRange range}.
	 * Intersection is a common part of both of those ranges. If ranges has no common part, returns `null`.
	 *
	 * Examples:
	 *
	 * ```ts
	 * let range = model.createRange(
	 * 	model.createPositionFromPath( root, [ 2, 7 ] ),
	 * 	model.createPositionFromPath( root, [ 4, 0, 1 ] )
	 * );
	 * let otherRange = model.createRange( model.createPositionFromPath( root, [ 1 ] ), model.createPositionFromPath( root, [ 2 ] ) );
	 * let transformed = range.getIntersection( otherRange ); // null - ranges have no common part
	 *
	 * otherRange = model.createRange( model.createPositionFromPath( root, [ 3 ] ), model.createPositionFromPath( root, [ 5 ] ) );
	 * transformed = range.getIntersection( otherRange ); // range from [ 3 ] to [ 4, 0, 1 ]
	 * ```
	 *
	 * @param otherRange Range to check for intersection.
	 * @returns A common part of given ranges or `null` if ranges have no common part.
	 */ getIntersection(otherRange) {
        if (this.isIntersecting(otherRange)) {
            // Ranges intersect, so a common range will be returned.
            // At most, it will be same as this range.
            let commonRangeStart = this.start;
            let commonRangeEnd = this.end;
            if (this.containsPosition(otherRange.start)) {
                // Given range start is inside this range. This means thaNt we have to
                // shrink common range to the given range start.
                commonRangeStart = otherRange.start;
            }
            if (this.containsPosition(otherRange.end)) {
                // Given range end is inside this range. This means that we have to
                // shrink common range to the given range end.
                commonRangeEnd = otherRange.end;
            }
            return new ModelRange(commonRangeStart, commonRangeEnd);
        }
        // Ranges do not intersect, so they do not have common part.
        return null;
    }
    /**
	 * Returns a range created by joining this {@link ~ModelRange range} with the given {@link ~ModelRange range}.
	 * If ranges have no common part, returns `null`.
	 *
	 * Examples:
	 *
	 * ```ts
	 * let range = model.createRange(
	 * 	model.createPositionFromPath( root, [ 2, 7 ] ),
	 * 	model.createPositionFromPath( root, [ 4, 0, 1 ] )
	 * );
	 * let otherRange = model.createRange(
	 * 	model.createPositionFromPath( root, [ 1 ] ),
	 * 	model.createPositionFromPath( root, [ 2 ] )
 	 * );
	 * let transformed = range.getJoined( otherRange ); // null - ranges have no common part
	 *
	 * otherRange = model.createRange(
	 * 	model.createPositionFromPath( root, [ 3 ] ),
	 * 	model.createPositionFromPath( root, [ 5 ] )
	 * );
	 * transformed = range.getJoined( otherRange ); // range from [ 2, 7 ] to [ 5 ]
	 * ```
	 *
	 * @param otherRange Range to be joined.
	 * @param loose Whether the intersection check is loose or strict. If the check is strict (`false`),
	 * ranges are tested for intersection or whether start/end positions are equal. If the check is loose (`true`),
	 * compared range is also checked if it's {@link module:engine/model/position~ModelPosition#isTouching touching} current range.
	 * @returns A sum of given ranges or `null` if ranges have no common part.
	 */ getJoined(otherRange, loose = false) {
        let shouldJoin = this.isIntersecting(otherRange);
        if (!shouldJoin) {
            if (this.start.isBefore(otherRange.start)) {
                shouldJoin = loose ? this.end.isTouching(otherRange.start) : this.end.isEqual(otherRange.start);
            } else {
                shouldJoin = loose ? otherRange.end.isTouching(this.start) : otherRange.end.isEqual(this.start);
            }
        }
        if (!shouldJoin) {
            return null;
        }
        let startPosition = this.start;
        let endPosition = this.end;
        if (otherRange.start.isBefore(startPosition)) {
            startPosition = otherRange.start;
        }
        if (otherRange.end.isAfter(endPosition)) {
            endPosition = otherRange.end;
        }
        return new ModelRange(startPosition, endPosition);
    }
    /**
	 * Computes and returns the smallest set of {@link #isFlat flat} ranges, that covers this range in whole.
	 *
	 * See an example of a model structure (`[` and `]` are range boundaries):
	 *
	 * ```
	 * root                                                            root
	 *  |- element DIV                         DIV             P2              P3             DIV
	 *  |   |- element H                   H        P1        f o o           b a r       H         P4
	 *  |   |   |- "fir[st"             fir[st     lorem                               se]cond     ipsum
	 *  |   |- element P1
	 *  |   |   |- "lorem"                                              ||
	 *  |- element P2                                                   ||
	 *  |   |- "foo"                                                    VV
	 *  |- element P3
	 *  |   |- "bar"                                                   root
	 *  |- element DIV                         DIV             [P2             P3]             DIV
	 *  |   |- element H                   H       [P1]       f o o           b a r        H         P4
	 *  |   |   |- "se]cond"            fir[st]    lorem                               [se]cond     ipsum
	 *  |   |- element P4
	 *  |   |   |- "ipsum"
	 * ```
	 *
	 * As it can be seen, letters contained in the range are: `stloremfoobarse`, spread across different parents.
	 * We are looking for minimal set of flat ranges that contains the same nodes.
	 *
	 * Minimal flat ranges for above range `( [ 0, 0, 3 ], [ 3, 0, 2 ] )` will be:
	 *
	 * ```
	 * ( [ 0, 0, 3 ], [ 0, 0, 5 ] ) = "st"
	 * ( [ 0, 1 ], [ 0, 2 ] ) = element P1 ("lorem")
	 * ( [ 1 ], [ 3 ] ) = element P2, element P3 ("foobar")
	 * ( [ 3, 0, 0 ], [ 3, 0, 2 ] ) = "se"
	 * ```
	 *
	 * **Note:** if an {@link module:engine/model/element~ModelElement element} is not wholly contained in this range, it won't be returned
	 * in any of the returned flat ranges. See in the example how `H` elements at the beginning and at the end of the range
	 * were omitted. Only their parts that were wholly in the range were returned.
	 *
	 * **Note:** this method is not returning flat ranges that contain no nodes.
	 *
	 * @returns Array of flat ranges covering this range.
	 */ getMinimalFlatRanges() {
        const ranges = [];
        const diffAt = this.start.getCommonPath(this.end).length;
        const pos = ModelPosition._createAt(this.start);
        let posParent = pos.parent;
        // Go up.
        while(pos.path.length > diffAt + 1){
            const howMany = posParent.maxOffset - pos.offset;
            if (howMany !== 0) {
                ranges.push(new ModelRange(pos, pos.getShiftedBy(howMany)));
            }
            pos.path = pos.path.slice(0, -1);
            pos.offset++;
            posParent = posParent.parent;
        }
        // Go down.
        while(pos.path.length <= this.end.path.length){
            const offset = this.end.path[pos.path.length - 1];
            const howMany = offset - pos.offset;
            if (howMany !== 0) {
                ranges.push(new ModelRange(pos, pos.getShiftedBy(howMany)));
            }
            pos.offset = offset;
            pos.path.push(0);
        }
        return ranges;
    }
    /**
	 * Creates a {@link module:engine/model/treewalker~ModelTreeWalker TreeWalker} instance with this range as a boundary.
	 *
	 * For example, to iterate over all items in the entire document root:
	 *
	 * ```ts
	 * // Create a range spanning over the entire root content:
	 * const range = editor.model.createRangeIn( editor.model.document.getRoot() );
	 *
	 * // Iterate over all items in this range:
	 * for ( const value of range.getWalker() ) {
	 * 	console.log( value.item );
	 * }
	 * ```
	 *
	 * @param options Object with configuration options. See {@link module:engine/model/treewalker~ModelTreeWalker}.
	 */ getWalker(options = {}) {
        options.boundaries = this;
        return new ModelTreeWalker(options);
    }
    /**
	 * Returns an iterator that iterates over all {@link module:engine/model/item~ModelItem items} that are in this range and returns
	 * them.
	 *
	 * This method uses {@link module:engine/model/treewalker~ModelTreeWalker} with `boundaries` set to this range and
	 * `ignoreElementEnd` option set to `true`. However it returns only {@link module:engine/model/item~ModelItem model items},
	 * not {@link module:engine/model/treewalker~ModelTreeWalkerValue}.
	 *
	 * You may specify additional options for the tree walker. See {@link module:engine/model/treewalker~ModelTreeWalker} for
	 * a full list of available options.
	 *
	 * @param options Object with configuration options. See {@link module:engine/model/treewalker~ModelTreeWalker}.
	 */ *getItems(options = {}) {
        options.boundaries = this;
        options.ignoreElementEnd = true;
        const treeWalker = new ModelTreeWalker(options);
        for (const value of treeWalker){
            yield value.item;
        }
    }
    /**
	 * Returns an iterator that iterates over all {@link module:engine/model/position~ModelPosition positions} that are boundaries or
	 * contained in this range.
	 *
	 * This method uses {@link module:engine/model/treewalker~ModelTreeWalker} with `boundaries` set to this range. However it returns only
	 * {@link module:engine/model/position~ModelPosition positions}, not {@link module:engine/model/treewalker~ModelTreeWalkerValue}.
	 *
	 * You may specify additional options for the tree walker. See {@link module:engine/model/treewalker~ModelTreeWalker} for
	 * a full list of available options.
	 *
	 * @param options Object with configuration options. See {@link module:engine/model/treewalker~ModelTreeWalker}.
	 */ *getPositions(options = {}) {
        options.boundaries = this;
        const treeWalker = new ModelTreeWalker(options);
        yield treeWalker.position;
        for (const value of treeWalker){
            yield value.nextPosition;
        }
    }
    /**
	 * Returns a range that is a result of transforming this range by given `operation`.
	 *
	 * **Note:** transformation may break one range into multiple ranges (for example, when a part of the range is
	 * moved to a different part of document tree). For this reason, an array is returned by this method and it
	 * may contain one or more `Range` instances.
	 *
	 * @param operation Operation to transform range by.
	 * @returns Range which is the result of transformation.
	 */ getTransformedByOperation(operation) {
        switch(operation.type){
            case 'insert':
                return this._getTransformedByInsertOperation(operation);
            case 'move':
            case 'remove':
            case 'reinsert':
                return this._getTransformedByMoveOperation(operation);
            case 'split':
                return [
                    this._getTransformedBySplitOperation(operation)
                ];
            case 'merge':
                return [
                    this._getTransformedByMergeOperation(operation)
                ];
        }
        return [
            new ModelRange(this.start, this.end)
        ];
    }
    /**
	 * Returns a range that is a result of transforming this range by multiple `operations`.
	 *
	 * @see ~ModelRange#getTransformedByOperation
	 * @param operations Operations to transform the range by.
	 * @returns Range which is the result of transformation.
	 */ getTransformedByOperations(operations) {
        const ranges = [
            new ModelRange(this.start, this.end)
        ];
        for (const operation of operations){
            for(let i = 0; i < ranges.length; i++){
                const result = ranges[i].getTransformedByOperation(operation);
                ranges.splice(i, 1, ...result);
                i += result.length - 1;
            }
        }
        // It may happen that a range is split into two, and then the part of second "piece" is moved into first
        // "piece". In this case we will have incorrect third range, which should not be included in the result --
        // because it is already included in the first "piece". In this loop we are looking for all such ranges that
        // are inside other ranges and we simply remove them.
        for(let i = 0; i < ranges.length; i++){
            const range = ranges[i];
            for(let j = i + 1; j < ranges.length; j++){
                const next = ranges[j];
                if (range.containsRange(next) || next.containsRange(range) || range.isEqual(next)) {
                    ranges.splice(j, 1);
                }
            }
        }
        return ranges;
    }
    /**
	 * Returns an {@link module:engine/model/element~ModelElement} or {@link module:engine/model/documentfragment~ModelDocumentFragment}
	 * which is a common ancestor of the range's both ends (in which the entire range is contained).
	 */ getCommonAncestor() {
        return this.start.getCommonAncestor(this.end);
    }
    /**
	 * Returns an {@link module:engine/model/element~ModelElement Element} contained by the range.
	 * The element will be returned when it is the **only** node within the range and **fully–contained**
	 * at the same time.
	 */ getContainedElement() {
        if (this.isCollapsed) {
            return null;
        }
        const nodeAfterStart = this.start.nodeAfter;
        const nodeBeforeEnd = this.end.nodeBefore;
        if (nodeAfterStart && nodeAfterStart.is('element') && nodeAfterStart === nodeBeforeEnd) {
            return nodeAfterStart;
        }
        return null;
    }
    /**
	 * Converts `Range` to plain object and returns it.
	 *
	 * @returns `Range` converted to plain object.
	 */ toJSON() {
        return {
            start: this.start.toJSON(),
            end: this.end.toJSON()
        };
    }
    /**
	 * Returns a new range that is equal to current range.
	 */ clone() {
        return new this.constructor(this.start, this.end);
    }
    /**
	 * Returns a result of transforming a copy of this range by insert operation.
	 *
	 * One or more ranges may be returned as a result of this transformation.
	 *
	 * @internal
	 */ _getTransformedByInsertOperation(operation, spread = false) {
        return this._getTransformedByInsertion(operation.position, operation.howMany, spread);
    }
    /**
	 * Returns a result of transforming a copy of this range by move operation.
	 *
	 * One or more ranges may be returned as a result of this transformation.
	 *
	 * @internal
	 */ _getTransformedByMoveOperation(operation, spread = false) {
        const sourcePosition = operation.sourcePosition;
        const howMany = operation.howMany;
        const targetPosition = operation.targetPosition;
        return this._getTransformedByMove(sourcePosition, targetPosition, howMany, spread);
    }
    /**
	 * Returns a result of transforming a copy of this range by split operation.
	 *
	 * Always one range is returned. The transformation is done in a way to not break the range.
	 *
	 * @internal
	 */ _getTransformedBySplitOperation(operation) {
        const start = this.start._getTransformedBySplitOperation(operation);
        let end = this.end._getTransformedBySplitOperation(operation);
        if (this.end.isEqual(operation.insertionPosition)) {
            end = this.end.getShiftedBy(1);
        }
        // Below may happen when range contains graveyard element used by split operation.
        if (start.root != end.root) {
            // End position was next to the moved graveyard element and was moved with it.
            // Fix it by using old `end` which has proper `root`.
            end = this.end.getShiftedBy(-1);
        }
        return new ModelRange(start, end);
    }
    /**
	 * Returns a result of transforming a copy of this range by merge operation.
	 *
	 * Always one range is returned. The transformation is done in a way to not break the range.
	 *
	 * @internal
	 */ _getTransformedByMergeOperation(operation) {
        // Special case when the marker is set on "the closing tag" of an element. Marker can be set like that during
        // transformations, especially when a content of a few block elements were removed. For example:
        //
        // {} is the transformed range, [] is the removed range.
        // <p>F[o{o</p><p>B}ar</p><p>Xy]z</p>
        //
        // <p>Fo{o</p><p>B}ar</p><p>z</p>
        // <p>F{</p><p>B}ar</p><p>z</p>
        // <p>F{</p>}<p>z</p>
        // <p>F{}z</p>
        //
        if (this.start.isEqual(operation.targetPosition) && this.end.isEqual(operation.deletionPosition)) {
            return new ModelRange(this.start);
        }
        let start = this.start._getTransformedByMergeOperation(operation);
        let end = this.end._getTransformedByMergeOperation(operation);
        if (start.root != end.root) {
            // This happens when the end position was next to the merged (deleted) element.
            // Then, the end position was moved to the graveyard root. In this case we need to fix
            // the range cause its boundaries would be in different roots.
            end = this.end.getShiftedBy(-1);
        }
        if (start.isAfter(end)) {
            // This happens in three following cases:
            //
            // Case 1: Merge operation source position is before the target position (due to some transformations, OT, etc.)
            //         This means that start can be moved before the end of the range.
            //
            // Before: <p>a{a</p><p>b}b</p><p>cc</p>
            // Merge:  <p>b}b</p><p>cca{a</p>
            // Fix:    <p>{b}b</p><p>ccaa</p>
            //
            // Case 2: Range start is before merged node but not directly.
            //         Result should include all nodes that were in the original range.
            //
            // Before: <p>aa</p>{<p>cc</p><p>b}b</p>
            // Merge:  <p>aab}b</p>{<p>cc</p>
            // Fix:    <p>aa{bb</p><p>cc</p>}
            //
            //         The range is expanded by an additional `b` letter but it is better than dropping the whole `cc` paragraph.
            //
            // Case 3: Range start is directly before merged node.
            //         Resulting range should include only nodes from the merged element:
            //
            // Before: <p>aa</p>{<p>b}b</p><p>cc</p>
            // Merge:  <p>aab}b</p>{<p>cc</p>
            // Fix:    <p>aa{b}b</p><p>cc</p>
            //
            if (operation.sourcePosition.isBefore(operation.targetPosition)) {
                // Case 1.
                start = ModelPosition._createAt(end);
                start.offset = 0;
            } else {
                if (!operation.deletionPosition.isEqual(start)) {
                    // Case 2.
                    end = operation.deletionPosition;
                }
                // In both case 2 and 3 start is at the end of the merge-to element.
                start = operation.targetPosition;
            }
            return new ModelRange(start, end);
        }
        return new ModelRange(start, end);
    }
    /**
	 * Returns an array containing one or two {@link ~ModelRange ranges} that are a result of transforming this
	 * {@link ~ModelRange range} by inserting `howMany` nodes at `insertPosition`. Two {@link ~ModelRange ranges} are
	 * returned if the insertion was inside this {@link ~ModelRange range} and `spread` is set to `true`.
	 *
	 * Examples:
	 *
	 * ```ts
	 * let range = model.createRange(
	 * 	model.createPositionFromPath( root, [ 2, 7 ] ),
	 * 	model.createPositionFromPath( root, [ 4, 0, 1 ] )
	 * );
	 * let transformed = range._getTransformedByInsertion( model.createPositionFromPath( root, [ 1 ] ), 2 );
	 * // transformed array has one range from [ 4, 7 ] to [ 6, 0, 1 ]
	 *
	 * transformed = range._getTransformedByInsertion( model.createPositionFromPath( root, [ 4, 0, 0 ] ), 4 );
	 * // transformed array has one range from [ 2, 7 ] to [ 4, 0, 5 ]
	 *
	 * transformed = range._getTransformedByInsertion( model.createPositionFromPath( root, [ 3, 2 ] ), 4 );
	 * // transformed array has one range, which is equal to original range
	 *
	 * transformed = range._getTransformedByInsertion( model.createPositionFromPath( root, [ 3, 2 ] ), 4, true );
	 * // transformed array has two ranges: from [ 2, 7 ] to [ 3, 2 ] and from [ 3, 6 ] to [ 4, 0, 1 ]
	 * ```
	 *
	 * @internal
	 * @param insertPosition Position where nodes are inserted.
	 * @param howMany How many nodes are inserted.
	 * @param spread Flag indicating whether this range should be spread if insertion
	 * was inside the range. Defaults to `false`.
	 * @returns Result of the transformation.
	 */ _getTransformedByInsertion(insertPosition, howMany, spread = false) {
        if (spread && this.containsPosition(insertPosition)) {
            // Range has to be spread. The first part is from original start to the spread point.
            // The other part is from spread point to the original end, but transformed by
            // insertion to reflect insertion changes.
            return [
                new ModelRange(this.start, insertPosition),
                new ModelRange(insertPosition.getShiftedBy(howMany), this.end._getTransformedByInsertion(insertPosition, howMany))
            ];
        } else {
            const range = new ModelRange(this.start, this.end);
            range.start = range.start._getTransformedByInsertion(insertPosition, howMany);
            range.end = range.end._getTransformedByInsertion(insertPosition, howMany);
            return [
                range
            ];
        }
    }
    /**
	 * Returns an array containing {@link ~ModelRange ranges} that are a result of transforming this
	 * {@link ~ModelRange range} by moving `howMany` nodes from `sourcePosition` to `targetPosition`.
	 *
	 * @internal
	 * @param sourcePosition Position from which nodes are moved.
	 * @param targetPosition Position to where nodes are moved.
	 * @param howMany How many nodes are moved.
	 * @param spread Whether the range should be spread if the move points inside the range.
	 * @returns  Result of the transformation.
	 */ _getTransformedByMove(sourcePosition, targetPosition, howMany, spread = false) {
        // Special case for transforming a collapsed range. Just transform it like a position.
        if (this.isCollapsed) {
            const newPos = this.start._getTransformedByMove(sourcePosition, targetPosition, howMany);
            return [
                new ModelRange(newPos)
            ];
        }
        // Special case for transformation when a part of the range is moved towards the range.
        //
        // Examples:
        //
        // <div><p>ab</p><p>c[d</p></div><p>e]f</p> --> <div><p>ab</p></div><p>c[d</p><p>e]f</p>
        // <p>e[f</p><div><p>a]b</p><p>cd</p></div> --> <p>e[f</p><p>a]b</p><div><p>cd</p></div>
        //
        // Without this special condition, the default algorithm leaves an "artifact" range from one of `differenceSet` parts:
        //
        // <div><p>ab</p><p>c[d</p></div><p>e]f</p> --> <div><p>ab</p>{</div>}<p>c[d</p><p>e]f</p>
        //
        // This special case is applied only if the range is to be kept together (not spread).
        const moveRange = ModelRange._createFromPositionAndShift(sourcePosition, howMany);
        const insertPosition = targetPosition._getTransformedByDeletion(sourcePosition, howMany);
        if (this.containsPosition(targetPosition) && !spread) {
            if (moveRange.containsPosition(this.start) || moveRange.containsPosition(this.end)) {
                const start = this.start._getTransformedByMove(sourcePosition, targetPosition, howMany);
                const end = this.end._getTransformedByMove(sourcePosition, targetPosition, howMany);
                return [
                    new ModelRange(start, end)
                ];
            }
        }
        // Default algorithm.
        let result;
        const differenceSet = this.getDifference(moveRange);
        let difference = null;
        const common = this.getIntersection(moveRange);
        if (differenceSet.length == 1) {
            // `moveRange` and this range may intersect but may be separate.
            difference = new ModelRange(differenceSet[0].start._getTransformedByDeletion(sourcePosition, howMany), differenceSet[0].end._getTransformedByDeletion(sourcePosition, howMany));
        } else if (differenceSet.length == 2) {
            // `moveRange` is inside this range.
            difference = new ModelRange(this.start, this.end._getTransformedByDeletion(sourcePosition, howMany));
        } // else, `moveRange` contains this range.
        if (difference) {
            result = difference._getTransformedByInsertion(insertPosition, howMany, common !== null || spread);
        } else {
            result = [];
        }
        if (common) {
            const transformedCommon = new ModelRange(common.start._getCombined(moveRange.start, insertPosition), common.end._getCombined(moveRange.start, insertPosition));
            if (result.length == 2) {
                result.splice(1, 0, transformedCommon);
            } else {
                result.push(transformedCommon);
            }
        }
        return result;
    }
    /**
	 * Returns a copy of this range that is transformed by deletion of `howMany` nodes from `deletePosition`.
	 *
	 * If the deleted range is intersecting with the transformed range, the transformed range will be shrank.
	 *
	 * If the deleted range contains transformed range, `null` will be returned.
	 *
	 * @internal
	 * @param deletePosition Position from which nodes are removed.
	 * @param howMany How many nodes are removed.
	 * @returns Result of the transformation.
	 */ _getTransformedByDeletion(deletePosition, howMany) {
        let newStart = this.start._getTransformedByDeletion(deletePosition, howMany);
        let newEnd = this.end._getTransformedByDeletion(deletePosition, howMany);
        if (newStart == null && newEnd == null) {
            return null;
        }
        if (newStart == null) {
            newStart = deletePosition;
        }
        if (newEnd == null) {
            newEnd = deletePosition;
        }
        return new ModelRange(newStart, newEnd);
    }
    /**
	 * Creates a new range, spreading from specified {@link module:engine/model/position~ModelPosition position} to a position moved by
	 * given `shift`. If `shift` is a negative value, shifted position is treated as the beginning of the range.
	 *
	 * @internal
	 * @param position Beginning of the range.
	 * @param shift How long the range should be.
	 */ static _createFromPositionAndShift(position, shift) {
        const start = position;
        const end = position.getShiftedBy(shift);
        return shift > 0 ? new this(start, end) : new this(end, start);
    }
    /**
	 * Creates a range inside an {@link module:engine/model/element~ModelElement element} which starts before the first child of
	 * that element and ends after the last child of that element.
	 *
	 * @internal
	 * @param element Element which is a parent for the range.
	 */ static _createIn(element) {
        return new this(ModelPosition._createAt(element, 0), ModelPosition._createAt(element, element.maxOffset));
    }
    /**
	 * Creates a range that starts before given {@link module:engine/model/item~ModelItem model item} and ends after it.
	 *
	 * @internal
	 */ static _createOn(item) {
        return this._createFromPositionAndShift(ModelPosition._createBefore(item), item.offsetSize);
    }
    /**
	 * Combines all ranges from the passed array into a one range. At least one range has to be passed.
	 * Passed ranges must not have common parts.
	 *
	 * The first range from the array is a reference range. If other ranges start or end on the exactly same position where
	 * the reference range, they get combined into one range.
	 *
	 * ```
	 * [  ][]  [    ][ ][             ][ ][]  [  ]  // Passed ranges, shown sorted
	 * [    ]                                       // The result of the function if the first range was a reference range.
	 *         [                           ]        // The result of the function if the third-to-seventh range was a reference range.
	 *                                        [  ]  // The result of the function if the last range was a reference range.
	 * ```
	 *
	 * @internal
	 * @param ranges Ranges to combine.
	 * @returns Combined range.
	 */ static _createFromRanges(ranges) {
        if (ranges.length === 0) {
            /**
			 * At least one range has to be passed to
			 * {@link module:engine/model/range~ModelRange._createFromRanges `Range._createFromRanges()`}.
			 *
			 * @error range-create-from-ranges-empty-array
			 */ throw new CKEditorError('range-create-from-ranges-empty-array', null);
        } else if (ranges.length == 1) {
            return ranges[0].clone();
        }
        // 1. Set the first range in `ranges` array as a reference range.
        // If we are going to return just a one range, one of the ranges need to be the reference one.
        // Other ranges will be stuck to that range, if possible.
        const ref = ranges[0];
        // 2. Sort all the ranges, so it's easier to process them.
        ranges.sort((a, b)=>{
            return a.start.isAfter(b.start) ? 1 : -1;
        });
        // 3. Check at which index the reference range is now.
        const refIndex = ranges.indexOf(ref);
        // 4. At this moment we don't need the original range.
        // We are going to modify the result, and we need to return a new instance of Range.
        // We have to create a copy of the reference range.
        const result = new this(ref.start, ref.end);
        // 5. Ranges should be checked and glued starting from the range that is closest to the reference range.
        // Since ranges are sorted, start with the range with index that is closest to reference range index.
        for(let i = refIndex - 1; i >= 0; i--){
            if (ranges[i].end.isEqual(result.start)) {
                result.start = ModelPosition._createAt(ranges[i].start);
            } else {
                break;
            }
        }
        // 6. Ranges should be checked and glued starting from the range that is closest to the reference range.
        // Since ranges are sorted, start with the range with index that is closest to reference range index.
        for(let i = refIndex + 1; i < ranges.length; i++){
            if (ranges[i].start.isEqual(result.end)) {
                result.end = ModelPosition._createAt(ranges[i].end);
            } else {
                break;
            }
        }
        return result;
    }
    /**
	 * Creates a `Range` instance from given plain object (i.e. parsed JSON string).
	 *
	 * @param json Plain object to be converted to `Range`.
	 * @param doc Document object that will be range owner.
	 * @returns `Range` instance created using given plain object.
	 */ static fromJSON(json, doc) {
        return new this(ModelPosition.fromJSON(json.start, doc), ModelPosition.fromJSON(json.end, doc));
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ModelRange.prototype.is = function(type) {
    return type === 'range' || type === 'model:range';
};

/**
 * Maps elements, positions and markers between the {@link module:engine/view/document~ViewDocument view} and
 * the {@link module:engine/model/model model}.
 *
 * The instance of the Mapper used for the editing pipeline is available in
 * {@link module:engine/controller/editingcontroller~EditingController#mapper `editor.editing.mapper`}.
 *
 * Mapper uses bound elements to find corresponding elements and positions, so, to get proper results,
 * all model elements should be {@link module:engine/conversion/mapper~Mapper#bindElements bound}.
 *
 * To map the complex model to/from view relations, you may provide custom callbacks for the
 * {@link module:engine/conversion/mapper~Mapper#event:modelToViewPosition modelToViewPosition event} and
 * {@link module:engine/conversion/mapper~Mapper#event:viewToModelPosition viewToModelPosition event} that are fired whenever
 * a position mapping request occurs.
 * Those events are fired by the {@link module:engine/conversion/mapper~Mapper#toViewPosition toViewPosition}
 * and {@link module:engine/conversion/mapper~Mapper#toModelPosition toModelPosition} methods. `Mapper` adds its own default callbacks
 * with `'lowest'` priority. To override default `Mapper` mapping, add custom callback with higher priority and
 * stop the event.
 */ class Mapper extends /* #__PURE__ */ EmitterMixin() {
    /**
	 * Model element to view element mapping.
	 */ _modelToViewMapping = new WeakMap();
    /**
	 * View element to model element mapping.
	 */ _viewToModelMapping = new WeakMap();
    /**
	 * A map containing callbacks between view element names and functions evaluating length of view elements
	 * in model.
	 */ _viewToModelLengthCallbacks = new Map();
    /**
	 * Model marker name to view elements mapping.
	 *
	 * Keys are `String`s while values are `Set`s with {@link module:engine/view/element~ViewElement view elements}.
	 * One marker (name) can be mapped to multiple elements.
	 */ _markerNameToElements = new Map();
    /**
	 * View element to model marker names mapping.
	 *
	 * This is reverse to {@link ~Mapper#_markerNameToElements} map.
	 */ _elementToMarkerNames = new Map();
    /**
	 * The map of removed view elements with their current root (used for deferred unbinding).
	 */ _deferredBindingRemovals = new Map();
    /**
	 * Stores marker names of markers which have changed due to unbinding a view element (so it is assumed that the view element
	 * has been removed, moved or renamed).
	 */ _unboundMarkerNames = new Set();
    /**
	 * Manages dynamic cache for the `Mapper` to improve the performance.
	 */ _cache = new MapperCache();
    /**
	 * Creates an instance of the mapper.
	 */ constructor(){
        super();
        // Default mapper algorithm for mapping model position to view position.
        this.on('modelToViewPosition', (evt, data)=>{
            if (data.viewPosition) {
                return;
            }
            const viewContainer = this._modelToViewMapping.get(data.modelPosition.parent);
            if (!viewContainer) {
                /**
				 * A model position could not be mapped to the view because the parent of the model position
				 * does not have a mapped view element (might have not been converted yet or it has no converter).
				 *
				 * Make sure that the model element is correctly converted to the view.
				 *
				 * @error mapping-model-position-view-parent-not-found
				 */ throw new CKEditorError('mapping-model-position-view-parent-not-found', this, {
                    modelPosition: data.modelPosition
                });
            }
            data.viewPosition = this.findPositionIn(viewContainer, data.modelPosition.offset);
        }, {
            priority: 'low'
        });
        // Default mapper algorithm for mapping view position to model position.
        this.on('viewToModelPosition', (evt, data)=>{
            if (data.modelPosition) {
                return;
            }
            const viewBlock = this.findMappedViewAncestor(data.viewPosition);
            const modelParent = this._viewToModelMapping.get(viewBlock);
            const modelOffset = this._toModelOffset(data.viewPosition.parent, data.viewPosition.offset, viewBlock);
            data.modelPosition = ModelPosition._createAt(modelParent, modelOffset);
        }, {
            priority: 'low'
        });
    }
    /**
	 * Marks model and view elements as corresponding. Corresponding elements can be retrieved by using
	 * the {@link module:engine/conversion/mapper~Mapper#toModelElement toModelElement} and
	 * {@link module:engine/conversion/mapper~Mapper#toViewElement toViewElement} methods.
	 * The information that elements are bound is also used to translate positions.
	 *
	 * @param modelElement Model element.
	 * @param viewElement View element.
	 */ bindElements(modelElement, viewElement) {
        this._modelToViewMapping.set(modelElement, viewElement);
        this._viewToModelMapping.set(viewElement, modelElement);
    }
    /**
	 * Unbinds the given {@link module:engine/view/element~ViewElement view element} from the map.
	 *
	 * **Note:** view-to-model binding will be removed, if it existed. However, corresponding model-to-view binding
	 * will be removed only if model element is still bound to the passed `viewElement`.
	 *
	 * This behavior allows for re-binding model element to another view element without fear of losing the new binding
	 * when the previously bound view element is unbound.
	 *
	 * @param viewElement View element to unbind.
	 * @param options The options object.
	 * @param options.defer Controls whether the binding should be removed immediately or deferred until a
	 * {@link #flushDeferredBindings `flushDeferredBindings()`} call.
	 */ unbindViewElement(viewElement, options = {}) {
        const modelElement = this.toModelElement(viewElement);
        if (this._elementToMarkerNames.has(viewElement)) {
            for (const markerName of this._elementToMarkerNames.get(viewElement)){
                this._unboundMarkerNames.add(markerName);
            }
        }
        if (options.defer) {
            this._deferredBindingRemovals.set(viewElement, viewElement.root);
        } else {
            const wasFound = this._viewToModelMapping.delete(viewElement);
            if (wasFound) {
                // Stop tracking after the element is no longer mapped. We want to track all mapped elements and only mapped elements.
                this._cache.stopTracking(viewElement);
            }
            if (this._modelToViewMapping.get(modelElement) == viewElement) {
                this._modelToViewMapping.delete(modelElement);
            }
        }
    }
    /**
	 * Unbinds the given {@link module:engine/model/element~ModelElement model element} from the map.
	 *
	 * **Note:** the model-to-view binding will be removed, if it existed. However, the corresponding view-to-model binding
	 * will be removed only if the view element is still bound to the passed `modelElement`.
	 *
	 * This behavior lets for re-binding view element to another model element without fear of losing the new binding
	 * when the previously bound model element is unbound.
	 *
	 * @param modelElement Model element to unbind.
	 */ unbindModelElement(modelElement) {
        const viewElement = this.toViewElement(modelElement);
        this._modelToViewMapping.delete(modelElement);
        if (this._viewToModelMapping.get(viewElement) == modelElement) {
            const wasFound = this._viewToModelMapping.delete(viewElement);
            if (wasFound) {
                // Stop tracking after the element is no longer mapped. We want to track all mapped elements and only mapped elements.
                this._cache.stopTracking(viewElement);
            }
        }
    }
    /**
	 * Binds the given marker name with the given {@link module:engine/view/element~ViewElement view element}. The element
	 * will be added to the current set of elements bound with the given marker name.
	 *
	 * @param element Element to bind.
	 * @param name Marker name.
	 */ bindElementToMarker(element, name) {
        const elements = this._markerNameToElements.get(name) || new Set();
        elements.add(element);
        const names = this._elementToMarkerNames.get(element) || new Set();
        names.add(name);
        this._markerNameToElements.set(name, elements);
        this._elementToMarkerNames.set(element, names);
    }
    /**
	 * Unbinds an element from given marker name.
	 *
	 * @param element Element to unbind.
	 * @param name Marker name.
	 */ unbindElementFromMarkerName(element, name) {
        const nameToElements = this._markerNameToElements.get(name);
        if (nameToElements) {
            nameToElements.delete(element);
            if (nameToElements.size == 0) {
                this._markerNameToElements.delete(name);
            }
        }
        const elementToNames = this._elementToMarkerNames.get(element);
        if (elementToNames) {
            elementToNames.delete(name);
            if (elementToNames.size == 0) {
                this._elementToMarkerNames.delete(element);
            }
        }
    }
    /**
	 * Returns all marker names of markers which have changed due to unbinding a view element (so it is assumed that the view element
	 * has been removed, moved or renamed) since the last flush. After returning, the marker names list is cleared.
	 */ flushUnboundMarkerNames() {
        const markerNames = Array.from(this._unboundMarkerNames);
        this._unboundMarkerNames.clear();
        return markerNames;
    }
    /**
	 * Unbinds all deferred binding removals of view elements that in the meantime were not re-attached to some root or document fragment.
	 *
	 * See: {@link #unbindViewElement `unbindViewElement()`}.
	 */ flushDeferredBindings() {
        for (const [viewElement, root] of this._deferredBindingRemovals){
            // Unbind it only if it wasn't re-attached to some root or document fragment.
            if (viewElement.root == root) {
                this.unbindViewElement(viewElement);
            }
        }
        this._deferredBindingRemovals = new Map();
    }
    /**
	 * Removes all model to view and view to model bindings.
	 */ clearBindings() {
        this._modelToViewMapping = new WeakMap();
        this._viewToModelMapping = new WeakMap();
        this._markerNameToElements = new Map();
        this._elementToMarkerNames = new Map();
        this._unboundMarkerNames = new Set();
        this._deferredBindingRemovals = new Map();
    }
    toModelElement(viewElement) {
        return this._viewToModelMapping.get(viewElement);
    }
    toViewElement(modelElement) {
        return this._modelToViewMapping.get(modelElement);
    }
    /**
	 * Gets the corresponding model range.
	 *
	 * @param viewRange View range.
	 * @returns Corresponding model range.
	 */ toModelRange(viewRange) {
        return new ModelRange(this.toModelPosition(viewRange.start), this.toModelPosition(viewRange.end));
    }
    /**
	 * Gets the corresponding view range.
	 *
	 * @param modelRange Model range.
	 * @returns Corresponding view range.
	 */ toViewRange(modelRange) {
        return new ViewRange(this.toViewPosition(modelRange.start), this.toViewPosition(modelRange.end));
    }
    /**
	 * Gets the corresponding model position.
	 *
	 * @fires viewToModelPosition
	 * @param viewPosition View position.
	 * @returns Corresponding model position.
	 */ toModelPosition(viewPosition) {
        const data = {
            viewPosition,
            mapper: this
        };
        this.fire('viewToModelPosition', data);
        return data.modelPosition;
    }
    /**
	 * Gets the corresponding view position.
	 *
	 * @fires modelToViewPosition
	 * @param modelPosition Model position.
	 * @param options Additional options for position mapping process.
	 * @param options.isPhantom Should be set to `true` if the model position to map is pointing to a place
	 * in model tree which no longer exists. For example, it could be an end of a removed model range.
	 * @returns Corresponding view position.
	 */ toViewPosition(modelPosition, options = {}) {
        const data = {
            modelPosition,
            mapper: this,
            isPhantom: options.isPhantom
        };
        this.fire('modelToViewPosition', data);
        return data.viewPosition;
    }
    /**
	 * Gets all view elements bound to the given marker name.
	 *
	 * @param name Marker name.
	 * @returns View elements bound with the given marker name or `null`
	 * if no elements are bound to the given marker name.
	 */ markerNameToElements(name) {
        const boundElements = this._markerNameToElements.get(name);
        if (!boundElements) {
            return null;
        }
        const elements = new Set();
        for (const element of boundElements){
            if (element.is('attributeElement')) {
                for (const clone of element.getElementsWithSameId()){
                    elements.add(clone);
                }
            } else {
                elements.add(element);
            }
        }
        return elements;
    }
    /**
	 * **This method is deprecated and will be removed in one of the future CKEditor 5 releases.**
	 *
	 * **Using this method will turn off `Mapper` caching system and may degrade performance when operating on bigger documents.**
	 *
	 * Registers a callback that evaluates the length in the model of a view element with the given name.
	 *
	 * The callback is fired with one argument, which is a view element instance. The callback is expected to return
	 * a number representing the length of the view element in the model.
	 *
	 * ```ts
	 * // List item in view may contain nested list, which have other list items. In model though,
	 * // the lists are represented by flat structure. Because of those differences, length of list view element
	 * // may be greater than one. In the callback it's checked how many nested list items are in evaluated list item.
	 *
	 * function getViewListItemLength( element ) {
	 * 	let length = 1;
	 *
	 * 	for ( let child of element.getChildren() ) {
	 * 		if ( child.name == 'ul' || child.name == 'ol' ) {
	 * 			for ( let item of child.getChildren() ) {
	 * 				length += getViewListItemLength( item );
	 * 			}
	 * 		}
	 * 	}
	 *
	 * 	return length;
	 * }
	 *
	 * mapper.registerViewToModelLength( 'li', getViewListItemLength );
	 * ```
	 *
	 * @param viewElementName Name of view element for which callback is registered.
	 * @param lengthCallback Function return a length of view element instance in model.
	 * @deprecated
	 */ registerViewToModelLength(viewElementName, lengthCallback) {
        this._viewToModelLengthCallbacks.set(viewElementName, lengthCallback);
    }
    /**
	 * For the given `viewPosition`, finds and returns the closest ancestor of this position that has a mapping to
	 * the model.
	 *
	 * @param viewPosition Position for which a mapped ancestor should be found.
	 */ findMappedViewAncestor(viewPosition) {
        let parent = viewPosition.parent;
        while(!this._viewToModelMapping.has(parent)){
            parent = parent.parent;
        }
        return parent;
    }
    /**
	 * Calculates model offset based on the view position and the block element.
	 *
	 * Example:
	 *
	 * ```html
	 * <p>foo<b>ba|r</b></p> // _toModelOffset( b, 2, p ) -> 5
	 * ```
	 *
	 * Is a sum of:
	 *
	 * ```html
	 * <p>foo|<b>bar</b></p> // _toModelOffset( p, 3, p ) -> 3
	 * <p>foo<b>ba|r</b></p> // _toModelOffset( b, 2, b ) -> 2
	 * ```
	 *
	 * @param viewParent Position parent.
	 * @param viewOffset Position offset.
	 * @param viewBlock Block used as a base to calculate offset.
	 * @returns Offset in the model.
	 */ _toModelOffset(viewParent, viewOffset, viewBlock) {
        if (viewBlock != viewParent) {
            // See example.
            const offsetToParentStart = this._toModelOffset(viewParent.parent, viewParent.index, viewBlock);
            const offsetInParent = this._toModelOffset(viewParent, viewOffset, viewParent);
            return offsetToParentStart + offsetInParent;
        }
        // viewBlock == viewParent, so we need to calculate the offset in the parent element.
        // If the position is a text it is simple ("ba|r" -> 2).
        if (viewParent.is('$text')) {
            return viewOffset;
        }
        // If the position is in an element we need to sum lengths of siblings ( <b> bar </b> foo | -> 3 + 3 = 6 ).
        let modelOffset = 0;
        for(let i = 0; i < viewOffset; i++){
            modelOffset += this.getModelLength(viewParent.getChild(i));
        }
        return modelOffset;
    }
    /**
	 * Gets the length of the view element in the model.
	 *
	 * The length is calculated as follows:
	 * * if a {@link #registerViewToModelLength length mapping callback} is provided for the given `viewNode`, it is used to
	 * evaluate the model length (`viewNode` is used as first and only parameter passed to the callback),
	 * * length of a {@link module:engine/view/text~ViewText text node} is equal to the length of its
	 * {@link module:engine/view/text~ViewText#data data},
	 * * length of a {@link module:engine/view/uielement~ViewUIElement ui element} is equal to 0,
	 * * length of a mapped {@link module:engine/view/element~ViewElement element} is equal to 1,
	 * * length of a non-mapped {@link module:engine/view/element~ViewElement element} is equal to the length of its children.
	 *
	 * Examples:
	 *
	 * ```
	 * foo                          -> 3 // Text length is equal to its data length.
	 * <p>foo</p>                   -> 1 // Length of an element which is mapped is by default equal to 1.
	 * <b>foo</b>                   -> 3 // Length of an element which is not mapped is a length of its children.
	 * <div><p>x</p><p>y</p></div>  -> 2 // Assuming that <div> is not mapped and <p> are mapped.
	 * ```
	 *
	 * @param viewNode View node.
	 * @returns Length of the node in the tree model.
	 */ getModelLength(viewNode) {
        const stack = [
            viewNode
        ];
        let len = 0;
        while(stack.length > 0){
            const node = stack.pop();
            const callback = node.name && this._viewToModelLengthCallbacks.size > 0 && this._viewToModelLengthCallbacks.get(node.name);
            if (callback) {
                len += callback(node);
            } else if (this._viewToModelMapping.has(node)) {
                len += 1;
            } else if (node.is('$text')) {
                len += node.data.length;
            } else if (node.is('uiElement')) {
                continue;
            } else {
                for (const child of node.getChildren()){
                    stack.push(child);
                }
            }
        }
        return len;
    }
    /**
	 * Finds the position in a view element or view document fragment node (or in its children) with the expected model offset.
	 *
	 * If the passed `viewContainer` is bound to model, `Mapper` will use caching mechanism to improve performance.
	 *
	 * @param viewContainer Tree view element in which we are looking for the position.
	 * @param modelOffset Expected offset.
	 * @returns Found position.
	 */ findPositionIn(viewContainer, modelOffset) {
        if (modelOffset === 0) {
            // Quickly return if asked for a position at the beginning of the container. No need to fire complex mechanisms to find it.
            return this._moveViewPositionToTextNode(new ViewPosition(viewContainer, 0));
        }
        // Use cache only if there are no custom view-to-model length callbacks and only for bound elements.
        // View-to-model length callbacks are deprecated and should be removed in one of the following releases.
        // Then it will be possible to simplify some logic inside `Mapper`.
        // Note: we could consider requiring `viewContainer` to be a mapped item.
        const useCache = this._viewToModelLengthCallbacks.size == 0 && this._viewToModelMapping.has(viewContainer);
        if (useCache) {
            const cacheItem = this._cache.getClosest(viewContainer, modelOffset);
            return this._findPositionStartingFrom(cacheItem.viewPosition, cacheItem.modelOffset, modelOffset, viewContainer, true);
        } else {
            return this._findPositionStartingFrom(new ViewPosition(viewContainer, 0), 0, modelOffset, viewContainer, false);
        }
    }
    /**
	 * Performs most of the logic for `Mapper#findPositionIn()`.
	 *
	 * It allows to start looking for the requested model offset from a given starting position, to enable caching. Using the cache,
	 * you can set the starting point and skip all the calculations that were already previously done.
	 *
	 * This method uses recursion to find positions inside deep structures. Example:
	 *
	 * ```
	 * <p>fo<b>bar</b>bom</p>  -> target offset: 4
	 * <p>|fo<b>bar</b>bom</p> -> target offset: 4, traversed offset: 0
	 * <p>fo|<b>bar</b>bom</p> -> target offset: 4, traversed offset: 2
	 * <p>fo<b>bar</b>|bom</p> -> target offset: 4, traversed offset: 5 -> we are too far, look recursively in <b>.
	 *
	 * <p>fo<b>|bar</b>bom</p> -> target offset: 4, traversed offset: 2
	 * <p>fo<b>bar|</b>bom</p> -> target offset: 4, traversed offset: 5 -> we are too far, look inside "bar".
	 *
	 * <p>fo<b>ba|r</b>bom</p> -> target offset: 4, traversed offset: 2 -> position is inside text node at offset 4-2 = 2.
	 * ```
	 *
	 * @param startViewPosition View position to start looking from.
	 * @param startModelOffset Model offset related to `startViewPosition`.
	 * @param targetModelOffset Target model offset to find.
	 * @param viewContainer Mapped ancestor of `startViewPosition`. `startModelOffset` is the offset inside a model element or model
	 * document fragment mapped to `viewContainer`.
	 * @param useCache Whether `Mapper` should cache positions while traversing the view tree looking for `expectedModelOffset`.
	 * @returns View position mapped to `targetModelOffset`.
	 */ _findPositionStartingFrom(startViewPosition, startModelOffset, targetModelOffset, viewContainer, useCache) {
        let viewParent = startViewPosition.parent;
        let viewOffset = startViewPosition.offset;
        // In the text node it is simple: the offset in the model equals the offset in the text.
        if (viewParent.is('$text')) {
            return new ViewPosition(viewParent, targetModelOffset - startModelOffset);
        }
        // Last scanned view node.
        let viewNode;
        // Total model offset of the view nodes that were visited so far.
        let traversedModelOffset = startModelOffset;
        // Model length of the last traversed view node.
        let lastLength = 0;
        while(traversedModelOffset < targetModelOffset){
            viewNode = viewParent.getChild(viewOffset);
            if (!viewNode) {
                // If we still haven't reached the model offset, but we reached end of this `viewParent`, then we need to "leave" this
                // element and "go up", looking further for the target model offset. This can happen when cached model offset is "deeper"
                // but target model offset is "higher" in the view tree.
                //
                // Example: `<p>Foo<strong><em>Bar</em>^Baz</strong>Xyz</p>`
                //
                // Consider `^` is last cached position, when the `targetModelOffset` is `12`. In such case, we need to "go up" from
                // `<strong>` and continue traversing in `<p>`.
                //
                if (viewParent == viewContainer) {
                    /**
					 * A model position could not be mapped to the view because specified model offset was too big and could not be
					 * found inside the mapped view element or view document fragment.
					 *
					 * @error mapping-model-offset-not-found
					 */ throw new CKEditorError('mapping-model-offset-not-found', this, {
                        modelOffset: targetModelOffset,
                        viewContainer
                    });
                } else {
                    viewOffset = viewParent.parent.getChildIndex(viewParent) + 1;
                    viewParent = viewParent.parent;
                    // Cache view position after stepping out of the view element to make sure that all visited view positions are cached.
                    // Otherwise, cache invalidation may work incorrectly.
                    if (useCache) {
                        this._cache.save(viewParent, viewOffset, viewContainer, traversedModelOffset);
                    }
                    continue;
                }
            }
            if (useCache) {
                lastLength = this._getModelLengthAndCache(viewNode, viewContainer, traversedModelOffset);
            } else {
                lastLength = this.getModelLength(viewNode);
            }
            traversedModelOffset += lastLength;
            viewOffset++;
        }
        let viewPosition = new ViewPosition(viewParent, viewOffset);
        if (useCache) {
            // Make sure to hoist view position and save cache for all view positions along the way that have the same `modelOffset`.
            //
            // Consider example view:
            //
            // <p>Foo<strong><i>bar<em>xyz</em></i></strong>abc</p>
            //
            // Lets assume that we looked for model offset `9`, starting  from `6` (as it was previously cached value).
            // In this case we will only traverse over `<em>xyz</em>` and cache view positions after "xyz" and after `</em>`.
            // After stepping over `<em>xyz</em>`, we will stop processing this view, as we will reach target model offset `9`.
            //
            // However, position before `</strong>` and before `abc` are also valid positions for model offset `9`.
            // Additionally, `Mapper` is supposed to return "hoisted" view positions, that is, we prefer positions that are closer to
            // the mapped `viewContainer`. If a position is nested inside attribute elements, it should be "moved up" if possible.
            //
            // As we hoist the view position, we need to make sure that all view positions valid for model offset `9` are cached.
            // This is necessary for cache invalidation to work correctly.
            //
            // To hoist a view position, we "go up" as long as the position is at the end of a non-mapped view element. We also cache
            // all necessary values. See example:
            //
            // <p>Foo<strong><i>bar<em>xyz</em>^</i></strong>abc</p>
            // <p>Foo<strong><i>bar<em>xyz</em></i>^</strong>abc</p>
            // <p>Foo<strong><i>bar<em>xyz</em></i></strong>^abc</p>
            //
            while(viewPosition.isAtEnd && viewPosition.parent !== viewContainer && viewPosition.parent.parent){
                const cacheViewParent = viewPosition.parent.parent;
                const cacheViewOffset = cacheViewParent.getChildIndex(viewPosition.parent) + 1;
                this._cache.save(cacheViewParent, cacheViewOffset, viewContainer, traversedModelOffset);
                viewPosition = new ViewPosition(cacheViewParent, cacheViewOffset);
            }
        }
        if (traversedModelOffset == targetModelOffset) {
            // If it equals we found the position.
            //
            // Try moving view position into a text node if possible, as the editor engine prefers positions inside view text nodes.
            //
            // <p>Foo<strong><i>bar<em>xyz</em></i></strong>[]abc</p>    -->     <p>Foo<strong><i>bar<em>xyz</em></i></strong>{}abc</p>
            //
            return this._moveViewPositionToTextNode(viewPosition);
        } else {
            // If it is higher we overstepped with the last traversed view node.
            // We need to "enter" it, and look for the view position / model offset inside the last visited view node.
            return this._findPositionStartingFrom(new ViewPosition(viewNode, 0), traversedModelOffset - lastLength, targetModelOffset, viewContainer, useCache);
        }
    }
    /**
	 * Gets the length of the view element in the model and updates cache values after each view item it visits.
	 *
	 * See also {@link #getModelLength}.
	 *
	 * @param viewNode View node.
	 * @param viewContainer Ancestor of `viewNode` that is a mapped view element.
	 * @param modelOffset Model offset at which the `viewNode` starts.
	 * @returns Length of the node in the tree model.
	 */ _getModelLengthAndCache(viewNode, viewContainer, modelOffset) {
        let len = 0;
        if (this._viewToModelMapping.has(viewNode)) {
            len = 1;
        } else if (viewNode.is('$text')) {
            len = viewNode.data.length;
        } else if (!viewNode.is('uiElement')) {
            for (const child of viewNode.getChildren()){
                len += this._getModelLengthAndCache(child, viewContainer, modelOffset + len);
            }
        }
        this._cache.save(viewNode.parent, viewNode.index + 1, viewContainer, modelOffset + len);
        return len;
    }
    /**
	 * Because we prefer positions in the text nodes over positions next to text nodes, if the view position was next to a text node,
	 * it moves it into the text node instead.
	 *
	 * ```
	 * <p>[]<b>foo</b></p> -> <p>[]<b>foo</b></p> // do not touch if position is not directly next to text
	 * <p>foo[]<b>foo</b></p> -> <p>foo{}<b>foo</b></p> // move to text node
	 * <p><b>[]foo</b></p> -> <p><b>{}foo</b></p> // move to text node
	 * ```
	 *
	 * @param viewPosition Position potentially next to the text node.
	 * @returns Position in the text node if possible.
	 */ _moveViewPositionToTextNode(viewPosition) {
        // If the position is just after a text node, put it at the end of that text node.
        // If the position is just before a text node, put it at the beginning of that text node.
        const nodeBefore = viewPosition.nodeBefore;
        const nodeAfter = viewPosition.nodeAfter;
        if (nodeBefore && nodeBefore.is('view:$text')) {
            return new ViewPosition(nodeBefore, nodeBefore.data.length);
        } else if (nodeAfter && nodeAfter.is('view:$text')) {
            return new ViewPosition(nodeAfter, 0);
        }
        // Otherwise, just return the given position.
        return viewPosition;
    }
}
/**
 * Cache mechanism for {@link module:engine/conversion/mapper~Mapper Mapper}.
 *
 * `MapperCache` improves performance for model-to-view position mapping, which is the main `Mapper` task. Asking for a mapping is much
 * more frequent than actually performing changes, and even if the change happens, we can still partially keep the cache. This makes
 * caching a useful strategy for `Mapper`.
 *
 * `MapperCache` will store some data for view elements or view document fragments that are mapped by the `Mapper`. These view items
 * are "tracked" by the `MapperCache`. For such view items, we will keep entries of model offsets inside their mapped counterpart. For
 * the cached model offsets, we will keep a view position that is inside the tracked item. This allows us to either get the mapping
 * instantly, or at least in less steps than when calculating it from the beginning.
 *
 * Important problem related to caching is invalidating the cache. The cache must be invalidated each time the tracked view item changes.
 * Additionally, we should invalidate as small part of the cache as possible. Since all the logic is encapsulated inside `MapperCache`,
 * the `MapperCache` listens to view items {@link module:engine/view/node~ViewNodeChangeEvent `change` event} and reacts to it.
 * Then, it invalidates just the part of the cache that is "after" the changed part of the view.
 *
 * As mentioned, `MapperCache` currently is used only for model-to-view position mapping as it was much bigger problem than view-to-model
 * mapping. However, it should be possible to use it also for view-to-model.
 *
 * The main assumptions regarding `MapperCache` are:
 *
 * * it is an internal tool, used by `Mapper`, transparent to the outside (no additional effort when developing a plugin or a converter),
 * * it stores all the necessary data internally, which makes it easier to disable or debug,
 * * it is optimized for initial downcast process (long insertions), which is crucial for editor init and data save,
 * * it does not save all possible positions mapping for memory considerations, although it is a possible improvement, which may increase
 *   performance, as well as simplify some parts of the `MapperCache` logic.
 *
 * @internal
 */ class MapperCache extends /* #__PURE__ */ EmitterMixin() {
    /**
	 * For every view element or document fragment tracked by `MapperCache`, it holds currently cached data, or more precisely,
	 * model offset to view position mappings. See also `MappingCache` and `CacheItem`.
	 *
	 * If an item is tracked by `MapperCache` it has an entry in this structure, so this structure can be used to check which items
	 * are tracked by `MapperCache`. When an item is no longer tracked, it is removed from this structure.
	 *
	 * Although `MappingCache` and `CacheItem` structures allows for caching any model offsets and view positions, we only cache
	 * values for model offsets that are after a view node. So, in essence, positions inside text nodes are not cached. However, it takes
	 * from one to at most a few steps, to get from a cached position to a position that is inside a view text node.
	 *
	 * Additionally, only one item per `modelOffset` is cached. There can be several view positions that map to the same `modelOffset`.
	 * Only the first save for `modelOffset` is stored.
	 */ _cachedMapping = new WeakMap();
    /**
	 * When `MapperCache` {@link ~MapperCache#save saves} view position -> model offset mapping, a `CacheItem` is inserted into certain
	 * `MappingCache#cacheList` at some index. Additionally, we store that index with the view node that is before the cached view position.
	 *
	 * This allows to quickly get a cache list item related to certain view node, and hence, for fast cache invalidation.
	 *
	 * For example, consider view: `<p>Some <strong>bold</strong> text.</p>`, where `<p>` is a view element tracked by `MapperCache`.
	 * If all `<p>` children were visited by `MapperCache`, then `<p>` cache list would have four items, related to following model offsets:
	 * `0`, `5`, `9`, `15`. Then, view node `"Some "` would have index `1`, `<strong>` index `2`, and `" text." index `3`.
	 *
	 * Note that the index related with a node is always greater than `0`. The first item in cache list is always for model offset `0`
	 * (and view offset `0`), and it is not related to any node.
	 */ _nodeToCacheListIndex = new WeakMap();
    /**
	 * Callback fired whenever there is a direct or indirect children change in tracked view element or tracked view document fragment.
	 *
	 * This is specified as a property to make it easier to set as an event callback and to later turn off that event.
	 */ _invalidateOnChildrenChangeCallback = (evt, viewNode, data)=>{
        // View element or document fragment changed its children at `data.index`. Clear all cache starting from before that index.
        this._clearCacheInsideParent(viewNode, data.index);
    };
    /**
	 * Callback fired whenever a view text node directly or indirectly inside a tracked view element or tracked view document fragment
	 * changes its text data.
	 *
	 * This is specified as a property to make it easier to set as an event callback and to later turn off that event.
	 */ _invalidateOnTextChangeCallback = (evt, viewNode)=>{
        // It is enough to validate starting from "after the text node", because the first cache entry that we might want to invalidate
        // is the position after this text node.
        //
        // For example - assume following view and following view positions cached (marked by `^`): `<p>Foo^<strong>bar^</strong>^abc^</p>`.
        //
        // If we change text "bar", we only need to invalidate cached positions after it. Cached positions before the text are not changed.
        //
        this._clearCacheAfter(viewNode);
    };
    /**
	 * Saves cache for given view position mapping <-> model offset mapping. The view position should be after a node (i.e. it cannot
	 * be the first position inside its parent, or in other words, `viewOffset` must be greater than `0`).
	 *
	 * Note, that if `modelOffset` for given `viewContainer` was already saved, the stored view position (i.e. parent+offset) will not
	 * be overwritten. However, it is important to still save it, as we still store additional data related to cached view positions.
	 *
	 * @param viewParent View position parent.
	 * @param viewOffset View position offset. Must be greater than `0`.
	 * @param viewContainer Tracked view position ascendant (it may be the direct parent of the view position).
	 * @param modelOffset Model offset in the model element or document fragment which is mapped to `viewContainer`.
	 */ save(viewParent, viewOffset, viewContainer, modelOffset) {
        // Get current cache for the tracked ancestor.
        const cache = this._cachedMapping.get(viewContainer);
        // See if there is already a cache defined for `modelOffset`.
        const cacheItem = cache.cacheMap.get(modelOffset);
        if (cacheItem) {
            // We already cached this offset. Don't overwrite the cache.
            // However, we still need to set a proper entry in `_nodeToCacheListIndex`. We can figure it out based on existing `cacheItem`.
            //
            // We have a `cacheItem` for the `modelOffset`, so we can get a `viewPosition` from there. Before that view position, there
            // must be a node. That node must have an index set. This will be the index we will want to use.
            // Since we expect `viewOffset` to be greater than 0, then in almost all cases `modelOffset` will be greater than 0 as well.
            // As a result, we can expect `cacheItem.viewPosition.nodeBefore` to be set.
            //
            // However, in an edge case, were the tracked element contains a 0-model-length view element as the first child (UI element or
            // an empty attribute element), then `modelOffset` will be 0, and `cacheItem.viewPosition` will be before any view node.
            // In such edge case, `cacheItem.viewPosition.nodeBefore` is `undefined`, so we set index to `0`.
            //
            const viewChild = viewParent.getChild(viewOffset - 1);
            const index = cacheItem.viewPosition.nodeBefore ? this._nodeToCacheListIndex.get(cacheItem.viewPosition.nodeBefore) : 0;
            this._nodeToCacheListIndex.set(viewChild, index);
            return;
        }
        const viewPosition = new ViewPosition(viewParent, viewOffset);
        const newCacheItem = {
            viewPosition,
            modelOffset
        };
        // Extend the valid cache range.
        cache.maxModelOffset = modelOffset > cache.maxModelOffset ? modelOffset : cache.maxModelOffset;
        // Save the new cache item to the `cacheMap`.
        cache.cacheMap.set(modelOffset, newCacheItem);
        // Save the new cache item to the `cacheList`.
        let i = cache.cacheList.length - 1;
        // Mostly, we cache elements at the end of `cacheList` and the loop does not execute even once. But when we recursively visit nodes
        // in `Mapper#_findPositionIn()`, then we will first cache the parent, and then it's children, and they will not be added at the
        // end of `cacheList`. This is why we need to find correct index to insert them.
        while(i >= 0 && cache.cacheList[i].modelOffset > modelOffset){
            i--;
        }
        cache.cacheList.splice(i + 1, 0, newCacheItem);
        if (viewOffset > 0) {
            const viewChild = viewParent.getChild(viewOffset - 1);
            // There was an idea to also cache `viewContainer` here but, it could lead to wrong results. If we wanted to cache
            // `viewContainer`, we probably would need to clear `this._nodeToCacheListIndex` when cache is cleared.
            // Also, there was no gain from caching this value, the results were almost the same (statistical error).
            this._nodeToCacheListIndex.set(viewChild, i + 1);
        }
    }
    /**
	 * For given `modelOffset` inside a model element mapped to given `viewContainer`, it returns the closest saved cache item
	 * (view position and related model offset) to the requested one.
	 *
	 * It can be exactly the requested mapping, or it can be mapping that is the closest starting point to look for the requested mapping.
	 *
	 * `viewContainer` must be a view element or document fragment that is mapped by the {@link ~Mapper Mapper}.
	 *
	 * If `viewContainer` is not yet tracked by the `MapperCache`, it will be automatically tracked after calling this method.
	 *
	 * Note: this method will automatically "hoist" cached positions, i.e. it will return a position that is closest to the tracked element.
	 *
	 * For example, if `<p>` is tracked element, and `^` is cached position:
	 *
	 * ```
	 * <p>This is <strong>some <em>heavily <u>formatted</u>^</em></strong> text.</p>
	 * ```
	 *
	 * If this position would be returned, instead, a position directly in `<p>` would be returned:
	 *
	 * ```
	 * <p>This is <strong>some <em>heavily <u>formatted</u></em></strong>^ text.</p>
	 * ```
	 *
	 * Note, that `modelOffset` for both positions is the same.
	 *
	 * @param viewContainer Tracked view element or document fragment, which cache will be used.
	 * @param modelOffset Model offset in a model element or document fragment, which is mapped to `viewContainer`.
	 */ getClosest(viewContainer, modelOffset) {
        const cache = this._cachedMapping.get(viewContainer);
        let result;
        if (cache) {
            if (modelOffset > cache.maxModelOffset) {
                result = cache.cacheList[cache.cacheList.length - 1];
            } else {
                const cacheItem = cache.cacheMap.get(modelOffset);
                if (cacheItem) {
                    result = cacheItem;
                } else {
                    result = this._findInCacheList(cache.cacheList, modelOffset);
                }
            }
        } else {
            result = this.startTracking(viewContainer);
        }
        return {
            modelOffset: result.modelOffset,
            viewPosition: result.viewPosition.clone()
        };
    }
    /**
	 * Starts tracking given `viewContainer`, which must be mapped to a model element or model document fragment.
	 *
	 * Note, that this method is automatically called by
	 * {@link module:engine/conversion/mapper~MapperCache#getClosest `MapperCache#getClosest()`} and there is no need to call it manually.
	 *
	 * This method initializes the cache for `viewContainer` and adds callbacks for
	 * {@link module:engine/view/node~ViewNodeChangeEvent `change` event} fired by `viewContainer`. `MapperCache` listens to `change` event
	 * on the tracked elements to invalidate the stored cache.
	 */ startTracking(viewContainer) {
        const viewPosition = new ViewPosition(viewContainer, 0);
        const initialCacheItem = {
            viewPosition,
            modelOffset: 0
        };
        const initialCache = {
            maxModelOffset: 0,
            cacheList: [
                initialCacheItem
            ],
            cacheMap: new Map([
                [
                    0,
                    initialCacheItem
                ]
            ])
        };
        this._cachedMapping.set(viewContainer, initialCache);
        // Listen to changes in tracked view containers in order to invalidate the cache.
        //
        // Possible performance improvement. This event bubbles, so if there are multiple tracked (mapped) elements that are ancestors
        // then this will be unnecessarily fired for each ancestor. This could be rewritten to listen only to roots and document fragments.
        viewContainer.on('change:children', this._invalidateOnChildrenChangeCallback);
        viewContainer.on('change:text', this._invalidateOnTextChangeCallback);
        return initialCacheItem;
    }
    /**
	 * Stops tracking given `viewContainer`.
	 *
	 * It removes the cached data and stops listening to {@link module:engine/view/node~ViewNodeChangeEvent `change` event} on the
	 * `viewContainer`.
	 */ stopTracking(viewContainer) {
        viewContainer.off('change:children', this._invalidateOnChildrenChangeCallback);
        viewContainer.off('change:text', this._invalidateOnTextChangeCallback);
        this._cachedMapping.delete(viewContainer);
    }
    /**
	 * Invalidates cache inside `viewParent`, starting from given `index` in that parent.
	 *
	 * This method may clear a bit more cache than just what was saved after given `index`, but it is guaranteed that at least it
	 * will invalidate everything after `index`.
	 */ _clearCacheInsideParent(viewParent, index) {
        if (index == 0) {
            // Change at the beginning of the parent.
            if (this._cachedMapping.has(viewParent)) {
                // If this is a tracked element, clear all cache.
                this._clearCacheAll(viewParent);
            } else {
                // If this is not a tracked element, remove cache starting from before this element.
                // Since it is not a tracked element, it has to have a parent.
                this._clearCacheInsideParent(viewParent.parent, viewParent.index);
            }
        } else {
            // Change in the middle of the parent. Get a view node that's before the change.
            const lastValidNode = viewParent.getChild(index - 1);
            // Then, clear all cache after this view node.
            //
            this._clearCacheAfter(lastValidNode);
        }
    }
    /**
	 * Clears all the cache for given tracked `viewContainer`.
	 */ _clearCacheAll(viewContainer) {
        const cache = this._cachedMapping.get(viewContainer);
        // TODO: Should clear `_nodeToCacheListIndex` too?
        if (cache.maxModelOffset > 0) {
            cache.maxModelOffset = 0;
            cache.cacheList.length = 1;
            cache.cacheMap.clear();
            cache.cacheMap.set(0, cache.cacheList[0]);
        }
    }
    /**
	 * Clears all the stored cache that is after given `viewNode`. The `viewNode` can be any node that is inside a tracked view element
	 * or view document fragment.
	 *
	 * In reality, this function may clear a bit more cache than just "starting after" `viewNode`, but it is guaranteed that at least
	 * all cache after `viewNode` is invalidated.
	 */ _clearCacheAfter(viewNode) {
        // To quickly invalidate the cache, we base on the cache list index stored with the node. See docs for `this._nodeToCacheListIndex`.
        const cacheListIndex = this._nodeToCacheListIndex.get(viewNode);
        // If there is no index stored, it means that this `viewNode` has not been cached yet.
        if (cacheListIndex === undefined) {
            // If the node is not cached, maybe it's parent is. We will try to invalidate the cache using the parent.
            const viewParent = viewNode.parent;
            // If the parent is a non-tracked element, try clearing the cache starting from the position before it.
            //
            // For example: `<p>Abc<strong>def<em>ghi</em></strong></p>`.
            //
            // If `viewNode` is `<em>` in this case, and it was not cached yet, we will try to clear cache starting from before `<strong>`.
            //
            // If the parent is a tracked element, then it means there's no cache to clear (nothing after the element is cached).
            // In this case, there's nothing to do. We assume that there are no "holes" in caching in direct children of tracked element
            // (that is if some children is cached, then its previous sibling is cached too, and we would not end up inside this `if`).
            //
            // TODO: Most probably this `if` could be removed altogether, after recent changes in Mapper.
            // TODO: Now we cache all items one after another, so there aren't any "holes" anywhere, not only on top-level.
            //
            if (!this._cachedMapping.has(viewParent)) {
                this._clearCacheInsideParent(viewParent.parent, viewParent.index);
            }
            return;
        }
        let viewContainer = viewNode.parent;
        while(!this._cachedMapping.has(viewContainer)){
            viewContainer = viewContainer.parent;
        }
        this._clearCacheFromCacheIndex(viewContainer, cacheListIndex);
    }
    /**
	 * Clears all the cache in the cache list related to given `viewContainer`, starting from `index` (inclusive).
	 */ _clearCacheFromCacheIndex(viewContainer, index) {
        if (index === 0) {
            // Don't remove the first entry in the cache (this entry is always a mapping between view offset 0 <-> model offset 0,
            // and it is a default value that is always expected to be in the cache list).
            //
            // The cache mechanism may ask to clear from index `0` in a case where a 0-model-length view element (UI element or empty
            // attribute element) was at the beginning of tracked element. In such scenario, the view element is mapped through
            // `nodeToCacheListIndex` to index `0`.
            index = 1;
        }
        // Cache is always available here because we initialize it just before adding a listener that fires `_clearCacheFromIndex()`.
        const cache = this._cachedMapping.get(viewContainer);
        const cacheItem = cache.cacheList[index - 1];
        if (!cacheItem) {
            return;
        }
        cache.maxModelOffset = cacheItem.modelOffset;
        // Remove from cache all `CacheItem`s that are "after" the index to clear from.
        const clearedItems = cache.cacheList.splice(index);
        // For each removed item, make sure to also remove it from `cacheMap` and clear related entry in `_nodeToCacheListIndex`.
        for (const item of clearedItems){
            cache.cacheMap.delete(item.modelOffset);
            const viewNode = item.viewPosition.nodeBefore;
            this._nodeToCacheListIndex.delete(viewNode);
        }
    }
    /**
	 * Finds a cache item in the given cache list, which `modelOffset` is closest (but smaller or equal) to given `offset`.
	 *
	 * Since `cacheList` is a sorted array, this uses binary search to retrieve the item quickly.
	 */ _findInCacheList(cacheList, offset) {
        let start = 0;
        let end = cacheList.length - 1;
        let index = end - start >> 1;
        let item = cacheList[index];
        while(start < end){
            if (item.modelOffset < offset) {
                start = index + 1;
            } else {
                end = index - 1;
            }
            index = start + (end - start >> 1);
            item = cacheList[index];
        }
        return item.modelOffset <= offset ? item : cacheList[index - 1];
    }
}

/**
 * Manages a list of consumable values for the {@link module:engine/model/item~ModelItem model items}.
 *
 * Consumables are various aspects of the model. A model item can be broken down into separate, single properties that might be
 * taken into consideration when converting that item.
 *
 * `ModelConsumable` is used by {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher} while analyzing the changed
 * parts of {@link module:engine/model/document~ModelDocument the document}. The added / changed / removed model items are broken down
 * into singular properties (the item itself and its attributes). All those parts are saved in `ModelConsumable`. Then,
 * during conversion, when the given part of a model item is converted (i.e. the view element has been inserted into the view,
 * but without attributes), the consumable value is removed from `ModelConsumable`.
 *
 * For model items, `ModelConsumable` stores consumable values of one of following types: `insert`, `addattribute:<attributeKey>`,
 * `changeattributes:<attributeKey>`, `removeattributes:<attributeKey>`.
 *
 * In most cases, it is enough to let th {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher}
 * gather consumable values, so there is no need to use
 * the {@link module:engine/conversion/modelconsumable~ModelConsumable#add add method} directly.
 * However, it is important to understand how consumable values can be
 * {@link module:engine/conversion/modelconsumable~ModelConsumable#consume consumed}.
 * See {@link module:engine/conversion/downcasthelpers default downcast converters} for more information.
 *
 * Keep in mind that one conversion event may have multiple callbacks (converters) attached to it. Each of those is
 * able to convert one or more parts of the model. However, when one of those callbacks actually converts
 * something, the others should not, because they would duplicate the results. Using `ModelConsumable` helps to avoid
 * this situation, because callbacks should only convert these values that were not yet consumed from `ModelConsumable`.
 *
 * Consuming multiple values in a single callback:
 *
 * ```ts
 * // Converter for custom `imageBlock` element that might have a `caption` element inside which changes
 * // how the image is displayed in the view:
 * //
 * // Model:
 * //
 * // [imageBlock]
 * //   └─ [caption]
 * //       └─ foo
 * //
 * // View:
 * //
 * // <figure>
 * //   ├─ <img />
 * //   └─ <caption>
 * //       └─ foo
 * modelConversionDispatcher.on( 'insert:imageBlock', ( evt, data, conversionApi ) => {
 * 	// First, consume the `imageBlock` element.
 * 	conversionApi.consumable.consume( data.item, 'insert' );
 *
 * 	// Just create normal image element for the view.
 * 	// Maybe it will be "decorated" later.
 * 	const viewImage = new ViewElement( 'img' );
 * 	const insertPosition = conversionApi.mapper.toViewPosition( data.range.start );
 * 	const viewWriter = conversionApi.writer;
 *
 * 	// Check if the `imageBlock` element has children.
 * 	if ( data.item.childCount > 0 ) {
 * 		const modelCaption = data.item.getChild( 0 );
 *
 * 		// `modelCaption` insertion change is consumed from consumable values.
 * 		// It will not be converted by other converters, but it's children (probably some text) will be.
 * 		// Through mapping, converters for text will know where to insert contents of `modelCaption`.
 * 		if ( conversionApi.consumable.consume( modelCaption, 'insert' ) ) {
 * 			const viewCaption = new ViewElement( 'figcaption' );
 *
 * 			const viewImageHolder = new ViewElement( 'figure', null, [ viewImage, viewCaption ] );
 *
 * 			conversionApi.mapper.bindElements( modelCaption, viewCaption );
 * 			conversionApi.mapper.bindElements( data.item, viewImageHolder );
 * 			viewWriter.insert( insertPosition, viewImageHolder );
 * 		}
 * 	} else {
 * 		conversionApi.mapper.bindElements( data.item, viewImage );
 * 		viewWriter.insert( insertPosition, viewImage );
 * 	}
 *
 * 	evt.stop();
 * } );
 * ```
 */ class ModelConsumable {
    /**
	 * Contains list of consumable values.
	 */ _consumable = new Map();
    /**
	 * For each {@link module:engine/model/textproxy~ModelTextProxy} added to `ModelConsumable`, this registry holds a parent
	 * of that `ModelTextProxy` and the start and end indices of that `ModelTextProxy`. This allows identification of the `ModelTextProxy`
	 * instances that point to the same part of the model but are different instances. Each distinct `ModelTextProxy`
	 * is given a unique `Symbol` which is then registered as consumable. This process is transparent for the `ModelConsumable`
	 * API user because whenever `ModelTextProxy` is added, tested, consumed or reverted, the internal mechanisms of
	 * `ModelConsumable` translate `ModelTextProxy` to that unique `Symbol`.
	 */ _textProxyRegistry = new Map();
    /**
	 * Adds a consumable value to the consumables list and links it with a given model item.
	 *
	 * ```ts
	 * modelConsumable.add( modelElement, 'insert' ); // Add `modelElement` insertion change to consumable values.
	 * modelConsumable.add( modelElement, 'addAttribute:bold' ); // Add `bold` attribute insertion on `modelElement` change.
	 * modelConsumable.add( modelElement, 'removeAttribute:bold' ); // Add `bold` attribute removal on `modelElement` change.
	 * modelConsumable.add( modelSelection, 'selection' ); // Add `modelSelection` to consumable values.
	 * modelConsumable.add( modelRange, 'range' ); // Add `modelRange` to consumable values.
	 * ```
	 *
	 * @param item Model item, range or selection that has the consumable.
	 * @param type Consumable type. Will be normalized to a proper form, that is either `<word>` or `<part>:<part>`.
	 * Second colon and everything after will be cut. Passing event name is a safe and good practice.
	 */ add(item, type) {
        type = _normalizeConsumableType(type);
        if (item instanceof ModelTextProxy) {
            item = this._getSymbolForTextProxy(item);
        }
        if (!this._consumable.has(item)) {
            this._consumable.set(item, new Map());
        }
        this._consumable.get(item).set(type, true);
    }
    /**
	 * Removes a given consumable value from a given model item.
	 *
	 * ```ts
	 * modelConsumable.consume( modelElement, 'insert' ); // Remove `modelElement` insertion change from consumable values.
	 * modelConsumable.consume( modelElement, 'addAttribute:bold' ); // Remove `bold` attribute insertion on `modelElement` change.
	 * modelConsumable.consume( modelElement, 'removeAttribute:bold' ); // Remove `bold` attribute removal on `modelElement` change.
	 * modelConsumable.consume( modelSelection, 'selection' ); // Remove `modelSelection` from consumable values.
	 * modelConsumable.consume( modelRange, 'range' ); // Remove 'modelRange' from consumable values.
	 * ```
	 *
	 * @param item Model item, range or selection from which consumable will be consumed.
	 * @param type Consumable type. Will be normalized to a proper form, that is either `<word>` or `<part>:<part>`.
	 * Second colon and everything after will be cut. Passing event name is a safe and good practice.
	 * @returns `true` if consumable value was available and was consumed, `false` otherwise.
	 */ consume(item, type) {
        type = _normalizeConsumableType(type);
        if (item instanceof ModelTextProxy) {
            item = this._getSymbolForTextProxy(item);
        }
        if (this.test(item, type)) {
            this._consumable.get(item).set(type, false);
            return true;
        } else {
            return false;
        }
    }
    /**
	 * Tests whether there is a consumable value of a given type connected with a given model item.
	 *
	 * ```ts
	 * modelConsumable.test( modelElement, 'insert' ); // Check for `modelElement` insertion change.
	 * modelConsumable.test( modelElement, 'addAttribute:bold' ); // Check for `bold` attribute insertion on `modelElement` change.
	 * modelConsumable.test( modelElement, 'removeAttribute:bold' ); // Check for `bold` attribute removal on `modelElement` change.
	 * modelConsumable.test( modelSelection, 'selection' ); // Check if `modelSelection` is consumable.
	 * modelConsumable.test( modelRange, 'range' ); // Check if `modelRange` is consumable.
	 * ```
	 *
	 * @param item Model item, range or selection to be tested.
	 * @param type Consumable type. Will be normalized to a proper form, that is either `<word>` or `<part>:<part>`.
	 * Second colon and everything after will be cut. Passing event name is a safe and good practice.
	 * @returns `null` if such consumable was never added, `false` if the consumable values was
	 * already consumed or `true` if it was added and not consumed yet.
	 */ test(item, type) {
        type = _normalizeConsumableType(type);
        if (item instanceof ModelTextProxy) {
            item = this._getSymbolForTextProxy(item);
        }
        const itemConsumables = this._consumable.get(item);
        if (itemConsumables === undefined) {
            return null;
        }
        const value = itemConsumables.get(type);
        if (value === undefined) {
            return null;
        }
        return value;
    }
    /**
	 * Reverts consuming of a consumable value.
	 *
	 * ```ts
	 * modelConsumable.revert( modelElement, 'insert' ); // Revert consuming `modelElement` insertion change.
	 * modelConsumable.revert( modelElement, 'addAttribute:bold' ); // Revert consuming `bold` attribute insert from `modelElement`.
	 * modelConsumable.revert( modelElement, 'removeAttribute:bold' ); // Revert consuming `bold` attribute remove from `modelElement`.
	 * modelConsumable.revert( modelSelection, 'selection' ); // Revert consuming `modelSelection`.
	 * modelConsumable.revert( modelRange, 'range' ); // Revert consuming `modelRange`.
	 * ```
	 *
	 * @param item Model item, range or selection to be reverted.
	 * @param type Consumable type.
	 * @returns `true` if consumable has been reversed, `false` otherwise. `null` if the consumable has
	 * never been added.
	 */ revert(item, type) {
        type = _normalizeConsumableType(type);
        if (item instanceof ModelTextProxy) {
            item = this._getSymbolForTextProxy(item);
        }
        const test = this.test(item, type);
        if (test === false) {
            this._consumable.get(item).set(type, true);
            return true;
        } else if (test === true) {
            return false;
        }
        return null;
    }
    /**
	 * Verifies if all events from the specified group were consumed.
	 *
	 * @param eventGroup The events group to verify.
	 */ verifyAllConsumed(eventGroup) {
        const items = [];
        for (const [item, consumables] of this._consumable){
            for (const [event, canConsume] of consumables){
                const eventPrefix = event.split(':')[0];
                if (canConsume && eventGroup == eventPrefix) {
                    items.push({
                        event,
                        item: item.name || item.description
                    });
                }
            }
        }
        if (items.length) {
            /**
			 * Some of the {@link module:engine/model/item~ModelItem model items} were not consumed while downcasting the model to view.
			 *
			 * This might be the effect of:
			 *
			 * * A missing converter for some model elements. Make sure that you registered downcast converters for all model elements.
			 * * A custom converter that does not consume converted items. Make sure that you
			 * {@link module:engine/conversion/modelconsumable~ModelConsumable#consume consumed} all model elements that you converted
			 * from the model to the view.
			 * * A custom converter that called `event.stop()`. When providing a custom converter, keep in mind that you should not stop
			 * the event. If you stop it then the default converter at the `lowest` priority will not trigger the conversion of this node's
			 * attributes and child nodes.
			 *
			 * @error conversion-model-consumable-not-consumed
			 * @param {Array.<module:engine/model/item~ModelItem>} items Items that were not consumed.
			 */ throw new CKEditorError('conversion-model-consumable-not-consumed', null, {
                items
            });
        }
    }
    /**
	 * Gets a unique symbol for the passed {@link module:engine/model/textproxy~ModelTextProxy} instance.
	 * All `ModelTextProxy` instances that have same parent, same start index and same end index will get the same symbol.
	 *
	 * Used internally to correctly consume `ModelTextProxy` instances.
	 *
	 * @internal
	 * @param textProxy `ModelTextProxy` instance to get a symbol for.
	 * @returns Symbol representing all equal instances of `ModelTextProxy`.
	 */ _getSymbolForTextProxy(textProxy) {
        let symbol = null;
        const startMap = this._textProxyRegistry.get(textProxy.startOffset);
        if (startMap) {
            const endMap = startMap.get(textProxy.endOffset);
            if (endMap) {
                symbol = endMap.get(textProxy.parent);
            }
        }
        if (!symbol) {
            symbol = this._addSymbolForTextProxy(textProxy);
        }
        return symbol;
    }
    /**
	 * Adds a symbol for the given {@link module:engine/model/textproxy~ModelTextProxy} instance.
	 *
	 * Used internally to correctly consume `ModelTextProxy` instances.
	 *
	 * @param textProxy Text proxy instance.
	 * @returns Symbol generated for given `ModelTextProxy`.
	 */ _addSymbolForTextProxy(textProxy) {
        const start = textProxy.startOffset;
        const end = textProxy.endOffset;
        const parent = textProxy.parent;
        const symbol = Symbol('$textProxy:' + textProxy.data);
        let startMap;
        let endMap;
        startMap = this._textProxyRegistry.get(start);
        if (!startMap) {
            startMap = new Map();
            this._textProxyRegistry.set(start, startMap);
        }
        endMap = startMap.get(end);
        if (!endMap) {
            endMap = new Map();
            startMap.set(end, endMap);
        }
        endMap.set(parent, symbol);
        return symbol;
    }
}
/**
 * Returns a normalized consumable type name from the given string. A normalized consumable type name is a string that has
 * at most one colon, for example: `insert` or `addMarker:highlight`. If a string to normalize has more "parts" (more colons),
 * the further parts are dropped, for example: `addattribute:bold:$text` -> `addattributes:bold`.
 *
 * @param type Consumable type.
 * @returns Normalized consumable type.
 */ function _normalizeConsumableType(type) {
    const parts = type.split(':');
    // For inserts allow passing event name, it's stored in the context of a specified element so the element name is not needed.
    if (parts[0] == 'insert') {
        return parts[0];
    }
    // Markers are identified by the whole name (otherwise we would consume the whole markers group).
    if (parts[0] == 'addMarker' || parts[0] == 'removeMarker') {
        return type;
    }
    return parts.length > 1 ? parts[0] + ':' + parts[1] : parts[0];
}

/**
 * The downcast dispatcher is a central point of downcasting (conversion from the model to the view), which is a process of reacting
 * to changes in the model and firing a set of events. The callbacks listening to these events are called converters. The
 * converters' role is to convert the model changes to changes in view (for example, adding view nodes or
 * changing attributes on view elements).
 *
 * During the conversion process, downcast dispatcher fires events basing on the state of the model and prepares
 * data for these events. It is important to understand that the events are connected with the changes done on the model,
 * for example: "a node has been inserted" or "an attribute has changed". This is in contrary to upcasting (a view-to-model conversion)
 * where you convert the view state (view nodes) to a model tree.
 *
 * The events are prepared basing on a diff created by the {@link module:engine/model/differ~Differ Differ}, which buffers them
 * and then passes to the downcast dispatcher as a diff between the old model state and the new model state.
 *
 * Note that because the changes are converted, there is a need to have a mapping between the model structure and the view structure.
 * To map positions and elements during the downcast (a model-to-view conversion), use {@link module:engine/conversion/mapper~Mapper}.
 *
 * Downcast dispatcher fires the following events for model tree changes:
 *
 * * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:insert `insert`} &ndash;
 * If a range of nodes was inserted to the model tree.
 * * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:remove `remove`} &ndash;
 * If a range of nodes was removed from the model tree.
 * * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:attribute `attribute`} &ndash;
 * If an attribute was added, changed or removed from a model node.
 *
 * For {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:insert `insert`}
 * and {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:attribute `attribute`},
 * the downcast dispatcher generates {@link module:engine/conversion/modelconsumable~ModelConsumable consumables}.
 * These are used to have control over which changes have already been consumed. It is useful when some converters
 * overwrite others or convert multiple changes (for example, it converts an insertion of an element and also converts that
 * element's attributes during the insertion).
 *
 * Additionally, downcast dispatcher fires events for {@link module:engine/model/markercollection~Marker marker} changes:
 *
 * * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:addMarker `addMarker`} &ndash; If a marker was added.
 * * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:removeMarker `removeMarker`} &ndash; If a marker was
 * removed.
 *
 * Note that changing a marker is done through removing the marker from the old range and adding it to the new range,
 * so both of these events are fired.
 *
 * Finally, a downcast dispatcher also handles firing events for the {@link module:engine/model/selection model selection}
 * conversion:
 *
 * * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:selection `selection`}
 * &ndash; Converts the selection from the model to the view.
 * * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:attribute `attribute`}
 * &ndash; Fired for every selection attribute.
 * * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:addMarker `addMarker`}
 * &ndash; Fired for every marker that contains a selection.
 *
 * Unlike the model tree and the markers, the events for selection are not fired for changes but for a selection state.
 *
 * When providing custom listeners for a downcast dispatcher, remember to check whether a given change has not been
 * {@link module:engine/conversion/modelconsumable~ModelConsumable#consume consumed} yet.
 *
 * When providing custom listeners for a downcast dispatcher, keep in mind that you **should not** stop the event. If you stop it,
 * then the default converter at the `lowest` priority will not trigger the conversion of this node's attributes and child nodes.
 *
 * When providing custom listeners for a downcast dispatcher, remember to use the provided
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter view downcast writer} to apply changes to the view document.
 *
 * You can read more about conversion in the following guide:
 *
 * * {@glink framework/deep-dive/conversion/downcast Downcast conversion}
 *
 * An example of a custom converter for the downcast dispatcher:
 *
 * ```ts
 * // You will convert inserting a "paragraph" model element into the model.
 * downcastDispatcher.on( 'insert:paragraph', ( evt, data, conversionApi ) => {
 * 	// Remember to check whether the change has not been consumed yet and consume it.
 * 	if ( !conversionApi.consumable.consume( data.item, 'insert' ) ) {
 * 		return;
 * 	}
 *
 * 	// Translate the position in the model to a position in the view.
 * 	const viewPosition = conversionApi.mapper.toViewPosition( data.range.start );
 *
 * 	// Create a <p> element that will be inserted into the view at the `viewPosition`.
 * 	const viewElement = conversionApi.writer.createContainerElement( 'p' );
 *
 * 	// Bind the newly created view element to the model element so positions will map accordingly in the future.
 * 	conversionApi.mapper.bindElements( data.item, viewElement );
 *
 * 	// Add the newly created view element to the view.
 * 	conversionApi.writer.insert( viewPosition, viewElement );
 * } );
 * ```
 */ class DowncastDispatcher extends /* #__PURE__ */ EmitterMixin() {
    /**
	 * A template for an interface passed by the dispatcher to the event callbacks.
	 *
	 * @internal
	 */ _conversionApi;
    /**
	 * A map of already fired events for a given `ModelConsumable`.
	 */ _firedEventsMap;
    /**
	 * Creates a downcast dispatcher instance.
	 *
	 * @see module:engine/conversion/downcastdispatcher~DowncastConversionApi
	 *
	 * @param conversionApi Additional properties for an interface that will be passed to events fired
	 * by the downcast dispatcher.
	 */ constructor(conversionApi){
        super();
        this._conversionApi = {
            dispatcher: this,
            ...conversionApi
        };
        this._firedEventsMap = new WeakMap();
    }
    /**
	 * Converts changes buffered in the given {@link module:engine/model/differ~Differ model differ}
	 * and fires conversion events based on it.
	 *
	 * @fires insert
	 * @fires remove
	 * @fires attribute
	 * @fires addMarker
	 * @fires removeMarker
	 * @fires reduceChanges
	 * @param differ The differ object with buffered changes.
	 * @param markers Markers related to the model fragment to convert.
	 * @param writer The view writer that should be used to modify the view document.
	 */ convertChanges(differ, markers, writer) {
        const conversionApi = this._createConversionApi(writer, differ.getRefreshedItems());
        // Before the view is updated, remove markers which have changed.
        for (const change of differ.getMarkersToRemove()){
            this._convertMarkerRemove(change.name, change.range, conversionApi);
        }
        // Let features modify the change list (for example to allow reconversion).
        const changes = this._reduceChanges(differ.getChanges());
        // Convert changes that happened on model tree.
        for (const entry of changes){
            if (entry.type === 'insert') {
                this._convertInsert(ModelRange._createFromPositionAndShift(entry.position, entry.length), conversionApi);
            } else if (entry.type === 'reinsert') {
                this._convertReinsert(ModelRange._createFromPositionAndShift(entry.position, entry.length), conversionApi);
            } else if (entry.type === 'remove') {
                this._convertRemove(entry.position, entry.length, entry.name, conversionApi);
            } else {
                // Defaults to 'attribute' change.
                this._convertAttribute(entry.range, entry.attributeKey, entry.attributeOldValue, entry.attributeNewValue, conversionApi);
            }
        }
        // Remove mappings for all removed view elements.
        // Remove these mappings as soon as they are not needed (https://github.com/ckeditor/ckeditor5/issues/15411).
        conversionApi.mapper.flushDeferredBindings();
        for (const markerName of conversionApi.mapper.flushUnboundMarkerNames()){
            const markerRange = markers.get(markerName).getRange();
            this._convertMarkerRemove(markerName, markerRange, conversionApi);
            this._convertMarkerAdd(markerName, markerRange, conversionApi);
        }
        // After the view is updated, convert markers which have changed.
        for (const change of differ.getMarkersToAdd()){
            this._convertMarkerAdd(change.name, change.range, conversionApi);
        }
        // Verify if all insert consumables were consumed.
        conversionApi.consumable.verifyAllConsumed('insert');
    }
    /**
	 * Starts a conversion of a model range and the provided markers.
	 *
	 * @fires insert
	 * @fires attribute
	 * @fires addMarker
	 * @param range The inserted range.
	 * @param markers The map of markers that should be down-casted.
	 * @param writer The view writer that should be used to modify the view document.
	 * @param options Optional options object passed to `convertionApi.options`.
	 */ convert(range, markers, writer, options = {}) {
        const conversionApi = this._createConversionApi(writer, undefined, options);
        this._convertInsert(range, conversionApi);
        for (const [name, range] of markers){
            this._convertMarkerAdd(name, range, conversionApi);
        }
        // Verify if all insert consumables were consumed.
        conversionApi.consumable.verifyAllConsumed('insert');
    }
    /**
	 * Starts the model selection conversion.
	 *
	 * Fires events for a given {@link module:engine/model/selection~ModelSelection selection} to start the selection conversion.
	 *
	 * @fires selection
	 * @fires addMarker
	 * @fires attribute
	 * @param selection The selection to convert.
	 * @param markers Markers connected with the converted model.
	 * @param writer View writer that should be used to modify the view document.
	 */ convertSelection(selection, markers, writer) {
        const conversionApi = this._createConversionApi(writer);
        // First perform a clean-up at the current position of the selection.
        this.fire('cleanSelection', {
            selection
        }, conversionApi);
        // Don't convert selection if it is in a model root that does not have a view root (for now this is only the graveyard root).
        const modelRoot = selection.getFirstPosition().root;
        if (!conversionApi.mapper.toViewElement(modelRoot)) {
            return;
        }
        // Now, perform actual selection conversion.
        const markersAtSelection = Array.from(markers.getMarkersAtPosition(selection.getFirstPosition()));
        this._addConsumablesForSelection(conversionApi.consumable, selection, markersAtSelection);
        this.fire('selection', {
            selection
        }, conversionApi);
        if (!selection.isCollapsed) {
            return;
        }
        for (const marker of markersAtSelection){
            // Do not fire event if the marker has been consumed.
            if (conversionApi.consumable.test(selection, 'addMarker:' + marker.name)) {
                const markerRange = marker.getRange();
                if (!shouldMarkerChangeBeConverted(selection.getFirstPosition(), marker, conversionApi.mapper)) {
                    continue;
                }
                const data = {
                    item: selection,
                    markerName: marker.name,
                    markerRange
                };
                this.fire(`addMarker:${marker.name}`, data, conversionApi);
            }
        }
        for (const key of selection.getAttributeKeys()){
            // Do not fire event if the attribute has been consumed.
            if (conversionApi.consumable.test(selection, 'attribute:' + key)) {
                const data = {
                    item: selection,
                    range: selection.getFirstRange(),
                    attributeKey: key,
                    attributeOldValue: null,
                    attributeNewValue: selection.getAttribute(key)
                };
                this.fire(`attribute:${key}:$text`, data, conversionApi);
            }
        }
    }
    /**
	 * Fires insertion conversion of a range of nodes.
	 *
	 * For each node in the range, {@link #event:insert `insert` event is fired}. For each attribute on each node,
	 * {@link #event:attribute `attribute` event is fired}.
	 *
	 * @fires insert
	 * @fires attribute
	 * @param range The inserted range.
	 * @param conversionApi The conversion API object.
	 * @param options.doNotAddConsumables Whether the ModelConsumable should not get populated
	 * for items in the provided range.
	 */ _convertInsert(range, conversionApi, options = {}) {
        if (!options.doNotAddConsumables) {
            // Collect a list of things that can be consumed, consisting of nodes and their attributes.
            this._addConsumablesForInsert(conversionApi.consumable, range);
        }
        // Fire a separate insert event for each node and text fragment contained in the range.
        for (const data of range.getWalker({
            shallow: true
        })){
            this._testAndFire('insert', walkerValueToEventData(data), conversionApi);
        }
    }
    /**
	 * Fires conversion of a single node removal. Fires {@link #event:remove remove event} with provided data.
	 *
	 * @param position Position from which node was removed.
	 * @param length Offset size of removed node.
	 * @param name Name of removed node.
	 * @param conversionApi The conversion API object.
	 */ _convertRemove(position, length, name, conversionApi) {
        this.fire(`remove:${name}`, {
            position,
            length
        }, conversionApi);
    }
    /**
	 * Starts a conversion of an attribute change on a given `range`.
	 *
	 * For each node in the given `range`, {@link #event:attribute attribute event} is fired with the passed data.
	 *
	 * @fires attribute
	 * @param range Changed range.
	 * @param key Key of the attribute that has changed.
	 * @param oldValue Attribute value before the change or `null` if the attribute has not been set before.
	 * @param newValue New attribute value or `null` if the attribute has been removed.
	 * @param conversionApi The conversion API object.
	 */ _convertAttribute(range, key, oldValue, newValue, conversionApi) {
        // Create a list with attributes to consume.
        this._addConsumablesForRange(conversionApi.consumable, range, `attribute:${key}`);
        // Create a separate attribute event for each node in the range.
        for (const value of range){
            const data = {
                item: value.item,
                range: ModelRange._createFromPositionAndShift(value.previousPosition, value.length),
                attributeKey: key,
                attributeOldValue: oldValue,
                attributeNewValue: newValue
            };
            this._testAndFire(`attribute:${key}`, data, conversionApi);
        }
    }
    /**
	 * Fires re-insertion conversion (with a `reconversion` flag passed to `insert` events)
	 * of a range of elements (only elements on the range depth, without children).
	 *
	 * For each node in the range on its depth (without children), {@link #event:insert `insert` event} is fired.
	 * For each attribute on each node, {@link #event:attribute `attribute` event} is fired.
	 *
	 * @fires insert
	 * @fires attribute
	 * @param range The range to reinsert.
	 * @param conversionApi The conversion API object.
	 */ _convertReinsert(range, conversionApi) {
        // Convert the elements - without converting children.
        const walkerValues = Array.from(range.getWalker({
            shallow: true
        }));
        // Collect a list of things that can be consumed, consisting of nodes and their attributes.
        this._addConsumablesForInsert(conversionApi.consumable, walkerValues);
        // Fire a separate insert event for each node and text fragment contained shallowly in the range.
        for (const data of walkerValues.map(walkerValueToEventData)){
            this._testAndFire('insert', {
                ...data,
                reconversion: true
            }, conversionApi);
        }
    }
    /**
	 * Converts the added marker. Fires the {@link #event:addMarker `addMarker`} event for each item
	 * in the marker's range. If the range is collapsed, a single event is dispatched. See the event description for more details.
	 *
	 * @fires addMarker
	 * @param markerName Marker name.
	 * @param markerRange The marker range.
	 * @param conversionApi The conversion API object.
	 */ _convertMarkerAdd(markerName, markerRange, conversionApi) {
        // Do not convert if range is in graveyard.
        if (markerRange.root.rootName == '$graveyard') {
            return;
        }
        // In markers' case, event name == consumable name.
        const eventName = `addMarker:${markerName}`;
        //
        // First, fire an event for the whole marker.
        //
        conversionApi.consumable.add(markerRange, eventName);
        this.fire(eventName, {
            markerName,
            markerRange
        }, conversionApi);
        //
        // Do not fire events for each item inside the range if the range got consumed.
        // Also consume the whole marker consumable if it wasn't consumed.
        //
        if (!conversionApi.consumable.consume(markerRange, eventName)) {
            return;
        }
        //
        // Then, fire an event for each item inside the marker range.
        //
        this._addConsumablesForRange(conversionApi.consumable, markerRange, eventName);
        for (const item of markerRange.getItems()){
            // Do not fire event for already consumed items.
            if (!conversionApi.consumable.test(item, eventName)) {
                continue;
            }
            const data = {
                item,
                range: ModelRange._createOn(item),
                markerName,
                markerRange
            };
            this.fire(eventName, data, conversionApi);
        }
    }
    /**
	 * Fires the conversion of the marker removal. Fires the {@link #event:removeMarker `removeMarker`} event with the provided data.
	 *
	 * @fires removeMarker
	 * @param markerName Marker name.
	 * @param markerRange The marker range.
	 * @param conversionApi The conversion API object.
	 */ _convertMarkerRemove(markerName, markerRange, conversionApi) {
        // Do not convert if range is in graveyard.
        if (markerRange.root.rootName == '$graveyard') {
            return;
        }
        this.fire(`removeMarker:${markerName}`, {
            markerName,
            markerRange
        }, conversionApi);
    }
    /**
	 * Fires the reduction of changes buffered in the {@link module:engine/model/differ~Differ `Differ`}.
	 *
	 * Features can replace selected {@link module:engine/model/differ~DifferItem `DifferItem`}s with `reinsert` entries to trigger
	 * reconversion. The {@link module:engine/conversion/downcasthelpers~DowncastHelpers#elementToStructure
	 * `DowncastHelpers.elementToStructure()`} is using this event to trigger reconversion.
	 *
	 * @fires reduceChanges
	 */ _reduceChanges(changes) {
        const data = {
            changes
        };
        this.fire('reduceChanges', data);
        return data.changes;
    }
    /**
	 * Populates provided {@link module:engine/conversion/modelconsumable~ModelConsumable} with values to consume from a given range,
	 * assuming that the range has just been inserted to the model.
	 *
	 * @param consumable The consumable.
	 * @param walkerValues The walker values for the inserted range.
	 * @returns The values to consume.
	 */ _addConsumablesForInsert(consumable, walkerValues) {
        for (const value of walkerValues){
            const item = value.item;
            // Add consumable if it wasn't there yet.
            if (consumable.test(item, 'insert') === null) {
                consumable.add(item, 'insert');
                for (const key of item.getAttributeKeys()){
                    consumable.add(item, 'attribute:' + key);
                }
            }
        }
        return consumable;
    }
    /**
	 * Populates provided {@link module:engine/conversion/modelconsumable~ModelConsumable} with values to consume for a given range.
	 *
	 * @param consumable The consumable.
	 * @param range The affected range.
	 * @param type Consumable type.
	 * @returns The values to consume.
	 */ _addConsumablesForRange(consumable, range, type) {
        for (const item of range.getItems()){
            consumable.add(item, type);
        }
        return consumable;
    }
    /**
	 * Populates provided {@link module:engine/conversion/modelconsumable~ModelConsumable} with selection consumable values.
	 *
	 * @param consumable The consumable.
	 * @param selection The selection to create the consumable from.
	 * @param markers Markers that contain the selection.
	 * @returns The values to consume.
	 */ _addConsumablesForSelection(consumable, selection, markers) {
        consumable.add(selection, 'selection');
        for (const marker of markers){
            consumable.add(selection, 'addMarker:' + marker.name);
        }
        for (const key of selection.getAttributeKeys()){
            consumable.add(selection, 'attribute:' + key);
        }
        return consumable;
    }
    /**
	 * Tests whether given event wasn't already fired and if so, fires it.
	 *
	 * @fires insert
	 * @fires attribute
	 * @param type Event type.
	 * @param data Event data.
	 * @param conversionApi The conversion API object.
	 */ _testAndFire(type, data, conversionApi) {
        const eventName = getEventName(type, data);
        const itemKey = data.item.is('$textProxy') ? conversionApi.consumable._getSymbolForTextProxy(data.item) : data.item;
        const eventsFiredForConversion = this._firedEventsMap.get(conversionApi);
        const eventsFiredForItem = eventsFiredForConversion.get(itemKey);
        if (!eventsFiredForItem) {
            eventsFiredForConversion.set(itemKey, new Set([
                eventName
            ]));
        } else if (!eventsFiredForItem.has(eventName)) {
            eventsFiredForItem.add(eventName);
        } else {
            return;
        }
        this.fire(eventName, data, conversionApi);
    }
    /**
	 * Fires not already fired events for setting attributes on just inserted item.
	 *
	 * @param item The model item to convert attributes for.
	 * @param conversionApi The conversion API object.
	 */ _testAndFireAddAttributes(item, conversionApi) {
        const data = {
            item,
            range: ModelRange._createOn(item)
        };
        for (const key of data.item.getAttributeKeys()){
            data.attributeKey = key;
            data.attributeOldValue = null;
            data.attributeNewValue = data.item.getAttribute(key);
            this._testAndFire(`attribute:${key}`, data, conversionApi);
        }
    }
    /**
	 * Builds an instance of the {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi} from a template and a given
	 * {@link module:engine/view/downcastwriter~ViewDowncastWriter `ViewDowncastWriter`} and options object.
	 *
	 * @param writer View writer that should be used to modify the view document.
	 * @param refreshedItems A set of model elements that should not reuse their
	 * previous view representations.
	 * @param options Optional options passed to `convertionApi.options`.
	 * @return The conversion API object.
	 */ _createConversionApi(writer, refreshedItems = new Set(), options = {}) {
        const conversionApi = {
            ...this._conversionApi,
            consumable: new ModelConsumable(),
            writer,
            options,
            convertItem: (item)=>this._convertInsert(ModelRange._createOn(item), conversionApi),
            convertChildren: (element)=>this._convertInsert(ModelRange._createIn(element), conversionApi, {
                    doNotAddConsumables: true
                }),
            convertAttributes: (item)=>this._testAndFireAddAttributes(item, conversionApi),
            canReuseView: (viewElement)=>!refreshedItems.has(conversionApi.mapper.toModelElement(viewElement))
        };
        this._firedEventsMap.set(conversionApi, new Map());
        return conversionApi;
    }
}
/**
 * Helper function, checks whether change of `marker` at `modelPosition` should be converted. Marker changes are not
 * converted if they happen inside an element with custom conversion method.
 */ function shouldMarkerChangeBeConverted(modelPosition, marker, mapper) {
    const range = marker.getRange();
    const ancestors = Array.from(modelPosition.getAncestors());
    ancestors.shift(); // Remove root element. It cannot be passed to `model.Range#containsItem`.
    ancestors.reverse();
    const hasCustomHandling = ancestors.some((element)=>{
        if (range.containsItem(element)) {
            const viewElement = mapper.toViewElement(element);
            return !!viewElement.getCustomProperty('addHighlight');
        }
    });
    return !hasCustomHandling;
}
function getEventName(type, data) {
    const name = data.item.is('element') ? data.item.name : '$text';
    return `${type}:${name}`;
}
function walkerValueToEventData(value) {
    return {
        item: value.item,
        range: ModelRange._createFromPositionAndShift(value.previousPosition, value.length)
    };
}

/**
 * Model node. Most basic structure of model tree.
 *
 * This is an abstract class that is a base for other classes representing different nodes in model.
 *
 * **Note:** If a node is detached from the model tree, you can manipulate it using it's API.
 * However, it is **very important** that nodes already attached to model tree should be only changed through
 * {@link module:engine/model/writer~ModelWriter Writer API}.
 *
 * Changes done by `Node` methods, like {@link module:engine/model/element~ModelElement#_insertChild _insertChild} or
 * {@link module:engine/model/node~ModelNode#_setAttribute _setAttribute}
 * do not generate {@link module:engine/model/operation/operation~Operation operations}
 * which are essential for correct editor work if you modify nodes in {@link module:engine/model/document~ModelDocument document} root.
 *
 * The flow of working on `Node` (and classes that inherits from it) is as such:
 * 1. You can create a `Node` instance, modify it using it's API.
 * 2. Add `Node` to the model using `Batch` API.
 * 3. Change `Node` that was already added to the model using `Batch` API.
 *
 * Similarly, you cannot use `Batch` API on a node that has not been added to the model tree, with the exception
 * of {@link module:engine/model/writer~ModelWriter#insert inserting} that node to the model tree.
 *
 * Be aware that using {@link module:engine/model/writer~ModelWriter#remove remove from Batch API} does not allow to use `Node` API because
 * the information about `Node` is still kept in model document.
 *
 * In case of {@link module:engine/model/element~ModelElement element node}, adding and removing children also counts as changing a node and
 * follows same rules.
 */ class ModelNode extends ModelTypeCheckable {
    /**
	 * Parent of this node. It could be {@link module:engine/model/element~ModelElement}
	 * or {@link module:engine/model/documentfragment~ModelDocumentFragment}.
	 * Equals to `null` if the node has no parent.
	 */ parent = null;
    /**
	 * Attributes set on this node.
	 */ _attrs;
    /**
	 * Index of this node in its parent or `null` if the node has no parent.
	 *
	 * @internal
	 */ _index = null;
    /**
	 * Offset at which this node starts in its parent or `null` if the node has no parent.
	 *
	 * @internal
	 */ _startOffset = null;
    /**
	 * Creates a model node.
	 *
	 * This is an abstract class, so this constructor should not be used directly.
	 *
	 * @param attrs Node's attributes. See {@link module:utils/tomap~toMap} for a list of accepted values.
	 */ constructor(attrs){
        super();
        this._attrs = toMap(attrs);
    }
    /**
	 * {@link module:engine/model/document~ModelDocument Document} that owns this root element.
	 */ get document() {
        return null;
    }
    /**
	 * Index of this node in its parent or `null` if the node has no parent.
	 */ get index() {
        return this._index;
    }
    /**
	 * Offset at which this node starts in its parent. It is equal to the sum of {@link #offsetSize offsetSize}
	 * of all its previous siblings. Equals to `null` if node has no parent.
	 */ get startOffset() {
        return this._startOffset;
    }
    /**
	 * Offset size of this node.
	 *
	 * Represents how much "offset space" is occupied by the node in its parent. It is important for
	 * {@link module:engine/model/position~ModelPosition position}. When node has `offsetSize` greater
	 * than `1`, position can be placed between that node start and end. `offsetSize` greater than `1` is for
	 * nodes that represents more than one entity, i.e. a {@link module:engine/model/text~ModelText text node}.
	 */ get offsetSize() {
        return 1;
    }
    /**
	 * Offset at which this node ends in its parent. It is equal to the sum of this node's
	 * {@link module:engine/model/node~ModelNode#startOffset start offset} and {@link #offsetSize offset size}.
	 * Equals to `null` if the node has no parent.
	 */ get endOffset() {
        if (this.startOffset === null) {
            return null;
        }
        return this.startOffset + this.offsetSize;
    }
    /**
	 * Node's next sibling or `null` if the node is a last child of it's parent or if the node has no parent.
	 */ get nextSibling() {
        const index = this.index;
        return index !== null && this.parent.getChild(index + 1) || null;
    }
    /**
	 * Node's previous sibling or `null` if the node is a first child of it's parent or if the node has no parent.
	 */ get previousSibling() {
        const index = this.index;
        return index !== null && this.parent.getChild(index - 1) || null;
    }
    /**
	 * The top-most ancestor of the node. If node has no parent it is the root itself. If the node is a part
	 * of {@link module:engine/model/documentfragment~ModelDocumentFragment}, it's `root` is equal to that `DocumentFragment`.
	 */ get root() {
        // eslint-disable-next-line @typescript-eslint/no-this-alias, consistent-this
        let root = this;
        while(root.parent){
            root = root.parent;
        }
        return root;
    }
    /**
	 * Returns `true` if the node is inside a document root that is attached to the document.
	 */ isAttached() {
        // If the node has no parent it means that it is a root.
        // But this is not a `RootElement`, so it means that it is not attached.
        //
        // If this is not the root, check if this element's root is attached.
        return this.parent === null ? false : this.root.isAttached();
    }
    /**
	 * Gets path to the node. The path is an array containing starting offsets of consecutive ancestors of this node,
	 * beginning from {@link module:engine/model/node~ModelNode#root root}, down to this node's starting offset. The path can be used to
	 * create {@link module:engine/model/position~ModelPosition Position} instance.
	 *
	 * ```ts
	 * const abc = new Text( 'abc' );
	 * const foo = new Text( 'foo' );
	 * const h1 = new ModelElement( 'h1', null, new Text( 'header' ) );
	 * const p = new ModelElement( 'p', null, [ abc, foo ] );
	 * const div = new ModelElement( 'div', null, [ h1, p ] );
	 * foo.getPath(); // Returns [ 1, 3 ]. `foo` is in `p` which is in `div`. `p` starts at offset 1, while `foo` at 3.
	 * h1.getPath(); // Returns [ 0 ].
	 * div.getPath(); // Returns [].
	 * ```
	 */ getPath() {
        const path = [];
        // eslint-disable-next-line @typescript-eslint/no-this-alias, consistent-this
        let node = this;
        while(node.parent){
            path.unshift(node.startOffset);
            node = node.parent;
        }
        return path;
    }
    /**
	 * Returns ancestors array of this node.
	 *
	 * @param options Options object.
	 * @param options.includeSelf When set to `true` this node will be also included in parent's array.
	 * @param options.parentFirst When set to `true`, array will be sorted from node's parent to root element,
	 * otherwise root element will be the first item in the array.
	 * @returns Array with ancestors.
	 */ getAncestors(options = {}) {
        const ancestors = [];
        let parent = options.includeSelf ? this : this.parent;
        while(parent){
            ancestors[options.parentFirst ? 'push' : 'unshift'](parent);
            parent = parent.parent;
        }
        return ancestors;
    }
    /**
	 * Returns a {@link module:engine/model/element~ModelElement} or {@link module:engine/model/documentfragment~ModelDocumentFragment}
	 * which is a common ancestor of both nodes.
	 *
	 * @param node The second node.
	 * @param options Options object.
	 * @param options.includeSelf When set to `true` both nodes will be considered "ancestors" too.
	 * Which means that if e.g. node A is inside B, then their common ancestor will be B.
	 */ getCommonAncestor(node, options = {}) {
        const ancestorsA = this.getAncestors(options);
        const ancestorsB = node.getAncestors(options);
        let i = 0;
        while(ancestorsA[i] == ancestorsB[i] && ancestorsA[i]){
            i++;
        }
        return i === 0 ? null : ancestorsA[i - 1];
    }
    /**
	 * Returns whether this node is before given node. `false` is returned if nodes are in different trees (for example,
	 * in different {@link module:engine/model/documentfragment~ModelDocumentFragment}s).
	 *
	 * @param node Node to compare with.
	 */ isBefore(node) {
        // Given node is not before this node if they are same.
        if (this == node) {
            return false;
        }
        // Return `false` if it is impossible to compare nodes.
        if (this.root !== node.root) {
            return false;
        }
        const thisPath = this.getPath();
        const nodePath = node.getPath();
        const result = compareArrays(thisPath, nodePath);
        switch(result){
            case 'prefix':
                return true;
            case 'extension':
                return false;
            default:
                return thisPath[result] < nodePath[result];
        }
    }
    /**
	 * Returns whether this node is after given node. `false` is returned if nodes are in different trees (for example,
	 * in different {@link module:engine/model/documentfragment~ModelDocumentFragment}s).
	 *
	 * @param node Node to compare with.
	 */ isAfter(node) {
        // Given node is not before this node if they are same.
        if (this == node) {
            return false;
        }
        // Return `false` if it is impossible to compare nodes.
        if (this.root !== node.root) {
            return false;
        }
        // In other cases, just check if the `node` is before, and return the opposite.
        return !this.isBefore(node);
    }
    /**
	 * Checks if the node has an attribute with given key.
	 *
	 * @param key Key of attribute to check.
	 * @returns `true` if attribute with given key is set on node, `false` otherwise.
	 */ hasAttribute(key) {
        return this._attrs.has(key);
    }
    /**
	 * Gets an attribute value for given key or `undefined` if that attribute is not set on node.
	 *
	 * @param key Key of attribute to look for.
	 * @returns Attribute value or `undefined`.
	 */ getAttribute(key) {
        return this._attrs.get(key);
    }
    /**
	 * Returns iterator that iterates over this node's attributes.
	 *
	 * Attributes are returned as arrays containing two items. First one is attribute key and second is attribute value.
	 * This format is accepted by native `Map` object and also can be passed in `Node` constructor.
	 */ getAttributes() {
        return this._attrs.entries();
    }
    /**
	 * Returns iterator that iterates over this node's attribute keys.
	 */ getAttributeKeys() {
        return this._attrs.keys();
    }
    /**
	 * Converts `Node` to plain object and returns it.
	 *
	 * @returns `Node` converted to plain object.
	 */ toJSON() {
        const json = {};
        // Serializes attributes to the object.
        // attributes = { a: 'foo', b: 1, c: true }.
        if (this._attrs.size) {
            json.attributes = Array.from(this._attrs).reduce((result, attr)=>{
                result[attr[0]] = attr[1];
                return result;
            }, {});
        }
        return json;
    }
    /**
	 * Creates a copy of this node, that is a node with exactly same attributes, and returns it.
	 *
	 * @internal
	 * @returns Node with same attributes as this node.
	 */ _clone(_deep) {
        return new this.constructor(this._attrs);
    }
    /**
	 * Removes this node from its parent.
	 *
	 * @internal
	 * @see module:engine/model/writer~ModelWriter#remove
	 */ _remove() {
        this.parent._removeChildren(this.index);
    }
    /**
	 * Sets attribute on the node. If attribute with the same key already is set, it's value is overwritten.
	 *
	 * @see module:engine/model/writer~ModelWriter#setAttribute
	 * @internal
	 * @param key Key of attribute to set.
	 * @param value Attribute value.
	 */ _setAttribute(key, value) {
        this._attrs.set(key, value);
    }
    /**
	 * Removes all attributes from the node and sets given attributes.
	 *
	 * @see module:engine/model/writer~ModelWriter#setAttributes
	 * @internal
	 * @param attrs Attributes to set. See {@link module:utils/tomap~toMap} for a list of accepted values.
	 */ _setAttributesTo(attrs) {
        this._attrs = toMap(attrs);
    }
    /**
	 * Removes an attribute with given key from the node.
	 *
	 * @see module:engine/model/writer~ModelWriter#removeAttribute
	 * @internal
	 * @param key Key of attribute to remove.
	 * @returns `true` if the attribute was set on the element, `false` otherwise.
	 */ _removeAttribute(key) {
        return this._attrs.delete(key);
    }
    /**
	 * Removes all attributes from the node.
	 *
	 * @see module:engine/model/writer~ModelWriter#clearAttributes
	 * @internal
	 */ _clearAttributes() {
        this._attrs.clear();
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ModelNode.prototype.is = function(type) {
    return type === 'node' || type === 'model:node';
};

/**
 * Selection is a set of {@link module:engine/model/range~ModelRange ranges}. It has a direction specified by its
 * {@link module:engine/model/selection~ModelSelection#anchor anchor} and {@link module:engine/model/selection~ModelSelection#focus focus}
 * (it can be {@link module:engine/model/selection~ModelSelection#isBackward forward or backward}).
 * Additionally, selection may have its own attributes (think – whether text typed in in this selection
 * should have those attributes – e.g. whether you type a bolded text).
 */ class ModelSelection extends /* #__PURE__ */ EmitterMixin(ModelTypeCheckable) {
    /**
	 * Specifies whether the last added range was added as a backward or forward range.
	 */ _lastRangeBackward = false;
    /**
	 * List of attributes set on current selection.
	 */ _attrs = new Map();
    /** @internal */ _ranges = [];
    /**
	 * Creates a new selection instance based on the given {@link module:engine/model/selection~ModelSelectable selectable}
	 * or creates an empty selection if no arguments were passed.
	 *
	 * ```ts
	 * // Creates empty selection without ranges.
	 * const selection = writer.createSelection();
	 *
	 * // Creates selection at the given range.
	 * const range = writer.createRange( start, end );
	 * const selection = writer.createSelection( range );
	 *
	 * // Creates selection at the given ranges
	 * const ranges = [ writer.createRange( start1, end2 ), writer.createRange( star2, end2 ) ];
	 * const selection = writer.createSelection( ranges );
	 *
	 * // Creates selection from the other selection.
	 * // Note: It doesn't copy selection attributes.
	 * const otherSelection = writer.createSelection();
	 * const selection = writer.createSelection( otherSelection );
	 *
	 * // Creates selection from the given document selection.
	 * // Note: It doesn't copy selection attributes.
	 * const documentSelection = model.document.selection;
	 * const selection = writer.createSelection( documentSelection );
	 *
	 * // Creates selection at the given position.
	 * const position = writer.createPositionFromPath( root, path );
	 * const selection = writer.createSelection( position );
	 *
	 * // Creates selection at the given offset in the given element.
	 * const paragraph = writer.createElement( 'paragraph' );
	 * const selection = writer.createSelection( paragraph, offset );
	 *
	 * // Creates a range inside an {@link module:engine/model/element~ModelElement element} which starts before the
	 * // first child of that element and ends after the last child of that element.
	 * const selection = writer.createSelection( paragraph, 'in' );
	 *
	 * // Creates a range on an {@link module:engine/model/item~ModelItem item} which starts before the item and ends
	 * // just after the item.
	 * const selection = writer.createSelection( paragraph, 'on' );
	 * ```
	 *
	 * Selection's constructor allow passing additional options (`'backward'`) as the last argument.
	 *
	 * ```ts
	 * // Creates backward selection.
	 * const selection = writer.createSelection( range, { backward: true } );
	 * ```
	 *
	 * @internal
	 */ constructor(...args){
        super();
        if (args.length) {
            this.setTo(...args);
        }
    }
    /**
	 * Selection anchor. Anchor is the position from which the selection was started. If a user is making a selection
	 * by dragging the mouse, the anchor is where the user pressed the mouse button (the beginning of the selection).
	 *
	 * Anchor and {@link #focus} define the direction of the selection, which is important
	 * when expanding/shrinking selection. The focus moves, while the anchor should remain in the same place.
	 *
	 * Anchor is always set to the {@link module:engine/model/range~ModelRange#start start} or
	 * {@link module:engine/model/range~ModelRange#end end} position of the last of selection's ranges. Whether it is
	 * the `start` or `end` depends on the specified `options.backward`. See the {@link #setTo `setTo()`} method.
	 *
	 * May be set to `null` if there are no ranges in the selection.
	 *
	 * @see #focus
	 */ get anchor() {
        if (this._ranges.length > 0) {
            const range = this._ranges[this._ranges.length - 1];
            return this._lastRangeBackward ? range.end : range.start;
        }
        return null;
    }
    /**
	 * Selection focus. Focus is the position where the selection ends. If a user is making a selection
	 * by dragging the mouse, the focus is where the mouse cursor is.
	 *
	 * May be set to `null` if there are no ranges in the selection.
	 *
	 * @see #anchor
	 */ get focus() {
        if (this._ranges.length > 0) {
            const range = this._ranges[this._ranges.length - 1];
            return this._lastRangeBackward ? range.start : range.end;
        }
        return null;
    }
    /**
	 * Whether the selection is collapsed. Selection is collapsed when there is exactly one range in it
	 * and it is collapsed.
	 */ get isCollapsed() {
        const length = this._ranges.length;
        if (length === 1) {
            return this._ranges[0].isCollapsed;
        } else {
            return false;
        }
    }
    /**
	 * Returns the number of ranges in the selection.
	 */ get rangeCount() {
        return this._ranges.length;
    }
    /**
	 * Specifies whether the selection's {@link #focus} precedes the selection's {@link #anchor}.
	 */ get isBackward() {
        return !this.isCollapsed && this._lastRangeBackward;
    }
    /**
	 * Checks whether this selection is equal to the given selection. Selections are equal if they have the same directions,
	 * the same number of ranges and all ranges from one selection equal to ranges from the another selection.
	 *
	 * @param otherSelection Selection to compare with.
	 * @returns `true` if selections are equal, `false` otherwise.
	 */ isEqual(otherSelection) {
        if (this.rangeCount != otherSelection.rangeCount) {
            return false;
        } else if (this.rangeCount === 0) {
            return true;
        }
        if (!this.anchor.isEqual(otherSelection.anchor) || !this.focus.isEqual(otherSelection.focus)) {
            return false;
        }
        for (const thisRange of this._ranges){
            let found = false;
            for (const otherRange of otherSelection._ranges){
                if (thisRange.isEqual(otherRange)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                return false;
            }
        }
        return true;
    }
    /**
	 * Returns an iterable object that iterates over copies of selection ranges.
	 */ *getRanges() {
        for (const range of this._ranges){
            yield new ModelRange(range.start, range.end);
        }
    }
    /**
	 * Returns a copy of the first range in the selection.
	 * First range is the one which {@link module:engine/model/range~ModelRange#start start} position
	 * {@link module:engine/model/position~ModelPosition#isBefore is before} start position of all other ranges
	 * (not to confuse with the first range added to the selection).
	 *
	 * Returns `null` if there are no ranges in selection.
	 */ getFirstRange() {
        let first = null;
        for (const range of this._ranges){
            if (!first || range.start.isBefore(first.start)) {
                first = range;
            }
        }
        return first ? new ModelRange(first.start, first.end) : null;
    }
    /**
	 * Returns a copy of the last range in the selection.
	 * Last range is the one which {@link module:engine/model/range~ModelRange#end end} position
	 * {@link module:engine/model/position~ModelPosition#isAfter is after} end position of all other
	 * ranges (not to confuse with the range most recently added to the selection).
	 *
	 * Returns `null` if there are no ranges in selection.
	 */ getLastRange() {
        let last = null;
        for (const range of this._ranges){
            if (!last || range.end.isAfter(last.end)) {
                last = range;
            }
        }
        return last ? new ModelRange(last.start, last.end) : null;
    }
    /**
	 * Returns the first position in the selection.
	 * First position is the position that {@link module:engine/model/position~ModelPosition#isBefore is before}
	 * any other position in the selection.
	 *
	 * Returns `null` if there are no ranges in selection.
	 */ getFirstPosition() {
        const first = this.getFirstRange();
        return first ? first.start.clone() : null;
    }
    /**
	 * Returns the last position in the selection.
	 * Last position is the position that {@link module:engine/model/position~ModelPosition#isAfter is after}
	 * any other position in the selection.
	 *
	 * Returns `null` if there are no ranges in selection.
	 */ getLastPosition() {
        const lastRange = this.getLastRange();
        return lastRange ? lastRange.end.clone() : null;
    }
    /**
	 * Sets this selection's ranges and direction to the specified location based on the given
	 * {@link module:engine/model/selection~ModelSelectable selectable}.
	 *
	 * ```ts
	 * // Removes all selection's ranges.
	 * selection.setTo( null );
	 *
	 * // Sets selection to the given range.
	 * const range = writer.createRange( start, end );
	 * selection.setTo( range );
	 *
	 * // Sets selection to given ranges.
	 * const ranges = [ writer.createRange( start1, end2 ), writer.createRange( star2, end2 ) ];
	 * selection.setTo( ranges );
	 *
	 * // Sets selection to other selection.
	 * // Note: It doesn't copy selection attributes.
	 * const otherSelection = writer.createSelection();
	 * selection.setTo( otherSelection );
	 *
	 * // Sets selection to the given document selection.
	 * // Note: It doesn't copy selection attributes.
	 * const documentSelection = new ModelDocumentSelection( doc );
	 * selection.setTo( documentSelection );
	 *
	 * // Sets collapsed selection at the given position.
	 * const position = writer.createPositionFromPath( root, path );
	 * selection.setTo( position );
	 *
	 * // Sets collapsed selection at the position of the given node and an offset.
	 * selection.setTo( paragraph, offset );
	 * ```
	 *
	 * Creates a range inside an {@link module:engine/model/element~ModelElement element} which starts before the first child of
 	 * that element and ends after the last child of that element.
	 *
	 * ```ts
	 * selection.setTo( paragraph, 'in' );
	 * ```
	 *
	 * Creates a range on an {@link module:engine/model/item~ModelItem item} which starts before the item and ends just after the item.
	 *
	 * ```ts
	 * selection.setTo( paragraph, 'on' );
	 * ```
	 *
	 * `Selection#setTo()`' method allow passing additional options (`backward`) as the last argument.
	 *
	 * ```ts
	 * // Sets backward selection.
	 * const selection = writer.createSelection( range, { backward: true } );
	 * ```
	 */ setTo(...args) {
        let [selectable, placeOrOffset, options] = args;
        if (typeof placeOrOffset == 'object') {
            options = placeOrOffset;
            placeOrOffset = undefined;
        }
        if (selectable === null) {
            this._setRanges([]);
        } else if (selectable instanceof ModelSelection) {
            this._setRanges(selectable.getRanges(), selectable.isBackward);
        } else if (selectable && typeof selectable.getRanges == 'function') {
            // We assume that the selectable is a ModelDocumentSelection.
            // It can't be imported here, because it would lead to circular imports.
            this._setRanges(selectable.getRanges(), selectable.isBackward);
        } else if (selectable instanceof ModelRange) {
            this._setRanges([
                selectable
            ], !!options && !!options.backward);
        } else if (selectable instanceof ModelPosition) {
            this._setRanges([
                new ModelRange(selectable)
            ]);
        } else if (selectable instanceof ModelNode) {
            const backward = !!options && !!options.backward;
            let range;
            if (placeOrOffset == 'in') {
                range = ModelRange._createIn(selectable);
            } else if (placeOrOffset == 'on') {
                range = ModelRange._createOn(selectable);
            } else if (placeOrOffset !== undefined) {
                range = new ModelRange(ModelPosition._createAt(selectable, placeOrOffset));
            } else {
                /**
				 * selection.setTo requires the second parameter when the first parameter is a node.
				 *
				 * @error model-selection-setto-required-second-parameter
				 */ throw new CKEditorError('model-selection-setto-required-second-parameter', [
                    this,
                    selectable
                ]);
            }
            this._setRanges([
                range
            ], backward);
        } else if (isIterable(selectable)) {
            // We assume that the selectable is an iterable of ranges.
            this._setRanges(selectable, options && !!options.backward);
        } else {
            /**
			 * Cannot set the selection to the given place.
			 *
			 * Invalid parameters were specified when setting the selection. Common issues:
			 *
			 * * A {@link module:engine/model/textproxy~ModelTextProxy} instance was passed instead of
			 * a real {@link module:engine/model/text~ModelText}.
			 * * View nodes were passed instead of model nodes.
			 * * `null`/`undefined` was passed.
			 *
			 * @error model-selection-setto-not-selectable
			 */ throw new CKEditorError('model-selection-setto-not-selectable', [
                this,
                selectable
            ]);
        }
    }
    /**
	 * Replaces all ranges that were added to the selection with given array of ranges. Last range of the array
	 * is treated like the last added range and is used to set {@link module:engine/model/selection~ModelSelection#anchor} and
	 * {@link module:engine/model/selection~ModelSelection#focus}. Accepts a flag describing in which direction the selection is made.
	 *
	 * @fires change:range
	 * @param newRanges Ranges to set.
	 * @param isLastBackward Flag describing if last added range was selected forward - from start to end (`false`)
	 * or backward - from end to start (`true`).
	 */ _setRanges(newRanges, isLastBackward = false) {
        const ranges = Array.from(newRanges);
        // Check whether there is any range in new ranges set that is different than all already added ranges.
        const anyNewRange = ranges.some((newRange)=>{
            if (!(newRange instanceof ModelRange)) {
                /**
				 * Selection range set to an object that is not an instance of {@link module:engine/model/range~ModelRange}.
				 *
				 * Only {@link module:engine/model/range~ModelRange} instances can be used to set a selection.
				 * Common mistakes leading to this error are:
				 *
				 * * using DOM `Range` object,
				 * * incorrect CKEditor 5 installation with multiple `ckeditor5-engine` packages having different versions.
				 *
				 * @error model-selection-set-ranges-not-range
				 */ throw new CKEditorError('model-selection-set-ranges-not-range', [
                    this,
                    newRanges
                ]);
            }
            return this._ranges.every((oldRange)=>{
                return !oldRange.isEqual(newRange);
            });
        });
        // Don't do anything if nothing changed.
        if (ranges.length === this._ranges.length && !anyNewRange) {
            return;
        }
        this._replaceAllRanges(ranges);
        this._lastRangeBackward = !!isLastBackward;
        this.fire('change:range', {
            directChange: true
        });
    }
    /**
	 * Moves {@link module:engine/model/selection~ModelSelection#focus} to the specified location.
	 *
	 * The location can be specified in the same form as
	 * {@link module:engine/model/writer~ModelWriter#createPositionAt writer.createPositionAt()} parameters.
	 *
	 * @fires change:range
	 * @param offset Offset or one of the flags. Used only when first parameter is a {@link module:engine/model/item~ModelItem model item}.
	 */ setFocus(itemOrPosition, offset) {
        if (this.anchor === null) {
            /**
			 * Cannot set selection focus if there are no ranges in selection.
			 *
			 * @error model-selection-setfocus-no-ranges
			 */ throw new CKEditorError('model-selection-setfocus-no-ranges', [
                this,
                itemOrPosition
            ]);
        }
        const newFocus = ModelPosition._createAt(itemOrPosition, offset);
        if (newFocus.compareWith(this.focus) == 'same') {
            return;
        }
        const anchor = this.anchor;
        if (this._ranges.length) {
            this._popRange();
        }
        if (newFocus.compareWith(anchor) == 'before') {
            this._pushRange(new ModelRange(newFocus, anchor));
            this._lastRangeBackward = true;
        } else {
            this._pushRange(new ModelRange(anchor, newFocus));
            this._lastRangeBackward = false;
        }
        this.fire('change:range', {
            directChange: true
        });
    }
    /**
	 * Gets an attribute value for given key or `undefined` if that attribute is not set on the selection.
	 *
	 * @param key Key of attribute to look for.
	 * @returns Attribute value or `undefined`.
	 */ getAttribute(key) {
        return this._attrs.get(key);
    }
    /**
	 * Returns iterable that iterates over this selection's attributes.
	 *
	 * Attributes are returned as arrays containing two items. First one is attribute key and second is attribute value.
	 * This format is accepted by native `Map` object and also can be passed in `Node` constructor.
	 */ getAttributes() {
        return this._attrs.entries();
    }
    /**
	 * Returns iterable that iterates over this selection's attribute keys.
	 */ getAttributeKeys() {
        return this._attrs.keys();
    }
    /**
	 * Checks if the selection has an attribute for given key.
	 *
	 * @param key Key of attribute to check.
	 * @returns `true` if attribute with given key is set on selection, `false` otherwise.
	 */ hasAttribute(key) {
        return this._attrs.has(key);
    }
    /**
	 * Removes an attribute with given key from the selection.
	 *
	 * If given attribute was set on the selection, fires the {@link #event:change:range} event with
	 * removed attribute key.
	 *
	 * @fires change:attribute
	 * @param key Key of attribute to remove.
	 */ removeAttribute(key) {
        if (this.hasAttribute(key)) {
            this._attrs.delete(key);
            this.fire('change:attribute', {
                attributeKeys: [
                    key
                ],
                directChange: true
            });
        }
    }
    /**
	 * Sets attribute on the selection. If attribute with the same key already is set, it's value is overwritten.
	 *
	 * If the attribute value has changed, fires the {@link #event:change:range} event with
	 * the attribute key.
	 *
	 * @fires change:attribute
	 * @param key Key of attribute to set.
	 * @param value Attribute value.
	 */ setAttribute(key, value) {
        if (this.getAttribute(key) !== value) {
            this._attrs.set(key, value);
            this.fire('change:attribute', {
                attributeKeys: [
                    key
                ],
                directChange: true
            });
        }
    }
    /**
	 * Returns the selected element. {@link module:engine/model/element~ModelElement Element} is considered as selected if there is only
	 * one range in the selection, and that range contains exactly one element.
	 * Returns `null` if there is no selected element.
	 */ getSelectedElement() {
        if (this.rangeCount !== 1) {
            return null;
        }
        return this.getFirstRange().getContainedElement();
    }
    /**
	 * Gets elements of type {@link module:engine/model/schema~ModelSchema#isBlock "block"} touched by the selection.
	 *
	 * This method's result can be used for example to apply block styling to all blocks covered by this selection.
	 *
	 * **Note:** `getSelectedBlocks()` returns blocks that are nested in other non-block elements
	 * but will not return blocks nested in other blocks.
	 *
	 * In this case the function will return exactly all 3 paragraphs (note: `<blockQuote>` is not a block itself):
	 *
	 * ```xml
	 * <paragraph>[a</paragraph>
	 * <blockQuote>
	 * 	<paragraph>b</paragraph>
	 * </blockQuote>
	 * <paragraph>c]d</paragraph>
	 * ```
	 *
	 * In this case the paragraph will also be returned, despite the collapsed selection:
	 *
	 * ```xml
	 * <paragraph>[]a</paragraph>
	 * ```
	 *
	 * In such a scenario, however, only blocks A, B & E will be returned as blocks C & D are nested in block B:
	 *
	 * ```xml
	 * [<blockA></blockA>
	 * <blockB>
	 * 	<blockC></blockC>
	 * 	<blockD></blockD>
	 * </blockB>
	 * <blockE></blockE>]
	 * ```
	 *
	 * If the selection is inside a block all the inner blocks (A & B) are returned:
	 *
	 * ```xml
	 * <block>
	 * 	<blockA>[a</blockA>
	 * 	<blockB>b]</blockB>
	 * </block>
	 * ```
	 *
	 * **Special case**: Selection ignores first and/or last blocks if nothing (from user perspective) is selected in them.
	 *
	 * ```xml
	 * // Selection ends and the beginning of the last block.
	 * <paragraph>[a</paragraph>
	 * <paragraph>b</paragraph>
	 * <paragraph>]c</paragraph> // This block will not be returned
	 *
	 * // Selection begins at the end of the first block.
	 * <paragraph>a[</paragraph> // This block will not be returned
	 * <paragraph>b</paragraph>
	 * <paragraph>c]</paragraph>
	 *
	 * // Selection begings at the end of the first block and ends at the beginning of the last block.
	 * <paragraph>a[</paragraph> // This block will not be returned
	 * <paragraph>b</paragraph>
	 * <paragraph>]c</paragraph> // This block will not be returned
	 * ```
	 */ *getSelectedBlocks() {
        const visited = new WeakSet();
        for (const range of this.getRanges()){
            // Get start block of range in case of a collapsed range.
            const startBlock = getParentBlock$1(range.start, visited);
            if (isStartBlockSelected(startBlock, range)) {
                yield startBlock;
            }
            const treewalker = range.getWalker();
            for (const value of treewalker){
                const block = value.item;
                if (value.type == 'elementEnd' && isUnvisitedTopBlock(block, visited, range)) {
                    yield block;
                } else if (value.type == 'elementStart' && block.is('model:element') && block.root.document.model.schema.isBlock(block)) {
                    treewalker.jumpTo(ModelPosition._createAt(block, 'end'));
                }
            }
            const endBlock = getParentBlock$1(range.end, visited);
            if (isEndBlockSelected(endBlock, range)) {
                yield endBlock;
            }
        }
    }
    /**
	 * Checks whether the selection contains the entire content of the given element. This means that selection must start
	 * at a position {@link module:engine/model/position~ModelPosition#isTouching touching} the element's start and ends at position
	 * touching the element's end.
	 *
	 * By default, this method will check whether the entire content of the selection's current root is selected.
	 * Useful to check if e.g. the user has just pressed <kbd>Ctrl</kbd> + <kbd>A</kbd>.
	 */ containsEntireContent(element = this.anchor.root) {
        const limitStartPosition = ModelPosition._createAt(element, 0);
        const limitEndPosition = ModelPosition._createAt(element, 'end');
        return limitStartPosition.isTouching(this.getFirstPosition()) && limitEndPosition.isTouching(this.getLastPosition());
    }
    /**
	 * Converts `Selection` to plain object and returns it.
	 *
	 * @returns `Selection` converted to plain object.
	 */ toJSON() {
        const json = {
            ranges: Array.from(this.getRanges()).map((range)=>range.toJSON())
        };
        const attributes = Object.fromEntries(this.getAttributes());
        if (Object.keys(attributes).length) {
            json.attributes = attributes;
        }
        if (this.isBackward) {
            json.isBackward = true;
        }
        return json;
    }
    /**
	 * Adds given range to internal {@link #_ranges ranges array}. Throws an error
	 * if given range is intersecting with any range that is already stored in this selection.
	 */ _pushRange(range) {
        this._checkRange(range);
        this._ranges.push(new ModelRange(range.start, range.end));
    }
    /**
	 * Checks if given range intersects with ranges that are already in the selection. Throws an error if it does.
	 */ _checkRange(range) {
        for(let i = 0; i < this._ranges.length; i++){
            if (range.isIntersecting(this._ranges[i])) {
                /**
				 * Trying to add a range that intersects with another range in the selection.
				 *
				 * @error model-selection-range-intersects
				 * @param {module:engine/model/range~ModelRange} addedRange Range that was added to the selection.
				 * @param {module:engine/model/range~ModelRange} intersectingRange Range in the selection that intersects with `addedRange`.
				 */ throw new CKEditorError('model-selection-range-intersects', [
                    this,
                    range
                ], {
                    addedRange: range,
                    intersectingRange: this._ranges[i]
                });
            }
        }
    }
    /**
	 * Replaces all the ranges by the given ones.
	 * Uses {@link #_popRange _popRange} and {@link #_pushRange _pushRange} to ensure proper ranges removal and addition.
	 */ _replaceAllRanges(ranges) {
        this._removeAllRanges();
        for (const range of ranges){
            this._pushRange(range);
        }
    }
    /**
	 * Deletes ranges from internal range array. Uses {@link #_popRange _popRange} to
	 * ensure proper ranges removal.
	 */ _removeAllRanges() {
        while(this._ranges.length > 0){
            this._popRange();
        }
    }
    /**
	 * Removes most recently added range from the selection.
	 */ _popRange() {
        this._ranges.pop();
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ModelSelection.prototype.is = function(type) {
    return type === 'selection' || type === 'model:selection';
};
/**
 * Checks whether the given element extends $block in the schema and has a parent (is not a root).
 * Marks it as already visited.
 */ function isUnvisitedBlock(element, visited) {
    if (visited.has(element)) {
        return false;
    }
    visited.add(element);
    return element.root.document.model.schema.isBlock(element) && !!element.parent;
}
/**
 * Checks if the given element is a $block was not previously visited and is a top block in a range.
 */ function isUnvisitedTopBlock(element, visited, range) {
    return isUnvisitedBlock(element, visited) && isTopBlockInRange(element, range);
}
/**
 * Finds the lowest element in position's ancestors which is a block.
 * It will search until first ancestor that is a limit element.
 * Marks all ancestors as already visited to not include any of them later on.
 */ function getParentBlock$1(position, visited) {
    const element = position.parent;
    const schema = element.root.document.model.schema;
    const ancestors = position.parent.getAncestors({
        parentFirst: true,
        includeSelf: true
    });
    let hasParentLimit = false;
    const block = ancestors.find((element)=>{
        // Stop searching after first parent node that is limit element.
        if (hasParentLimit) {
            return false;
        }
        hasParentLimit = schema.isLimit(element);
        return !hasParentLimit && isUnvisitedBlock(element, visited);
    });
    // Mark all ancestors of this position's parent, because find() might've stopped early and
    // the found block may be a child of another block.
    ancestors.forEach((element)=>visited.add(element));
    return block;
}
/**
 * Checks if the blocks is not nested in other block inside a range.
 */ function isTopBlockInRange(block, range) {
    const parentBlock = findAncestorBlock(block);
    if (!parentBlock) {
        return true;
    }
    // Add loose flag to check as parentRange can be equal to range.
    const isParentInRange = range.containsRange(ModelRange._createOn(parentBlock), true);
    return !isParentInRange;
}
/**
 * If a selection starts at the end of a block, that block is not returned as from the user's perspective this block wasn't selected.
 * See [#11585](https://github.com/ckeditor/ckeditor5/issues/11585) for more details.
 *
 * ```xml
 * <paragraph>a[</paragraph> // This block will not be returned
 * <paragraph>b</paragraph>
 * <paragraph>c]</paragraph>
 * ```
 *
 * Collapsed selection is not affected by it:
 *
 * ```xml
 * <paragraph>a[]</paragraph> // This block will be returned
 * ```
 */ function isStartBlockSelected(startBlock, range) {
    if (!startBlock) {
        return false;
    }
    if (range.isCollapsed || startBlock.isEmpty) {
        return true;
    }
    if (range.start.isTouching(ModelPosition._createAt(startBlock, startBlock.maxOffset))) {
        return false;
    }
    return isTopBlockInRange(startBlock, range);
}
/**
 * If a selection ends at the beginning of a block, that block is not returned as from the user's perspective this block wasn't selected.
 * See [#984](https://github.com/ckeditor/ckeditor5-engine/issues/984) for more details.
 *
 * ```xml
 * <paragraph>[a</paragraph>
 * <paragraph>b</paragraph>
 * <paragraph>]c</paragraph> // this block will not be returned
 * ```
 *
 * Collapsed selection is not affected by it:
 *
 * ```xml
 * <paragraph>[]a</paragraph> // this block will be returned
 * ```
 */ function isEndBlockSelected(endBlock, range) {
    if (!endBlock) {
        return false;
    }
    if (range.isCollapsed || endBlock.isEmpty) {
        return true;
    }
    if (range.end.isTouching(ModelPosition._createAt(endBlock, 0))) {
        return false;
    }
    return isTopBlockInRange(endBlock, range);
}
/**
 * Returns first ancestor block of a node.
 */ function findAncestorBlock(node) {
    const schema = node.root.document.model.schema;
    let parent = node.parent;
    while(parent){
        if (schema.isBlock(parent)) {
            return parent;
        }
        parent = parent.parent;
    }
}

/**
 * `ModelLiveRange` is a type of {@link module:engine/model/range~ModelRange Range}
 * that updates itself as {@link module:engine/model/document~ModelDocument document}
 * is changed through operations. It may be used as a bookmark.
 *
 * **Note:** Be very careful when dealing with `ModelLiveRange`. Each `ModelLiveRange` instance bind events that might
 * have to be unbound. Use {@link module:engine/model/liverange~ModelLiveRange#detach detach} whenever you don't need
 * `ModelLiveRange` anymore.
 */ class ModelLiveRange extends /* #__PURE__ */ EmitterMixin(ModelRange) {
    /**
	 * Creates a live range.
	 *
	 * @see module:engine/model/range~ModelRange
	 */ constructor(start, end){
        super(start, end);
        bindWithDocument$1.call(this);
    }
    /**
	 * Unbinds all events previously bound by `ModelLiveRange`. Use it whenever you don't need `ModelLiveRange` instance
	 * anymore (i.e. when leaving scope in which it was declared or before re-assigning variable that was
	 * referring to it).
	 */ detach() {
        this.stopListening();
    }
    /**
	 * Creates a {@link module:engine/model/range~ModelRange range instance} that is equal to this live range.
	 */ toRange() {
        return new ModelRange(this.start, this.end);
    }
    /**
	 * Creates a `ModelLiveRange` instance that is equal to the given range.
	 */ static fromRange(range) {
        return new ModelLiveRange(range.start, range.end);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ModelLiveRange.prototype.is = function(type) {
    return type === 'liveRange' || type === 'model:liveRange' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
    type == 'range' || type === 'model:range';
};
/**
 * Binds this `ModelLiveRange` to the {@link module:engine/model/document~ModelDocument document}
 * that owns this range's {@link module:engine/model/range~ModelRange#root root}.
 */ function bindWithDocument$1() {
    this.listenTo(this.root.document.model, 'applyOperation', (event, args)=>{
        const operation = args[0];
        if (!operation.isDocumentOperation) {
            return;
        }
        transform$2.call(this, operation);
    }, {
        priority: 'low'
    });
}
/**
 * Updates this range accordingly to the updates applied to the model. Bases on change events.
 */ function transform$2(operation) {
    // Transform the range by the operation. Join the result ranges if needed.
    const ranges = this.getTransformedByOperation(operation);
    const result = ModelRange._createFromRanges(ranges);
    const boundariesChanged = !result.isEqual(this);
    const contentChanged = doesOperationChangeRangeContent(this, operation);
    let deletionPosition = null;
    if (boundariesChanged) {
        // If range boundaries have changed, fire `change:range` event.
        //
        if (result.root.rootName == '$graveyard') {
            // If the range was moved to the graveyard root, set `deletionPosition`.
            if (operation.type == 'remove') {
                deletionPosition = operation.sourcePosition;
            } else {
                // Merge operation.
                deletionPosition = operation.deletionPosition;
            }
        }
        const oldRange = this.toRange();
        this.start = result.start;
        this.end = result.end;
        this.fire('change:range', oldRange, {
            deletionPosition
        });
    } else if (contentChanged) {
        // If range boundaries have not changed, but there was change inside the range, fire `change:content` event.
        this.fire('change:content', this.toRange(), {
            deletionPosition
        });
    }
}
/**
 * Checks whether given operation changes something inside the range (even if it does not change boundaries).
 */ function doesOperationChangeRangeContent(range, operation) {
    switch(operation.type){
        case 'insert':
            return range.containsPosition(operation.position);
        case 'move':
        case 'remove':
        case 'reinsert':
        case 'merge':
            return range.containsPosition(operation.sourcePosition) || range.start.isEqual(operation.sourcePosition) || range.containsPosition(operation.targetPosition);
        case 'split':
            return range.containsPosition(operation.splitPosition) || range.containsPosition(operation.insertionPosition);
    }
    return false;
}

// @if CK_DEBUG_ENGINE // const { convertMapToStringifiedObject } = require( '../dev-utils/utils' );
/**
 * Model text node. Type of {@link module:engine/model/node~ModelNode node} that
 * contains {@link module:engine/model/text~ModelText#data text data}.
 *
 * **Important:** see {@link module:engine/model/node~ModelNode} to read about restrictions using `Text` and `Node` API.
 *
 * **Note:** keep in mind that `Text` instances might indirectly got removed from model tree when model is changed.
 * This happens when {@link module:engine/model/writer~ModelWriter model writer} is used to change model and the text node is merged with
 * another text node. Then, both text nodes are removed and a new text node is inserted into the model. Because of
 * this behavior, keeping references to `Text` is not recommended. Instead, consider creating
 * {@link module:engine/model/liveposition~ModelLivePosition live position} placed before the text node.
 */ class ModelText extends ModelNode {
    /**
	 * Text data contained in this text node.
	 *
	 * @internal
	 */ _data;
    /**
	 * Creates a text node.
	 *
	 * **Note:** Constructor of this class shouldn't be used directly in the code.
	 * Use the {@link module:engine/model/writer~ModelWriter#createText} method instead.
	 *
	 * @internal
	 * @param data Node's text.
	 * @param attrs Node's attributes. See {@link module:utils/tomap~toMap} for a list of accepted values.
	 */ constructor(data, attrs){
        super(attrs);
        this._data = data || '';
    }
    /**
	 * @inheritDoc
	 */ get offsetSize() {
        return this.data.length;
    }
    /**
	 * Returns a text data contained in the node.
	 */ get data() {
        return this._data;
    }
    /**
	 * Converts `Text` instance to plain object and returns it.
	 *
	 * @returns`Text` instance converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        json.data = this.data;
        return json;
    }
    /**
	 * Creates a copy of this text node and returns it. Created text node has same text data and attributes as original text node.
	 *
	 * @internal
	 * @returns `Text` instance created using given plain object.
	 */ _clone() {
        return new ModelText(this.data, this.getAttributes());
    }
    /**
	 * Creates a `Text` instance from given plain object (i.e. parsed JSON string).
	 *
	 * @param json Plain object to be converted to `Text`.
	 * @returns `Text` instance created using given plain object.
	 */ static fromJSON(json) {
        return new ModelText(json.data, json.attributes);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ModelText.prototype.is = function(type) {
    return type === '$text' || type === 'model:$text' || // This are legacy values kept for backward compatibility.
    type === 'text' || type === 'model:text' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
    type === 'node' || type === 'model:node';
};

const storePrefix = 'selection:';
/**
 * `ModelDocumentSelection` is a special selection which is used as the
 * {@link module:engine/model/document~ModelDocument#selection document's selection}.
 * There can be only one instance of `ModelDocumentSelection` per document.
 *
 * Document selection can only be changed by using the {@link module:engine/model/writer~ModelWriter} instance
 * inside the {@link module:engine/model/model~Model#change `change()`} block, as it provides a secure way to modify model.
 *
 * `ModelDocumentSelection` is automatically updated upon changes in the {@link module:engine/model/document~ModelDocument document}
 * to always contain valid ranges. Its attributes are inherited from the text unless set explicitly.
 *
 * Differences between {@link module:engine/model/selection~ModelSelection} and `ModelDocumentSelection` are:
 * * there is always a range in `ModelDocumentSelection` - even if no ranges were added there is a "default range"
 * present in the selection,
 * * ranges added to this selection updates automatically when the document changes,
 * * attributes of `ModelDocumentSelection` are updated automatically according to selection ranges.
 *
 * Since `ModelDocumentSelection` uses {@link module:engine/model/liverange~ModelLiveRange live ranges}
 * and is updated when {@link module:engine/model/document~ModelDocument document}
 * changes, it cannot be set on {@link module:engine/model/node~ModelNode nodes}
 * that are inside {@link module:engine/model/documentfragment~ModelDocumentFragment document fragment}.
 * If you need to represent a selection in document fragment,
 * use {@link module:engine/model/selection~ModelSelection Selection class} instead.
 */ class ModelDocumentSelection extends /* #__PURE__ */ EmitterMixin(ModelTypeCheckable) {
    /**
	 * Selection used internally by that class (`ModelDocumentSelection` is a proxy to that selection).
	 */ _selection;
    /**
	 * Creates an empty live selection for given {@link module:engine/model/document~ModelDocument}.
	 *
	 * @param doc Document which owns this selection.
	 */ constructor(doc){
        super();
        this._selection = new LiveSelection(doc);
        this._selection.delegate('change:range').to(this);
        this._selection.delegate('change:attribute').to(this);
        this._selection.delegate('change:marker').to(this);
    }
    /**
	 * Describes whether the selection is collapsed. Selection is collapsed when there is exactly one range which is
	 * collapsed.
	 */ get isCollapsed() {
        return this._selection.isCollapsed;
    }
    /**
	 * Selection anchor. Anchor may be described as a position where the most recent part of the selection starts.
	 * Together with {@link #focus} they define the direction of selection, which is important
	 * when expanding/shrinking selection. Anchor is always {@link module:engine/model/range~ModelRange#start start} or
	 * {@link module:engine/model/range~ModelRange#end end} position of the most recently added range.
	 *
	 * Is set to `null` if there are no ranges in selection.
	 *
	 * @see #focus
	 */ get anchor() {
        return this._selection.anchor;
    }
    /**
	 * Selection focus. Focus is a position where the selection ends.
	 *
	 * Is set to `null` if there are no ranges in selection.
	 *
	 * @see #anchor
	 */ get focus() {
        return this._selection.focus;
    }
    /**
	 * Number of ranges in selection.
	 */ get rangeCount() {
        return this._selection.rangeCount;
    }
    /**
	 * Describes whether `Documentselection` has own range(s) set, or if it is defaulted to
	 * {@link module:engine/model/document~ModelDocument#_getDefaultRange document's default range}.
	 */ get hasOwnRange() {
        return this._selection.hasOwnRange;
    }
    /**
	 * Specifies whether the {@link #focus}
	 * precedes {@link #anchor}.
	 *
	 * @readonly
	 * @type {Boolean}
	 */ get isBackward() {
        return this._selection.isBackward;
    }
    /**
	 * Describes whether the gravity is overridden (using {@link module:engine/model/writer~ModelWriter#overrideSelectionGravity}) or not.
	 *
	 * Note that the gravity remains overridden as long as will not be restored the same number of times as it was overridden.
	 */ get isGravityOverridden() {
        return this._selection.isGravityOverridden;
    }
    /**
	 * A collection of selection {@link module:engine/model/markercollection~Marker markers}.
	 * Marker is a selection marker when selection range is inside the marker range.
	 *
	 * **Note**: Only markers from {@link ~ModelDocumentSelection#observeMarkers observed markers groups} are collected.
	 */ get markers() {
        return this._selection.markers;
    }
    /**
	 * Used for the compatibility with the {@link module:engine/model/selection~ModelSelection#isEqual} method.
	 *
	 * @internal
	 */ get _ranges() {
        return this._selection._ranges;
    }
    /**
	 * Returns an iterable that iterates over copies of selection ranges.
	 */ getRanges() {
        return this._selection.getRanges();
    }
    /**
	 * Returns the first position in the selection.
	 * First position is the position that {@link module:engine/model/position~ModelPosition#isBefore is before}
	 * any other position in the selection.
	 *
	 * Returns `null` if there are no ranges in selection.
	 */ getFirstPosition() {
        return this._selection.getFirstPosition();
    }
    /**
	 * Returns the last position in the selection.
	 * Last position is the position that {@link module:engine/model/position~ModelPosition#isAfter is after}
	 * any other position in the selection.
	 *
	 * Returns `null` if there are no ranges in selection.
	 */ getLastPosition() {
        return this._selection.getLastPosition();
    }
    /**
	 * Returns a copy of the first range in the selection.
	 * First range is the one which {@link module:engine/model/range~ModelRange#start start} position
	 * {@link module:engine/model/position~ModelPosition#isBefore is before} start position of all other ranges
	 * (not to confuse with the first range added to the selection).
	 *
	 * Returns `null` if there are no ranges in selection.
	 */ getFirstRange() {
        return this._selection.getFirstRange();
    }
    /**
	 * Returns a copy of the last range in the selection.
	 * Last range is the one which {@link module:engine/model/range~ModelRange#end end} position
	 * {@link module:engine/model/position~ModelPosition#isAfter is after} end position of all
	 * other ranges (not to confuse with the range most recently added to the selection).
	 *
	 * Returns `null` if there are no ranges in selection.
	 */ getLastRange() {
        return this._selection.getLastRange();
    }
    /**
	 * Gets elements of type {@link module:engine/model/schema~ModelSchema#isBlock "block"} touched by the selection.
	 *
	 * This method's result can be used for example to apply block styling to all blocks covered by this selection.
	 *
	 * **Note:** `getSelectedBlocks()` returns blocks that are nested in other non-block elements
	 * but will not return blocks nested in other blocks.
	 *
	 * In this case the function will return exactly all 3 paragraphs (note: `<blockQuote>` is not a block itself):
	 *
	 * ```
	 * <paragraph>[a</paragraph>
	 * <blockQuote>
	 * 	<paragraph>b</paragraph>
	 * </blockQuote>
	 * <paragraph>c]d</paragraph>
	 * ```
	 *
	 * In this case the paragraph will also be returned, despite the collapsed selection:
	 *
	 * ```
	 * <paragraph>[]a</paragraph>
	 * ```
	 *
	 * In such a scenario, however, only blocks A, B & E will be returned as blocks C & D are nested in block B:
	 *
	 * ```
	 * [<blockA></blockA>
	 * <blockB>
	 * 	<blockC></blockC>
	 * 	<blockD></blockD>
	 * </blockB>
	 * <blockE></blockE>]
	 * ```
	 *
	 * If the selection is inside a block all the inner blocks (A & B) are returned:
	 *
	 * ```
	 * <block>
	 * 	<blockA>[a</blockA>
	 * 	<blockB>b]</blockB>
	 * </block>
	 * ```
	 *
	 * **Special case**: If a selection ends at the beginning of a block, that block is not returned as from user perspective
	 * this block wasn't selected. See [#984](https://github.com/ckeditor/ckeditor5-engine/issues/984) for more details.
	 *
	 * ```
	 * <paragraph>[a</paragraph>
	 * <paragraph>b</paragraph>
	 * <paragraph>]c</paragraph> // this block will not be returned
	 * ```
	 */ getSelectedBlocks() {
        return this._selection.getSelectedBlocks();
    }
    /**
	 * Returns the selected element. {@link module:engine/model/element~ModelElement Element} is considered as selected if there is only
	 * one range in the selection, and that range contains exactly one element.
	 * Returns `null` if there is no selected element.
	 */ getSelectedElement() {
        return this._selection.getSelectedElement();
    }
    /**
	 * Checks whether the selection contains the entire content of the given element. This means that selection must start
	 * at a position {@link module:engine/model/position~ModelPosition#isTouching touching} the element's start and ends at position
	 * touching the element's end.
	 *
	 * By default, this method will check whether the entire content of the selection's current root is selected.
	 * Useful to check if e.g. the user has just pressed <kbd>Ctrl</kbd> + <kbd>A</kbd>.
	 */ containsEntireContent(element) {
        return this._selection.containsEntireContent(element);
    }
    /**
	 * Unbinds all events previously bound by document selection.
	 */ destroy() {
        this._selection.destroy();
    }
    /**
	 * Returns iterable that iterates over this selection's attribute keys.
	 */ getAttributeKeys() {
        return this._selection.getAttributeKeys();
    }
    /**
	 * Returns iterable that iterates over this selection's attributes.
	 *
	 * Attributes are returned as arrays containing two items. First one is attribute key and second is attribute value.
	 * This format is accepted by native `Map` object and also can be passed in `Node` constructor.
	 */ getAttributes() {
        return this._selection.getAttributes();
    }
    /**
	 * Gets an attribute value for given key or `undefined` if that attribute is not set on the selection.
	 *
	 * @param key Key of attribute to look for.
	 * @returns Attribute value or `undefined`.
	 */ getAttribute(key) {
        return this._selection.getAttribute(key);
    }
    /**
	 * Checks if the selection has an attribute for given key.
	 *
	 * @param key Key of attribute to check.
	 * @returns `true` if attribute with given key is set on selection, `false` otherwise.
	 */ hasAttribute(key) {
        return this._selection.hasAttribute(key);
    }
    /**
	 * Refreshes selection attributes and markers according to the current position in the model.
	 */ refresh() {
        this._selection.updateMarkers();
        this._selection._updateAttributes(false);
    }
    /**
	 * Registers a marker group prefix or a marker name to be collected in the
	 * {@link ~ModelDocumentSelection#markers selection markers collection}.
	 *
	 * See also {@link module:engine/model/markercollection~MarkerCollection#getMarkersGroup `MarkerCollection#getMarkersGroup()`}.
	 *
	 * @param prefixOrName The marker group prefix or marker name.
	 */ observeMarkers(prefixOrName) {
        this._selection.observeMarkers(prefixOrName);
    }
    /**
	 * Converts `DocumentSelection` to plain object and returns it.
	 *
	 * @returns `DocumentSelection` converted to plain object.
	 */ toJSON() {
        return this._selection.toJSON();
    }
    /**
	 * Moves {@link module:engine/model/documentselection~ModelDocumentSelection#focus} to the specified location.
	 * Should be used only within the {@link module:engine/model/writer~ModelWriter#setSelectionFocus} method.
	 *
	 * The location can be specified in the same form as
	 * {@link module:engine/model/writer~ModelWriter#createPositionAt writer.createPositionAt()} parameters.
	 *
	 * @see module:engine/model/writer~ModelWriter#setSelectionFocus
	 * @internal
	 * @param offset Offset or one of the flags. Used only when
	 * first parameter is a {@link module:engine/model/item~ModelItem model item}.
	 */ _setFocus(itemOrPosition, offset) {
        this._selection.setFocus(itemOrPosition, offset);
    }
    /**
	 * Sets this selection's ranges and direction to the specified location based on the given
	 * {@link module:engine/model/selection~ModelSelectable selectable}.
	 * Should be used only within the {@link module:engine/model/writer~ModelWriter#setSelection} method.
	 *
	 * @see module:engine/model/writer~ModelWriter#setSelection
	 * @internal
	 */ _setTo(...args) {
        this._selection.setTo(...args);
    }
    /**
	 * Sets attribute on the selection. If attribute with the same key already is set, it's value is overwritten.
	 * Should be used only within the {@link module:engine/model/writer~ModelWriter#setSelectionAttribute} method.
	 *
	 * @see module:engine/model/writer~ModelWriter#setSelectionAttribute
	 * @internal
	 * @param key Key of the attribute to set.
	 * @param value Attribute value.
	 */ _setAttribute(key, value) {
        this._selection.setAttribute(key, value);
    }
    /**
	 * Removes an attribute with given key from the selection.
	 * If the given attribute was set on the selection, fires the {@link module:engine/model/selection~ModelSelection#event:change:range}
	 * event with removed attribute key.
	 * Should be used only within the {@link module:engine/model/writer~ModelWriter#removeSelectionAttribute} method.
	 *
	 * @see module:engine/model/writer~ModelWriter#removeSelectionAttribute
	 * @internal
	 * @param key Key of the attribute to remove.
	 */ _removeAttribute(key) {
        this._selection.removeAttribute(key);
    }
    /**
	 * Returns an iterable that iterates through all selection attributes stored in current selection's parent.
	 *
	 * @internal
	 */ _getStoredAttributes() {
        return this._selection.getStoredAttributes();
    }
    /**
	 * Temporarily changes the gravity of the selection from the left to the right.
	 *
	 * The gravity defines from which direction the selection inherits its attributes. If it's the default left
	 * gravity, the selection (after being moved by the the user) inherits attributes from its left hand side.
	 * This method allows to temporarily override this behavior by forcing the gravity to the right.
	 *
	 * It returns an unique identifier which is required to restore the gravity. It guarantees the symmetry
	 * of the process.
	 *
	 * @see module:engine/model/writer~ModelWriter#overrideSelectionGravity
	 * @internal
	 * @returns The unique id which allows restoring the gravity.
	 */ _overrideGravity() {
        return this._selection.overrideGravity();
    }
    /**
	 * Restores the {@link ~ModelDocumentSelection#_overrideGravity overridden gravity}.
	 *
	 * Restoring the gravity is only possible using the unique identifier returned by
	 * {@link ~ModelDocumentSelection#_overrideGravity}. Note that the gravity remains overridden as long as won't be restored
	 * the same number of times it was overridden.
	 *
	 * @see module:engine/model/writer~ModelWriter#restoreSelectionGravity
	 * @internal
	 * @param uid The unique id returned by {@link #_overrideGravity}.
	 */ _restoreGravity(uid) {
        this._selection.restoreGravity(uid);
    }
    /**
	 * Generates and returns an attribute key for selection attributes store, basing on original attribute key.
	 *
	 * @internal
	 * @param key Attribute key to convert.
	 * @returns Converted attribute key, applicable for selection store.
	 */ static _getStoreAttributeKey(key) {
        return storePrefix + key;
    }
    /**
	 * Checks whether the given attribute key is an attribute stored on an element.
	 *
	 * @internal
	 */ static _isStoreAttributeKey(key) {
        return key.startsWith(storePrefix);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ModelDocumentSelection.prototype.is = function(type) {
    return type === 'selection' || type == 'model:selection' || type == 'documentSelection' || type == 'model:documentSelection';
};
/**
 * `LiveSelection` is used internally by {@link module:engine/model/documentselection~ModelDocumentSelection}
 * and shouldn't be used directly.
 *
 * `LiveSelection` is automatically updated upon changes in the {@link module:engine/model/document~ModelDocument document}
 * to always contain valid ranges. Its attributes are inherited from the text unless set explicitly.
 *
 * Differences between {@link module:engine/model/selection~ModelSelection} and `LiveSelection` are:
 * * there is always a range in `LiveSelection` - even if no ranges were added there is a "default range"
 * present in the selection,
 * * ranges added to this selection updates automatically when the document changes,
 * * attributes of `LiveSelection` are updated automatically according to selection ranges.
 */ class LiveSelection extends ModelSelection {
    /**
	 * List of selection markers.
	 * Marker is a selection marker when selection range is inside the marker range.
	 */ markers = new Collection({
        idProperty: 'name'
    });
    /**
	 * Document which owns this selection.
	 */ _model;
    /**
	 * Document which owns this selection.
	 */ _document;
    /**
	 * Keeps mapping of attribute name to priority with which the attribute got modified (added/changed/removed)
	 * last time. Possible values of priority are: `'low'` and `'normal'`.
	 *
	 * Priorities are used by internal `LiveSelection` mechanisms. All attributes set using `LiveSelection`
	 * attributes API are set with `'normal'` priority.
	 */ _attributePriority = new Map();
    /**
	 * Position to which the selection should be set if the last selection range was moved to the graveyard.
	 */ _selectionRestorePosition = null;
    /**
	 * Flag that informs whether the selection ranges have changed. It is changed on true when `LiveRange#change:range` event is fired.
	 */ _hasChangedRange = false;
    /**
	 * Each overriding gravity adds an UID to the set and each removal removes it.
	 * Gravity is overridden when there's at least one UID in the set.
	 * Gravity is restored when the set is empty.
	 * This is to prevent conflicts when gravity is overridden by more than one feature at the same time.
	 */ _overriddenGravityRegister = new Set();
    /**
	 * Prefixes of marker names that should affect `LiveSelection#markers` collection.
	 */ _observedMarkers = new Set();
    /**
	 * Creates an empty live selection for given {@link module:engine/model/document~ModelDocument}.
	 *
	 * @param doc Document which owns this selection.
	 */ constructor(doc){
        super();
        this._model = doc.model;
        this._document = doc;
        // Ensure selection is correct after each operation.
        this.listenTo(this._model, 'applyOperation', (evt, args)=>{
            const operation = args[0];
            if (!operation.isDocumentOperation || operation.type == 'marker' || operation.type == 'rename' || operation.type == 'noop') {
                return;
            }
            // Fix selection if the last range was removed from it and we have a position to which we can restore the selection.
            if (this._ranges.length == 0 && this._selectionRestorePosition) {
                this._fixGraveyardSelection(this._selectionRestorePosition);
            }
            // "Forget" the restore position even if it was not "used".
            this._selectionRestorePosition = null;
            if (this._hasChangedRange) {
                this._hasChangedRange = false;
                this.fire('change:range', {
                    directChange: false
                });
            }
        }, {
            priority: 'lowest'
        });
        // Ensure selection is correct and up to date after each range change.
        this.on('change:range', ()=>{
            this._validateSelectionRanges(this.getRanges());
        });
        // Update markers data stored by the selection after each marker change.
        // This handles only marker changes done through marker operations (not model tree changes).
        this.listenTo(this._model.markers, 'update', (evt, marker, oldRange, newRange)=>{
            this._updateMarker(marker, newRange);
        });
        // Ensure selection is up to date after each change block.
        this.listenTo(this._document, 'change', (evt, batch)=>{
            clearAttributesStoredInElement(this._model, batch);
        });
    }
    get isCollapsed() {
        const length = this._ranges.length;
        return length === 0 ? this._document._getDefaultRange().isCollapsed : super.isCollapsed;
    }
    get anchor() {
        return super.anchor || this._document._getDefaultRange().start;
    }
    get focus() {
        return super.focus || this._document._getDefaultRange().end;
    }
    get rangeCount() {
        return this._ranges.length ? this._ranges.length : 1;
    }
    /**
	 * Describes whether `LiveSelection` has own range(s) set, or if it is defaulted to
	 * {@link module:engine/model/document~ModelDocument#_getDefaultRange document's default range}.
	 */ get hasOwnRange() {
        return this._ranges.length > 0;
    }
    /**
	 * When set to `true` then selection attributes on node before the caret won't be taken
	 * into consideration while updating selection attributes.
	 */ get isGravityOverridden() {
        return !!this._overriddenGravityRegister.size;
    }
    /**
	 * Unbinds all events previously bound by live selection.
	 */ destroy() {
        for(let i = 0; i < this._ranges.length; i++){
            this._ranges[i].detach();
        }
        this.stopListening();
    }
    *getRanges() {
        if (this._ranges.length) {
            yield* super.getRanges();
        } else {
            yield this._document._getDefaultRange();
        }
    }
    getFirstRange() {
        return super.getFirstRange() || this._document._getDefaultRange();
    }
    getLastRange() {
        return super.getLastRange() || this._document._getDefaultRange();
    }
    setTo(...args) {
        super.setTo(...args);
        this._updateAttributes(true);
        this.updateMarkers();
    }
    setFocus(itemOrPosition, offset) {
        super.setFocus(itemOrPosition, offset);
        this._updateAttributes(true);
        this.updateMarkers();
    }
    setAttribute(key, value) {
        if (this._setAttribute(key, value)) {
            // Fire event with exact data.
            const attributeKeys = [
                key
            ];
            this.fire('change:attribute', {
                attributeKeys,
                directChange: true
            });
        }
    }
    removeAttribute(key) {
        if (this._removeAttribute(key)) {
            // Fire event with exact data.
            const attributeKeys = [
                key
            ];
            this.fire('change:attribute', {
                attributeKeys,
                directChange: true
            });
        }
    }
    overrideGravity() {
        const overrideUid = uid();
        // Remember that another overriding has been requested. It will need to be removed
        // before the gravity is to be restored.
        this._overriddenGravityRegister.add(overrideUid);
        if (this._overriddenGravityRegister.size === 1) {
            this._updateAttributes(true);
        }
        return overrideUid;
    }
    restoreGravity(uid) {
        if (!this._overriddenGravityRegister.has(uid)) {
            /**
			 * Restoring gravity for an unknown UID is not possible. Make sure you are using a correct
			 * UID obtained from the {@link module:engine/model/writer~ModelWriter#overrideSelectionGravity} to restore.
			 *
			 * @error document-selection-gravity-wrong-restore
			 * @param {string} uid The unique identifier returned by
			 * {@link module:engine/model/documentselection~ModelDocumentSelection#_overrideGravity}.
			 */ throw new CKEditorError('document-selection-gravity-wrong-restore', this, {
                uid
            });
        }
        this._overriddenGravityRegister.delete(uid);
        // Restore gravity only when all overriding have been restored.
        if (!this.isGravityOverridden) {
            this._updateAttributes(true);
        }
    }
    observeMarkers(prefixOrName) {
        this._observedMarkers.add(prefixOrName);
        this.updateMarkers();
    }
    _replaceAllRanges(ranges) {
        this._validateSelectionRanges(ranges);
        super._replaceAllRanges(ranges);
    }
    _popRange() {
        this._ranges.pop().detach();
    }
    _pushRange(range) {
        const liveRange = this._prepareRange(range);
        // `undefined` is returned when given `range` is in graveyard root.
        if (liveRange) {
            this._ranges.push(liveRange);
        }
    }
    /**
	 * Converts `LiveSelection` to plain object and returns it.
	 *
	 * @returns `LiveSelection` converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        if (this.markers.length) {
            json.markers = this.markers.map((marker)=>marker.toJSON());
        }
        return json;
    }
    _validateSelectionRanges(ranges) {
        for (const range of ranges){
            if (!this._document._validateSelectionRange(range)) {
                /**
				 * Range from {@link module:engine/model/documentselection~ModelDocumentSelection document selection}
				 * starts or ends at incorrect position.
				 *
				 * @error document-selection-wrong-position
				 * @param {module:engine/model/range~ModelRange} range The invalid range.
				 */ throw new CKEditorError('document-selection-wrong-position', this, {
                    range
                });
            }
        }
    }
    /**
	 * Prepares given range to be added to selection. Checks if it is correct,
	 * converts it to {@link module:engine/model/liverange~ModelLiveRange ModelLiveRange}
	 * and sets listeners listening to the range's change event.
	 */ _prepareRange(range) {
        this._checkRange(range);
        if (range.root == this._document.graveyard) {
            // @if CK_DEBUG // console.warn( 'Trying to add a Range that is in the graveyard root. Range rejected.' );
            return;
        }
        const liveRange = ModelLiveRange.fromRange(range);
        // If selection range is moved to the graveyard remove it from the selection object.
        // Also, save some data that can be used to restore selection later, on `Model#applyOperation` event.
        liveRange.on('change:range', (evt, oldRange, data)=>{
            this._hasChangedRange = true;
            if (liveRange.root == this._document.graveyard) {
                this._selectionRestorePosition = data.deletionPosition;
                const index = this._ranges.indexOf(liveRange);
                this._ranges.splice(index, 1);
                liveRange.detach();
            }
        });
        return liveRange;
    }
    updateMarkers() {
        if (!this._observedMarkers.size) {
            return;
        }
        const markers = [];
        let changed = false;
        for (const marker of this._model.markers){
            const markerGroup = marker.name.split(':', 1)[0];
            if (!this._observedMarkers.has(markerGroup)) {
                continue;
            }
            const markerRange = marker.getRange();
            for (const selectionRange of this.getRanges()){
                if (markerRange.containsRange(selectionRange, !selectionRange.isCollapsed)) {
                    markers.push(marker);
                }
            }
        }
        const oldMarkers = Array.from(this.markers);
        for (const marker of markers){
            if (!this.markers.has(marker)) {
                this.markers.add(marker);
                changed = true;
            }
        }
        for (const marker of Array.from(this.markers)){
            if (!markers.includes(marker)) {
                this.markers.remove(marker);
                changed = true;
            }
        }
        if (changed) {
            this.fire('change:marker', {
                oldMarkers,
                directChange: false
            });
        }
    }
    _updateMarker(marker, markerRange) {
        const markerGroup = marker.name.split(':', 1)[0];
        if (!this._observedMarkers.has(markerGroup)) {
            return;
        }
        let changed = false;
        const oldMarkers = Array.from(this.markers);
        const hasMarker = this.markers.has(marker);
        if (!markerRange) {
            if (hasMarker) {
                this.markers.remove(marker);
                changed = true;
            }
        } else {
            let contained = false;
            for (const selectionRange of this.getRanges()){
                if (markerRange.containsRange(selectionRange, !selectionRange.isCollapsed)) {
                    contained = true;
                    break;
                }
            }
            if (contained && !hasMarker) {
                this.markers.add(marker);
                changed = true;
            } else if (!contained && hasMarker) {
                this.markers.remove(marker);
                changed = true;
            }
        }
        if (changed) {
            this.fire('change:marker', {
                oldMarkers,
                directChange: false
            });
        }
    }
    /**
	 * Updates this selection attributes according to its ranges and the {@link module:engine/model/document~ModelDocument model document}.
	 */ _updateAttributes(clearAll) {
        const newAttributes = toMap(this._getSurroundingAttributes());
        const oldAttributes = toMap(this.getAttributes());
        if (clearAll) {
            // If `clearAll` remove all attributes and reset priorities.
            this._attributePriority = new Map();
            this._attrs = new Map();
        } else {
            // If not, remove only attributes added with `low` priority.
            for (const [key, priority] of this._attributePriority){
                if (priority == 'low') {
                    this._attrs.delete(key);
                    this._attributePriority.delete(key);
                }
            }
        }
        this._setAttributesTo(newAttributes);
        // Let's evaluate which attributes really changed.
        const changed = [];
        // First, loop through all attributes that are set on selection right now.
        // Check which of them are different than old attributes.
        for (const [newKey, newValue] of this.getAttributes()){
            if (!oldAttributes.has(newKey) || oldAttributes.get(newKey) !== newValue) {
                changed.push(newKey);
            }
        }
        // Then, check which of old attributes got removed.
        for (const [oldKey] of oldAttributes){
            if (!this.hasAttribute(oldKey)) {
                changed.push(oldKey);
            }
        }
        // Fire event with exact data (fire only if anything changed).
        if (changed.length > 0) {
            this.fire('change:attribute', {
                attributeKeys: changed,
                directChange: false
            });
        }
    }
    /**
	 * Internal method for setting `LiveSelection` attribute. Supports attribute priorities (through `directChange`
	 * parameter).
	 */ _setAttribute(key, value, directChange = true) {
        const priority = directChange ? 'normal' : 'low';
        if (priority == 'low' && this._attributePriority.get(key) == 'normal') {
            // Priority too low.
            return false;
        }
        const oldValue = super.getAttribute(key);
        // Don't do anything if value has not changed.
        if (oldValue === value) {
            return false;
        }
        this._attrs.set(key, value);
        // Update priorities map.
        this._attributePriority.set(key, priority);
        return true;
    }
    /**
	 * Internal method for removing `LiveSelection` attribute. Supports attribute priorities (through `directChange`
	 * parameter).
	 *
	 * NOTE: Even if attribute is not present in the selection but is provided to this method, it's priority will
	 * be changed according to `directChange` parameter.
	 */ _removeAttribute(key, directChange = true) {
        const priority = directChange ? 'normal' : 'low';
        if (priority == 'low' && this._attributePriority.get(key) == 'normal') {
            // Priority too low.
            return false;
        }
        // Update priorities map.
        this._attributePriority.set(key, priority);
        // Don't do anything if value has not changed.
        if (!super.hasAttribute(key)) {
            return false;
        }
        this._attrs.delete(key);
        return true;
    }
    /**
	 * Internal method for setting multiple `LiveSelection` attributes. Supports attribute priorities (through
	 * `directChange` parameter).
	 */ _setAttributesTo(attrs) {
        const changed = new Set();
        for (const [oldKey, oldValue] of this.getAttributes()){
            // Do not remove attribute if attribute with same key and value is about to be set.
            if (attrs.get(oldKey) === oldValue) {
                continue;
            }
            // All rest attributes will be removed so changed attributes won't change .
            this._removeAttribute(oldKey, false);
        }
        for (const [key, value] of attrs){
            // Attribute may not be set because of attributes or because same key/value is already added.
            const gotAdded = this._setAttribute(key, value, false);
            if (gotAdded) {
                changed.add(key);
            }
        }
        return changed;
    }
    /**
	 * Returns an iterable that iterates through all selection attributes stored in current selection's parent.
	 */ *getStoredAttributes() {
        const selectionParent = this.getFirstPosition().parent;
        if (this.isCollapsed && selectionParent.isEmpty) {
            for (const key of selectionParent.getAttributeKeys()){
                if (key.startsWith(storePrefix)) {
                    const realKey = key.substr(storePrefix.length);
                    yield [
                        realKey,
                        selectionParent.getAttribute(key)
                    ];
                }
            }
        }
    }
    /**
	 * Checks model text nodes that are closest to the selection's first position and returns attributes of first
	 * found element. If there are no text nodes in selection's first position parent, it returns selection
	 * attributes stored in that parent.
	 */ _getSurroundingAttributes() {
        const position = this.getFirstPosition();
        const schema = this._model.schema;
        if (position.root.rootName == '$graveyard') {
            return null;
        }
        let attrs = null;
        if (!this.isCollapsed) {
            // 1. If selection is a range...
            const range = this.getFirstRange();
            // ...look for a first character node in that range and take attributes from it.
            for (const value of range){
                // If the item is an object, we don't want to get attributes from its children...
                if (value.item.is('element') && schema.isObject(value.item)) {
                    // ...but collect attributes from inline object.
                    attrs = getTextAttributes(value.item, schema);
                    break;
                }
                if (value.type == 'text') {
                    attrs = value.item.getAttributes();
                    break;
                }
            }
        } else {
            // 2. If the selection is a caret or the range does not contain a character node...
            const nodeBefore = position.textNode ? position.textNode : position.nodeBefore;
            const nodeAfter = position.textNode ? position.textNode : position.nodeAfter;
            // When gravity is overridden then don't take node before into consideration.
            if (!this.isGravityOverridden) {
                // ...look at the node before caret and take attributes from it if it is a character node.
                attrs = getTextAttributes(nodeBefore, schema);
            }
            // 3. If not, look at the node after caret...
            if (!attrs) {
                attrs = getTextAttributes(nodeAfter, schema);
            }
            // 4. If not, try to find the first character on the left, that is in the same node.
            // When gravity is overridden then don't take node before into consideration.
            if (!this.isGravityOverridden && !attrs) {
                let node = nodeBefore;
                while(node && !attrs){
                    node = node.previousSibling;
                    attrs = getTextAttributes(node, schema);
                }
            }
            // 5. If not found, try to find the first character on the right, that is in the same node.
            if (!attrs) {
                let node = nodeAfter;
                while(node && !attrs){
                    node = node.nextSibling;
                    attrs = getTextAttributes(node, schema);
                }
            }
            // 6. If not found, selection should retrieve attributes from parent.
            if (!attrs) {
                attrs = this.getStoredAttributes();
            }
        }
        return attrs;
    }
    /**
	 * Fixes the selection after all its ranges got removed.
	 * @param deletionPosition Position where the deletion happened.
	 */ _fixGraveyardSelection(deletionPosition) {
        // Find a range that is a correct selection range and is closest to the position where the deletion happened.
        const selectionRange = this._model.schema.getNearestSelectionRange(deletionPosition);
        // If nearest valid selection range has been found - add it in the place of old range.
        if (selectionRange) {
            // Check the range, convert it to live range, bind events, etc.
            this._pushRange(selectionRange);
        }
    // If nearest valid selection range cannot be found don't add any range. Selection will be set to the default range.
    }
}
/**
 * Helper function for {@link module:engine/model/liveselection~LiveSelection#_updateAttributes}.
 *
 * It checks if the passed model item is a text node (or text proxy) and, if so, returns it's attributes.
 * If not, it checks if item is an inline object and does the same. Otherwise it returns `null`.
 */ function getTextAttributes(node, schema) {
    if (!node) {
        return null;
    }
    if (node instanceof ModelTextProxy || node instanceof ModelText) {
        return node.getAttributes();
    }
    if (!schema.isInline(node)) {
        return null;
    }
    // Stop on inline elements (such as `<softBreak>`) that are not objects (such as `<imageInline>` or `<mathml>`).
    if (!schema.isObject(node)) {
        return [];
    }
    const attributes = [];
    // Collect all attributes that can be applied to the text node.
    for (const [key, value] of node.getAttributes()){
        if (schema.checkAttribute('$text', key) && schema.getAttributeProperties(key).copyFromObject !== false) {
            attributes.push([
                key,
                value
            ]);
        }
    }
    return attributes;
}
/**
 * Removes selection attributes from element which is not empty anymore.
 */ function clearAttributesStoredInElement(model, batch) {
    const differ = model.document.differ;
    for (const entry of differ.getChanges()){
        if (entry.type != 'insert') {
            continue;
        }
        const changeParent = entry.position.parent;
        const isNoLongerEmpty = entry.length === changeParent.maxOffset;
        if (isNoLongerEmpty) {
            model.enqueueChange(batch, (writer)=>{
                const storedAttributes = Array.from(changeParent.getAttributeKeys()).filter((key)=>key.startsWith(storePrefix));
                for (const key of storedAttributes){
                    writer.removeAttribute(key, changeParent);
                }
            });
        }
    }
}

/**
 * Provides an interface to operate on a list of {@link module:engine/model/node~ModelNode nodes}. `NodeList` is used internally
 * in classes like {@link module:engine/model/element~ModelElement Element}
 * or {@link module:engine/model/documentfragment~ModelDocumentFragment ModelDocumentFragment}.
 */ class ModelNodeList {
    /**
	 * Nodes contained in this node list.
	 */ _nodes = [];
    /**
	 * This array maps numbers (offsets) to node that is placed at that offset.
	 *
	 * This array is similar to `_nodes` with the difference that one node may occupy multiple consecutive items in the array.
	 *
	 * This array is needed to quickly retrieve a node that is placed at given offset.
	 */ _offsetToNode = [];
    /**
	 * Creates a node list.
	 *
	 * @internal
	 * @param nodes Nodes contained in this node list.
	 */ constructor(nodes){
        if (nodes) {
            this._insertNodes(0, nodes);
        }
    }
    /**
	 * Iterable interface.
	 *
	 * Iterates over all nodes contained inside this node list.
	 */ [Symbol.iterator]() {
        return this._nodes[Symbol.iterator]();
    }
    /**
	 * Number of nodes contained inside this node list.
	 */ get length() {
        return this._nodes.length;
    }
    /**
	 * Sum of {@link module:engine/model/node~ModelNode#offsetSize offset sizes} of all nodes contained inside this node list.
	 */ get maxOffset() {
        return this._offsetToNode.length;
    }
    /**
	 * Gets the node at the given index. Returns `null` if incorrect index was passed.
	 */ getNode(index) {
        return this._nodes[index] || null;
    }
    /**
	 * Gets the node at the given offset. Returns `null` if incorrect offset was passed.
	 */ getNodeAtOffset(offset) {
        return this._offsetToNode[offset] || null;
    }
    /**
	 * Returns an index of the given node or `null` if given node does not have a parent.
	 *
	 * This is an alias to {@link module:engine/model/node~ModelNode#index}.
	 */ getNodeIndex(node) {
        return node.index;
    }
    /**
	 * Returns the offset at which given node is placed in its parent or `null` if given node does not have a parent.
	 *
	 * This is an alias to {@link module:engine/model/node~ModelNode#startOffset}.
	 */ getNodeStartOffset(node) {
        return node.startOffset;
    }
    /**
	 * Converts index to offset in node list.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `model-nodelist-index-out-of-bounds` if given index is less
	 * than `0` or more than {@link #length}.
	 */ indexToOffset(index) {
        if (index == this._nodes.length) {
            return this.maxOffset;
        }
        const node = this._nodes[index];
        if (!node) {
            /**
			 * Given index cannot be found in the node list.
			 *
			 * @error model-nodelist-index-out-of-bounds
			 */ throw new CKEditorError('model-nodelist-index-out-of-bounds', this);
        }
        return this.getNodeStartOffset(node);
    }
    /**
	 * Converts offset in node list to index.
	 *
	 * Throws {@link module:utils/ckeditorerror~CKEditorError CKEditorError} `model-nodelist-offset-out-of-bounds` if given offset is less
	 * than `0` or more than {@link #maxOffset}.
	 */ offsetToIndex(offset) {
        if (offset == this._offsetToNode.length) {
            return this._nodes.length;
        }
        const node = this._offsetToNode[offset];
        if (!node) {
            /**
			 * Given offset cannot be found in the node list.
			 *
			 * @error model-nodelist-offset-out-of-bounds
			 * @param {number} offset The offset value.
			 * @param {module:engine/model/nodelist~ModelNodeList} nodeList Stringified node list.
			 */ throw new CKEditorError('model-nodelist-offset-out-of-bounds', this, {
                offset,
                nodeList: this
            });
        }
        return this.getNodeIndex(node);
    }
    /**
	 * Inserts given nodes at given index.
	 *
	 * @internal
	 * @param index Index at which nodes should be inserted.
	 * @param nodes Nodes to be inserted.
	 */ _insertNodes(index, nodes) {
        const nodesArray = [];
        // Validation.
        for (const node of nodes){
            if (!(node instanceof ModelNode)) {
                /**
				 * Trying to insert an object which is not a Node instance.
				 *
				 * @error model-nodelist-insertnodes-not-node
				 */ throw new CKEditorError('model-nodelist-insertnodes-not-node', this);
            }
            nodesArray.push(node);
        }
        let offset = this.indexToOffset(index);
        // Splice nodes array and offsets array into the nodelist.
        spliceArray(this._nodes, nodesArray, index);
        spliceArray(this._offsetToNode, makeOffsetsArray(nodesArray), offset);
        // Refresh indexes and offsets for nodes inside this node list. We need to do this for all inserted nodes and all nodes after them.
        for(let i = index; i < this._nodes.length; i++){
            this._nodes[i]._index = i;
            this._nodes[i]._startOffset = offset;
            offset += this._nodes[i].offsetSize;
        }
    }
    /**
	 * Removes one or more nodes starting at the given index.
	 *
	 * @internal
	 * @param indexStart Index of the first node to remove.
	 * @param howMany Number of nodes to remove.
	 * @returns Array containing removed nodes.
	 */ _removeNodes(indexStart, howMany = 1) {
        if (howMany == 0) {
            return [];
        }
        // Remove nodes from this nodelist.
        let offset = this.indexToOffset(indexStart);
        const nodes = this._nodes.splice(indexStart, howMany);
        const lastNode = nodes[nodes.length - 1];
        const removedOffsetSum = lastNode.startOffset + lastNode.offsetSize - offset;
        this._offsetToNode.splice(offset, removedOffsetSum);
        // Reset index and start offset properties for the removed nodes -- they do not have a parent anymore.
        for (const node of nodes){
            node._index = null;
            node._startOffset = null;
        }
        for(let i = indexStart; i < this._nodes.length; i++){
            this._nodes[i]._index = i;
            this._nodes[i]._startOffset = offset;
            offset += this._nodes[i].offsetSize;
        }
        return nodes;
    }
    /**
	 * Removes children nodes provided as an array. These nodes do not need to be direct siblings.
	 *
	 * This method is faster than removing nodes one by one, as it recalculates offsets only once.
	 *
	 * @internal
	 * @param nodes Array of nodes.
	 */ _removeNodesArray(nodes) {
        if (nodes.length == 0) {
            return;
        }
        for (const node of nodes){
            node._index = null;
            node._startOffset = null;
        }
        this._nodes = this._nodes.filter((node)=>node.index !== null);
        this._offsetToNode = this._offsetToNode.filter((node)=>node.index !== null);
        let offset = 0;
        for(let i = 0; i < this._nodes.length; i++){
            this._nodes[i]._index = i;
            this._nodes[i]._startOffset = offset;
            offset += this._nodes[i].offsetSize;
        }
    }
    /**
	 * Converts `NodeList` instance to an array containing nodes that were inserted in the node list. Nodes
	 * are also converted to their plain object representation.
	 *
	 * @returns `NodeList` instance converted to `Array`.
	 */ toJSON() {
        return this._nodes.map((node)=>node.toJSON());
    }
}
/**
 * Creates an array of nodes in the format as in {@link module:engine/model/nodelist~ModelNodeList#_offsetToNode}, i.e. one node will
 * occupy multiple items if its offset size is greater than one.
 */ function makeOffsetsArray(nodes) {
    const offsets = [];
    let index = 0;
    for (const node of nodes){
        for(let i = 0; i < node.offsetSize; i++){
            offsets[index++] = node;
        }
    }
    return offsets;
}

// @if CK_DEBUG_ENGINE // const { stringifyMap, convertMapToStringifiedObject, convertMapToTags } = require( '../dev-utils/utils' );
/**
 * Model element. Type of {@link module:engine/model/node~ModelNode node} that has a
 * {@link module:engine/model/element~ModelElement#name name} and {@link module:engine/model/element~ModelElement#getChildren child nodes}.
 *
 * **Important**: see {@link module:engine/model/node~ModelNode} to read about restrictions using `Element` and `Node` API.
 */ class ModelElement extends ModelNode {
    /**
	 * Element name.
	 */ name;
    /**
	 * List of children nodes.
	 */ _children = new ModelNodeList();
    /**
	 * Creates a model element.
	 *
	 * **Note:** Constructor of this class shouldn't be used directly in the code.
	 * Use the {@link module:engine/model/writer~ModelWriter#createElement} method instead.
	 *
	 * @internal
	 * @param name Element's name.
	 * @param attrs Element's attributes. See {@link module:utils/tomap~toMap} for a list of accepted values.
	 * @param children One or more nodes to be inserted as children of created element.
	 */ constructor(name, attrs, children){
        super(attrs);
        this.name = name;
        if (children) {
            this._insertChild(0, children);
        }
    }
    /**
	 * Number of this element's children.
	 */ get childCount() {
        return this._children.length;
    }
    /**
	 * Sum of {@link module:engine/model/node~ModelNode#offsetSize offset sizes} of all of this element's children.
	 */ get maxOffset() {
        return this._children.maxOffset;
    }
    /**
	 * Is `true` if there are no nodes inside this element, `false` otherwise.
	 */ get isEmpty() {
        return this.childCount === 0;
    }
    /**
	 * Gets the child at the given index. Returns `null` if incorrect index was passed.
	 *
	 * @param index Index in this element.
	 * @returns Child node.
	 */ getChild(index) {
        return this._children.getNode(index);
    }
    /**
	 * Gets the child at the given offset. Returns `null` if incorrect index was passed.
	 *
	 * @param offset Offset in this element.
	 * @returns Child node.
	 */ getChildAtOffset(offset) {
        return this._children.getNodeAtOffset(offset);
    }
    /**
	 * Returns an iterator that iterates over all of this element's children.
	 */ getChildren() {
        return this._children[Symbol.iterator]();
    }
    /**
	 * Returns an index of the given child node. Returns `null` if given node is not a child of this element.
	 *
	 * @param node Child node to look for.
	 * @returns Child node's index in this element.
	 */ getChildIndex(node) {
        return this._children.getNodeIndex(node);
    }
    /**
	 * Returns the starting offset of given child. Starting offset is equal to the sum of
	 * {@link module:engine/model/node~ModelNode#offsetSize offset sizes} of all node's siblings that are before it. Returns `null` if
	 * given node is not a child of this element.
	 *
	 * @param node Child node to look for.
	 * @returns Child node's starting offset.
	 */ getChildStartOffset(node) {
        return this._children.getNodeStartOffset(node);
    }
    /**
	 * Returns index of a node that occupies given offset. If given offset is too low, returns `0`. If given offset is
	 * too high, returns {@link module:engine/model/element~ModelElement#getChildIndex index after last child}.
	 *
	 * ```ts
	 * const textNode = new Text( 'foo' );
	 * const pElement = new Element( 'p' );
	 * const divElement = new Element( [ textNode, pElement ] );
	 * divElement.offsetToIndex( -1 ); // Returns 0, because offset is too low.
	 * divElement.offsetToIndex( 0 ); // Returns 0, because offset 0 is taken by `textNode` which is at index 0.
	 * divElement.offsetToIndex( 1 ); // Returns 0, because `textNode` has `offsetSize` equal to 3, so it occupies offset 1 too.
	 * divElement.offsetToIndex( 2 ); // Returns 0.
	 * divElement.offsetToIndex( 3 ); // Returns 1.
	 * divElement.offsetToIndex( 4 ); // Returns 2. There are no nodes at offset 4, so last available index is returned.
	 * ```
	 */ offsetToIndex(offset) {
        return this._children.offsetToIndex(offset);
    }
    /**
	 * Returns a descendant node by its path relative to this element.
	 *
	 * ```ts
	 * // <this>a<b>c</b></this>
	 * this.getNodeByPath( [ 0 ] );     // -> "a"
	 * this.getNodeByPath( [ 1 ] );     // -> <b>
	 * this.getNodeByPath( [ 1, 0 ] );  // -> "c"
	 * ```
	 *
	 * @param relativePath Path of the node to find, relative to this element.
	 */ getNodeByPath(relativePath) {
        // eslint-disable-next-line @typescript-eslint/no-this-alias, consistent-this
        let node = this;
        for (const offset of relativePath){
            node = node.getChildAtOffset(offset);
        }
        return node;
    }
    /**
	 * Returns the parent element of the given name. Returns null if the element is not inside the desired parent.
	 *
	 * @param parentName The name of the parent element to find.
	 * @param options Options object.
	 * @param options.includeSelf When set to `true` this node will be also included while searching.
	 */ findAncestor(parentName, options = {}) {
        let parent = options.includeSelf ? this : this.parent;
        while(parent){
            if (parent.name === parentName) {
                return parent;
            }
            parent = parent.parent;
        }
        return null;
    }
    /**
	 * Converts `Element` instance to plain object and returns it. Takes care of converting all of this element's children.
	 *
	 * @returns `Element` instance converted to plain object.
	 */ toJSON() {
        const json = super.toJSON();
        json.name = this.name;
        if (this._children.length > 0) {
            json.children = [];
            for (const node of this._children){
                json.children.push(node.toJSON());
            }
        }
        return json;
    }
    /**
	 * Creates a copy of this element and returns it. Created element has the same name and attributes as the original element.
	 * If clone is deep, the original element's children are also cloned. If not, then empty element is returned.
	 *
	 * @internal
	 * @param deep If set to `true` clones element and all its children recursively. When set to `false`,
	 * element will be cloned without any child.
	 */ _clone(deep = false) {
        const children = deep ? cloneNodes(this._children) : undefined;
        return new ModelElement(this.name, this.getAttributes(), children);
    }
    /**
	 * {@link module:engine/model/element~ModelElement#_insertChild Inserts} one or more nodes at the end of this element.
	 *
	 * @see module:engine/model/writer~ModelWriter#append
	 * @internal
	 * @param nodes Nodes to be inserted.
	 */ _appendChild(nodes) {
        this._insertChild(this.childCount, nodes);
    }
    /**
	 * Inserts one or more nodes at the given index and sets {@link module:engine/model/node~ModelNode#parent parent} of these nodes
	 * to this element.
	 *
	 * @see module:engine/model/writer~ModelWriter#insert
	 * @internal
	 * @param index Index at which nodes should be inserted.
	 * @param items Items to be inserted.
	 */ _insertChild(index, items) {
        const nodes = normalize$1(items);
        for (const node of nodes){
            // If node that is being added to this element is already inside another element, first remove it from the old parent.
            if (node.parent !== null) {
                node._remove();
            }
            node.parent = this;
        }
        this._children._insertNodes(index, nodes);
    }
    /**
	 * Removes one or more nodes starting at the given index and sets
	 * {@link module:engine/model/node~ModelNode#parent parent} of these nodes to `null`.
	 *
	 * @see module:engine/model/writer~ModelWriter#remove
	 * @internal
	 * @param index Index of the first node to remove.
	 * @param howMany Number of nodes to remove.
	 * @returns Array containing removed nodes.
	 */ _removeChildren(index, howMany = 1) {
        const nodes = this._children._removeNodes(index, howMany);
        for (const node of nodes){
            node.parent = null;
        }
        return nodes;
    }
    /**
	 * Removes children nodes provided as an array and sets
	 * the {@link module:engine/model/node~ModelNode#parent parent} of these nodes to `null`.
	 *
	 * These nodes do not need to be direct siblings.
	 *
	 * This method is faster than removing nodes one by one, as it recalculates offsets only once.
	 *
	 * @internal
	 * @param nodes Array of nodes.
	 */ _removeChildrenArray(nodes) {
        this._children._removeNodesArray(nodes);
        for (const node of nodes){
            node.parent = null;
        }
    }
    /**
	 * Creates an `Element` instance from given plain object (i.e. parsed JSON string).
	 * Converts `Element` children to proper nodes.
	 *
	 * @param json Plain object to be converted to `Element`.
	 * @returns `Element` instance created using given plain object.
	 */ static fromJSON(json) {
        let children;
        if (json.children) {
            children = [];
            for (const child of json.children){
                if (child.name) {
                    // If child has name property, it is an Element.
                    children.push(ModelElement.fromJSON(child));
                } else {
                    // Otherwise, it is a Text node.
                    children.push(ModelText.fromJSON(child));
                }
            }
        }
        return new ModelElement(json.name, json.attributes, children);
    }
}
// The magic of type inference using `is` method is centralized in `TypeCheckable` class.
// Proper overload would interfere with that.
ModelElement.prototype.is = function(type, name) {
    if (!name) {
        return type === 'element' || type === 'model:element' || // From super.is(). This is highly utilised method and cannot call super. See ckeditor/ckeditor5#6529.
        type === 'node' || type === 'model:node';
    }
    return name === this.name && (type === 'element' || type === 'model:element');
};
/**
 * Converts strings to Text and non-iterables to arrays.
 */ function normalize$1(nodes) {
    // Separate condition because string is iterable.
    if (typeof nodes == 'string') {
        return [
            new ModelText(nodes)
        ];
    }
    if (!isIterable(nodes)) {
        nodes = [
            nodes
        ];
    }
    const normalizedNodes = [];
    for (const node of nodes){
        if (typeof node == 'string') {
            normalizedNodes.push(new ModelText(node));
        } else if (node instanceof ModelTextProxy) {
            normalizedNodes.push(new ModelText(node.data, node.getAttributes()));
        } else {
            normalizedNodes.push(node);
        }
    }
    return normalizedNodes;
}
function cloneNodes(nodes) {
    const clonedNodes = [];
    for (const node of nodes){
        clonedNodes.push(node._clone(true));
    }
    return clonedNodes;
}

/**
 * @license Copyright (c) 2003-2025, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-licensing-options
 */ /**
 * @module engine/conversion/conversionhelpers
 */ /**
 * Base class for conversion helpers.
 */ class ConversionHelpers {
    _dispatchers;
    /**
	 * Creates a conversion helpers instance.
	 */ constructor(dispatchers){
        this._dispatchers = dispatchers;
    }
    /**
	 * Registers a conversion helper.
	 *
	 * **Note**: See full usage example in the `{@link module:engine/conversion/conversion~Conversion#for conversion.for()}`
	 * method description.
	 *
	 * @param conversionHelper The function to be called on event.
	 */ add(conversionHelper) {
        for (const dispatcher of this._dispatchers){
            conversionHelper(dispatcher);
        }
        return this;
    }
}

/**
 * Downcast conversion helper functions.
 *
 * Learn more about {@glink framework/deep-dive/conversion/downcast downcast helpers}.
 *
 * @extends module:engine/conversion/conversionhelpers~ConversionHelpers
 */ class DowncastHelpers extends ConversionHelpers {
    /**
	 * Model element to view element conversion helper.
	 *
	 * This conversion results in creating a view element. For example, model `<paragraph>Foo</paragraph>` becomes `<p>Foo</p>` in the view.
	 *
	 * ```ts
	 * editor.conversion.for( 'downcast' ).elementToElement( {
	 * 	model: 'paragraph',
	 * 	view: 'p'
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).elementToElement( {
	 * 	model: 'paragraph',
	 * 	view: 'div',
	 * 	converterPriority: 'high'
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).elementToElement( {
	 * 	model: 'fancyParagraph',
	 * 	view: {
	 * 		name: 'p',
	 * 		classes: 'fancy'
	 * 	}
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).elementToElement( {
	 * 	model: 'heading',
	 * 	view: ( modelElement, conversionApi ) => {
	 * 		const { writer } = conversionApi;
	 *
	 * 		return writer.createContainerElement( 'h' + modelElement.getAttribute( 'level' ) );
	 * 	}
	 * } );
	 * ```
	 *
	 * The element-to-element conversion supports the reconversion mechanism. It can be enabled by using either the `attributes` or
	 * the `children` props on a model description. You will find a couple examples below.
	 *
	 * In order to reconvert an element if any of its direct children have been added or removed, use the `children` property on a `model`
	 * description. For example, this model:
	 *
	 * ```xml
	 * <box>
	 * 	<paragraph>Some text.</paragraph>
	 * </box>
	 * ```
	 *
	 * will be converted into this structure in the view:
	 *
	 * ```html
	 * <div class="box" data-type="single">
	 * 	<p>Some text.</p>
	 * </div>
	 * ```
	 *
	 * But if more items were inserted in the model:
	 *
	 * ```xml
	 * <box>
	 * 	<paragraph>Some text.</paragraph>
	 * 	<paragraph>Other item.</paragraph>
	 * </box>
	 * ```
	 *
	 * it will be converted into this structure in the view (note the element `data-type` change):
	 *
	 * ```html
	 * <div class="box" data-type="multiple">
	 * 	<p>Some text.</p>
	 * 	<p>Other item.</p>
	 * </div>
	 * ```
	 *
	 * Such a converter would look like this (note that the `paragraph` elements are converted separately):
	 *
	 * ```ts
	 * editor.conversion.for( 'downcast' ).elementToElement( {
	 * 	model: {
	 * 		name: 'box',
	 * 		children: true
	 * 	},
	 * 	view: ( modelElement, conversionApi ) => {
	 * 		const { writer } = conversionApi;
	 *
	 * 		return writer.createContainerElement( 'div', {
	 * 			class: 'box',
	 * 			'data-type': modelElement.childCount == 1 ? 'single' : 'multiple'
	 * 		} );
	 * 	}
	 * } );
	 * ```
	 *
	 * In order to reconvert element if any of its attributes have been updated, use the `attributes` property on a `model`
	 * description. For example, this model:
	 *
	 * ```xml
	 * <heading level="2">Some text.</heading>
	 * ```
	 *
	 * will be converted into this structure in the view:
	 *
	 * ```html
	 * <h2>Some text.</h2>
	 * ```
	 *
	 * But if the `heading` element's `level` attribute has been updated to `3` for example, then
	 * it will be converted into this structure in the view:
	 *
	 * ```html
	 * <h3>Some text.</h3>
	 * ```
	 *
	 * Such a converter would look as follows:
	 *
	 * ```ts
	 * editor.conversion.for( 'downcast' ).elementToElement( {
	 * 	model: {
	 * 		name: 'heading',
	 * 		attributes: 'level'
	 * 	},
	 * 	view: ( modelElement, conversionApi ) => {
	 * 		const { writer } = conversionApi;
	 *
	 * 		return writer.createContainerElement( 'h' + modelElement.getAttribute( 'level' ) );
	 * 	}
	 * } );
	 * ```
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * You can read more about the element-to-element conversion in the
	 * {@glink framework/deep-dive/conversion/downcast downcast conversion} guide.
	 *
	 * @param config Conversion configuration.
	 * @param config.model The description or a name of the model element to convert.
	 * @param config.view A view element definition or a function that takes the model element and
	 * {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi downcast conversion API}
	 * as parameters and returns a view container element.
	 * @param config.converterPriority Converter priority.
	 */ elementToElement(config) {
        return this.add(downcastElementToElement(config));
    }
    /**
	 * The model element to view structure (several elements) conversion helper.
	 *
	 * This conversion results in creating a view structure with one or more slots defined for the child nodes.
	 * For example, a model `<table>` may become this structure in the view:
	 *
	 * ```html
	 * <figure class="table">
	 * 	<table>
	 * 		<tbody>${ slot for table rows }</tbody>
	 * 	</table>
	 * </figure>
	 * ```
	 *
	 * The children of the model's `<table>` element will be inserted into the `<tbody>` element.
	 * If the `elementToElement()` helper was used, the children would be inserted into the `<figure>`.
	 *
	 * Imagine a table feature where for this model structure:
	 *
	 * ```xml
	 * <table headingRows="1">
	 * 	<tableRow> ... table cells 1 ... </tableRow>
	 * 	<tableRow> ... table cells 2 ... </tableRow>
	 * 	<tableRow> ... table cells 3 ... </tableRow>
	 * 	<caption>Caption text</caption>
	 * </table>
	 * ```
	 *
	 * we want to generate this view structure:
	 *
	 * ```html
	 * <figure class="table">
	 * 	<table>
	 * 		<thead>
	 * 			<tr> ... table cells 1 ... </tr>
	 * 		</thead>
	 * 		<tbody>
	 * 			<tr> ... table cells 2 ... </tr>
	 * 			<tr> ... table cells 3 ... </tr>
	 * 		</tbody>
	 * 	</table>
	 * 	<figcaption>Caption text</figcaption>
	 * </figure>
	 * ```
	 *
	 * The converter has to take the `headingRows` attribute into consideration when allocating the `<tableRow>` elements
	 * into the `<tbody>` and `<thead>` elements. Hence, we need two slots and need to define proper filter callbacks for them.
	 *
	 * Additionally, all elements other than `<tableRow>` should be placed outside the `<table>` tag.
	 * In the example above, this will handle the table caption.
	 *
	 * Such a converter would look like this:
	 *
	 * ```ts
	 * editor.conversion.for( 'downcast' ).elementToStructure( {
	 * 	model: {
	 * 		name: 'table',
	 * 		attributes: [ 'headingRows' ]
	 * 	},
	 * 	view: ( modelElement, conversionApi ) => {
	 * 		const { writer } = conversionApi;
	 *
	 * 		const figureElement = writer.createContainerElement( 'figure', { class: 'table' } );
	 * 		const tableElement = writer.createContainerElement( 'table' );
	 *
	 * 		writer.insert( writer.createPositionAt( figureElement, 0 ), tableElement );
	 *
	 * 		const headingRows = modelElement.getAttribute( 'headingRows' ) || 0;
	 *
	 * 		if ( headingRows > 0 ) {
	 * 			const tableHead = writer.createContainerElement( 'thead' );
	 *
	 * 			const headSlot = writer.createSlot( node => node.is( 'element', 'tableRow' ) && node.index < headingRows );
	 *
	 * 			writer.insert( writer.createPositionAt( tableElement, 'end' ), tableHead );
	 * 			writer.insert( writer.createPositionAt( tableHead, 0 ), headSlot );
	 * 		}
	 *
	 * 		if ( headingRows < tableUtils.getRows( table ) ) {
	 * 			const tableBody = writer.createContainerElement( 'tbody' );
	 *
	 * 			const bodySlot = writer.createSlot( node => node.is( 'element', 'tableRow' ) && node.index >= headingRows );
	 *
	 * 			writer.insert( writer.createPositionAt( tableElement, 'end' ), tableBody );
	 * 			writer.insert( writer.createPositionAt( tableBody, 0 ), bodySlot );
	 * 		}
	 *
	 * 		const restSlot = writer.createSlot( node => !node.is( 'element', 'tableRow' ) );
	 *
	 * 		writer.insert( writer.createPositionAt( figureElement, 'end' ), restSlot );
	 *
	 * 		return figureElement;
	 * 	}
	 * } );
	 * ```
	 *
	 * Note: The children of a model element that's being converted must be allocated in the same order in the view
	 * in which they are placed in the model.
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * @param config Conversion configuration.
 	 * @param config.model The description or a name of the model element to convert.
	 * @param config.view A function that takes the model element and
	 * {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi downcast conversion API} as parameters
	 * and returns a view container element with slots for model child nodes to be converted into.
	 * @param config.converterPriority Converter priority.
	 */ elementToStructure(config) {
        return this.add(downcastElementToStructure(config));
    }
    /**
	 * Model attribute to view element conversion helper.
	 *
	 * This conversion results in wrapping view nodes with a view attribute element. For example, a model text node with
	 * `"Foo"` as data and the `bold` attribute becomes `<strong>Foo</strong>` in the view.
	 *
	 * ```ts
	 * editor.conversion.for( 'downcast' ).attributeToElement( {
	 * 	model: 'bold',
	 * 	view: 'strong'
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).attributeToElement( {
	 * 	model: 'bold',
	 * 	view: 'b',
	 * 	converterPriority: 'high'
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).attributeToElement( {
	 * 	model: 'invert',
	 * 	view: {
	 * 		name: 'span',
	 * 		classes: [ 'font-light', 'bg-dark' ]
	 * 	}
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).attributeToElement( {
	 * 	model: {
	 * 		key: 'fontSize',
	 * 		values: [ 'big', 'small' ]
	 * 	},
	 * 	view: {
	 * 		big: {
	 * 			name: 'span',
	 * 			styles: {
	 * 				'font-size': '1.2em'
	 * 			}
	 * 		},
	 * 		small: {
	 * 			name: 'span',
	 * 			styles: {
	 * 				'font-size': '0.8em'
	 * 			}
	 * 		}
	 * 	}
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).attributeToElement( {
	 * 	model: 'bold',
	 * 	view: ( modelAttributeValue, conversionApi ) => {
	 * 		const { writer } = conversionApi;
	 *
	 * 		return writer.createAttributeElement( 'span', {
	 * 			style: 'font-weight:' + modelAttributeValue
	 * 		} );
	 * 	}
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).attributeToElement( {
	 * 	model: {
	 * 		key: 'color',
	 * 		name: '$text'
	 * 	},
	 * 	view: ( modelAttributeValue, conversionApi ) => {
	 * 		const { writer } = conversionApi;
	 *
	 * 		return writer.createAttributeElement( 'span', {
	 * 			style: 'color:' + modelAttributeValue
	 * 		} );
	 * 	}
	 * } );
	 * ```
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * @param config Conversion configuration.
	 * @param config.model The key of the attribute to convert from or a `{ key, values }` object. `values` is an array
	 * of `String`s with possible values if the model attribute is an enumerable.
	 * @param config.view A view element definition or a function
	 * that takes the model attribute value and
	 * {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi downcast conversion API} as parameters and returns a view
	 * attribute element. If `config.model.values` is given, `config.view` should be an object assigning values from `config.model.values`
	 * to view element definitions or functions.
	 * @param config.converterPriority Converter priority.
	 */ attributeToElement(config) {
        return this.add(downcastAttributeToElement(config));
    }
    /**
	 * Model attribute to view attribute conversion helper.
	 *
	 * This conversion results in adding an attribute to a view node, basing on an attribute from a model node. For example,
	 * `<imageInline src='foo.jpg'></imageInline>` is converted to `<img src='foo.jpg'></img>`.
	 *
	 * ```ts
	 * editor.conversion.for( 'downcast' ).attributeToAttribute( {
	 * 	model: 'source',
	 * 	view: 'src'
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).attributeToAttribute( {
	 * 	model: 'source',
	 * 	view: 'href',
	 * 	converterPriority: 'high'
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).attributeToAttribute( {
	 * 	model: {
	 * 		name: 'imageInline',
	 * 		key: 'source'
	 * 	},
	 * 	view: 'src'
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).attributeToAttribute( {
	 * 	model: {
	 * 		name: 'styled',
	 * 		values: [ 'dark', 'light' ]
	 * 	},
	 * 	view: {
	 * 		dark: {
	 * 			key: 'class',
	 * 			value: [ 'styled', 'styled-dark' ]
	 * 		},
	 * 		light: {
	 * 			key: 'class',
	 * 			value: [ 'styled', 'styled-light' ]
	 * 		}
	 * 	}
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).attributeToAttribute( {
	 * 	model: 'styled',
	 * 	view: modelAttributeValue => ( {
	 * 		key: 'class',
	 * 		value: 'styled-' + modelAttributeValue
	 * 	} )
	 * } );
	 * ```
	 *
	 * **Note**: Downcasting to a style property requires providing `value` as an object:
	 *
	 * ```ts
	 * editor.conversion.for( 'downcast' ).attributeToAttribute( {
	 * 	model: 'lineHeight',
	 * 	view: modelAttributeValue => ( {
	 * 		key: 'style',
	 * 		value: {
	 * 			'line-height': modelAttributeValue,
	 * 			'border-bottom': '1px dotted #ba2'
	 * 		}
	 * 	} )
	 * } );
	 * ```
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * @param config Conversion configuration.
	 * @param config.model The key of the attribute to convert from or a `{ key, values, [ name ] }` object describing
	 * the attribute key, possible values and, optionally, an element name to convert from.
	 * @param config.view A view attribute key, or a `{ key, value }` object or a function that takes the model attribute value and
	 * {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi downcast conversion API}
	 * as parameters and returns a `{ key, value }` object. If the `key` is `'class'`, the `value` can be a `String` or an
	 * array of `String`s. If the `key` is `'style'`, the `value` is an object with key-value pairs. In other cases, `value` is a `String`.
	 * If `config.model.values` is set, `config.view` should be an object assigning values from `config.model.values` to
	 * `{ key, value }` objects or a functions.
	 * @param config.converterPriority Converter priority.
	 */ attributeToAttribute(config) {
        return this.add(downcastAttributeToAttribute(config));
    }
    /**
	 * Model marker to view element conversion helper.
	 *
	 * **Note**: This method should be used mainly for editing the downcast and it is recommended
	 * to use the {@link #markerToData `#markerToData()`} helper instead.
	 *
	 * This helper may produce invalid HTML code (e.g. a span between table cells).
	 * It should only be used when you are sure that the produced HTML will be semantically correct.
	 *
	 * This conversion results in creating a view element on the boundaries of the converted marker. If the converted marker
	 * is collapsed, only one element is created. For example, a model marker set like this: `<paragraph>F[oo b]ar</paragraph>`
	 * becomes `<p>F<span data-marker="search"></span>oo b<span data-marker="search"></span>ar</p>` in the view.
	 *
	 * ```ts
	 * editor.conversion.for( 'editingDowncast' ).markerToElement( {
	 * 	model: 'search',
	 * 	view: 'marker-search'
	 * } );
	 *
	 * editor.conversion.for( 'editingDowncast' ).markerToElement( {
	 * 	model: 'search',
	 * 	view: 'search-result',
	 * 	converterPriority: 'high'
	 * } );
	 *
	 * editor.conversion.for( 'editingDowncast' ).markerToElement( {
	 * 	model: 'search',
	 * 	view: {
	 * 		name: 'span',
	 * 		attributes: {
	 * 			'data-marker': 'search'
	 * 		}
	 * 	}
	 * } );
	 *
	 * editor.conversion.for( 'editingDowncast' ).markerToElement( {
	 * 	model: 'search',
	 * 	view: ( markerData, conversionApi ) => {
	 * 		const { writer } = conversionApi;
	 *
	 * 		return writer.createUIElement( 'span', {
	 * 			'data-marker': 'search',
	 * 			'data-start': markerData.isOpening
	 * 		} );
	 * 	}
	 * } );
	 * ```
	 *
	 * If a function is passed as the `config.view` parameter, it will be used to generate both boundary elements. The function
	 * receives the `data` object and {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi downcast conversion API}
	 * as a parameters and should return an instance of the
	 * {@link module:engine/view/uielement~ViewUIElement view UI element}. The `data` object and
	 * {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi `conversionApi`} are passed from
	 * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:addMarker}. Additionally,
	 * the `data.isOpening` parameter is passed, which is set to `true` for the marker start boundary element, and `false` for
	 * the marker end boundary element.
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * @param config Conversion configuration.
	 * @param config.model The name of the model marker (or model marker group) to convert.
	 * @param config.view A view element definition or a function that takes the model marker data and
	 * {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi downcast conversion API} as a parameters
	 * and returns a view UI element.
	 * @param config.converterPriority Converter priority.
	 */ markerToElement(config) {
        return this.add(downcastMarkerToElement(config));
    }
    /**
	 * Model marker to highlight conversion helper.
	 *
	 * This conversion results in creating a highlight on view nodes. For this kind of conversion,
	 * the {@link module:engine/conversion/downcasthelpers~DowncastHighlightDescriptor} should be provided.
	 *
	 * For text nodes, a `<span>` {@link module:engine/view/attributeelement~ViewAttributeElement} is created and it wraps all text nodes
	 * in the converted marker range. For example, a model marker set like this: `<paragraph>F[oo b]ar</paragraph>` becomes
	 * `<p>F<span class="comment">oo b</span>ar</p>` in the view.
	 *
	 * {@link module:engine/view/containerelement~ViewContainerElement} may provide a custom way of handling highlight. Most often,
	 * the element itself is given classes and attributes described in the highlight descriptor (instead of being wrapped in `<span>`).
	 * For example, a model marker set like this:
	 * `[<imageInline src="foo.jpg"></imageInline>]` becomes `<img src="foo.jpg" class="comment"></img>` in the view.
	 *
	 * For container elements, the conversion is two-step. While the converter processes the highlight descriptor and passes it
	 * to a container element, it is the container element instance itself that applies values from the highlight descriptor.
	 * So, in a sense, the converter takes care of stating what should be applied on what, while the element decides how to apply that.
	 *
	 * ```ts
	 * editor.conversion.for( 'downcast' ).markerToHighlight( { model: 'comment', view: { classes: 'comment' } } );
	 *
	 * editor.conversion.for( 'downcast' ).markerToHighlight( {
	 * 	model: 'comment',
	 * 	view: { classes: 'comment' },
	 * 	converterPriority: 'high'
	 * } );
	 *
	 * editor.conversion.for( 'downcast' ).markerToHighlight( {
	 * 	model: 'comment',
	 * 	view: ( data, conversionApi ) => {
	 * 		// Assuming that the marker name is in a form of comment:commentType:commentId.
	 * 		const [ , commentType, commentId ] = data.markerName.split( ':' );
	 *
	 * 		return {
	 * 			classes: [ 'comment', 'comment-' + commentType ],
	 * 			attributes: { 'data-comment-id': commentId }
	 * 		};
	 * 	}
	 * } );
	 * ```
	 *
	 * If a function is passed as the `config.view` parameter, it will be used to generate the highlight descriptor. The function
	 * receives the `data` object and {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi downcast conversion API}
	 * as the parameters and should return a
	 * {@link module:engine/conversion/downcasthelpers~DowncastHighlightDescriptor highlight descriptor}.
	 * The `data` object properties are passed from {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:addMarker}.
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * @param config Conversion configuration.
	 * @param config.model The name of the model marker (or model marker group) to convert.
	 * @param config.view A highlight descriptor that will be used for highlighting or a function that takes the model marker data and
	 * {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi downcast conversion API} as a parameters
	 * and returns a highlight descriptor.
	 * @param config.converterPriority Converter priority.
	 */ markerToHighlight(config) {
        return this.add(downcastMarkerToHighlight(config));
    }
    /**
	 * Model marker converter for data downcast.
	 *
	 * This conversion creates a representation for model marker boundaries in the view:
	 *
	 * * If the marker boundary is before or after a model element, a view attribute is set on a corresponding view element.
	 * * In other cases, a view element with the specified tag name is inserted at the corresponding view position.
	 *
	 * Typically, the marker names use the `group:uniqueId:otherData` convention. For example: `comment:e34zfk9k2n459df53sjl34:zx32c`.
	 * The default configuration for this conversion is that the first part is the `group` part and the rest of
	 * the marker name becomes the `name` part.
	 *
	 * Tag and attribute names and values are generated from the marker name:
	 *
	 * * The templates for attributes are `data-[group]-start-before="[name]"`, `data-[group]-start-after="[name]"`,
	 * `data-[group]-end-before="[name]"` and `data-[group]-end-after="[name]"`.
	 * * The templates for view elements are `<[group]-start name="[name]">` and `<[group]-end name="[name]">`.
	 *
	 * Attributes mark whether the given marker's start or end boundary is before or after the given element.
	 * The `data-[group]-start-before` and `data-[group]-end-after` attributes are favored.
	 * The other two are used when the former two cannot be used.
	 *
	 * The conversion configuration can take a function that will generate different group and name parts.
	 * If such a function is set as the `config.view` parameter, it is passed a marker name and it is expected to return an object with two
	 * properties: `group` and `name`. If the function returns a falsy value, the conversion will not take place.
	 *
	 * Basic usage:
	 *
	 * ```ts
	 * // Using the default conversion.
	 * // In this case, all markers with names starting with 'comment:' will be converted.
	 * // The `group` parameter will be set to `comment`.
	 * // The `name` parameter will be the rest of the marker name (without the `:`).
	 * editor.conversion.for( 'dataDowncast' ).markerToData( {
	 * 	model: 'comment'
	 * } );
	 * ```
	 *
	 * An example of a view that may be generated by this conversion (assuming a marker with the name `comment:commentId:uid` marked
	 * by `[]`):
	 *
	 * ```
	 * // Model:
	 * <paragraph>Foo[bar</paragraph>
	 * <imageBlock src="abc.jpg"></imageBlock>]
	 *
	 * // View:
	 * <p>Foo<comment-start name="commentId:uid"></comment-start>bar</p>
	 * <figure data-comment-end-after="commentId:uid" class="image"><img src="abc.jpg" /></figure>
	 * ```
	 *
	 * In the example above, the comment starts before "bar" and ends after the image.
	 *
	 * If the `name` part is empty, the following view may be generated:
	 *
	 * ```html
	 * <p>Foo <myMarker-start></myMarker-start>bar</p>
	 * <figure data-myMarker-end-after="" class="image"><img src="abc.jpg" /></figure>
	 * ```
	 *
	 * **Note:** A situation where some markers have the `name` part and some do not, is incorrect and should be avoided.
	 *
	 * Examples where `data-group-start-after` and `data-group-end-before` are used:
	 *
	 * ```
	 * // Model:
	 * <blockQuote>[]<paragraph>Foo</paragraph></blockQuote>
	 *
	 * // View:
	 * <blockquote><p data-group-end-before="name" data-group-start-before="name">Foo</p></blockquote>
	 * ```
	 *
	 * Similarly, when a marker is collapsed after the last element:
	 *
	 * ```
	 * // Model:
	 * <blockQuote><paragraph>Foo</paragraph>[]</blockQuote>
	 *
	 * // View:
	 * <blockquote><p data-group-end-after="name" data-group-start-after="name">Foo</p></blockquote>
	 * ```
	 *
	 * When there are multiple markers from the same group stored in the same attribute of the same element, their
	 * name parts are put together in the attribute value, for example: `data-group-start-before="name1,name2,name3"`.
	 *
	 * Other examples of usage:
	 *
	 * ```ts
	 * // Using a custom function which is the same as the default conversion:
	 * editor.conversion.for( 'dataDowncast' ).markerToData( {
	 * 	model: 'comment',
	 * 	view: markerName => ( {
	 * 		group: 'comment',
	 * 		name: markerName.substr( 8 ) // Removes 'comment:' part.
	 * 	} )
	 * } );
	 *
	 * // Using the converter priority:
	 * editor.conversion.for( 'dataDowncast' ).markerToData( {
	 * 	model: 'comment',
	 * 	view: markerName => ( {
	 * 		group: 'comment',
	 * 		name: markerName.substr( 8 ) // Removes 'comment:' part.
	 * 	} ),
	 * 	converterPriority: 'high'
	 * } );
	 * ```
	 *
	 * This kind of conversion is useful for saving data into the database, so it should be used in the data conversion pipeline.
	 *
	 * See the {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} API guide to learn how to
	 * add a converter to the conversion process.
	 *
	 * @param config Conversion configuration.
	 * @param config.model The name of the model marker (or the model marker group) to convert.
	 * @param config.view A function that takes the model marker name and
	 * {@link module:engine/conversion/downcastdispatcher~DowncastConversionApi downcast conversion API} as the parameters
	 * and returns an object with the `group` and `name` properties.
	 * @param config.converterPriority Converter priority.
	 */ markerToData(config) {
        return this.add(downcastMarkerToData(config));
    }
}
/**
 * Function factory that creates a default downcast converter for text insertion changes.
 *
 * The converter automatically consumes the corresponding value from the consumables list and stops the event (see
 * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher}).
 *
 * ```ts
 * modelDispatcher.on( 'insert:$text', insertText() );
 * ```
 *
 * @returns Insert text event converter.
 * @internal
 */ function insertText() {
    return (evt, data, conversionApi)=>{
        if (!conversionApi.consumable.consume(data.item, evt.name)) {
            return;
        }
        const viewWriter = conversionApi.writer;
        const viewPosition = conversionApi.mapper.toViewPosition(data.range.start);
        const viewText = viewWriter.createText(data.item.data);
        viewWriter.insert(viewPosition, viewText);
    };
}
/**
 * Function factory that creates a default downcast converter for triggering attributes and children conversion.
 *
 * @returns The converter.
 * @internal
 */ function insertAttributesAndChildren() {
    return (evt, data, conversionApi)=>{
        conversionApi.convertAttributes(data.item);
        // Start converting children of the current item.
        // In case of reconversion children were already re-inserted or converted separately.
        if (!data.reconversion && data.item.is('element') && !data.item.isEmpty) {
            conversionApi.convertChildren(data.item);
        }
    };
}
/**
 * Function factory that creates a default downcast converter for node remove changes.
 *
 * ```ts
 * modelDispatcher.on( 'remove', remove() );
 * ```
 *
 * @returns Remove event converter.
 * @internal
 */ function remove() {
    return (evt, data, conversionApi)=>{
        // Find the view range start position by mapping the model position at which the remove happened.
        const viewStart = conversionApi.mapper.toViewPosition(data.position);
        const modelEnd = data.position.getShiftedBy(data.length);
        const viewEnd = conversionApi.mapper.toViewPosition(modelEnd, {
            isPhantom: true
        });
        const viewRange = conversionApi.writer.createRange(viewStart, viewEnd);
        // Trim the range to remove in case some UI elements are on the view range boundaries.
        const removed = conversionApi.writer.remove(viewRange.getTrimmed());
        // After the range is removed, unbind all view elements from the model.
        // Range inside view document fragment is used to unbind deeply.
        for (const child of conversionApi.writer.createRangeIn(removed).getItems()){
            conversionApi.mapper.unbindViewElement(child, {
                defer: true
            });
        }
    };
}
/**
 * Creates a `<span>` {@link module:engine/view/attributeelement~ViewAttributeElement view attribute element} from the information
 * provided by the {@link module:engine/conversion/downcasthelpers~DowncastHighlightDescriptor highlight descriptor} object. If the priority
 * is not provided in the descriptor, the default priority will be used.
 *
 * @internal
 */ function createViewElementFromDowncastHighlightDescriptor(writer, descriptor) {
    const viewElement = writer.createAttributeElement('span', descriptor.attributes);
    if (descriptor.classes) {
        viewElement._addClass(descriptor.classes);
    }
    if (typeof descriptor.priority === 'number') {
        viewElement._priority = descriptor.priority;
    }
    viewElement._id = descriptor.id;
    return viewElement;
}
/**
 * Function factory that creates a converter which converts a non-collapsed
 * {@link module:engine/model/selection~ModelSelection model selection}
 * to a {@link module:engine/view/documentselection~ViewDocumentSelection view selection}. The converter consumes appropriate
 * value from the `consumable` object and maps model positions from the selection to view positions.
 *
 * ```ts
 * modelDispatcher.on( 'selection', convertRangeSelection() );
 * ```
 *
 * @returns Selection converter.
 * @internal
 */ function convertRangeSelection() {
    return (evt, data, conversionApi)=>{
        const selection = data.selection;
        if (selection.isCollapsed) {
            return;
        }
        if (!conversionApi.consumable.consume(selection, 'selection')) {
            return;
        }
        const viewRanges = [];
        for (const range of selection.getRanges()){
            viewRanges.push(conversionApi.mapper.toViewRange(range));
        }
        conversionApi.writer.setSelection(viewRanges, {
            backward: selection.isBackward
        });
    };
}
/**
 * Function factory that creates a converter which converts a collapsed
 * {@link module:engine/model/selection~ModelSelection model selection} to
 * a {@link module:engine/view/documentselection~ViewDocumentSelection view selection}. The converter consumes appropriate
 * value from the `consumable` object, maps the model selection position to the view position and breaks
 * {@link module:engine/view/attributeelement~ViewAttributeElement attribute elements} at the selection position.
 *
 * ```ts
 * modelDispatcher.on( 'selection', convertCollapsedSelection() );
 * ```
 *
 * An example of the view state before and after converting the collapsed selection:
 *
 * ```
 *    <p><strong>f^oo<strong>bar</p>
 * -> <p><strong>f</strong>^<strong>oo</strong>bar</p>
 * ```
 *
 * By breaking attribute elements like `<strong>`, the selection is in a correct element. Then, when the selection attribute is
 * converted, broken attributes might be merged again, or the position where the selection is may be wrapped
 * with different, appropriate attribute elements.
 *
 * See also {@link module:engine/conversion/downcasthelpers~cleanSelection} which does a clean-up
 * by merging attributes.
 *
 * @returns Selection converter.
 * @internal
 */ function convertCollapsedSelection() {
    return (evt, data, conversionApi)=>{
        const selection = data.selection;
        if (!selection.isCollapsed) {
            return;
        }
        if (!conversionApi.consumable.consume(selection, 'selection')) {
            return;
        }
        const viewWriter = conversionApi.writer;
        const modelPosition = selection.getFirstPosition();
        const viewPosition = conversionApi.mapper.toViewPosition(modelPosition);
        const brokenPosition = viewWriter.breakAttributes(viewPosition);
        viewWriter.setSelection(brokenPosition);
    };
}
/**
 * Function factory that creates a converter which cleans artifacts after the previous
 * {@link module:engine/model/selection~ModelSelection model selection} conversion. It removes all empty
 * {@link module:engine/view/attributeelement~ViewAttributeElement view attribute elements} and merges
 * sibling attributes at all start and end positions of all ranges.
 *
 * ```
 *    <p><strong>^</strong></p>
 * -> <p>^</p>
 *
 *    <p><strong>foo</strong>^<strong>bar</strong>bar</p>
 * -> <p><strong>foo^bar<strong>bar</p>
 *
 *    <p><strong>foo</strong><em>^</em><strong>bar</strong>bar</p>
 * -> <p><strong>foo^bar<strong>bar</p>
 * ```
 *
 * This listener should be assigned before any converter for the new selection:
 *
 * ```ts
 * modelDispatcher.on( 'cleanSelection', cleanSelection() );
 * ```
 *
 * See {@link module:engine/conversion/downcasthelpers~convertCollapsedSelection}
 * which does the opposite by breaking attributes in the selection position.
 *
 * @returns Selection converter.
 * @internal
 */ function cleanSelection() {
    return (evt, data, conversionApi)=>{
        const viewWriter = conversionApi.writer;
        const viewSelection = viewWriter.document.selection;
        for (const range of viewSelection.getRanges()){
            // Not collapsed selection should not have artifacts.
            if (range.isCollapsed) {
                // Position might be in the node removed by the view writer.
                if (range.end.parent.isAttached()) {
                    conversionApi.writer.mergeAttributes(range.start);
                }
            }
        }
        viewWriter.setSelection(null);
    };
}
/**
 * Function factory that creates a converter which converts the set/change/remove attribute changes from the model to the view.
 * It can also be used to convert selection attributes. In that case, an empty attribute element will be created and the
 * selection will be put inside it.
 *
 * Attributes from the model are converted to a view element that will be wrapping these view nodes that are bound to
 * model elements having the given attribute. This is useful for attributes like `bold` that may be set on text nodes in the model
 * but are represented as an element in the view:
 *
 * ```
 * [paragraph]              MODEL ====> VIEW        <p>
 * 	|- a {bold: true}                             |- <b>
 * 	|- b {bold: true}                             |   |- ab
 * 	|- c                                          |- c
 * 	```
 *
 * Passed `Function` will be provided with the attribute value and then all the parameters of the
 * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:attribute `attribute` event}.
 * It is expected that the function returns an {@link module:engine/view/element~ViewElement}.
 * The result of the function will be the wrapping element.
 * When the provided `Function` does not return any element, no conversion will take place.
 *
 * The converter automatically consumes the corresponding value from the consumables list and stops the event (see
 * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher}).
 *
 * ```ts
 * modelDispatcher.on( 'attribute:bold', wrap( ( modelAttributeValue, { writer } ) => {
 * 	return writer.createAttributeElement( 'strong' );
 * } );
 * ```
 *
 * @internal
 * @param elementCreator Function returning a view element that will be used for wrapping.
 * @returns Set/change attribute converter.
 */ function wrap(elementCreator) {
    return (evt, data, conversionApi)=>{
        if (!conversionApi.consumable.test(data.item, evt.name)) {
            return;
        }
        // Recreate current wrapping node. It will be used to unwrap view range if the attribute value has changed
        // or the attribute was removed.
        const oldViewElement = elementCreator(data.attributeOldValue, conversionApi, data);
        // Create node to wrap with.
        const newViewElement = elementCreator(data.attributeNewValue, conversionApi, data);
        if (!oldViewElement && !newViewElement) {
            return;
        }
        conversionApi.consumable.consume(data.item, evt.name);
        const viewWriter = conversionApi.writer;
        const viewSelection = viewWriter.document.selection;
        if (data.item instanceof ModelSelection || data.item instanceof ModelDocumentSelection) {
            // Selection attribute conversion.
            viewWriter.wrap(viewSelection.getFirstRange(), newViewElement);
        } else {
            // Node attribute conversion.
            let viewRange = conversionApi.mapper.toViewRange(data.range);
            // First, unwrap the range from current wrapper.
            if (data.attributeOldValue !== null && oldViewElement) {
                viewRange = viewWriter.unwrap(viewRange, oldViewElement);
            }
            if (data.attributeNewValue !== null && newViewElement) {
                viewWriter.wrap(viewRange, newViewElement);
            }
        }
    };
}
/**
 * Function factory that creates a converter which converts node insertion changes from the model to the view.
 * The function passed will be provided with all the parameters of the dispatcher's
 * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher#event:insert `insert` event}.
 * It is expected that the function returns an {@link module:engine/view/element~ViewElement}.
 * The result of the function will be inserted into the view.
 *
 * The converter automatically consumes the corresponding value from the consumables list and binds the model and view elements.
 *
 * ```ts
 * downcastDispatcher.on(
 * 	'insert:myElem',
 * 	insertElement( ( modelItem, { writer } ) => {
 * 		const text = writer.createText( 'myText' );
 * 		const myElem = writer.createElement( 'myElem', { myAttr: 'my-' + modelItem.getAttribute( 'myAttr' ) }, text );
 *
 * 		// Do something fancy with `myElem` using `modelItem` or other parameters.
 *
 * 		return myElem;
 * 	}
 * ) );
 * ```
 *
 * @internal
 * @param  elementCreator Function returning a view element, which will be inserted.
 * @param consumer Function defining element consumption process.
 * By default this function just consume passed item insertion.
 * @returns Insert element event converter.
 */ function insertElement(elementCreator, consumer = defaultConsumer) {
    return (evt, data, conversionApi)=>{
        if (!consumer(data.item, conversionApi.consumable, {
            preflight: true
        })) {
            return;
        }
        const viewElement = elementCreator(data.item, conversionApi, data);
        if (!viewElement) {
            return;
        }
        // Consume an element insertion and all present attributes that are specified as a reconversion triggers.
        consumer(data.item, conversionApi.consumable);
        const viewPosition = conversionApi.mapper.toViewPosition(data.range.start);
        conversionApi.mapper.bindElements(data.item, viewElement);
        conversionApi.writer.insert(viewPosition, viewElement);
        // Convert attributes before converting children.
        conversionApi.convertAttributes(data.item);
        // Convert children or reinsert previous view elements.
        reinsertOrConvertNodes(viewElement, data.item.getChildren(), conversionApi, {
            reconversion: data.reconversion
        });
    };
}
/**
 * Function factory that creates a converter which converts a single model node insertion to a view structure.
 *
 * It is expected that the passed element creator function returns an {@link module:engine/view/element~ViewElement} with attached slots
 * created with `writer.createSlot()` to indicate where child nodes should be converted.
 *
 * @see module:engine/conversion/downcasthelpers~DowncastHelpers#elementToStructure
 *
 * @internal
 * @param elementCreator Function returning a view structure, which will be inserted.
 * @param consumer A callback that is expected to consume all the consumables
 * that were used by the element creator.
 * @returns Insert element event converter.
*/ function insertStructure(elementCreator, consumer) {
    return (evt, data, conversionApi)=>{
        if (!consumer(data.item, conversionApi.consumable, {
            preflight: true
        })) {
            return;
        }
        const slotsMap = new Map();
        conversionApi.writer._registerSlotFactory(createSlotFactory(data.item, slotsMap, conversionApi));
        // View creation.
        const viewElement = elementCreator(data.item, conversionApi, data);
        conversionApi.writer._clearSlotFactory();
        if (!viewElement) {
            return;
        }
        // Check if all children are covered by slots and there is no child that landed in multiple slots.
        validateSlotsChildren(data.item, slotsMap, conversionApi);
        // Consume an element insertion and all present attributes that are specified as a reconversion triggers.
        consumer(data.item, conversionApi.consumable);
        const viewPosition = conversionApi.mapper.toViewPosition(data.range.start);
        conversionApi.mapper.bindElements(data.item, viewElement);
        conversionApi.writer.insert(viewPosition, viewElement);
        // Convert attributes before converting children.
        conversionApi.convertAttributes(data.item);
        // Fill view slots with previous view elements or create new ones.
        fillSlots(viewElement, slotsMap, conversionApi, {
            reconversion: data.reconversion
        });
    };
}
/**
 * Function factory that creates a converter which converts marker adding change to the
 * {@link module:engine/view/uielement~ViewUIElement view UI element}.
 *
 * The view UI element that will be added to the view depends on the passed parameter. See {@link ~insertElement}.
 * In case of a non-collapsed range, the UI element will not wrap nodes but separate elements will be placed at the beginning
 * and at the end of the range.
 *
 * This converter binds created UI elements with the marker name using {@link module:engine/conversion/mapper~Mapper#bindElementToMarker}.
 *
 * @internal
 * @param elementCreator A view UI element or a function returning the view element that will be inserted.
 * @returns Insert element event converter.
 */ function insertUIElement(elementCreator) {
    return (evt, data, conversionApi)=>{
        // Create two view elements. One will be inserted at the beginning of marker, one at the end.
        // If marker is collapsed, only "opening" element will be inserted.
        data.isOpening = true;
        const viewStartElement = elementCreator(data, conversionApi);
        data.isOpening = false;
        const viewEndElement = elementCreator(data, conversionApi);
        if (!viewStartElement || !viewEndElement) {
            return;
        }
        const markerRange = data.markerRange;
        // Marker that is collapsed has consumable build differently that non-collapsed one.
        // For more information see `addMarker` event description.
        // If marker's range is collapsed - check if it can be consumed.
        if (markerRange.isCollapsed && !conversionApi.consumable.consume(markerRange, evt.name)) {
            return;
        }
        // If marker's range is not collapsed - consume all items inside.
        for (const value of markerRange){
            if (!conversionApi.consumable.consume(value.item, evt.name)) {
                return;
            }
        }
        const mapper = conversionApi.mapper;
        const viewWriter = conversionApi.writer;
        // Add "opening" element.
        viewWriter.insert(mapper.toViewPosition(markerRange.start), viewStartElement);
        conversionApi.mapper.bindElementToMarker(viewStartElement, data.markerName);
        // Add "closing" element only if range is not collapsed.
        if (!markerRange.isCollapsed) {
            viewWriter.insert(mapper.toViewPosition(markerRange.end), viewEndElement);
            conversionApi.mapper.bindElementToMarker(viewEndElement, data.markerName);
        }
        evt.stop();
    };
}
/**
 * Function factory that returns a default downcast converter for removing a {@link module:engine/view/uielement~ViewUIElement UI element}
 * based on marker remove change.
 *
 * This converter unbinds elements from the marker name.
 *
 * @returns Removed UI element converter.
 */ function removeUIElement() {
    return (evt, data, conversionApi)=>{
        const elements = conversionApi.mapper.markerNameToElements(data.markerName);
        if (!elements) {
            return;
        }
        for (const element of elements){
            conversionApi.mapper.unbindElementFromMarkerName(element, data.markerName);
            conversionApi.writer.clear(conversionApi.writer.createRangeOn(element), element);
        }
        conversionApi.writer.clearClonedElementsGroup(data.markerName);
        evt.stop();
    };
}
/**
 * Function factory that creates a default converter for model markers.
 *
 * See {@link DowncastHelpers#markerToData} for more information what type of view is generated.
 *
 * This converter binds created UI elements and affected view elements with the marker name
 * using {@link module:engine/conversion/mapper~Mapper#bindElementToMarker}.
 *
 * @returns Add marker converter.
 */ function insertMarkerData(viewCreator) {
    return (evt, data, conversionApi)=>{
        const viewMarkerData = viewCreator(data.markerName, conversionApi);
        if (!viewMarkerData) {
            return;
        }
        const markerRange = data.markerRange;
        if (!conversionApi.consumable.consume(markerRange, evt.name)) {
            return;
        }
        // Adding closing data first to keep the proper order in the view.
        handleMarkerBoundary(markerRange, false, conversionApi, data, viewMarkerData);
        handleMarkerBoundary(markerRange, true, conversionApi, data, viewMarkerData);
        evt.stop();
    };
}
/**
 * Helper function for `insertMarkerData()` that marks a marker boundary at the beginning or end of given `range`.
 */ function handleMarkerBoundary(range, isStart, conversionApi, data, viewMarkerData) {
    const modelPosition = isStart ? range.start : range.end;
    const elementAfter = modelPosition.nodeAfter && modelPosition.nodeAfter.is('element') ? modelPosition.nodeAfter : null;
    const elementBefore = modelPosition.nodeBefore && modelPosition.nodeBefore.is('element') ? modelPosition.nodeBefore : null;
    if (elementAfter || elementBefore) {
        let modelElement;
        let isBefore;
        // If possible, we want to add `data-group-start-before` and `data-group-end-after` attributes.
        if (isStart && elementAfter || !isStart && !elementBefore) {
            // [<elementAfter>...</elementAfter> -> <elementAfter data-group-start-before="...">...</elementAfter>
            // <parent>]<elementAfter> -> <parent><elementAfter data-group-end-before="...">
            modelElement = elementAfter;
            isBefore = true;
        } else {
            // <elementBefore>...</elementBefore>] -> <elementBefore data-group-end-after="...">...</elementBefore>
            // </elementBefore>[</parent> -> </elementBefore data-group-start-after="..."></parent>
            modelElement = elementBefore;
            isBefore = false;
        }
        const viewElement = conversionApi.mapper.toViewElement(modelElement);
        // In rare circumstances, the model element may be not mapped to any view element and that would cause an error.
        // One of those situations is a soft break inside code block.
        if (viewElement) {
            insertMarkerAsAttribute(viewElement, isStart, isBefore, conversionApi, data, viewMarkerData);
            return;
        }
    }
    const viewPosition = conversionApi.mapper.toViewPosition(modelPosition);
    insertMarkerAsElement(viewPosition, isStart, conversionApi, data, viewMarkerData);
}
/**
 * Helper function for `insertMarkerData()` that marks a marker boundary in the view as an attribute on a view element.
 */ function insertMarkerAsAttribute(viewElement, isStart, isBefore, conversionApi, data, viewMarkerData) {
    const attributeName = `data-${viewMarkerData.group}-${isStart ? 'start' : 'end'}-${isBefore ? 'before' : 'after'}`;
    const markerNames = viewElement.hasAttribute(attributeName) ? viewElement.getAttribute(attributeName).split(',') : [];
    // Adding marker name at the beginning to have the same order in the attribute as there is with marker elements.
    markerNames.unshift(viewMarkerData.name);
    conversionApi.writer.setAttribute(attributeName, markerNames.join(','), viewElement);
    conversionApi.mapper.bindElementToMarker(viewElement, data.markerName);
}
/**
 * Helper function for `insertMarkerData()` that marks a marker boundary in the view as a separate view ui element.
 */ function insertMarkerAsElement(position, isStart, conversionApi, data, viewMarkerData) {
    const viewElementName = `${viewMarkerData.group}-${isStart ? 'start' : 'end'}`;
    const attrs = viewMarkerData.name ? {
        'name': viewMarkerData.name
    } : null;
    const viewElement = conversionApi.writer.createUIElement(viewElementName, attrs);
    conversionApi.writer.insert(position, viewElement);
    conversionApi.mapper.bindElementToMarker(viewElement, data.markerName);
}
/**
 * Function factory that creates a converter for removing a model marker data added by the {@link #insertMarkerData} converter.
 *
 * @returns Remove marker converter.
 */ function removeMarkerData(viewCreator) {
    return (evt, data, conversionApi)=>{
        const viewData = viewCreator(data.markerName, conversionApi);
        if (!viewData) {
            return;
        }
        const elements = conversionApi.mapper.markerNameToElements(data.markerName);
        if (!elements) {
            return;
        }
        for (const element of elements){
            conversionApi.mapper.unbindElementFromMarkerName(element, data.markerName);
            if (element.is('containerElement')) {
                removeMarkerFromAttribute(`data-${viewData.group}-start-before`, element);
                removeMarkerFromAttribute(`data-${viewData.group}-start-after`, element);
                removeMarkerFromAttribute(`data-${viewData.group}-end-before`, element);
                removeMarkerFromAttribute(`data-${viewData.group}-end-after`, element);
            } else {
                conversionApi.writer.clear(conversionApi.writer.createRangeOn(element), element);
            }
        }
        conversionApi.writer.clearClonedElementsGroup(data.markerName);
        evt.stop();
        function removeMarkerFromAttribute(attributeName, element) {
            if (element.hasAttribute(attributeName)) {
                const markerNames = new Set(element.getAttribute(attributeName).split(','));
                markerNames.delete(viewData.name);
                if (markerNames.size == 0) {
                    conversionApi.writer.removeAttribute(attributeName, element);
                } else {
                    conversionApi.writer.setAttribute(attributeName, Array.from(markerNames).join(','), element);
                }
            }
        }
    };
}
/**
 * Function factory that creates a converter which converts the set/change/remove attribute changes from the model to the view.
 *
 * Attributes from the model are converted to the view element attributes in the view. You may provide a custom function to generate
 * a key-value attribute pair to add/change/remove. If not provided, model attributes will be converted to view element
 * attributes on a one-to-one basis.
 *
 * *Note:** The provided attribute creator should always return the same `key` for a given attribute from the model.
 *
 * The converter automatically consumes the corresponding value from the consumables list and stops the event (see
 * {@link module:engine/conversion/downcastdispatcher~DowncastDispatcher}).
 *
 * ```ts
 * modelDispatcher.on( 'attribute:customAttr:myElem', changeAttribute( ( value, data ) => {
 * 	// Change attribute key from `customAttr` to `class` in the view.
 * 	const key = 'class';
 * 	let value = data.attributeNewValue;
 *
 * 	// Force attribute value to 'empty' if the model element is empty.
 * 	if ( data.item.childCount === 0 ) {
 * 		value = 'empty';
 * 	}
 *
 * 	// Return the key-value pair.
 * 	return { key, value };
 * } ) );
 * ```
 *
 * @param attributeCreator Function returning an object with two properties: `key` and `value`, which
 * represent the attribute key and attribute value to be set on a {@link module:engine/view/element~ViewElement view element}.
 * The function is passed the model attribute value as the first parameter and additional data about the change as the second parameter.
 * @returns Set/change attribute converter.
 */ function changeAttribute(attributeCreator) {
    return (evt, data, conversionApi)=>{
        if (!conversionApi.consumable.test(data.item, evt.name)) {
            return;
        }
        const oldAttribute = attributeCreator(data.attributeOldValue, conversionApi, data);
        const newAttribute = attributeCreator(data.attributeNewValue, conversionApi, data);
        if (!oldAttribute && !newAttribute) {
            return;
        }
        conversionApi.consumable.consume(data.item, evt.name);
        const viewElement = conversionApi.mapper.toViewElement(data.item);
        const viewWriter = conversionApi.writer;
        // If model item cannot be mapped to a view element, it means item is not an `Element` instance but a `ModelTextProxy` node.
        // Only elements can have attributes in a view so do not proceed for anything else (#1587).
        if (!viewElement) {
            /**
			 * This error occurs when a {@link module:engine/model/textproxy~ModelTextProxy text node's} attribute is to be downcasted
			 * by an {@link module:engine/conversion/conversion~Conversion#attributeToAttribute `Attribute to Attribute converter`}.
			 * In most cases it is caused by converters misconfiguration when only "generic" converter is defined:
			 *
			 * ```ts
			 * editor.conversion.for( 'downcast' ).attributeToAttribute( {
			 * 	model: 'attribute-name',
			 * 	view: 'attribute-name'
			 * } ) );
			 * ```
			 *
			 * and given attribute is used on text node, for example:
			 *
			 * ```ts
			 * model.change( writer => {
			 * 	writer.insertText( 'Foo', { 'attribute-name': 'bar' }, parent, 0 );
			 * } );
			 * ```
			 *
			 * In such cases, to convert the same attribute for both {@link module:engine/model/element~ModelElement}
			 * and {@link module:engine/model/textproxy~ModelTextProxy `Text`} nodes, text specific
			 * {@link module:engine/conversion/conversion~Conversion#attributeToElement `Attribute to Element converter`}
			 * with higher {@link module:utils/priorities~PriorityString priority} must also be defined:
			 *
			 * ```ts
			 * editor.conversion.for( 'downcast' ).attributeToElement( {
			 * 	model: {
			 * 		key: 'attribute-name',
			 * 		name: '$text'
			 * 	},
			 * 	view: ( value, { writer } ) => {
			 * 		return writer.createAttributeElement( 'span', { 'attribute-name': value } );
			 * 	},
			 * 	converterPriority: 'high'
			 * } ) );
			 * ```
			 *
			 * @error conversion-attribute-to-attribute-on-text
			 * @param {object} data The conversion data.
			 */ throw new CKEditorError('conversion-attribute-to-attribute-on-text', conversionApi.dispatcher, data);
        }
        // First remove the old attribute if there was one.
        if (data.attributeOldValue !== null && oldAttribute) {
            let value = oldAttribute.value;
            if (oldAttribute.key == 'style') {
                if (typeof oldAttribute.value == 'string') {
                    value = new StylesMap(viewWriter.document.stylesProcessor).setTo(oldAttribute.value).getStylesEntries().map(([key])=>key);
                } else {
                    value = Object.keys(oldAttribute.value);
                }
            }
            viewWriter.removeAttribute(oldAttribute.key, value, viewElement);
        }
        // Then set the new attribute.
        if (data.attributeNewValue !== null && newAttribute) {
            let value = newAttribute.value;
            if (newAttribute.key == 'style' && typeof newAttribute.value == 'string') {
                value = Object.fromEntries(new StylesMap(viewWriter.document.stylesProcessor).setTo(newAttribute.value).getStylesEntries());
            }
            viewWriter.setAttribute(newAttribute.key, value, false, viewElement);
        }
    };
}
/**
 * Function factory that creates a converter which converts the text inside marker's range. The converter wraps the text with
 * {@link module:engine/view/attributeelement~ViewAttributeElement} created from the provided descriptor.
 * See {link module:engine/conversion/downcasthelpers~createViewElementFromDowncastHighlightDescriptor}.
 *
 * It can also be used to convert the selection that is inside a marker. In that case, an empty attribute element will be
 * created and the selection will be put inside it.
 *
 * If the highlight descriptor does not provide the `priority` property, `10` will be used.
 *
 * If the highlight descriptor does not provide the `id` property, the name of the marker will be used.
 *
 * This converter binds the created {@link module:engine/view/attributeelement~ViewAttributeElement attribute elemens} with the marker name
 * using the {@link module:engine/conversion/mapper~Mapper#bindElementToMarker} method.
 */ function highlightText(highlightDescriptor) {
    return (evt, data, conversionApi)=>{
        if (!data.item) {
            return;
        }
        if (!(data.item instanceof ModelSelection || data.item instanceof ModelDocumentSelection) && !data.item.is('$textProxy')) {
            return;
        }
        const descriptor = prepareDescriptor(highlightDescriptor, data, conversionApi);
        if (!descriptor) {
            return;
        }
        if (!conversionApi.consumable.consume(data.item, evt.name)) {
            return;
        }
        const viewWriter = conversionApi.writer;
        const viewElement = createViewElementFromDowncastHighlightDescriptor(viewWriter, descriptor);
        const viewSelection = viewWriter.document.selection;
        if (data.item instanceof ModelSelection || data.item instanceof ModelDocumentSelection) {
            viewWriter.wrap(viewSelection.getFirstRange(), viewElement);
        } else {
            const viewRange = conversionApi.mapper.toViewRange(data.range);
            const rangeAfterWrap = viewWriter.wrap(viewRange, viewElement);
            for (const element of rangeAfterWrap.getItems()){
                if (element.is('attributeElement') && element.isSimilar(viewElement)) {
                    conversionApi.mapper.bindElementToMarker(element, data.markerName);
                    break;
                }
            }
        }
    };
}
/**
 * Converter function factory. It creates a function which applies the marker's highlight to an element inside the marker's range.
 *
 * The converter checks if an element has the `addHighlight` function stored as a
 * {@link module:engine/view/element~ViewElement#_setCustomProperty custom property} and, if so, uses it to apply the highlight.
 * In such case the converter will consume all element's children, assuming that they were handled by the element itself.
 *
 * When the `addHighlight` custom property is not present, the element is not converted in any special way.
 * This means that converters will proceed to convert the element's child nodes.
 *
 * If the highlight descriptor does not provide the `priority` property, `10` will be used.
 *
 * If the highlight descriptor does not provide the `id` property, the name of the marker will be used.
 *
 * This converter binds altered {@link module:engine/view/containerelement~ViewContainerElement container elements}
 * with the marker name using the {@link module:engine/conversion/mapper~Mapper#bindElementToMarker} method.
 */ function highlightElement(highlightDescriptor) {
    return (evt, data, conversionApi)=>{
        if (!data.item) {
            return;
        }
        if (!(data.item instanceof ModelElement)) {
            return;
        }
        const descriptor = prepareDescriptor(highlightDescriptor, data, conversionApi);
        if (!descriptor) {
            return;
        }
        if (!conversionApi.consumable.test(data.item, evt.name)) {
            return;
        }
        const viewElement = conversionApi.mapper.toViewElement(data.item);
        if (viewElement && viewElement.getCustomProperty('addHighlight')) {
            // Consume element itself.
            conversionApi.consumable.consume(data.item, evt.name);
            // Consume all children nodes.
            for (const value of ModelRange._createIn(data.item)){
                conversionApi.consumable.consume(value.item, evt.name);
            }
            const addHighlightCallback = viewElement.getCustomProperty('addHighlight');
            addHighlightCallback(viewElement, descriptor, conversionApi.writer);
            conversionApi.mapper.bindElementToMarker(viewElement, data.markerName);
        }
    };
}
/**
 * Function factory that creates a converter which converts the removing model marker to the view.
 *
 * Both text nodes and elements are handled by this converter but they are handled a bit differently.
 *
 * Text nodes are unwrapped using the {@link module:engine/view/attributeelement~ViewAttributeElement attribute element} created from the
 * provided highlight descriptor. See {link module:engine/conversion/downcasthelpers~DowncastHighlightDescriptor}.
 *
 * For elements, the converter checks if an element has the `removeHighlight` function stored as a
 * {@link module:engine/view/element~ViewElement#_setCustomProperty custom property}. If so, it uses it to remove the highlight.
 * In such case, the children of that element will not be converted.
 *
 * When `removeHighlight` is not present, the element is not converted in any special way.
 * The converter will proceed to convert the element's child nodes instead.
 *
 * If the highlight descriptor does not provide the `priority` property, `10` will be used.
 *
 * If the highlight descriptor does not provide the `id` property, the name of the marker will be used.
 *
 * This converter unbinds elements from the marker name.
 */ function removeHighlight(highlightDescriptor) {
    return (evt, data, conversionApi)=>{
        // This conversion makes sense only for non-collapsed range.
        if (data.markerRange.isCollapsed) {
            return;
        }
        const descriptor = prepareDescriptor(highlightDescriptor, data, conversionApi);
        if (!descriptor) {
            return;
        }
        // View element that will be used to unwrap `AttributeElement`s.
        const viewHighlightElement = createViewElementFromDowncastHighlightDescriptor(conversionApi.writer, descriptor);
        // Get all elements bound with given marker name.
        const elements = conversionApi.mapper.markerNameToElements(data.markerName);
        if (!elements) {
            return;
        }
        for (const element of elements){
            conversionApi.mapper.unbindElementFromMarkerName(element, data.markerName);
            if (element.is('attributeElement')) {
                conversionApi.writer.unwrap(conversionApi.writer.createRangeOn(element), viewHighlightElement);
            } else {
                // if element.is( 'containerElement' ).
                const removeHighlightCallback = element.getCustomProperty('removeHighlight');
                removeHighlightCallback(element, descriptor.id, conversionApi.writer);
            }
        }
        conversionApi.writer.clearClonedElementsGroup(data.markerName);
        evt.stop();
    };
}
/**
 * Model element to view element conversion helper.
 *
 * See {@link ~DowncastHelpers#elementToElement `.elementToElement()` downcast helper} for examples and config params description.
 *
 * @param config Conversion configuration.
 * @param config.model The description or a name of the model element to convert.
 * @param config.model.attributes List of attributes triggering element reconversion.
 * @param config.model.children Should reconvert element if the list of model child nodes changed.
 * @returns Conversion helper.
 */ function downcastElementToElement(config) {
    const model = normalizeModelElementConfig(config.model);
    const view = normalizeToElementConfig(config.view, 'container');
    // Trigger reconversion on children list change if element is a subject to any reconversion.
    // This is required to be able to trigger Differ#refreshItem() on a direct child of the reconverted element.
    if (model.attributes.length) {
        model.children = true;
    }
    return (dispatcher)=>{
        dispatcher.on(`insert:${model.name}`, insertElement(view, createConsumer(model)), {
            priority: config.converterPriority || 'normal'
        });
        if (model.children || model.attributes.length) {
            dispatcher.on('reduceChanges', createChangeReducer(model), {
                priority: 'low'
            });
        }
    };
}
/**
 * Model element to view structure conversion helper.
 *
 * See {@link ~DowncastHelpers#elementToStructure `.elementToStructure()` downcast helper} for examples and config params description.
 *
 * @param config Conversion configuration.
 * @returns Conversion helper.
 */ function downcastElementToStructure(config) {
    const model = normalizeModelElementConfig(config.model);
    const view = normalizeToElementConfig(config.view, 'container');
    // Trigger reconversion on children list change because it always needs to use slots to put children in proper places.
    // This is required to be able to trigger Differ#refreshItem() on a direct child of the reconverted element.
    model.children = true;
    return (dispatcher)=>{
        if (dispatcher._conversionApi.schema.checkChild(model.name, '$text')) {
            /**
			 * This error occurs when a {@link module:engine/model/element~ModelElement model element} is downcasted
			 * via {@link module:engine/conversion/downcasthelpers~DowncastHelpers#elementToStructure} helper but the element was
			 * allowed to host `$text` by the {@link module:engine/model/schema~ModelSchema model schema}.
			 *
			 * For instance, this may be the result of `myElement` allowing the content of
			 * {@glink framework/deep-dive/schema#generic-items `$block`} in its schema definition:
			 *
			 * ```ts
			 * // Element definition in schema.
			 * schema.register( 'myElement', {
			 * 	allowContentOf: '$block',
			 *
			 * 	// ...
			 * } );
			 *
			 * // ...
			 *
			 * // Conversion of myElement with the use of elementToStructure().
			 * editor.conversion.for( 'downcast' ).elementToStructure( {
			 * 	model: 'myElement',
			 * 	view: ( modelElement, { writer } ) => {
			 * 		// ...
			 * 	}
			 * } );
			 * ```
			 *
			 * In such case, {@link module:engine/conversion/downcasthelpers~DowncastHelpers#elementToElement `elementToElement()`} helper
			 * can be used instead to get around this problem:
			 *
			 * ```ts
			 * editor.conversion.for( 'downcast' ).elementToElement( {
			 * 	model: 'myElement',
			 * 	view: ( modelElement, { writer } ) => {
			 * 		// ...
			 * 	}
			 * } );
			 * ```
			 *
			 * @error conversion-element-to-structure-disallowed-text
			 * @param {string} elementName The name of the element the structure is to be created for.
			 */ throw new CKEditorError('conversion-element-to-structure-disallowed-text', dispatcher, {
                elementName: model.name
            });
        }
        dispatcher.on(`insert:${model.name}`, insertStructure(view, createConsumer(model)), {
            priority: config.converterPriority || 'normal'
        });
        dispatcher.on('reduceChanges', createChangeReducer(model), {
            priority: 'low'
        });
    };
}
/**
 * Model attribute to view element conversion helper.
 *
 * See {@link ~DowncastHelpers#attributeToElement `.attributeToElement()` downcast helper} for examples.
 *
 * @param config Conversion configuration.
 * @param config.model The key of the attribute to convert from or a `{ key, values }` object. `values` is an array
 * of `String`s with possible values if the model attribute is an enumerable.
 * @param config.view A view element definition or a function that takes the model attribute value and
 * {@link module:engine/view/downcastwriter~ViewDowncastWriter view downcast writer} as parameters and returns a view attribute element.
 * If `config.model.values` is given, `config.view` should be an object assigning values from `config.model.values` to view element
 * definitions or functions.
 * @param config.converterPriority Converter priority.
 * @returns Conversion helper.
 */ function downcastAttributeToElement(config) {
    config = cloneDeep(config);
    let model = config.model;
    if (typeof model == 'string') {
        model = {
            key: model
        };
    }
    let eventName = `attribute:${model.key}`;
    if (model.name) {
        eventName += ':' + model.name;
    }
    if (model.values) {
        for (const modelValue of model.values){
            config.view[modelValue] = normalizeToElementConfig(config.view[modelValue], 'attribute');
        }
    } else {
        config.view = normalizeToElementConfig(config.view, 'attribute');
    }
    const elementCreator = getFromAttributeCreator(config);
    return (dispatcher)=>{
        dispatcher.on(eventName, wrap(elementCreator), {
            priority: config.converterPriority || 'normal'
        });
    };
}
/**
 * Model attribute to view attribute conversion helper.
 *
 * See {@link ~DowncastHelpers#attributeToAttribute `.attributeToAttribute()` downcast helper} for examples.
 *
 * @param config Conversion configuration.
 * @param config.model The key of the attribute to convert from or a `{ key, values, [ name ] }` object describing
 * the attribute key, possible values and, optionally, an element name to convert from.
 * @param config.view A view attribute key, or a `{ key, value }` object or a function that takes the model attribute value and returns
 * a `{ key, value }` object.
 * If `key` is `'class'`, `value` can be a `String` or an array of `String`s. If `key` is `'style'`, `value` is an object with
 * key-value pairs. In other cases, `value` is a `String`.
 * If `config.model.values` is set, `config.view` should be an object assigning values from `config.model.values` to
 * `{ key, value }` objects or a functions.
 * @param config.converterPriority Converter priority.
 * @returns Conversion helper.
 */ function downcastAttributeToAttribute(config) {
    config = cloneDeep(config);
    let model = config.model;
    if (typeof model == 'string') {
        model = {
            key: model
        };
    }
    let eventName = `attribute:${model.key}`;
    if (model.name) {
        eventName += ':' + model.name;
    }
    if (model.values) {
        for (const modelValue of model.values){
            config.view[modelValue] = normalizeToAttributeConfig(config.view[modelValue]);
        }
    } else {
        config.view = normalizeToAttributeConfig(config.view);
    }
    const elementCreator = getFromAttributeCreator(config);
    return (dispatcher)=>{
        dispatcher.on(eventName, changeAttribute(elementCreator), {
            priority: config.converterPriority || 'normal'
        });
    };
}
/**
 * Model marker to view element conversion helper.
 *
 * See {@link ~DowncastHelpers#markerToElement `.markerToElement()` downcast helper} for examples.
 *
 * @param config Conversion configuration.
 * @param config.model The name of the model marker (or model marker group) to convert.
 * @param config.view A view element definition or a function that takes the model marker data as a parameter and returns a view UI element.
 * @param config.converterPriority Converter priority.
 * @returns Conversion helper.
 */ function downcastMarkerToElement(config) {
    const view = normalizeToElementConfig(config.view, 'ui');
    return (dispatcher)=>{
        dispatcher.on(`addMarker:${config.model}`, insertUIElement(view), {
            priority: config.converterPriority || 'normal'
        });
        dispatcher.on(`removeMarker:${config.model}`, removeUIElement(), {
            priority: config.converterPriority || 'normal'
        });
    };
}
/**
 * Model marker to view data conversion helper.
 *
 * See {@link ~DowncastHelpers#markerToData `markerToData()` downcast helper} to learn more.
 *
 * @returns Conversion helper.
 */ function downcastMarkerToData(config) {
    config = cloneDeep(config);
    const group = config.model;
    let view = config.view;
    // Default conversion.
    if (!view) {
        view = (markerName)=>({
                group,
                name: markerName.substr(config.model.length + 1)
            });
    }
    return (dispatcher)=>{
        dispatcher.on(`addMarker:${group}`, insertMarkerData(view), {
            priority: config.converterPriority || 'normal'
        });
        dispatcher.on(`removeMarker:${group}`, removeMarkerData(view), {
            priority: config.converterPriority || 'normal'
        });
    };
}
/**
 * Model marker to highlight conversion helper.
 *
 * See {@link ~DowncastHelpers#markerToElement `.markerToElement()` downcast helper} for examples.
 *
 * @param config Conversion configuration.
 * @param config.model The name of the model marker (or model marker group) to convert.
 * @param config.view A highlight descriptor that will be used for highlighting or a function that takes
 * the model marker data as a parameter and returns a highlight descriptor.
 * @param config.converterPriority Converter priority.
 * @returns Conversion helper.
 */ function downcastMarkerToHighlight(config) {
    return (dispatcher)=>{
        dispatcher.on(`addMarker:${config.model}`, highlightText(config.view), {
            priority: config.converterPriority || 'normal'
        });
        dispatcher.on(`addMarker:${config.model}`, highlightElement(config.view), {
            priority: config.converterPriority || 'normal'
        });
        dispatcher.on(`removeMarker:${config.model}`, removeHighlight(config.view), {
            priority: config.converterPriority || 'normal'
        });
    };
}
/**
 * Takes `config.model`, and converts it to an object with normalized structure.
 *
 * @param model Model configuration or element name.
 */ function normalizeModelElementConfig(model) {
    if (typeof model == 'string') {
        model = {
            name: model
        };
    }
    return {
        name: model.name,
        attributes: model.attributes ? toArray(model.attributes) : [],
        children: !!model.children
    };
}
/**
 * Takes `config.view`, and if it is an {@link module:engine/view/elementdefinition~ViewElementDefinition}, converts it
 * to a function (because lower level converters accept only element creator functions).
 *
 * @param view View configuration.
 * @param viewElementType View element type to create.
 * @returns Element creator function to use in lower level converters.
 */ function normalizeToElementConfig(view, viewElementType) {
    if (typeof view == 'function') {
        // If `view` is already a function, don't do anything.
        return view;
    }
    return (modelData, conversionApi)=>createViewElementFromDefinition(view, conversionApi, viewElementType);
}
/**
 * Creates a view element instance from the provided {@link module:engine/view/elementdefinition~ViewElementDefinition} and class.
 */ function createViewElementFromDefinition(viewElementDefinition, conversionApi, viewElementType) {
    if (typeof viewElementDefinition == 'string') {
        // If `viewElementDefinition` is given as a `String`, normalize it to an object with `name` property.
        viewElementDefinition = {
            name: viewElementDefinition
        };
    }
    let element;
    const viewWriter = conversionApi.writer;
    const attributes = Object.assign({}, viewElementDefinition.attributes);
    if (viewElementType == 'container') {
        element = viewWriter.createContainerElement(viewElementDefinition.name, attributes);
    } else if (viewElementType == 'attribute') {
        const options = {
            priority: viewElementDefinition.priority || ViewAttributeElement.DEFAULT_PRIORITY
        };
        element = viewWriter.createAttributeElement(viewElementDefinition.name, attributes, options);
    } else {
        // 'ui'.
        element = viewWriter.createUIElement(viewElementDefinition.name, attributes);
    }
    if (viewElementDefinition.styles) {
        const keys = Object.keys(viewElementDefinition.styles);
        for (const key of keys){
            viewWriter.setStyle(key, viewElementDefinition.styles[key], element);
        }
    }
    if (viewElementDefinition.classes) {
        const classes = viewElementDefinition.classes;
        if (typeof classes == 'string') {
            viewWriter.addClass(classes, element);
        } else {
            for (const className of classes){
                viewWriter.addClass(className, element);
            }
        }
    }
    return element;
}
function getFromAttributeCreator(config) {
    if (config.model.values) {
        return (modelAttributeValue, conversionApi, data)=>{
            const view = config.view[modelAttributeValue];
            if (view) {
                return view(modelAttributeValue, conversionApi, data);
            }
            return null;
        };
    } else {
        return config.view;
    }
}
/**
 * Takes the configuration, adds default parameters if they do not exist and normalizes other parameters to be used in downcast converters
 * for generating a view attribute.
 *
 * @param view View configuration.
 */ function normalizeToAttributeConfig(view) {
    if (typeof view == 'string') {
        return (modelAttributeValue)=>({
                key: view,
                value: modelAttributeValue
            });
    } else if (typeof view == 'object') {
        // { key, value, ... }
        if (view.value) {
            return ()=>view;
        } else {
            return (modelAttributeValue)=>({
                    key: view.key,
                    value: modelAttributeValue
                });
        }
    } else {
        // function.
        return view;
    }
}
/**
 * Helper function for `highlight`. Prepares the actual descriptor object using value passed to the converter.
 */ function prepareDescriptor(highlightDescriptor, data, conversionApi) {
    // If passed descriptor is a creator function, call it. If not, just use passed value.
    const descriptor = typeof highlightDescriptor == 'function' ? highlightDescriptor(data, conversionApi) : highlightDescriptor;
    if (!descriptor) {
        return null;
    }
    // Apply default descriptor priority.
    if (!descriptor.priority) {
        descriptor.priority = 10;
    }
    // Default descriptor id is marker name.
    if (!descriptor.id) {
        descriptor.id = data.markerName;
    }
    return descriptor;
}
/**
 * Creates a function that checks a single differ diff item whether it should trigger reconversion.
 *
 * @param model A normalized `config.model` converter configuration.
 * @param model.name The name of element.
 * @param model.attributes The list of attribute names that should trigger reconversion.
 * @param model.children Whether the child list change should trigger reconversion.
 */ function createChangeReducerCallback(model) {
    return (node, change)=>{
        if (!node.is('element', model.name)) {
            return false;
        }
        if (change.type == 'attribute') {
            if (model.attributes.includes(change.attributeKey)) {
                return true;
            }
        } else {
            /* istanbul ignore else: This is always true because otherwise it would not register a reducer callback. -- @preserve */ if (model.children) {
                return true;
            }
        }
        return false;
    };
}
/**
 * Creates a `reduceChanges` event handler for reconversion.
 *
 * @param model A normalized `config.model` converter configuration.
 * @param model.name The name of element.
 * @param model.attributes The list of attribute names that should trigger reconversion.
 * @param model.children Whether the child list change should trigger reconversion.
 */ function createChangeReducer(model) {
    const shouldReplace = createChangeReducerCallback(model);
    return (evt, data)=>{
        const reducedChanges = [];
        if (!data.reconvertedElements) {
            data.reconvertedElements = new Set();
        }
        for (const change of data.changes){
            // For attribute use node affected by the change.
            // For insert or remove use parent element because we need to check if it's added/removed child.
            const node = change.type == 'attribute' ? change.range.start.nodeAfter : change.position.parent;
            if (!node || !shouldReplace(node, change)) {
                reducedChanges.push(change);
                continue;
            }
            // If it's already marked for reconversion, so skip this change, otherwise add the diff items.
            if (!data.reconvertedElements.has(node)) {
                data.reconvertedElements.add(node);
                const position = ModelPosition._createBefore(node);
                let changeIndex = reducedChanges.length;
                // We need to insert remove+reinsert before any other change on and inside the re-converted element.
                // This is important because otherwise we would remove element that had already been modified by the previous change.
                // Note that there could be some element removed before the re-converted element, so we must not break this behavior.
                for(let i = reducedChanges.length - 1; i >= 0; i--){
                    const change = reducedChanges[i];
                    const changePosition = change.type == 'attribute' ? change.range.start : change.position;
                    const positionRelation = changePosition.compareWith(position);
                    if (positionRelation == 'before' || change.type == 'remove' && positionRelation == 'same') {
                        break;
                    }
                    changeIndex = i;
                }
                reducedChanges.splice(changeIndex, 0, {
                    type: 'remove',
                    name: node.name,
                    position,
                    length: 1
                }, {
                    type: 'reinsert',
                    name: node.name,
                    position,
                    length: 1
                });
            }
        }
        data.changes = reducedChanges;
    };
}
/**
 * Creates a function that checks if an element and its watched attributes can be consumed and consumes them.
 *
 * @param model A normalized `config.model` converter configuration.
 * @param model.name The name of element.
 * @param model.attributes The list of attribute names that should trigger reconversion.
 * @param model.children Whether the child list change should trigger reconversion.
 */ function createConsumer(model) {
    return (node, consumable, options = {})=>{
        const events = [
            'insert'
        ];
        // Collect all set attributes that are triggering conversion.
        for (const attributeName of model.attributes){
            if (node.hasAttribute(attributeName)) {
                events.push(`attribute:${attributeName}`);
            }
        }
        if (!events.every((event)=>consumable.test(node, event))) {
            return false;
        }
        if (!options.preflight) {
            events.forEach((event)=>consumable.consume(node, event));
        }
        return true;
    };
}
/**
 * Creates a function that creates view slots.
 *
 * @returns Function exposed by the writer as `createSlot()`.
 */ function createSlotFactory(element, slotsMap, conversionApi) {
    return (writer, modeOrFilter)=>{
        const slot = writer.createContainerElement('$slot');
        let children = null;
        if (modeOrFilter === 'children') {
            children = Array.from(element.getChildren());
        } else if (typeof modeOrFilter == 'function') {
            children = Array.from(element.getChildren()).filter((element)=>modeOrFilter(element));
        } else {
            /**
			 * Unknown slot mode was provided to `writer.createSlot()` in the downcast converter.
			 *
			 * @error conversion-slot-mode-unknown
			 * @param {never} modeOrFilter The specified mode or filter.
			 */ throw new CKEditorError('conversion-slot-mode-unknown', conversionApi.dispatcher, {
                modeOrFilter
            });
        }
        slotsMap.set(slot, children);
        return slot;
    };
}
/**
 * Checks if all children are covered by slots and there is no child that landed in multiple slots.
 */ function validateSlotsChildren(element, slotsMap, conversionApi) {
    const childrenInSlots = Array.from(slotsMap.values()).flat();
    const uniqueChildrenInSlots = new Set(childrenInSlots);
    if (uniqueChildrenInSlots.size != childrenInSlots.length) {
        /**
		 * Filters provided to `writer.createSlot()` overlap (at least two filters accept the same child element).
		 *
		 * @error conversion-slot-filter-overlap
		 * @param {module:engine/model/element~ModelElement} element The element of which children would not be properly
		 * allocated to multiple slots.
		 */ throw new CKEditorError('conversion-slot-filter-overlap', conversionApi.dispatcher, {
            element
        });
    }
    if (uniqueChildrenInSlots.size != element.childCount) {
        /**
		 * Filters provided to `writer.createSlot()` are incomplete and exclude at least one children element (one of
		 * the children elements would not be assigned to any of the slots).
		 *
		 * @error conversion-slot-filter-incomplete
		 * @param {module:engine/model/element~ModelElement} element The element of which children would not be properly
		 * allocated to multiple slots.
		 */ throw new CKEditorError('conversion-slot-filter-incomplete', conversionApi.dispatcher, {
            element
        });
    }
}
/**
 * Fill slots with appropriate view elements.
 */ function fillSlots(viewElement, slotsMap, conversionApi, options) {
    // Set temporary position mapping to redirect child view elements into a proper slots.
    conversionApi.mapper.on('modelToViewPosition', toViewPositionMapping, {
        priority: 'highest'
    });
    let currentSlot = null;
    let currentSlotNodes = null;
    // Fill slots with nested view nodes.
    for ([currentSlot, currentSlotNodes] of slotsMap){
        reinsertOrConvertNodes(viewElement, currentSlotNodes, conversionApi, options);
        conversionApi.writer.move(conversionApi.writer.createRangeIn(currentSlot), conversionApi.writer.createPositionBefore(currentSlot));
        conversionApi.writer.remove(currentSlot);
    }
    conversionApi.mapper.off('modelToViewPosition', toViewPositionMapping);
    function toViewPositionMapping(evt, data) {
        const element = data.modelPosition.nodeAfter;
        // Find the proper offset within the slot.
        const index = currentSlotNodes.indexOf(element);
        if (index < 0) {
            return;
        }
        data.viewPosition = data.mapper.findPositionIn(currentSlot, index);
    }
}
/**
 * Inserts view representation of `nodes` into the `viewElement` either by bringing back just removed view nodes
 * or by triggering conversion for them.
 */ function reinsertOrConvertNodes(viewElement, modelNodes, conversionApi, options) {
    // Fill with nested view nodes.
    for (const modelChildNode of modelNodes){
        // Try reinserting the view node for the specified model node...
        if (!reinsertNode(viewElement.root, modelChildNode, conversionApi, options)) {
            // ...or else convert the model element to the view.
            conversionApi.convertItem(modelChildNode);
        }
    }
}
/**
 * Checks if the view for the given model element could be reused and reinserts it to the view.
 *
 * @returns `false` if view element can't be reused.
 */ function reinsertNode(viewRoot, modelNode, conversionApi, options) {
    const { writer, mapper } = conversionApi;
    // Don't reinsert if this is not a reconversion...
    if (!options.reconversion) {
        return false;
    }
    const viewChildNode = mapper.toViewElement(modelNode);
    // ...or there is no view to reinsert or it was already inserted to the view structure...
    if (!viewChildNode || viewChildNode.root == viewRoot) {
        return false;
    }
    // ...or it was strictly marked as not to be reused.
    if (!conversionApi.canReuseView(viewChildNode)) {
        return false;
    }
    // Otherwise reinsert the view node.
    writer.move(writer.createRangeOn(viewChildNode), mapper.toViewPosition(ModelPosition._createBefore(modelNode)));
    return true;
}
/**
 * The default consumer for insert events.
 *
 * @param item Model item.
 * @param consumable The model consumable.
 * @param options.preflight Whether should consume or just check if can be consumed.
 */ function defaultConsumer(item, consumable, { preflight } = {}) {
    if (preflight) {
        return consumable.test(item, 'insert');
    } else {
        return consumable.consume(item, 'insert');
    }
}

/**
 * @license Copyright (c) 2003-2025, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-licensing-options
 */ /**
 * @module engine/model/utils/autoparagraphing
 */ /**
 * Fixes all empty roots.
 *
 * @internal
 * @param writer The model writer.
 * @returns `true` if any change has been applied, `false` otherwise.
 */ function autoParagraphEmptyRoots(writer) {
    const { schema, document } = writer.model;
    for (const root of document.getRoots()){
        if (root.isEmpty && !schema.checkChild(root, '$text')) {
            // If paragraph element is allowed in the root, create paragraph element.
            if (schema.checkChild(root, 'paragraph')) {
                writer.insertElement('paragraph', root);
                // Other roots will get fixed in the next post-fixer round. Those will be triggered
                // in the same batch no matter if this method was triggered by the post-fixing or not
                // (the above insertElement call will trigger the post-fixers).
                return true;
            }
        }
    }
    return false;
}
/**
 * Checks if the given node wrapped with a paragraph would be accepted by the schema in the given position.
 *
 * @internal
 * @param position The position at which to check.
 * @param nodeOrType The child node or child type to check.
 * @param schema A schema instance used for element validation.
 */ function isParagraphable(position, nodeOrType, schema) {
    const context = schema.createContext(position);
    // When paragraph is allowed in this context...
    if (!schema.checkChild(context, 'paragraph')) {
        return false;
    }
    // And a node would be allowed in this paragraph...
    if (!schema.checkChild(context.push('paragraph'), nodeOrType)) {
        return false;
    }
    return true;
}
/**
 * Inserts a new paragraph at the given position and returns a position inside that paragraph.
 *
 * @internal
 * @param position The position where a paragraph should be inserted.
 * @param writer The model writer.
 * @returns  Position inside the created paragraph.
 */ function wrapInParagraph(position, writer) {
    const paragraph = writer.createElement('paragraph');
    writer.insert(paragraph, position);
    return writer.createPositionAt(paragraph, 0);
}

/**
 * Contains the {@link module:engine/view/view view} to {@link module:engine/model/model model} converters for
 * {@link module:engine/conversion/upcastdispatcher~UpcastDispatcher}.
 *
 * @module engine/conversion/upcasthelpers
 */ /**
 * Upcast conversion helper functions.
 *
 * Learn more about {@glink framework/deep-dive/conversion/upcast upcast helpers}.
 *
 * @extends module:engine/conversion/conversionhelpers~ConversionHelpers
 */ class UpcastHelpers extends ConversionHelpers {
    /**
	 * View element to model element conversion helper.
	 *
	 * This conversion results in creating a model element. For example,
	 * view `<p>Foo</p>` becomes `<paragraph>Foo</paragraph>` in the model.
	 *
	 * Keep in mind that the element will be inserted only if it is allowed
	 * by {@link module:engine/model/schema~ModelSchema schema} configuration.
	 *
	 * ```ts
	 * editor.conversion.for( 'upcast' ).elementToElement( {
	 * 	view: 'p',
	 * 	model: 'paragraph'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).elementToElement( {
	 * 	view: 'p',
	 * 	model: 'paragraph',
	 * 	converterPriority: 'high'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).elementToElement( {
	 * 	view: {
	 * 		name: 'p',
	 * 		classes: 'fancy'
	 * 	},
	 * 	model: 'fancyParagraph'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).elementToElement( {
	 * 	view: {
	 * 		name: 'p',
	 * 		classes: 'heading'
	 * 	},
	 * 	model: ( viewElement, conversionApi ) => {
	 * 		const modelWriter = conversionApi.writer;
	 *
	 * 		return modelWriter.createElement( 'heading', { level: viewElement.getAttribute( 'data-level' ) } );
	 * 	}
	 * } );
	 * ```
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * @param config Conversion configuration.
	 * @param config.view Pattern matching all view elements which should be converted. If not set, the converter
	 * will fire for every view element.
	 * @param config.model Name of the model element, a model element instance or a function that takes a view element
	 * and {@link module:engine/conversion/upcastdispatcher~UpcastConversionApi upcast conversion API}
	 * and returns a model element. The model element will be inserted in the model.
	 * @param config.converterPriority Converter priority.
	 */ elementToElement(config) {
        return this.add(upcastElementToElement(config));
    }
    /**
	 * View element to model attribute conversion helper.
	 *
	 * This conversion results in setting an attribute on a model node. For example, view `<strong>Foo</strong>` becomes
	 * `Foo` {@link module:engine/model/text~ModelText model text node} with `bold` attribute set to `true`.
	 *
	 * This helper is meant to set a model attribute on all the elements that are inside the converted element:
	 *
	 * ```
	 * <strong>Foo</strong>   -->   <strong><p>Foo</p></strong>   -->   <paragraph><$text bold="true">Foo</$text></paragraph>
	 * ```
	 *
	 * Above is a sample of HTML code, that goes through autoparagraphing (first step) and then is converted (second step).
	 * Even though `<strong>` is over `<p>` element, `bold="true"` was added to the text. See
	 * {@link module:engine/conversion/upcasthelpers~UpcastHelpers#attributeToAttribute} for comparison.
	 *
	 * Keep in mind that the attribute will be set only if it is allowed by
	 * {@link module:engine/model/schema~ModelSchema schema} configuration.
	 *
	 * ```ts
	 * editor.conversion.for( 'upcast' ).elementToAttribute( {
	 * 	view: 'strong',
	 * 	model: 'bold'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).elementToAttribute( {
	 * 	view: 'strong',
	 * 	model: 'bold',
	 * 	converterPriority: 'high'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).elementToAttribute( {
	 * 	view: {
	 * 		name: 'span',
	 * 		classes: 'bold'
	 * 	},
	 * 	model: 'bold'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).elementToAttribute( {
	 * 	view: {
	 * 		name: 'span',
	 * 		classes: [ 'styled', 'styled-dark' ]
	 * 	},
	 * 	model: {
	 * 		key: 'styled',
	 * 		value: 'dark'
	 * 	}
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).elementToAttribute( {
	 * 	view: {
	 * 		name: 'span',
	 * 		styles: {
	 * 			'font-size': /[\s\S]+/
	 * 		}
	 * 	},
	 * 	model: {
	 * 		key: 'fontSize',
	 * 		value: ( viewElement, conversionApi ) => {
	 * 			const fontSize = viewElement.getStyle( 'font-size' );
	 * 			const value = fontSize.substr( 0, fontSize.length - 2 );
	 *
	 * 			if ( value <= 10 ) {
	 * 				return 'small';
	 * 			} else if ( value > 12 ) {
	 * 				return 'big';
	 * 			}
	 *
	 * 			return null;
	 * 		}
	 * 	}
	 * } );
	 * ```
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * @param config Conversion configuration.
	 * @param config.view Pattern matching all view elements which should be converted.
	 * @param config.model Model attribute key or an object with `key` and `value` properties, describing
	 * the model attribute. `value` property may be set as a function that takes a view element and
	 * {@link module:engine/conversion/upcastdispatcher~UpcastConversionApi upcast conversion API} and returns the value.
	 * If `String` is given, the model attribute value will be set to `true`.
	 * @param config.converterPriority Converter priority. Defaults to `low`.
	 */ elementToAttribute(config) {
        return this.add(upcastElementToAttribute(config));
    }
    /**
	 * View attribute to model attribute conversion helper.
	 *
	 * This conversion results in setting an attribute on a model node. For example, view `<img src="foo.jpg"></img>` becomes
	 * `<imageBlock source="foo.jpg"></imageBlock>` in the model.
	 *
	 * This helper is meant to convert view attributes from view elements which got converted to the model, so the view attribute
	 * is set only on the corresponding model node:
	 *
	 * ```
	 * <div class="dark"><div>foo</div></div>    -->    <div dark="true"><div>foo</div></div>
	 * ```
	 *
	 * Above, `class="dark"` attribute is added only to the `<div>` elements that has it. This is in contrast to
	 * {@link module:engine/conversion/upcasthelpers~UpcastHelpers#elementToAttribute} which sets attributes for
	 * all the children in the model:
	 *
	 * ```
	 * <strong>Foo</strong>   -->   <strong><p>Foo</p></strong>   -->   <paragraph><$text bold="true">Foo</$text></paragraph>
	 * ```
	 *
	 * Above is a sample of HTML code, that goes through autoparagraphing (first step) and then is converted (second step).
	 * Even though `<strong>` is over `<p>` element, `bold="true"` was added to the text.
	 *
	 * Keep in mind that the attribute will be set only if it is allowed by
	 * {@link module:engine/model/schema~ModelSchema schema} configuration.
	 *
	 * ```ts
	 * editor.conversion.for( 'upcast' ).attributeToAttribute( {
	 * 	view: 'src',
	 * 	model: 'source'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).attributeToAttribute( {
	 * 	view: { key: 'src' },
	 * 	model: 'source'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).attributeToAttribute( {
	 * 	view: { key: 'src' },
	 * 	model: 'source',
	 * 	converterPriority: 'normal'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).attributeToAttribute( {
	 * 	view: {
	 * 		key: 'data-style',
	 * 		value: /[\s\S]+/
	 * 	},
	 * 	model: 'styled'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).attributeToAttribute( {
	 * 	view: {
	 * 		name: 'img',
	 * 		key: 'class',
	 * 		value: 'styled-dark'
	 * 	},
	 * 	model: {
	 * 		key: 'styled',
	 * 		value: 'dark'
	 * 	}
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).attributeToAttribute( {
	 * 	view: {
	 * 		key: 'class',
	 * 		value: /styled-[\S]+/
	 * 	},
	 * 	model: {
	 * 		key: 'styled'
	 * 		value: ( viewElement, conversionApi ) => {
	 * 			const regexp = /styled-([\S]+)/;
	 * 			const match = viewElement.getAttribute( 'class' ).match( regexp );
	 *
	 * 			return match[ 1 ];
	 * 		}
	 * 	}
	 * } );
	 * ```
	 *
	 * Converting styles works a bit differently as it requires `view.styles` to be an object and by default
	 * a model attribute will be set to `true` by such a converter. You can set the model attribute to any value by providing the `value`
	 * callback that returns the desired value.
	 *
	 * ```ts
	 * // Default conversion of font-weight style will result in setting bold attribute to true.
	 * editor.conversion.for( 'upcast' ).attributeToAttribute( {
	 * 	view: {
	 * 		styles: {
	 * 			'font-weight': 'bold'
	 * 		}
	 * 	},
	 * 	model: 'bold'
	 * } );
	 *
	 * // This converter will pass any style value to the `lineHeight` model attribute.
	 * editor.conversion.for( 'upcast' ).attributeToAttribute( {
	 * 	view: {
	 * 		styles: {
	 * 			'line-height': /[\s\S]+/
	 * 		}
	 * 	},
	 * 	model: {
	 * 		key: 'lineHeight',
	 * 		value: ( viewElement, conversionApi ) => viewElement.getStyle( 'line-height' )
	 * 	}
	 * } );
	 * ```
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * @param config Conversion configuration.
	 * @param config.view Specifies which view attribute will be converted. If a `String` is passed,
	 * attributes with given key will be converted. If an `Object` is passed, it must have a required `key` property,
	 * specifying view attribute key, and may have an optional `value` property, specifying view attribute value and optional `name`
	 * property specifying a view element name from/on which the attribute should be converted. `value` can be given as a `String`,
	 * a `RegExp` or a function callback, that takes view attribute value as the only parameter and returns `Boolean`.
	 * @param config.model Model attribute key or an object with `key` and `value` properties, describing
	 * the model attribute. `value` property may be set as a function that takes a view element and
	 * {@link module:engine/conversion/upcastdispatcher~UpcastConversionApi upcast conversion API} and returns the value.
	 * If `String` is given, the model attribute value will be same as view attribute value.
	 * @param config.converterPriority Converter priority. Defaults to `low`.
	 */ attributeToAttribute(config) {
        return this.add(upcastAttributeToAttribute(config));
    }
    /**
	 * View element to model marker conversion helper.
	 *
	 * This conversion results in creating a model marker. For example, if the marker was stored in a view as an element:
	 * `<p>Fo<span data-marker="comment" data-comment-id="7"></span>o</p><p>B<span data-marker="comment" data-comment-id="7"></span>ar</p>`,
	 * after the conversion is done, the marker will be available in
	 * {@link module:engine/model/model~Model#markers model document markers}.
	 *
	 * **Note**: When this helper is used in the data upcast in combination with
	 * {@link module:engine/conversion/downcasthelpers~DowncastHelpers#markerToData `#markerToData()`} in the data downcast,
	 * then invalid HTML code (e.g. a span between table cells) may be produced by the latter converter.
	 *
	 * In most of the cases, the {@link #dataToMarker} should be used instead.
	 *
	 * ```ts
	 * editor.conversion.for( 'upcast' ).elementToMarker( {
	 * 	view: 'marker-search',
	 * 	model: 'search'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).elementToMarker( {
	 * 	view: 'marker-search',
	 * 	model: 'search',
	 * 	converterPriority: 'high'
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).elementToMarker( {
	 * 	view: 'marker-search',
	 * 	model: ( viewElement, conversionApi ) => 'comment:' + viewElement.getAttribute( 'data-comment-id' )
	 * } );
	 *
	 * editor.conversion.for( 'upcast' ).elementToMarker( {
	 * 	view: {
	 * 		name: 'span',
	 * 		attributes: {
	 * 			'data-marker': 'search'
	 * 		}
	 * 	},
	 * 	model: 'search'
	 * } );
	 * ```
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * @param config Conversion configuration.
	 * @param config.view Pattern matching all view elements which should be converted.
	 * @param config.model Name of the model marker, or a function that takes a view element and returns
	 * a model marker name.
	 * @param config.converterPriority Converter priority.
	 */ elementToMarker(config) {
        return this.add(upcastElementToMarker(config));
    }
    /**
	 * View-to-model marker conversion helper.
	 *
	 * Converts view data created by {@link module:engine/conversion/downcasthelpers~DowncastHelpers#markerToData `#markerToData()`}
	 * back to a model marker.
	 *
	 * This converter looks for specific view elements and view attributes that mark marker boundaries. See
	 * {@link module:engine/conversion/downcasthelpers~DowncastHelpers#markerToData `#markerToData()`} to learn what view data
	 * is expected by this converter.
	 *
	 * The `config.view` property is equal to the marker group name to convert.
	 *
	 * By default, this converter creates markers with the `group:name` name convention (to match the default `markerToData` conversion).
	 *
	 * The conversion configuration can take a function that will generate a marker name.
	 * If such function is set as the `config.model` parameter, it is passed the `name` part from the view element or attribute and it is
	 * expected to return a string with the marker name.
	 *
	 * Basic usage:
	 *
	 * ```ts
	 * // Using the default conversion.
	 * // In this case, all markers from the `comment` group will be converted.
	 * // The conversion will look for `<comment-start>` and `<comment-end>` tags and
	 * // `data-comment-start-before`, `data-comment-start-after`,
	 * // `data-comment-end-before` and `data-comment-end-after` attributes.
	 * editor.conversion.for( 'upcast' ).dataToMarker( {
	 * 	view: 'comment'
	 * } );
	 * ```
	 *
	 * An example of a model that may be generated by this conversion:
	 *
	 * ```
	 * // View:
	 * <p>Foo<comment-start name="commentId:uid"></comment-start>bar</p>
	 * <figure data-comment-end-after="commentId:uid" class="image"><img src="abc.jpg" /></figure>
	 *
	 * // Model:
	 * <paragraph>Foo[bar</paragraph>
	 * <imageBlock src="abc.jpg"></imageBlock>]
	 * ```
	 *
	 * Where `[]` are boundaries of a marker that will receive the `comment:commentId:uid` name.
	 *
	 * Other examples of usage:
	 *
	 * ```ts
	 * // Using a custom function which is the same as the default conversion:
	 * editor.conversion.for( 'upcast' ).dataToMarker( {
	 * 	view: 'comment',
	 * 	model: ( name, conversionApi ) => 'comment:' + name,
	 * } );
	 *
	 * // Using the converter priority:
	 * editor.conversion.for( 'upcast' ).dataToMarker( {
	 * 	view: 'comment',
	 * 	model: ( name, conversionApi ) => 'comment:' + name,
	 * 	converterPriority: 'high'
	 * } );
	 * ```
	 *
	 * See {@link module:engine/conversion/conversion~Conversion#for `conversion.for()`} to learn how to add a converter
	 * to the conversion process.
	 *
	 * @param config Conversion configuration.
	 * @param config.view The marker group name to convert.
	 * @param config.model A function that takes the `name` part from the view element or attribute and
	 * {@link module:engine/conversion/upcastdispatcher~UpcastConversionApi upcast conversion API} and returns the marker name.
	 * @param config.converterPriority Converter priority.
	 */ dataToMarker(config) {
        return this.add(upcastDataToMarker(config));
    }
}
/**
 * Function factory, creates a converter that converts
 * {@link module:engine/view/documentfragment~ViewDocumentFragment view document fragment}
 * or all children of {@link module:engine/view/element~ViewElement} into
 * {@link module:engine/model/documentfragment~ModelDocumentFragment model document fragment}.
 * This is the "entry-point" converter for upcast (view to model conversion). This converter starts the conversion of all children
 * of passed view document fragment. Those children {@link module:engine/view/node~ViewNode view nodes} are then
 * handled by other converters.
 *
 * This also a "default", last resort converter for all view elements that has not been converted by other converters.
 * When a view element is being converted to the model but it does not have converter specified, that view element
 * will be converted to {@link module:engine/model/documentfragment~ModelDocumentFragment model document fragment} and returned.
 *
 * @returns Universal converter for view {@link module:engine/view/documentfragment~ViewDocumentFragment fragments} and
 * {@link module:engine/view/element~ViewElement elements} that returns
 * {@link module:engine/model/documentfragment~ModelDocumentFragment model fragment} with children of converted view item.
 *
 * @internal
 */ function convertToModelFragment$1() {
    return (evt, data, conversionApi)=>{
        // Second argument in `consumable.consume` is discarded for ViewDocumentFragment but is needed for ViewElement.
        if (!data.modelRange && conversionApi.consumable.consume(data.viewItem, {
            name: true
        })) {
            const { modelRange, modelCursor } = conversionApi.convertChildren(data.viewItem, data.modelCursor);
            data.modelRange = modelRange;
            data.modelCursor = modelCursor;
        }
    };
}
/**
 * Function factory, creates a converter that converts
 * {@link module:engine/view/text~ViewText} to {@link module:engine/model/text~ModelText}.
 *
 * @returns {@link module:engine/view/text~ViewText View text} converter.
 * @internal
 */ function convertText() {
    return (evt, data, { schema, consumable, writer })=>{
        let position = data.modelCursor;
        // When node is already converted then do nothing.
        if (!consumable.test(data.viewItem)) {
            return;
        }
        if (!schema.checkChild(position, '$text')) {
            if (!isParagraphable(position, '$text', schema)) {
                return;
            }
            // Do not auto-paragraph whitespaces.
            if (data.viewItem.data.trim().length == 0) {
                return;
            }
            position = wrapInParagraph(position, writer);
        }
        consumable.consume(data.viewItem);
        const text = writer.createText(data.viewItem.data);
        writer.insert(text, position);
        data.modelRange = writer.createRange(position, position.getShiftedBy(text.offsetSize));
        data.modelCursor = data.modelRange.end;
    };
}
/**
 * Function factory, creates a callback function which converts a {@link module:engine/view/selection~ViewSelection
 * view selection} taken from the {@link module:engine/view/document~ViewDocument#event:selectionChange} event
 * and sets in on the {@link module:engine/model/document~ModelDocument#selection model}.
 *
 * **Note**: because there is no view selection change dispatcher nor any other advanced view selection to model
 * conversion mechanism, the callback should be set directly on view document.
 *
 * ```ts
 * view.document.on( 'selectionChange', convertSelectionChange( modelDocument, mapper ) );
 * ```
 *
 * @param model Data model.
 * @param mapper Conversion mapper.
 * @returns {@link module:engine/view/document~ViewDocument#event:selectionChange} callback function.
 * @internal
 */ function convertSelectionChange(model, mapper) {
    return (evt, data)=>{
        const viewSelection = data.newSelection;
        const ranges = [];
        for (const viewRange of viewSelection.getRanges()){
            ranges.push(mapper.toModelRange(viewRange));
        }
        const modelSelection = model.createSelection(ranges, {
            backward: viewSelection.isBackward
        });
        if (!modelSelection.isEqual(model.document.selection)) {
            model.change((writer)=>{
                writer.setSelection(modelSelection);
            });
        }
    };
}
/**
 * View element to model element conversion helper.
 *
 * See {@link ~UpcastHelpers#elementToElement `.elementToElement()` upcast helper} for examples.
 *
 * @param config Conversion configuration.
 * @param config.view Pattern matching all view elements which should be converted. If not
 * set, the converter will fire for every view element.
 * @param config.model Name of the model element, a model element
 * instance or a function that takes a view element and returns a model element. The model element will be inserted in the model.
 * @param config.converterPriority Converter priority.
 * @returns Conversion helper.
 */ function upcastElementToElement(config) {
    config = cloneDeep(config);
    const converter = prepareToElementConverter(config);
    const elementName = getViewElementNameFromConfig(config.view);
    const eventName = elementName ? `element:${elementName}` : 'element';
    return (dispatcher)=>{
        dispatcher.on(eventName, converter, {
            priority: config.converterPriority || 'normal'
        });
    };
}
/**
 * View element to model attribute conversion helper.
 *
 * See {@link ~UpcastHelpers#elementToAttribute `.elementToAttribute()` upcast helper} for examples.
 *
 * @param config Conversion configuration.
 * @param config.view Pattern matching all view elements which should be converted.
 * @param config.model Model attribute key or an object with `key` and `value` properties, describing
 * the model attribute. `value` property may be set as a function that takes a view element and returns the value.
 * If `String` is given, the model attribute value will be set to `true`.
 * @param config.converterPriority Converter priority. Defaults to `low`.
 * @returns Conversion helper.
 */ function upcastElementToAttribute(config) {
    config = cloneDeep(config);
    normalizeModelAttributeConfig(config);
    const converter = prepareToAttributeConverter(config, false);
    const elementName = getViewElementNameFromConfig(config.view);
    const eventName = elementName ? `element:${elementName}` : 'element';
    return (dispatcher)=>{
        dispatcher.on(eventName, converter, {
            priority: config.converterPriority || 'low'
        });
    };
}
/**
 * View attribute to model attribute conversion helper.
 *
 * See {@link ~UpcastHelpers#attributeToAttribute `.attributeToAttribute()` upcast helper} for examples.
 *
 * @param config Conversion configuration.
 * @param config.view Specifies which view attribute will be converted. If a `String` is passed,
 * attributes with given key will be converted. If an `Object` is passed, it must have a required `key` property,
 * specifying view attribute key, and may have an optional `value` property, specifying view attribute value and optional `name`
 * property specifying a view element name from/on which the attribute should be converted. `value` can be given as a `String`,
 * a `RegExp` or a function callback, that takes view attribute value as the only parameter and returns `Boolean`.
 * @param config.model Model attribute key or an object with `key` and `value` properties, describing
 * the model attribute. `value` property may be set as a function that takes a view element and returns the value.
 * If `String` is given, the model attribute value will be same as view attribute value.
 * @param config.converterPriority Converter priority. Defaults to `low`.
 * @returns Conversion helper.
 */ function upcastAttributeToAttribute(config) {
    config = cloneDeep(config);
    let viewKey = null;
    if (typeof config.view == 'string' || config.view.key) {
        viewKey = normalizeViewAttributeKeyValueConfig(config);
    }
    normalizeModelAttributeConfig(config, viewKey);
    const converter = prepareToAttributeConverter(config, true);
    return (dispatcher)=>{
        dispatcher.on('element', converter, {
            priority: config.converterPriority || 'low'
        });
    };
}
/**
 * View element to model marker conversion helper.
 *
 * See {@link ~UpcastHelpers#elementToMarker `.elementToMarker()` upcast helper} for examples.
 *
 * @param config Conversion configuration.
 * @param config.view Pattern matching all view elements which should be converted.
 * @param config.model Name of the model marker, or a function that takes a view element and returns
 * a model marker name.
 * @param config.converterPriority Converter priority.
 * @returns Conversion helper.
 */ function upcastElementToMarker(config) {
    const model = normalizeElementToMarkerModelConfig(config.model);
    return upcastElementToElement({
        ...config,
        model
    });
}
/**
 * View data to model marker conversion helper.
 *
 * See {@link ~UpcastHelpers#dataToMarker} to learn more.
 *
 * @returns Conversion helper.
 */ function upcastDataToMarker(config) {
    config = cloneDeep(config);
    // Default conversion.
    if (!config.model) {
        config.model = (name)=>{
            return name ? config.view + ':' + name : config.view;
        };
    }
    const normalizedConfig = {
        view: config.view,
        model: config.model
    };
    const converterStart = prepareToElementConverter(normalizeDataToMarkerConfig(normalizedConfig, 'start'));
    const converterEnd = prepareToElementConverter(normalizeDataToMarkerConfig(normalizedConfig, 'end'));
    return (dispatcher)=>{
        dispatcher.on(`element:${config.view}-start`, converterStart, {
            priority: config.converterPriority || 'normal'
        });
        dispatcher.on(`element:${config.view}-end`, converterEnd, {
            priority: config.converterPriority || 'normal'
        });
        // Below is a hack that is needed to properly handle `converterPriority` for both elements and attributes.
        // Attribute conversion needs to be performed *after* element conversion.
        // This converter handles both element conversion and attribute conversion, which means that if a single
        // `config.converterPriority` is used, it will lead to problems. For example, if the `'high'` priority is used,
        // the attribute conversion will be performed before a lot of element upcast converters.
        // On the other hand, we want to support `config.converterPriority` and converter overwriting.
        //
        // To make it work, we need to do some extra processing for priority for attribute converter.
        // Priority `'low'` value should be the base value and then we will change it depending on `config.converterPriority` value.
        //
        // This hack probably would not be needed if attributes are upcasted separately.
        //
        const basePriority = priorities.low;
        const maxPriority = priorities.highest;
        const priorityFactor = priorities.get(config.converterPriority) / maxPriority; // Number in range [ -1, 1 ].
        dispatcher.on('element', upcastAttributeToMarker(normalizedConfig), {
            priority: basePriority + priorityFactor
        });
    };
}
/**
 * Function factory, returns a callback function which converts view attributes to a model marker.
 *
 * The converter looks for elements with `data-group-start-before`, `data-group-start-after`, `data-group-end-before`
 * and `data-group-end-after` attributes and inserts `$marker` model elements before/after those elements.
 * `group` part is specified in `config.view`.
 *
 * @returns Marker converter.
 */ function upcastAttributeToMarker(config) {
    return (evt, data, conversionApi)=>{
        const attrName = `data-${config.view}`;
        // Check if any attribute for the given view item can be consumed before changing the conversion data
        // and consuming view items with these attributes.
        if (!conversionApi.consumable.test(data.viewItem, {
            attributes: attrName + '-end-after'
        }) && !conversionApi.consumable.test(data.viewItem, {
            attributes: attrName + '-start-after'
        }) && !conversionApi.consumable.test(data.viewItem, {
            attributes: attrName + '-end-before'
        }) && !conversionApi.consumable.test(data.viewItem, {
            attributes: attrName + '-start-before'
        })) {
            return;
        }
        // This converter wants to add a model element, marking a marker, before/after an element (or maybe even group of elements).
        // To do that, we can use `data.modelRange` which is set on an element (or a group of elements) that has been upcasted.
        // But, if the processed view element has not been upcasted yet (it does not have been converted), we need to
        // fire conversion for its children first, then we will have `data.modelRange` available.
        if (!data.modelRange) {
            Object.assign(data, conversionApi.convertChildren(data.viewItem, data.modelCursor));
        }
        if (conversionApi.consumable.consume(data.viewItem, {
            attributes: attrName + '-end-after'
        })) {
            addMarkerElements(data.modelRange.end, data.viewItem.getAttribute(attrName + '-end-after').split(','));
        }
        if (conversionApi.consumable.consume(data.viewItem, {
            attributes: attrName + '-start-after'
        })) {
            addMarkerElements(data.modelRange.end, data.viewItem.getAttribute(attrName + '-start-after').split(','));
        }
        if (conversionApi.consumable.consume(data.viewItem, {
            attributes: attrName + '-end-before'
        })) {
            addMarkerElements(data.modelRange.start, data.viewItem.getAttribute(attrName + '-end-before').split(','));
        }
        if (conversionApi.consumable.consume(data.viewItem, {
            attributes: attrName + '-start-before'
        })) {
            addMarkerElements(data.modelRange.start, data.viewItem.getAttribute(attrName + '-start-before').split(','));
        }
        function addMarkerElements(position, markerViewNames) {
            for (const markerViewName of markerViewNames){
                const markerName = config.model(markerViewName, conversionApi);
                const element = conversionApi.writer.createElement('$marker', {
                    'data-name': mark