/**
 * @license Copyright (c) 2003-2025, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-licensing-options
 */
/**
 * @module engine/view/observer/mutationobserver
 */
import { Observer } from './observer.js';
import { type ViewDomConverter } from '../domconverter.js';
import { type EditingView } from '../view.js';
import { type ViewNode } from '../node.js';
import type { ViewDocumentChangeType } from '../document.js';
/**
 * Mutation observer's role is to watch for any DOM changes inside the editor that weren't
 * done by the editor's {@link module:engine/view/renderer~ViewRenderer} itself and reverting these changes.
 *
 * It does this by observing all mutations in the DOM, marking related view elements as changed and calling
 * {@link module:engine/view/renderer~ViewRenderer#render}. Because all mutated nodes are marked as
 * "to be rendered" and the {@link module:engine/view/renderer~ViewRenderer#render `render()`} method is called,
 * all changes are reverted in the DOM (the DOM is synced with the editor's view structure).
 *
 * Note that this observer is attached by the {@link module:engine/view/view~EditingView} and is available by default.
 */
export declare class MutationObserver extends Observer {
    /**
     * Reference to the {@link module:engine/view/view~EditingView#domConverter}.
     */
    readonly domConverter: ViewDomConverter;
    /**
     * Native mutation observer config.
     */
    private readonly _config;
    /**
     * Observed DOM elements.
     */
    private readonly _domElements;
    /**
     * Native mutation observer.
     */
    private _mutationObserver;
    /**
     * @inheritDoc
     */
    constructor(view: EditingView);
    /**
     * Synchronously handles mutations and empties the queue.
     */
    flush(): void;
    /**
     * @inheritDoc
     */
    observe(domElement: HTMLElement): void;
    /**
     * @inheritDoc
     */
    stopObserving(domElement: HTMLElement): void;
    /**
     * @inheritDoc
     */
    enable(): void;
    /**
     * @inheritDoc
     */
    disable(): void;
    /**
     * @inheritDoc
     */
    destroy(): void;
    /**
     * Handles mutations. Mark view elements to sync and call render.
     *
     * @param domMutations Array of native mutations.
     */
    private _onMutations;
    /**
     * Checks if mutation was generated by the browser inserting bogus br on the end of the block element.
     * Such mutations are generated while pressing space or performing native spellchecker correction
     * on the end of the block element in Firefox browser.
     *
     * @param mutation Native mutation object.
     */
    private _isBogusBrMutation;
}
/**
 * Event fired on DOM mutations detected.
 *
 * This event is introduced by {@link module:engine/view/observer/mutationobserver~MutationObserver} and available
 * by default in all editor instances (attached by {@link module:engine/view/view~EditingView}).
 *
 * @eventName module:engine/view/document~ViewDocument#mutations
 * @param data Event data containing detailed information about the event.
 */
export type ViewDocumentMutationsEvent = {
    name: 'mutations';
    args: [data: ViewDocumentMutationEventData];
};
/**
 * The value of {@link ~ViewDocumentMutationsEvent}.
 */
export type ViewDocumentMutationEventData = {
    mutations: Array<ObserverMutationData>;
};
/**
 * A single entry in {@link ~ViewDocumentMutationEventData} mutations array.
 */
export type ObserverMutationData = {
    /**
     * Type of mutation detected.
     */
    type: ViewDocumentChangeType;
    /**
     * The view node related to the detected mutation.
     */
    node: ViewNode;
};
