/**
 * @license Copyright (c) 2003-2025, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-licensing-options
 */
/**
 * @module engine/view/observer/observer
 */
import { DomEmitterMixin } from '@ckeditor/ckeditor5-utils';
/**
 * Abstract base observer class. Observers are classes which listen to DOM events, do the preliminary
 * processing and fire events on the {@link module:engine/view/document~ViewDocument} objects.
 * Observers can also add features to the view, for instance by updating its status or marking elements
 * which need a refresh on DOM events.
 */
export class Observer extends /* #__PURE__ */ DomEmitterMixin() {
    /**
     * An instance of the view controller.
     */
    view;
    /**
     * A reference to the {@link module:engine/view/document~ViewDocument} object.
     */
    document;
    /**
     * The state of the observer. If it is disabled, no events will be fired.
     */
    _isEnabled = false;
    /**
     * Creates an instance of the observer.
     */
    constructor(view) {
        super();
        this.view = view;
        this.document = view.document;
    }
    /**
     * The state of the observer. If it is disabled, no events will be fired.
     */
    get isEnabled() {
        return this._isEnabled;
    }
    /**
     * Enables the observer. This method is called when the observer is registered to the
     * {@link module:engine/view/view~EditingView} and after {@link module:engine/view/view~EditingView#forceRender rendering}
     * (all observers are {@link #disable disabled} before rendering).
     *
     * A typical use case for disabling observers is that mutation observers need to be disabled for the rendering.
     * However, a child class may not need to be disabled, so it can implement an empty method.
     *
     * @see module:engine/view/observer/observer~Observer#disable
     */
    enable() {
        this._isEnabled = true;
    }
    /**
     * Disables the observer. This method is called before
     * {@link module:engine/view/view~EditingView#forceRender rendering} to prevent firing events during rendering.
     *
     * @see module:engine/view/observer/observer~Observer#enable
     */
    disable() {
        this._isEnabled = false;
    }
    /**
     * Disables and destroys the observer, among others removes event listeners created by the observer.
     */
    destroy() {
        this.disable();
        this.stopListening();
    }
    /**
     * Checks whether a given DOM event should be ignored (should not be turned into a synthetic view document event).
     *
     * Currently, an event will be ignored only if its target or any of its ancestors has the `data-cke-ignore-events` attribute.
     * This attribute can be used inside the structures generated by
     * {@link module:engine/view/downcastwriter~ViewDowncastWriter#createUIElement `ViewDowncastWriter#createUIElement()`} to ignore events
     * fired within a UI that should be excluded from CKEditor 5's realms.
     *
     * @param domTarget The DOM event target to check (usually an element, sometimes a text node and
     * potentially sometimes a document, too).
     * @returns Whether this event should be ignored by the observer.
     */
    checkShouldIgnoreEventFromTarget(domTarget) {
        if (domTarget && domTarget.nodeType === 3) {
            domTarget = domTarget.parentNode;
        }
        if (!domTarget || domTarget.nodeType !== 1) {
            return false;
        }
        return domTarget.matches('[data-cke-ignore-events], [data-cke-ignore-events] *');
    }
}
