/**
 * @license Copyright (c) 2003-2025, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-licensing-options
 */
/**
 * @module list/listformatting/listitemitalicintegration
 */
import { Plugin } from 'ckeditor5/src/core.js';
import { env } from 'ckeditor5/src/utils.js';
import { ListEditing } from '../list/listediting.js';
/**
 * The list item italic integration plugin.
 */
export class ListItemItalicIntegration extends Plugin {
    /**
     * @inheritDoc
     */
    static get pluginName() {
        return 'ListItemItalicIntegration';
    }
    /**
     * @inheritDoc
     */
    static get isOfficialPlugin() {
        return true;
    }
    /**
     * @inheritDoc
     */
    static get requires() {
        return [ListEditing];
    }
    /**
     * @inheritDoc
     */
    init() {
        const editor = this.editor;
        const ListFormatting = editor.plugins.get('ListFormatting');
        const listEditing = editor.plugins.get(ListEditing);
        if (!editor.plugins.has('ItalicEditing') || !this.editor.config.get('list.enableListItemMarkerFormatting')) {
            return;
        }
        ListFormatting.registerFormatAttribute('italic', 'listItemItalic');
        // Register the downcast strategy in init() so that the attribute name is registered before the list editing
        // registers its converters.
        // This ensures that the attribute is recognized by downcast strategies and bogus paragraphs are handled correctly.
        listEditing.registerDowncastStrategy({
            scope: 'item',
            attributeName: 'listItemItalic',
            setAttributeOnDowncast(writer, value, viewElement, options) {
                if (value) {
                    writer.addClass('ck-list-marker-italic', viewElement);
                    // See: https://github.com/ckeditor/ckeditor5/issues/18790.
                    if (env.isSafari && !(options && options.dataPipeline)) {
                        writer.setStyle('--ck-content-list-marker-dummy-italic', '0', viewElement);
                    }
                }
            }
        });
    }
    /**
     * @inheritDoc
     */
    afterInit() {
        const editor = this.editor;
        const model = editor.model;
        if (!editor.plugins.has('ItalicEditing') || !this.editor.config.get('list.enableListItemMarkerFormatting')) {
            return;
        }
        model.schema.extend('$listItem', { allowAttributes: 'listItemItalic' });
        model.schema.setAttributeProperties('listItemItalic', {
            isFormatting: true
        });
        model.schema.addAttributeCheck(context => {
            const item = context.last;
            if (!item.getAttribute('listItemId')) {
                return false;
            }
        }, 'listItemItalic');
        editor.conversion.for('upcast').attributeToAttribute({
            model: 'listItemItalic',
            view: {
                name: 'li',
                classes: 'ck-list-marker-italic'
            }
        });
    }
}
